# vim:set fileencoding=utf-8 et ts=4 sts=4 sw=4:
#
#   apt-listchanges - Show changelog entries between the installed versions
#                     of a set of packages and the versions contained in
#                     corresponding .deb files
#
#   Copyright (C) 2000-2006  Matt Zimmerman  <mdz@debian.org>
#   Copyright (C) 2006       Pierre Habouzit <madcoder@debian.org>
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public
#   License along with this program; if not, write to the Free
#   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
#   MA 02111-1307 USA
#

import sys
import os
import os.path
import re
import locale
import email.Message
import email.Header
import email.Charset
import cStringIO
import tempfile
from ALChacks import *

# TODO:
# newt-like frontend, or maybe some GUI bit
# keep track of tar/dpkg-deb errors like in pre-2.0

def _parse_apt_bool(value):
    # Based on StringToBool() from apt-pkg/contrib/strutl.cc in apt source
    # and should return same result as StringToBool(value, false)
    return value.lower() in [ '1', 'yes', 'true', 'with', 'on', 'enable' ]

def _parse_apt_int(value):
    # This function should match Configuration::FindI() from apt's
    # apt-pkg/contrib/configuration.cc, except for values like '1something'
    try:
        return int(value)
    except:
        return 0

def read_apt_pipeline(config):
    if config.debug:
        sys.stderr.write("APT pipeline messages:\n")
    version = sys.stdin.readline().rstrip()
    if config.debug:
        sys.stderr.write("\t%s\n" % version)
    if version != "VERSION 2":
        sys.stderr.write(_('''Wrong or missing VERSION from apt pipeline
(is Dpkg::Tools::Options::/usr/bin/apt-listchanges::Version set to 2?)
'''))
        sys.exit(1)

    while True:
        line = sys.stdin.readline().rstrip()
        if config.debug:
            sys.stderr.write("\t%s\n" % line)
        if not line:
            break

        if (not config.ignore_apt_assume and
              line.startswith('APT::Get::Assume-Yes=') and
              _parse_apt_bool(line[len('APT::Get::Assume-Yes='):]) ):
            config.confirm = False
            # Set config.quiet as well to force non-interactive frontend
            config.quiet = max(1, config.quiet)
        elif line.startswith('quiet='):
            config.quiet = max(_parse_apt_int(line[len('quiet='):]), config.quiet)

    filenames = {}
    toconfig = []
    toremove = []
    hasupgrade = False

    for pkgline in sys.stdin.readlines():
        pkgline = pkgline.rstrip()
        if config.debug:
            sys.stderr.write("\t%s\n" % pkgline)
        if not pkgline:
            break

        (pkgname, oldversion, compare, newversion, filename) = pkgline.split()
        if compare != '<': # ignore downgrades or re-installations
            continue

        if filename == '**REMOVE**' or filename == '**ERROR**':
            toremove.append(pkgname)
            continue

        # New installs (oldversion equal to '-') are not ignored to support
        # a case when changelog is moved from one package to a dependent
        # package built from the same source (see p7zip-full 15.09+dfsg-3).
        if oldversion != '-':
            hasupgrade = True

        if filename == '**CONFIGURE**':
            toconfig.append(pkgname)
        else:
            filenames[pkgname] = filename

    # Quit early if no package has been upgraded (e.g. only new installs or removals)
    if not hasupgrade:
        return []

    # Sort by configuration order.  THIS IS IMPORTANT.  Sometimes, a
    # situation exists where package X contains changelog.gz (upstream
    # changelog) and depends on package Y which contains
    # changelog.Debian.gz (Debian changelog).  Until we have a more
    # reliable method for determining whether a package is Debian
    # native, this allows things to work, since Y will always be
    # configured first.

    # apt doesn't explicitly configure everything anymore, so sort
    # the things to be configured first, and then do everything else
    # in alphabetical order.  Also, drop from the list everything
    # that's to be removed.
    for pkg in toremove:
        if pkg in filenames:
            del filenames[pkg]

    ordered_filenames = []
    for pkg in toconfig:
        if pkg in filenames:
            ordered_filenames.append(filenames[pkg])
            del filenames[pkg]

    ordered_filenames.extend(sorted(filenames.values()))
    return ordered_filenames

def confirm_or_exit(config, frontend):
    if not config.confirm:
        return
    try:
        if not frontend.confirm():
            sys.stderr.write(_('Aborting')+'.\n')
            sys.exit(10)
    except KeyboardInterrupt:
        sys.exit(10)
    except Exception as e:
        sys.stderr.write(_("Confirmation failed: %s") % str(e) +'\n')

def mail_changes(address, email_format, changes, subject):
    print "apt-listchanges: " + _("Mailing %s: %s") % (address, subject)

    charset = email.Charset.Charset('utf-8')
    charset.body_encoding = '8bit'
    charset.header_encoding = email.Charset.QP
    message = email.Message.Message()
    if email_format == 'html':
        changes = html().convert_to_html(subject, changes)
        message['Content-Type'] = 'text/html; charset=utf-8'
    subject = unicode(subject.decode(locale.getpreferredencoding() or 'ascii', 'replace'))
    message['Auto-Submitted'] = 'auto-generated'
    message['Subject'] = email.Header.Header(subject, 'utf-8')
    message['To'] = address
    message.set_payload(changes, charset)

    fh = os.popen('/usr/sbin/sendmail -oi -t', 'w')
    fh.write(message.as_string())
    fh.close()


''' Check if the mail frontend is usable. When the second parameter is given
    print an appropriate error message '''
def can_send_emails(config, replacementFrontend = None):
    if not os.path.exists("/usr/sbin/sendmail"):
        if replacementFrontend:
            sys.stderr.write((_("The mail frontend needs a installed 'sendmail', using %s")
                                + '\n') % replacementFrontend)
        return False

    if not config.email_address:
        if replacementFrontend:
            sys.stderr.write((_("The mail frontend needs an e-mail address to be configured, using %s")
                                + '\n') % replacementFrontend)
        return False

    return True

''' Exception class to notify callers of make_frontend() that invalid frontend
    was given in the configuration'''
class EUnknownFrontend(Exception):
    pass

def _select_frontend(config, frontends):
    ''' Utility function used for testing purposes '''
    prompt = "\n" + _("Available apt-listchanges frontends:") + "\n" + \
             "".join(["  %d. %s\n"%(i+1,frontends[i]) for i in range(0, len(frontends))]) + \
             _("Choose a frontend by entering its number: ")

    for i in (1,2,3):
        try:
            response = ttyconfirm().ttyask(prompt).strip()
            if not response:
                break
            return frontends[int(response)-1]
        except Exception as ex:
            sys.stderr.write(_("Error: ") + str(ex) + "\n")

    print(_("Using default frontend: %s") % config.frontend)
    return config.frontend

def make_frontend(name, packages, config):
    frontends = { 'text' : text,
                  'pager' : pager,
                  'debconf': debconf,
                  'mail' : mail,
                  'browser' : browser,
                  'xterm-pager' : xterm_pager,
                  'xterm-browser' : xterm_browser,
                  'gtk' : None, # handled below
                  'none' : None }

    if config.select_frontend: # For testing purposes
        name = _select_frontend(config, sorted(list(frontends.keys())))
    else:
        name = config.frontend

    if name == 'none':
        return None

    # If user does not want any messages force either the mail frontend
    # or no frontend at all if mail is not usable
    if config.quiet >= 2:
        if can_send_emails(config):
            name = 'mail'
        else:
            return None

    # If apt is in quiet (loggable) mode, we should make our output
    # loggable too unless the mail frontend is used (see #788059)
    elif config.quiet == 1:
        if name != 'mail' or not can_send_emails(config, 'text'):
            name = 'text'

    # Non-quiet mode
    else:
        if name in ('newt', 'w3m', 'xterm-w3m'):
            sys.stderr.write((_("The %s frontend is deprecated, using pager") + '\n') % name)
            name = 'pager'

        if name == "mail" and not can_send_emails(config, 'pager'):
            name = 'pager'

        # TODO: it would probably be nice to have a frontends subdir and
        # import from that. that would mean a uniform mechanism for all
        # frontends (that would become small files inside
        if name == "gtk":
            if os.environ.has_key("DISPLAY"):
                try:
                    gtk = __import__("AptListChangesGtk")
                    frontends[name] = gtk.gtk2
                except ImportError, e:
                    sys.stderr.write(_("The gtk frontend needs a working python-gtk2 "
                                       "and python-glade2.\n"
                                       "Those imports can not be found. Falling back "
                                       "to pager.\n"
                                       "The error is: %s\n") % e)
                    name = 'pager'
            else:
                name = 'pager'

    config.frontend = name
    if not frontends.has_key(name):
        raise EUnknownFrontend
    return frontends[name](packages, config)

class frontend:
    def __init__(self, packages, config):
        self.packages = packages
        self.config = config

    def update_progress(self):
        pass

    def progress_done(self):
        pass

    def display_output(self, text):
        pass

    def _render(self, text):
        newtext = []
        for line in text.split('\n'):
            try:
                # changelogs are supposed to be in UTF-8
                uline = line.decode('utf-8')
            except UnicodeError:
                # ... but handle gracefully if they aren't.
                # (That's also the reason we do it line by line.)
                # This is possibly wrong, but our best guess.
                uline = line.decode('iso8859-1')
            newtext.append(uline.encode(locale.getpreferredencoding() or 'ascii', 'replace'))
        return '\n'.join(newtext)

    def confirm(self):
        return 1

    def set_title(self, text):
        pass

class debconf(frontend):
    def set_title(self, text):
        self.title = text

    def display_output(self, text):
        import socket
        import debconf as dc
        if 'DEBIAN_FRONTEND' not in os.environ or os.environ['DEBIAN_FRONTEND'] != 'passthrough':
            return
        sock = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM, 0)
        sock.connect(os.environ['DEBCONF_PIPE'])
        dcfd = sock.makefile()
        sock.close()
        db = dc.Debconf(read=dcfd, write=dcfd)
        tmp = tempfile.NamedTemporaryFile(prefix="apt-listchanges-tmp")
        os.fchmod(tmp.fileno(), 0644)
        tmp.write('''Template: apt-listchanges/info
Type: title
Description: NEWS

Template: apt-listchanges/title
Type: title
Description: ${title}

Template: apt-listchanges/news
Type: note
Description: ${packages} packages\n''')
        for line in text.split('\n'):
            if line.strip():
                tmp.write('  ' + line + '\n')
            else:
                tmp.write(' .\n')
        tmp.flush()
        db.command('x_loadtemplatefile', tmp.name)
        tmp.close()
        db.info('apt-listchanges/info')
        db.subst('apt-listchanges/title', 'title', self.title)
        db.subst('apt-listchanges/news', 'packages', self.packages)
        db.settitle('apt-listchanges/title')
        db.fset('apt-listchanges/news', 'seen', 'false')
        db.input('high', 'apt-listchanges/news')
        db.go()
        dcfd.close()

class ttyconfirm:
    def ttyask(self, prompt):
        tty = open('/dev/tty', 'r+')
        tty.write(prompt)
        tty.flush()
        return tty.readline()

    def confirm(self):
        response = self.ttyask('apt-listchanges: ' + _('Do you want to continue? [Y/n] '))
        return response == '\n' or re.search(locale.nl_langinfo(locale.YESEXPR),
                                             response)

class simpleprogress:
    def update_progress(self):
        if self.config.quiet > 1:
            return

        if not hasattr(self, 'message_printed'):
            self.message_printed = 1
            sys.stdout.write(_("Reading changelogs") + "...\n")

    def progress_done(self):
        pass

class mail(simpleprogress, frontend):
    pass

class text(simpleprogress, ttyconfirm, frontend):
    def display_output(self, text):
        sys.stdout.write(text)

class fancyprogress:
    def update_progress(self):
        if not hasattr(self, 'progress'):
            # First call
            self.progress = 0
            self.line_length = 0

        self.progress += 1
        line = _("Reading changelogs") + "... %d%%" % (self.progress * 100 / self.packages)
        self.line_length = len(line)
        sys.stdout.write(line + '\r')
        sys.stdout.flush()

    def progress_done(self):
        if hasattr(self, 'line_length'):
            sys.stdout.write(' ' * self.line_length + '\r')
            sys.stdout.write(_("Reading changelogs") + "... " + _("Done") + "\n")
            sys.stdout.flush()

class runcommand:
    mode = os.P_WAIT
    suffix = ''

    def display_output(self, text):
        if self.mode == os.P_NOWAIT:
            if os.fork() != 0:
                return

        tmp = tempfile.NamedTemporaryFile(prefix="apt-listchanges-tmp", suffix=self.suffix)
        tmp.write(self._render(text))
        tmp.flush()
        shellcommand = self.get_command() + ' ' + tmp.name

        status = os.spawnl(os.P_WAIT, '/bin/sh', 'sh', '-c', shellcommand)
        if status != 0:
            raise OSError('Subprocess ' + shellcommand + ' exited with status ' + str(status))

        if self.mode == os.P_NOWAIT:
            # We are a child; exit
            sys.exit(0)

    def get_command(self):
        return self.command

class pager(runcommand, ttyconfirm, fancyprogress, frontend):
    def __init__(self, *args):
        if not 'LESS' in os.environ:
            os.environ['LESS'] = "-P?e(q to quit)"
        apply(frontend.__init__, [self] + list(args))
        self.command = self.config.get('pager', 'sensible-pager')

class xterm(runcommand, ttyconfirm, fancyprogress, frontend):
    def __init__(self, *args):
        apply(frontend.__init__, [self] + list(args))
        self.mode = os.P_NOWAIT
        self.xterm = self.config.get('xterm', 'x-terminal-emulator')

    def get_command(self):
        return self.xterm + ' -T apt-listchanges -e ' + self.xterm_command

class xterm_pager(xterm):
    def __init__(self, *args):
        apply(xterm.__init__, [self] + list(args))
        self.xterm_command = self.config.get('pager', 'sensible-pager')

class html:
    suffix = '.html'

    # LP bug-closing format requires the colon after "LP", but many people
    # say "LP #123456" when talking informally about bugs.
    lp_bug_stanza_re = re.compile(r'(?:lp:?\s+\#\d+(?:,\s*\#\d+)*)', re.I)
    lp_bug_re        = re.compile('(?P<linktext>#(?P<bugnum>\d+))', re.I)
    lp_bug_fmt       = r'<a href="https://launchpad.net/bugs/\g<bugnum>">\g<linktext></a>'
    bug_stanza_re = re.compile(r'(?:closes:\s*(?:bug)?\#?\s?\d+(?:,\s*(?:bug)?\#?\s?\d+)*|(?<!">)#\d+)', re.I)
    bug_re        = re.compile('(?P<linktext>#?(?P<bugnum>\d+))', re.I)
    bug_fmt       = r'<a href="https://bugs.debian.org/\g<bugnum>">\g<linktext></a>'

    cve_re        = re.compile(r'\bC(VE|AN)-(19|20|21)\d\d-\d{4,7}\b')
    cve_fmt       = r'<a href="https://cve.mitre.org/cgi-bin/cvename.cgi?name=\g<0>">\g<0></a>'
    # regxlib.com
    email_re = re.compile(r'([a-zA-Z0-9_\-\.]+)@(([[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.)|(([a-zA-Z0-9\-]+\.)+))([a-zA-Z]{2,4}|[0-9]{1,3})(\]?)')
    email_fmt = r'<a href="mailto:\g<0>">\g<0></a>'
    url_re = re.compile(r'(ht|f)tps?\://[a-zA-Z0-9\-\.]+\.[a-zA-Z]{2,3}(:[a-zA-Z0-9]*)?/?([a-zA-Z0-9\-\._\?\,\'/\\\+&amp;%\$#\=~])*')
    url_fmt = r'<a href="\g<0>">\g<0></a>'

    title = '''apt-listchanges output'''

    def convert_to_html(self, title, text):
        self.title = title
        return self._render(text)

    def _render(self, text):
        htmltext = cStringIO.StringIO()
        htmltext.write('''<html>
        <head>
        <meta http-equiv="Content-Type" content="text/html; charset=UTF-8" />
        <title>''')
        htmltext.write(self.title)
        htmltext.write('''</title>
        </head>

        <body>
        <pre>''')

        for line in text.split('\n'):
            try:
                # changelogs are supposed to be in UTF-8
                uline = line.decode('utf-8')
            except UnicodeError:
                # ... but handle gracefully if they aren't.
                # This is possibly wrong, but our best guess.
                uline = line.decode('iso8859-1')
            line = uline.encode('utf-8').replace(
                '&', '&amp;').replace(
                '<', '&lt;').replace(
                '>', '&gt;')
            line = self.url_re.sub(self.url_fmt, line)
            line = self.lp_bug_stanza_re.sub(lambda m: self.lp_bug_re.sub(self.lp_bug_fmt, m.group(0)), line)
            line = self.bug_stanza_re.sub(lambda m: self.bug_re.sub(self.bug_fmt, m.group(0)), line)
            line = self.cve_re.sub(self.cve_fmt, line)
            line = self.email_re.sub(self.email_fmt, line)
            htmltext.write(line + '\n')
        htmltext.write('</pre></body></html>')

        return htmltext.getvalue()

class browser(html, pager):
    def __init__(self, *args):
        apply(pager.__init__, [self] + list(args))
        self.command = self.config.get('browser', 'sensible-browser')
    def set_title(self, text):
        self.title = text

class xterm_browser(html, xterm):
    def __init__(self, *args):
        apply(xterm.__init__, [self] + list(args))
        self.xterm_command = self.config.get('browser', 'sensible-browser')

