------------------------------------------------------------------------------
--                                                                          --
--                          GNATCHECK COMPONENTS                            --
--                                                                          --
--                     G N A T S Y N C . T H R E A D S                      --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                       Copyright (C) 2008, AdaCore                        --
--                                                                          --
-- GNATSYNC  is  free  software;  you  can redistribute it and/or modify it --
-- under terms of the  GNU  General Public License as published by the Free --
-- Software Foundation;  either version 2, or ( at your option)  any  later --
-- version.  GNATCHECK  is  distributed in the hope that it will be useful, --
-- but  WITHOUT  ANY  WARRANTY;   without  even  the  implied  warranty  of --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General --
-- Public License for more details.  You should have received a copy of the --
-- GNU  General Public License distributed with GNAT; see file  COPYING. If --
-- not,  write to the  Free Software Foundation,  51 Franklin Street, Fifth --
-- Floor, Boston, MA 02110-1301, USA.                                       --
--                                                                          --
-- GNATSYNC is maintained by AdaCore (http://www.adacore.com).              --
--                                                                          --
------------------------------------------------------------------------------

--  Defines the routines for processing the foreign threads

with Asis; use Asis;

package Gnatsync.Threads is

   type Thread_Info_Kinds is
     (Not_A_Thread_Info,
      Thread,
      Section_Start,
      Section_End);
   --  Classification for thread info items

   type Thread_Info_Id is  new Integer range 0 .. Integer'Last;
   --  Index of the nodes representing information items describing the threads

   No_Thread_Info    : constant Thread_Info_Id := Thread_Info_Id'First;
   --  Id of nonexistend info item.

   function Present (Id : Thread_Info_Id) return Boolean;
   --  Checks if the argument points to the thread info entry.

   function No (Id : Thread_Info_Id) return Boolean;
   --  Checks if the argument does not point to the thread info entry.

   function Thread_Info_Kind (Id : Thread_Info_Id) return Thread_Info_Kinds;
   --  Returns the info kind of the corresponding info item. Returns
   --  Not_A_Thread_Info if No (Id).

   procedure Store_Threads_Info (Thread_File_Name : String);
   --  Stores the names of the procedures that should be considered as foreign
   --  threads (and, therefore, entry points in the call graph) from the text
   --  file which name is provided as an actual parameter.
   --  The caller is responsible for making sure that the actual parameter
   --  is a name of some existing file.

   function Is_Foreign_Thread (El : Asis.Element) return Boolean;
   --  Checks if El represents a subprogram that is a foreign thread.

   function Get_Section_Border_Id (El : Asis.Element) return Thread_Info_Id;
   --  Assuming that El is a procedure declaration, checks if this procedure
   --  is defined as a start or end procedure for a foreign critical section,
   --  and if it is, returns the corresponding Id.

   function Closes_Section
     (Started_By   : Thread_Info_Id;
      Closing_Item : Thread_Info_Id)
      return         Boolean;
   --  Provided that Started_By represents a Section_Start info item, and
   --  Closing_Item represents a Section_End item, check if Closing_Item is a
   --  closing procedure for a critical section started by Started_By

   function Foring_Threads_Specified return Boolean;
   --  Indicates if there are foreign threads specified

   function Foreign_Critical_Sections_Specified return Boolean;
   --  Indicates if there are foreign critical sections specified

   procedure Print_Threads_Debug_Info;
   --  Prints into Stderr the debug image of the data structures containing
   --  information about foreign threads.

end Gnatsync.Threads;
