/*-
 * Copyright (c) 2008, 2011  Peter Pentchev
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Ringlet$
 */

#define _BSD_SOURCE

#include <ctype.h>
#include <err.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "confget.h"
#include "confget_common.h"
#include "confget_ini.h"

static int		 readfile_ini(void);

confget_backend	 confget_ini_backend = {
	"ini",
	common_openfile,
	readfile_ini,
	common_closefile,
};

#define RF_SKIPSPACE	while (*p != '\0' && isspace(*p)) p++;
#define RF_SKIPNONSPACE	while (*p != '\0' && *p != '=' && !isspace(*p)) p++;

/***
 * Function:
 *	readfile_ini		- scan an INI file for the requested variable
 * Inputs:
 *	None.
 * Returns:
 *	0 on success, -1 on error.
 * Modifies:
 *	May write to standard output if the variable has been found.
 */
static int
readfile_ini(void)
{
	size_t len, vlen, plen;
	char *line, *grp, *p, *start, *end, *val, *vname, *vvalue;
	int ingroup, incont, seenvars;

	line = grp = NULL;
	ingroup = incont = seenvars = 0;
	vname = vvalue = NULL;
	len = 0;
	if (section == NULL)
		ingroup = 1;
	while (confgetline(conffile, &line, &len) != NULL) {
		p = line;
		if (incont) {
			if (vvalue == NULL)
				errx(2, "INTERNAL ERROR: null vvalue incont");
			vlen = strlen(vvalue);
			plen = strlen(p);
			val = realloc(vvalue, vlen + plen + 1);
			if (val == NULL) {
				warnx("Out of memory for the variable value");
				free(line);
				return (-1);
			}
			vvalue = val;
			memcpy(vvalue + vlen, p, plen);
			vvalue[vlen + plen] = '\0';

			vlen += plen;
			if (vlen > 0 && vvalue[vlen - 1] == '\\') {
				vvalue[vlen - 1] = '\0';
				continue;
			}
			incont = 0;
			while (vlen > 0 && isspace(vvalue[vlen - 1]))
				vvalue[--vlen] = '\0';

			if (ingroup) {
				seenvars = 1;
				if (foundvar(grp, vname, vvalue) == -1) {
					free(line);
					return (-1);
				}
				free(vname);
				free(vvalue);
				if (!morevars) {
					free(grp);
					free(line);
					return (0);
				} else {
					vname = vvalue = NULL;
				}
			}
			continue;
		}
		RF_SKIPSPACE;
		/* Comment? */
		if (*p == '#' || *p == ';' || *p == '\0')
			continue;
		/* Section? */
		if (*p == '[') {
			if (grp != NULL) {
				free(grp);
				grp = NULL;
			}
			p++;
			RF_SKIPSPACE;
			if (*p == '\0') {
				warnx("Invalid group line: %s", line);
				free(line);
				return (-1);
			}
			start = p++;
			end = p;
			while (*p != '\0' && *p != ']') {
				if (!isspace(*p))
					end = p + 1;
				p++;
			}
			if (*p != ']') {
				warnx("Invalid group line: %s", line);
				free(line);
				return (-1);
			}
			p++;
			RF_SKIPSPACE;
			if (*p != '\0') {
				warnx("Invalid group line: %s", line);
				free(line);
				return (-1);
			}

			*end = '\0';
			if (section == NULL) {
				if (seenvars) {
					free(line);
					return (0);
				}
				section = strdup(start);
				if (section == NULL) {
					warnx("Out of memory for the "
					    "group name");
					free(line);
					return (-1);
				}
			}

			val = strdup(start);
			if (val == NULL) {
				warnx("Out of memory for the group name");
				free(line);
				return (-1);
			}
			grp = val;

			ingroup = !strcmp(grp, section);
		} else {
			/* Variable! */
			if (vname != NULL) {
				free(vname);
				vname = NULL;
			}
			if (vvalue != NULL) {
				free(vvalue);
				vvalue = NULL;
			}
			start = p++;
			RF_SKIPNONSPACE;
			if (*p == '\0') {
				warnx("No value on line: %s", line);
				free(line);
				return (-1);
			}
			end = p;
			RF_SKIPSPACE;
			if (*p != '=') {
				warnx("No value on line: %s", line);
				free(line);
				return (-1);
			}
			p++;
			RF_SKIPSPACE;

			*end = '\0';
			val = strdup(start);
			if (val == NULL) {
				warnx("Out of memory for the variable name");
				free(line);
				return (-1);
			}
			vname = val;
			val = strdup(p);
			if (val == NULL) {
				warnx("Out of memory for the variable value");
				free(line);
				return (-1);
			}
			vvalue = val;

			vlen = strlen(vvalue);
			if (vlen > 0 && vvalue[vlen - 1] == '\\') {
				vvalue[vlen - 1] = '\0';
				incont = 1;
				continue;
			}
			while (vlen > 0 && isspace(vvalue[vlen - 1]))
				vvalue[--vlen] = '\0';

			if (ingroup) {
				seenvars = 1;
				if (foundvar(grp, vname, vvalue) == -1) {
					free(line);
					return (-1);
				}
				free(vname);
				free(vvalue);
				if (!morevars) {
					free(grp);
					free(line);
					return (0);
				} else {
					vname = vvalue = NULL;
				}
			}
		}
	}
	free(grp);
	free(vname);
	free(vvalue);
	free(line);
	if (!feof(conffile)) {
		warnx("Error reading input file %s", filename);
		return (-1);
	}
	if (incont) {
		warnx("Invalid input file - continuation at the last line");
		return (-1);
	}
	return (0);
}
