{-# OPTIONS_GHC -O0 -fno-case-merge -fno-strictness -fno-cse #-}
{-# LANGUAGE OverloadedStrings           #-}

module Text.XmlHtml.HTML.Meta
  ( voidTags
  , rawTextTags
  , endOmittableLast
  , endOmittableNext
  , predefinedRefs
  ) where

import           Data.Monoid
import           Data.HashMap.Strict (HashMap)
import qualified Data.HashMap.Strict as M
import           Data.HashSet (HashSet)
import qualified Data.HashSet as S
import           Data.Text (Text)

------------------------------------------------------------------------------
-- Metadata used for HTML5 quirks mode.                                     --
------------------------------------------------------------------------------

------------------------------------------------------------------------------
-- | Void elements as defined by the HTML5 spec.
{-# NOINLINE voidTags #-}
voidTags :: HashSet Text
voidTags = S.fromList [
    "area", "base", "br", "col", "command", "embed", "hr", "img", "input",
    "keygen", "link", "meta", "param", "source", "track", "wbr"
    ]

------------------------------------------------------------------------------
-- | Elements that XmlHtml treats as raw text.  Raw text elements are not
-- allowed to have any other tags in them.  This is necessary to support the
-- Javascript less than operator inside a script tag, for example.
{-# NOINLINE rawTextTags #-}
rawTextTags :: HashSet Text
rawTextTags = S.fromList [ "script", "style" ]

------------------------------------------------------------------------------
-- | List of elements with omittable end tags.
{-# NOINLINE endOmittableLast #-}
endOmittableLast :: HashSet Text
endOmittableLast = S.fromList [

    -- Tags which can be implicitly ended in case they are the last element in
    -- their parent.  This list actually includes all of the elements that
    -- have any kind of omittable end tag, since in general when an element
    -- with an omittable end tag isn't specified to be omittable in this way,
    -- it's just because in a complete document it isn't expected to ever be
    -- the last thing in its parent.  We aren't interested in enforcing
    -- element structure rules, so we'll allow it anyway.

    "body", "colgroup", "dd", "dt", "head", "html", "li", "optgroup",
    "option", "p", "rp", "rt", "tbody", "td", "tfoot", "th", "thead", "tr"
    ]

------------------------------------------------------------------------------
-- | Tags which should be considered automatically ended in case one of a
-- certain set of tags pops up.
{-# NOINLINE endOmittableNext #-}
endOmittableNext :: HashMap Text (HashSet Text)
endOmittableNext = M.fromList [
    ("colgroup", S.fromList ["caption", "colgroup", "tbody",
                             "thead", "tfoot", "tr"]),
    ("dd",       S.fromList ["dd", "dt"]),
    ("dt",       S.fromList ["dd", "dt"]),
    ("head",     S.fromList ["body"]),
    ("li",       S.fromList ["li"]),
    ("optgroup", S.fromList ["optgroup"]),
    ("option",   S.fromList ["optgroup", "option"]),
    ("p",        S.fromList ["address", "article", "aside", "blockquote",
                             "dir", "div", "dl", "fieldset", "footer",
                             "form", "h1", "h2", "h3", "h4", "h5", "h6",
                             "header", "hgroup", "hr", "menu", "nav", "ol",
                             "p", "pre", "section", "table", "ul"]),
    ("rp",       S.fromList ["rp", "rt"]),
    ("rt",       S.fromList ["rp", "rt"]),
    ("tbody",    S.fromList ["tbody", "tfoot", "thead"]),
    ("td",       S.fromList ["td", "th"]),
    ("tfoot",    S.fromList ["tbody", "tfoot", "thead"]),
    ("th",       S.fromList ["td", "th"]),
    ("thead",    S.fromList ["tbody", "tfoot", "thead"]),
    ("tr",       S.fromList ["tr"])
    ]

------------------------------------------------------------------------------
-- | Predefined character entity references as defined by the HTML5 spec.
{-# NOINLINE predefinedRefs #-}
predefinedRefs :: HashMap Text Text
predefinedRefs = mconcat $ map M.fromList [
      reftab1
    , reftab2
    , reftab3
    , reftab4
    , reftab5
    , reftab6
    , reftab7
    , reftab8
    , reftab9
    , reftab10
    , reftab11
    , reftab12
    , reftab13
    , reftab14
    , reftab15
    , reftab16
    , reftab17
    , reftab18
    , reftab19
    , reftab20
    , reftab21
    , reftab22
    , reftab23
    , reftab24
    , reftab25
    , reftab26
    , reftab27
    , reftab28
    , reftab29
    , reftab30
    , reftab31
    , reftab32
    , reftab33
    , reftab34
    , reftab35
    , reftab36
    , reftab37
    , reftab38
    , reftab39
    , reftab40
    , reftab41
    , reftab42
    , reftab43
    , reftab44
    , reftab45
    , reftab46
    , reftab47
    , reftab48
    , reftab49
    , reftab50
    , reftab51
    , reftab52
    , reftab53
    , reftab54
    , reftab55
    , reftab56
    , reftab57
    , reftab58 ]


{-# NOINLINE reftab1 #-}
reftab1 :: [(Text,Text)]
reftab1 =
  [ ("AElig", "\x000C6"),
    ("AMP", "\x00026"),
    ("Aacute", "\x000C1"),
    ("Abreve", "\x00102"),
    ("Acirc", "\x000C2"),
    ("Acy", "\x00410"),
    ("Afr", "\x1D504"),
    ("Agrave", "\x000C0"),
    ("Alpha", "\x00391"),
    ("Amacr", "\x00100"),
    ("And", "\x02A53"),
    ("Aogon", "\x00104"),
    ("Aopf", "\x1D538"),
    ("ApplyFunction", "\x02061"),
    ("Aring", "\x000C5"),
    ("Ascr", "\x1D49C"),
    ("Assign", "\x02254"),
    ("Atilde", "\x000C3"),
    ("Auml", "\x000C4"),
    ("Backslash", "\x02216"),
    ("Barv", "\x02AE7"),
    ("Barwed", "\x02306"),
    ("Bcy", "\x00411"),
    ("Because", "\x02235"),
    ("Bernoullis", "\x0212C"),
    ("Beta", "\x00392"),
    ("Bfr", "\x1D505"),
    ("Bopf", "\x1D539"),
    ("Breve", "\x002D8"),
    ("Bscr", "\x0212C"),
    ("Bumpeq", "\x0224E"),
    ("CHcy", "\x00427"),
    ("COPY", "\x000A9"),
    ("Cacute", "\x00106"),
    ("Cap", "\x022D2"),
    ("CapitalDifferentialD", "\x02145"),
    ("Cayleys", "\x0212D") ]

{-# NOINLINE reftab2 #-}
reftab2 :: [(Text,Text)]
reftab2 =
  [ ("Ccaron", "\x0010C"),
    ("Ccedil", "\x000C7"),
    ("Ccirc", "\x00108"),
    ("Cconint", "\x02230"),
    ("Cdot", "\x0010A"),
    ("Cedilla", "\x000B8"),
    ("CenterDot", "\x000B7"),
    ("Cfr", "\x0212D"),
    ("Chi", "\x003A7"),
    ("CircleDot", "\x02299"),
    ("CircleMinus", "\x02296"),
    ("CirclePlus", "\x02295"),
    ("CircleTimes", "\x02297"),
    ("ClockwiseContourIntegral", "\x02232"),
    ("CloseCurlyDoubleQuote", "\x0201D"),
    ("CloseCurlyQuote", "\x02019"),
    ("Colon", "\x02237"),
    ("Colone", "\x02A74"),
    ("Congruent", "\x02261"),
    ("Conint", "\x0222F"),
    ("ContourIntegral", "\x0222E"),
    ("Copf", "\x02102"),
    ("Coproduct", "\x02210"),
    ("CounterClockwiseContourIntegral", "\x02233"),
    ("Cross", "\x02A2F"),
    ("Cscr", "\x1D49E"),
    ("Cup", "\x022D3"),
    ("CupCap", "\x0224D"),
    ("DD", "\x02145"),
    ("DDotrahd", "\x02911"),
    ("DJcy", "\x00402"),
    ("DScy", "\x00405"),
    ("DZcy", "\x0040F"),
    ("Dagger", "\x02021"),
    ("Darr", "\x021A1"),
    ("Dashv", "\x02AE4"),
    ("Dcaron", "\x0010E") ]

{-# NOINLINE reftab3 #-}
reftab3 :: [(Text,Text)]
reftab3 =
  [ ("Dcy", "\x00414"),
    ("Del", "\x02207"),
    ("Delta", "\x00394"),
    ("Dfr", "\x1D507"),
    ("DiacriticalAcute", "\x000B4"),
    ("DiacriticalDot", "\x002D9"),
    ("DiacriticalDoubleAcute", "\x002DD"),
    ("DiacriticalGrave", "\x00060"),
    ("DiacriticalTilde", "\x002DC"),
    ("Diamond", "\x022C4"),
    ("DifferentialD", "\x02146"),
    ("Dopf", "\x1D53B"),
    ("Dot", "\x000A8"),
    ("DotDot", "\x020DC"),
    ("DotEqual", "\x02250"),
    ("DoubleContourIntegral", "\x0222F"),
    ("DoubleDot", "\x000A8"),
    ("DoubleDownArrow", "\x021D3"),
    ("DoubleLeftArrow", "\x021D0"),
    ("DoubleLeftRightArrow", "\x021D4"),
    ("DoubleLeftTee", "\x02AE4"),
    ("DoubleLongLeftArrow", "\x027F8"),
    ("DoubleLongLeftRightArrow", "\x027FA"),
    ("DoubleLongRightArrow", "\x027F9"),
    ("DoubleRightArrow", "\x021D2"),
    ("DoubleRightTee", "\x022A8"),
    ("DoubleUpArrow", "\x021D1"),
    ("DoubleUpDownArrow", "\x021D5"),
    ("DoubleVerticalBar", "\x02225"),
    ("DownArrow", "\x02193"),
    ("DownArrowBar", "\x02913"),
    ("DownArrowUpArrow", "\x021F5"),
    ("DownBreve", "\x00311"),
    ("DownLeftRightVector", "\x02950"),
    ("DownLeftTeeVector", "\x0295E"),
    ("DownLeftVector", "\x021BD"),
    ("DownLeftVectorBar", "\x02956") ]

{-# NOINLINE reftab4 #-}
reftab4 :: [(Text,Text)]
reftab4 =
  [ ("DownRightTeeVector", "\x0295F"),
    ("DownRightVector", "\x021C1"),
    ("DownRightVectorBar", "\x02957"),
    ("DownTee", "\x022A4"),
    ("DownTeeArrow", "\x021A7"),
    ("Downarrow", "\x021D3"),
    ("Dscr", "\x1D49F"),
    ("Dstrok", "\x00110"),
    ("ENG", "\x0014A"),
    ("ETH", "\x000D0"),
    ("Eacute", "\x000C9"),
    ("Ecaron", "\x0011A"),
    ("Ecirc", "\x000CA"),
    ("Ecy", "\x0042D"),
    ("Edot", "\x00116"),
    ("Efr", "\x1D508"),
    ("Egrave", "\x000C8"),
    ("Element", "\x02208"),
    ("Emacr", "\x00112"),
    ("EmptySmallSquare", "\x025FB"),
    ("EmptyVerySmallSquare", "\x025AB"),
    ("Eogon", "\x00118"),
    ("Eopf", "\x1D53C"),
    ("Epsilon", "\x00395"),
    ("Equal", "\x02A75"),
    ("EqualTilde", "\x02242"),
    ("Equilibrium", "\x021CC"),
    ("Escr", "\x02130"),
    ("Esim", "\x02A73"),
    ("Eta", "\x00397"),
    ("Euml", "\x000CB"),
    ("Exists", "\x02203"),
    ("ExponentialE", "\x02147"),
    ("Fcy", "\x00424"),
    ("Ffr", "\x1D509"),
    ("FilledSmallSquare", "\x025FC"),
    ("FilledVerySmallSquare", "\x025AA") ]

{-# NOINLINE reftab5 #-}
reftab5 :: [(Text,Text)]
reftab5 =
  [ ("Fopf", "\x1D53D"),
    ("ForAll", "\x02200"),
    ("Fouriertrf", "\x02131"),
    ("Fscr", "\x02131"),
    ("GJcy", "\x00403"),
    ("GT", "\x0003E"),
    ("Gamma", "\x00393"),
    ("Gammad", "\x003DC"),
    ("Gbreve", "\x0011E"),
    ("Gcedil", "\x00122"),
    ("Gcirc", "\x0011C"),
    ("Gcy", "\x00413"),
    ("Gdot", "\x00120"),
    ("Gfr", "\x1D50A"),
    ("Gg", "\x022D9"),
    ("Gopf", "\x1D53E"),
    ("GreaterEqual", "\x02265"),
    ("GreaterEqualLess", "\x022DB"),
    ("GreaterFullEqual", "\x02267"),
    ("GreaterGreater", "\x02AA2"),
    ("GreaterLess", "\x02277"),
    ("GreaterSlantEqual", "\x02A7E"),
    ("GreaterTilde", "\x02273"),
    ("Gscr", "\x1D4A2"),
    ("Gt", "\x0226B"),
    ("HARDcy", "\x0042A"),
    ("Hacek", "\x002C7"),
    ("Hat", "\x0005E"),
    ("Hcirc", "\x00124"),
    ("Hfr", "\x0210C"),
    ("HilbertSpace", "\x0210B"),
    ("Hopf", "\x0210D"),
    ("HorizontalLine", "\x02500"),
    ("Hscr", "\x0210B"),
    ("Hstrok", "\x00126"),
    ("HumpDownHump", "\x0224E"),
    ("HumpEqual", "\x0224F") ]

{-# NOINLINE reftab6 #-}
reftab6 :: [(Text,Text)]
reftab6 =
  [ ("IEcy", "\x00415"),
    ("IJlig", "\x00132"),
    ("IOcy", "\x00401"),
    ("Iacute", "\x000CD"),
    ("Icirc", "\x000CE"),
    ("Icy", "\x00418"),
    ("Idot", "\x00130"),
    ("Ifr", "\x02111"),
    ("Igrave", "\x000CC"),
    ("Im", "\x02111"),
    ("Imacr", "\x0012A"),
    ("ImaginaryI", "\x02148"),
    ("Implies", "\x021D2"),
    ("Int", "\x0222C"),
    ("Integral", "\x0222B"),
    ("Intersection", "\x022C2"),
    ("InvisibleComma", "\x02063"),
    ("InvisibleTimes", "\x02062"),
    ("Iogon", "\x0012E"),
    ("Iopf", "\x1D540"),
    ("Iota", "\x00399"),
    ("Iscr", "\x02110"),
    ("Itilde", "\x00128"),
    ("Iukcy", "\x00406"),
    ("Iuml", "\x000CF"),
    ("Jcirc", "\x00134"),
    ("Jcy", "\x00419"),
    ("Jfr", "\x1D50D"),
    ("Jopf", "\x1D541"),
    ("Jscr", "\x1D4A5"),
    ("Jsercy", "\x00408"),
    ("Jukcy", "\x00404"),
    ("KHcy", "\x00425"),
    ("KJcy", "\x0040C"),
    ("Kappa", "\x0039A"),
    ("Kcedil", "\x00136"),
    ("Kcy", "\x0041A") ]

{-# NOINLINE reftab7 #-}
reftab7 :: [(Text,Text)]
reftab7 =
  [ ("Kfr", "\x1D50E"),
    ("Kopf", "\x1D542"),
    ("Kscr", "\x1D4A6"),
    ("LJcy", "\x00409"),
    ("LT", "\x0003C"),
    ("Lacute", "\x00139"),
    ("Lambda", "\x0039B"),
    ("Lang", "\x027EA"),
    ("Laplacetrf", "\x02112"),
    ("Larr", "\x0219E"),
    ("Lcaron", "\x0013D"),
    ("Lcedil", "\x0013B"),
    ("Lcy", "\x0041B"),
    ("LeftAngleBracket", "\x027E8"),
    ("LeftArrow", "\x02190"),
    ("LeftArrowBar", "\x021E4"),
    ("LeftArrowRightArrow", "\x021C6"),
    ("LeftCeiling", "\x02308"),
    ("LeftDoubleBracket", "\x027E6"),
    ("LeftDownTeeVector", "\x02961"),
    ("LeftDownVector", "\x021C3"),
    ("LeftDownVectorBar", "\x02959"),
    ("LeftFloor", "\x0230A"),
    ("LeftRightArrow", "\x02194"),
    ("LeftRightVector", "\x0294E"),
    ("LeftTee", "\x022A3"),
    ("LeftTeeArrow", "\x021A4"),
    ("LeftTeeVector", "\x0295A"),
    ("LeftTriangle", "\x022B2"),
    ("LeftTriangleBar", "\x029CF"),
    ("LeftTriangleEqual", "\x022B4"),
    ("LeftUpDownVector", "\x02951"),
    ("LeftUpTeeVector", "\x02960"),
    ("LeftUpVector", "\x021BF"),
    ("LeftUpVectorBar", "\x02958"),
    ("LeftVector", "\x021BC"),
    ("LeftVectorBar", "\x02952") ]

{-# NOINLINE reftab8 #-}
reftab8 :: [(Text,Text)]
reftab8 =
  [ ("Leftarrow", "\x021D0"),
    ("Leftrightarrow", "\x021D4"),
    ("LessEqualGreater", "\x022DA"),
    ("LessFullEqual", "\x02266"),
    ("LessGreater", "\x02276"),
    ("LessLess", "\x02AA1"),
    ("LessSlantEqual", "\x02A7D"),
    ("LessTilde", "\x02272"),
    ("Lfr", "\x1D50F"),
    ("Ll", "\x022D8"),
    ("Lleftarrow", "\x021DA"),
    ("Lmidot", "\x0013F"),
    ("LongLeftArrow", "\x027F5"),
    ("LongLeftRightArrow", "\x027F7"),
    ("LongRightArrow", "\x027F6"),
    ("Longleftarrow", "\x027F8"),
    ("Longleftrightarrow", "\x027FA"),
    ("Longrightarrow", "\x027F9"),
    ("Lopf", "\x1D543"),
    ("LowerLeftArrow", "\x02199"),
    ("LowerRightArrow", "\x02198"),
    ("Lscr", "\x02112"),
    ("Lsh", "\x021B0"),
    ("Lstrok", "\x00141"),
    ("Lt", "\x0226A"),
    ("Map", "\x02905"),
    ("Mcy", "\x0041C"),
    ("MediumSpace", "\x0205F"),
    ("Mellintrf", "\x02133"),
    ("Mfr", "\x1D510"),
    ("MinusPlus", "\x02213"),
    ("Mopf", "\x1D544"),
    ("Mscr", "\x02133"),
    ("Mu", "\x0039C"),
    ("NJcy", "\x0040A"),
    ("Nacute", "\x00143"),
    ("Ncaron", "\x00147") ]

{-# NOINLINE reftab9 #-}
reftab9 :: [(Text,Text)]
reftab9 =
  [ ("Ncedil", "\x00145"),
    ("Ncy", "\x0041D"),
    ("NegativeMediumSpace", "\x0200B"),
    ("NegativeThickSpace", "\x0200B"),
    ("NegativeThinSpace", "\x0200B"),
    ("NegativeVeryThinSpace", "\x0200B"),
    ("NestedGreaterGreater", "\x0226B"),
    ("NestedLessLess", "\x0226A"),
    ("NewLine", "\x0000A"),
    ("Nfr", "\x1D511"),
    ("NoBreak", "\x02060"),
    ("NonBreakingSpace", "\x000A0"),
    ("Nopf", "\x02115"),
    ("Not", "\x02AEC"),
    ("NotCongruent", "\x02262"),
    ("NotCupCap", "\x0226D"),
    ("NotDoubleVerticalBar", "\x02226"),
    ("NotElement", "\x02209"),
    ("NotEqual", "\x02260"),
    ("NotEqualTilde", "\x02242\x00338"),
    ("NotExists", "\x02204"),
    ("NotGreater", "\x0226F"),
    ("NotGreaterEqual", "\x02271"),
    ("NotGreaterFullEqual", "\x02267\x00338"),
    ("NotGreaterGreater", "\x0226B\x00338"),
    ("NotGreaterLess", "\x02279"),
    ("NotGreaterSlantEqual", "\x02A7E\x00338"),
    ("NotGreaterTilde", "\x02275"),
    ("NotHumpDownHump", "\x0224E\x00338"),
    ("NotHumpEqual", "\x0224F\x00338"),
    ("NotLeftTriangle", "\x022EA"),
    ("NotLeftTriangleBar", "\x029CF\x00338"),
    ("NotLeftTriangleEqual", "\x022EC"),
    ("NotLess", "\x0226E"),
    ("NotLessEqual", "\x02270"),
    ("NotLessGreater", "\x02278"),
    ("NotLessLess", "\x0226A\x00338") ]

{-# NOINLINE reftab10 #-}
reftab10 :: [(Text,Text)]
reftab10 =
  [ ("NotLessSlantEqual", "\x02A7D\x00338"),
    ("NotLessTilde", "\x02274"),
    ("NotNestedGreaterGreater", "\x02AA2\x00338"),
    ("NotNestedLessLess", "\x02AA1\x00338"),
    ("NotPrecedes", "\x02280"),
    ("NotPrecedesEqual", "\x02AAF\x00338"),
    ("NotPrecedesSlantEqual", "\x022E0"),
    ("NotReverseElement", "\x0220C"),
    ("NotRightTriangle", "\x022EB"),
    ("NotRightTriangleBar", "\x029D0\x00338"),
    ("NotRightTriangleEqual", "\x022ED"),
    ("NotSquareSubset", "\x0228F\x00338"),
    ("NotSquareSubsetEqual", "\x022E2"),
    ("NotSquareSuperset", "\x02290\x00338"),
    ("NotSquareSupersetEqual", "\x022E3"),
    ("NotSubset", "\x02282\x020D2"),
    ("NotSubsetEqual", "\x02288"),
    ("NotSucceeds", "\x02281"),
    ("NotSucceedsEqual", "\x02AB0\x00338"),
    ("NotSucceedsSlantEqual", "\x022E1"),
    ("NotSucceedsTilde", "\x0227F\x00338"),
    ("NotSuperset", "\x02283\x020D2"),
    ("NotSupersetEqual", "\x02289"),
    ("NotTilde", "\x02241"),
    ("NotTildeEqual", "\x02244"),
    ("NotTildeFullEqual", "\x02247"),
    ("NotTildeTilde", "\x02249"),
    ("NotVerticalBar", "\x02224"),
    ("Nscr", "\x1D4A9"),
    ("Ntilde", "\x000D1"),
    ("Nu", "\x0039D"),
    ("OElig", "\x00152"),
    ("Oacute", "\x000D3"),
    ("Ocirc", "\x000D4"),
    ("Ocy", "\x0041E"),
    ("Odblac", "\x00150"),
    ("Ofr", "\x1D512") ]

{-# NOINLINE reftab11 #-}
reftab11 :: [(Text,Text)]
reftab11 =
  [ ("Ograve", "\x000D2"),
    ("Omacr", "\x0014C"),
    ("Omega", "\x003A9"),
    ("Omicron", "\x0039F"),
    ("Oopf", "\x1D546"),
    ("OpenCurlyDoubleQuote", "\x0201C"),
    ("OpenCurlyQuote", "\x02018"),
    ("Or", "\x02A54"),
    ("Oscr", "\x1D4AA"),
    ("Oslash", "\x000D8"),
    ("Otilde", "\x000D5"),
    ("Otimes", "\x02A37"),
    ("Ouml", "\x000D6"),
    ("OverBar", "\x0203E"),
    ("OverBrace", "\x023DE"),
    ("OverBracket", "\x023B4"),
    ("OverParenthesis", "\x023DC"),
    ("PartialD", "\x02202"),
    ("Pcy", "\x0041F"),
    ("Pfr", "\x1D513"),
    ("Phi", "\x003A6"),
    ("Pi", "\x003A0"),
    ("PlusMinus", "\x000B1"),
    ("Poincareplane", "\x0210C"),
    ("Popf", "\x02119"),
    ("Pr", "\x02ABB"),
    ("Precedes", "\x0227A"),
    ("PrecedesEqual", "\x02AAF"),
    ("PrecedesSlantEqual", "\x0227C"),
    ("PrecedesTilde", "\x0227E"),
    ("Prime", "\x02033"),
    ("Product", "\x0220F"),
    ("Proportion", "\x02237"),
    ("Proportional", "\x0221D"),
    ("Pscr", "\x1D4AB"),
    ("Psi", "\x003A8"),
    ("QUOT", "\x00022") ]

{-# NOINLINE reftab12 #-}
reftab12 :: [(Text,Text)]
reftab12 =
  [ ("Qfr", "\x1D514"),
    ("Qopf", "\x0211A"),
    ("Qscr", "\x1D4AC"),
    ("RBarr", "\x02910"),
    ("REG", "\x000AE"),
    ("Racute", "\x00154"),
    ("Rang", "\x027EB"),
    ("Rarr", "\x021A0"),
    ("Rarrtl", "\x02916"),
    ("Rcaron", "\x00158"),
    ("Rcedil", "\x00156"),
    ("Rcy", "\x00420"),
    ("Re", "\x0211C"),
    ("ReverseElement", "\x0220B"),
    ("ReverseEquilibrium", "\x021CB"),
    ("ReverseUpEquilibrium", "\x0296F"),
    ("Rfr", "\x0211C"),
    ("Rho", "\x003A1"),
    ("RightAngleBracket", "\x027E9"),
    ("RightArrow", "\x02192"),
    ("RightArrowBar", "\x021E5"),
    ("RightArrowLeftArrow", "\x021C4"),
    ("RightCeiling", "\x02309"),
    ("RightDoubleBracket", "\x027E7"),
    ("RightDownTeeVector", "\x0295D"),
    ("RightDownVector", "\x021C2"),
    ("RightDownVectorBar", "\x02955"),
    ("RightFloor", "\x0230B"),
    ("RightTee", "\x022A2"),
    ("RightTeeArrow", "\x021A6"),
    ("RightTeeVector", "\x0295B"),
    ("RightTriangle", "\x022B3"),
    ("RightTriangleBar", "\x029D0"),
    ("RightTriangleEqual", "\x022B5"),
    ("RightUpDownVector", "\x0294F"),
    ("RightUpTeeVector", "\x0295C"),
    ("RightUpVector", "\x021BE") ]

{-# NOINLINE reftab13 #-}
reftab13 :: [(Text,Text)]
reftab13 =
  [ ("RightUpVectorBar", "\x02954"),
    ("RightVector", "\x021C0"),
    ("RightVectorBar", "\x02953"),
    ("Rightarrow", "\x021D2"),
    ("Ropf", "\x0211D"),
    ("RoundImplies", "\x02970"),
    ("Rrightarrow", "\x021DB"),
    ("Rscr", "\x0211B"),
    ("Rsh", "\x021B1"),
    ("RuleDelayed", "\x029F4"),
    ("SHCHcy", "\x00429"),
    ("SHcy", "\x00428"),
    ("SOFTcy", "\x0042C"),
    ("Sacute", "\x0015A"),
    ("Sc", "\x02ABC"),
    ("Scaron", "\x00160"),
    ("Scedil", "\x0015E"),
    ("Scirc", "\x0015C"),
    ("Scy", "\x00421"),
    ("Sfr", "\x1D516"),
    ("ShortDownArrow", "\x02193"),
    ("ShortLeftArrow", "\x02190"),
    ("ShortRightArrow", "\x02192"),
    ("ShortUpArrow", "\x02191"),
    ("Sigma", "\x003A3"),
    ("SmallCircle", "\x02218"),
    ("Sopf", "\x1D54A"),
    ("Sqrt", "\x0221A"),
    ("Square", "\x025A1"),
    ("SquareIntersection", "\x02293"),
    ("SquareSubset", "\x0228F"),
    ("SquareSubsetEqual", "\x02291"),
    ("SquareSuperset", "\x02290"),
    ("SquareSupersetEqual", "\x02292"),
    ("SquareUnion", "\x02294"),
    ("Sscr", "\x1D4AE"),
    ("Star", "\x022C6") ]

{-# NOINLINE reftab14 #-}
reftab14 :: [(Text,Text)]
reftab14 =
  [ ("Sub", "\x022D0"),
    ("Subset", "\x022D0"),
    ("SubsetEqual", "\x02286"),
    ("Succeeds", "\x0227B"),
    ("SucceedsEqual", "\x02AB0"),
    ("SucceedsSlantEqual", "\x0227D"),
    ("SucceedsTilde", "\x0227F"),
    ("SuchThat", "\x0220B"),
    ("Sum", "\x02211"),
    ("Sup", "\x022D1"),
    ("Superset", "\x02283"),
    ("SupersetEqual", "\x02287"),
    ("Supset", "\x022D1"),
    ("THORN", "\x000DE"),
    ("TRADE", "\x02122"),
    ("TSHcy", "\x0040B"),
    ("TScy", "\x00426"),
    ("Tab", "\x00009"),
    ("Tau", "\x003A4"),
    ("Tcaron", "\x00164"),
    ("Tcedil", "\x00162"),
    ("Tcy", "\x00422"),
    ("Tfr", "\x1D517"),
    ("Therefore", "\x02234"),
    ("Theta", "\x00398"),
    ("ThickSpace", "\x0205F\x0200A"),
    ("ThinSpace", "\x02009"),
    ("Tilde", "\x0223C"),
    ("TildeEqual", "\x02243"),
    ("TildeFullEqual", "\x02245"),
    ("TildeTilde", "\x02248"),
    ("Topf", "\x1D54B"),
    ("TripleDot", "\x020DB"),
    ("Tscr", "\x1D4AF"),
    ("Tstrok", "\x00166"),
    ("Uacute", "\x000DA"),
    ("Uarr", "\x0219F") ]

{-# NOINLINE reftab15 #-}
reftab15 :: [(Text,Text)]
reftab15 =
  [ ("Uarrocir", "\x02949"),
    ("Ubrcy", "\x0040E"),
    ("Ubreve", "\x0016C"),
    ("Ucirc", "\x000DB"),
    ("Ucy", "\x00423"),
    ("Udblac", "\x00170"),
    ("Ufr", "\x1D518"),
    ("Ugrave", "\x000D9"),
    ("Umacr", "\x0016A"),
    ("UnderBar", "\x0005F"),
    ("UnderBrace", "\x023DF"),
    ("UnderBracket", "\x023B5"),
    ("UnderParenthesis", "\x023DD"),
    ("Union", "\x022C3"),
    ("UnionPlus", "\x0228E"),
    ("Uogon", "\x00172"),
    ("Uopf", "\x1D54C"),
    ("UpArrow", "\x02191"),
    ("UpArrowBar", "\x02912"),
    ("UpArrowDownArrow", "\x021C5"),
    ("UpDownArrow", "\x02195"),
    ("UpEquilibrium", "\x0296E"),
    ("UpTee", "\x022A5"),
    ("UpTeeArrow", "\x021A5"),
    ("Uparrow", "\x021D1"),
    ("Updownarrow", "\x021D5"),
    ("UpperLeftArrow", "\x02196"),
    ("UpperRightArrow", "\x02197"),
    ("Upsi", "\x003D2"),
    ("Upsilon", "\x003A5"),
    ("Uring", "\x0016E"),
    ("Uscr", "\x1D4B0"),
    ("Utilde", "\x00168"),
    ("Uuml", "\x000DC"),
    ("VDash", "\x022AB"),
    ("Vbar", "\x02AEB"),
    ("Vcy", "\x00412") ]

{-# NOINLINE reftab16 #-}
reftab16 :: [(Text,Text)]
reftab16 =
  [ ("Vdash", "\x022A9"),
    ("Vdashl", "\x02AE6"),
    ("Vee", "\x022C1"),
    ("Verbar", "\x02016"),
    ("Vert", "\x02016"),
    ("VerticalBar", "\x02223"),
    ("VerticalLine", "\x0007C"),
    ("VerticalSeparator", "\x02758"),
    ("VerticalTilde", "\x02240"),
    ("VeryThinSpace", "\x0200A"),
    ("Vfr", "\x1D519"),
    ("Vopf", "\x1D54D"),
    ("Vscr", "\x1D4B1"),
    ("Vvdash", "\x022AA"),
    ("Wcirc", "\x00174"),
    ("Wedge", "\x022C0"),
    ("Wfr", "\x1D51A"),
    ("Wopf", "\x1D54E"),
    ("Wscr", "\x1D4B2"),
    ("Xfr", "\x1D51B"),
    ("Xi", "\x0039E"),
    ("Xopf", "\x1D54F"),
    ("Xscr", "\x1D4B3"),
    ("YAcy", "\x0042F"),
    ("YIcy", "\x00407"),
    ("YUcy", "\x0042E"),
    ("Yacute", "\x000DD"),
    ("Ycirc", "\x00176"),
    ("Ycy", "\x0042B"),
    ("Yfr", "\x1D51C"),
    ("Yopf", "\x1D550"),
    ("Yscr", "\x1D4B4"),
    ("Yuml", "\x00178"),
    ("ZHcy", "\x00416"),
    ("Zacute", "\x00179"),
    ("Zcaron", "\x0017D"),
    ("Zcy", "\x00417") ]

{-# NOINLINE reftab17 #-}
reftab17 :: [(Text,Text)]
reftab17 =
  [ ("Zdot", "\x0017B"),
    ("ZeroWidthSpace", "\x0200B"),
    ("Zeta", "\x00396"),
    ("Zfr", "\x02128"),
    ("Zopf", "\x02124"),
    ("Zscr", "\x1D4B5"),
    ("aacute", "\x000E1"),
    ("abreve", "\x00103"),
    ("ac", "\x0223E"),
    ("acE", "\x0223E\x00333"),
    ("acd", "\x0223F"),
    ("acirc", "\x000E2"),
    ("acute", "\x000B4"),
    ("acy", "\x00430"),
    ("aelig", "\x000E6"),
    ("af", "\x02061"),
    ("afr", "\x1D51E"),
    ("agrave", "\x000E0"),
    ("alefsym", "\x02135"),
    ("aleph", "\x02135"),
    ("alpha", "\x003B1"),
    ("amacr", "\x00101"),
    ("amalg", "\x02A3F"),
    ("amp", "\x00026"),
    ("and", "\x02227"),
    ("andand", "\x02A55"),
    ("andd", "\x02A5C"),
    ("andslope", "\x02A58"),
    ("andv", "\x02A5A"),
    ("ang", "\x02220"),
    ("ange", "\x029A4"),
    ("angle", "\x02220"),
    ("angmsd", "\x02221"),
    ("angmsdaa", "\x029A8"),
    ("angmsdab", "\x029A9"),
    ("angmsdac", "\x029AA"),
    ("angmsdad", "\x029AB") ]

{-# NOINLINE reftab18 #-}
reftab18 :: [(Text,Text)]
reftab18 =
  [ ("angmsdae", "\x029AC"),
    ("angmsdaf", "\x029AD"),
    ("angmsdag", "\x029AE"),
    ("angmsdah", "\x029AF"),
    ("angrt", "\x0221F"),
    ("angrtvb", "\x022BE"),
    ("angrtvbd", "\x0299D"),
    ("angsph", "\x02222"),
    ("angst", "\x000C5"),
    ("angzarr", "\x0237C"),
    ("aogon", "\x00105"),
    ("aopf", "\x1D552"),
    ("ap", "\x02248"),
    ("apE", "\x02A70"),
    ("apacir", "\x02A6F"),
    ("ape", "\x0224A"),
    ("apid", "\x0224B"),
    ("apos", "\x00027"),
    ("approx", "\x02248"),
    ("approxeq", "\x0224A"),
    ("aring", "\x000E5"),
    ("ascr", "\x1D4B6"),
    ("ast", "\x0002A"),
    ("asymp", "\x02248"),
    ("asympeq", "\x0224D"),
    ("atilde", "\x000E3"),
    ("auml", "\x000E4"),
    ("awconint", "\x02233"),
    ("awint", "\x02A11"),
    ("bNot", "\x02AED"),
    ("backcong", "\x0224C"),
    ("backepsilon", "\x003F6"),
    ("backprime", "\x02035"),
    ("backsim", "\x0223D"),
    ("backsimeq", "\x022CD"),
    ("barvee", "\x022BD"),
    ("barwed", "\x02305") ]

{-# NOINLINE reftab19 #-}
reftab19 :: [(Text,Text)]
reftab19 =
  [ ("barwedge", "\x02305"),
    ("bbrk", "\x023B5"),
    ("bbrktbrk", "\x023B6"),
    ("bcong", "\x0224C"),
    ("bcy", "\x00431"),
    ("bdquo", "\x0201E"),
    ("becaus", "\x02235"),
    ("because", "\x02235"),
    ("bemptyv", "\x029B0"),
    ("bepsi", "\x003F6"),
    ("bernou", "\x0212C"),
    ("beta", "\x003B2"),
    ("beth", "\x02136"),
    ("between", "\x0226C"),
    ("bfr", "\x1D51F"),
    ("bigcap", "\x022C2"),
    ("bigcirc", "\x025EF"),
    ("bigcup", "\x022C3"),
    ("bigodot", "\x02A00"),
    ("bigoplus", "\x02A01"),
    ("bigotimes", "\x02A02"),
    ("bigsqcup", "\x02A06"),
    ("bigstar", "\x02605"),
    ("bigtriangledown", "\x025BD"),
    ("bigtriangleup", "\x025B3"),
    ("biguplus", "\x02A04"),
    ("bigvee", "\x022C1"),
    ("bigwedge", "\x022C0"),
    ("bkarow", "\x0290D"),
    ("blacklozenge", "\x029EB"),
    ("blacksquare", "\x025AA"),
    ("blacktriangle", "\x025B4"),
    ("blacktriangledown", "\x025BE"),
    ("blacktriangleleft", "\x025C2"),
    ("blacktriangleright", "\x025B8"),
    ("blank", "\x02423"),
    ("blk12", "\x02592") ]

{-# NOINLINE reftab20 #-}
reftab20 :: [(Text,Text)]
reftab20 =
  [ ("blk14", "\x02591"),
    ("blk34", "\x02593"),
    ("block", "\x02588"),
    ("bne", "\x0003D\x020E5"),
    ("bnequiv", "\x02261\x020E5"),
    ("bnot", "\x02310"),
    ("bopf", "\x1D553"),
    ("bot", "\x022A5"),
    ("bottom", "\x022A5"),
    ("bowtie", "\x022C8"),
    ("boxDL", "\x02557"),
    ("boxDR", "\x02554"),
    ("boxDl", "\x02556"),
    ("boxDr", "\x02553"),
    ("boxH", "\x02550"),
    ("boxHD", "\x02566"),
    ("boxHU", "\x02569"),
    ("boxHd", "\x02564"),
    ("boxHu", "\x02567"),
    ("boxUL", "\x0255D"),
    ("boxUR", "\x0255A"),
    ("boxUl", "\x0255C"),
    ("boxUr", "\x02559"),
    ("boxV", "\x02551"),
    ("boxVH", "\x0256C"),
    ("boxVL", "\x02563"),
    ("boxVR", "\x02560"),
    ("boxVh", "\x0256B"),
    ("boxVl", "\x02562"),
    ("boxVr", "\x0255F"),
    ("boxbox", "\x029C9"),
    ("boxdL", "\x02555"),
    ("boxdR", "\x02552"),
    ("boxdl", "\x02510"),
    ("boxdr", "\x0250C"),
    ("boxh", "\x02500"),
    ("boxhD", "\x02565") ]

{-# NOINLINE reftab21 #-}
reftab21 :: [(Text,Text)]
reftab21 =
  [ ("boxhU", "\x02568"),
    ("boxhd", "\x0252C"),
    ("boxhu", "\x02534"),
    ("boxminus", "\x0229F"),
    ("boxplus", "\x0229E"),
    ("boxtimes", "\x022A0"),
    ("boxuL", "\x0255B"),
    ("boxuR", "\x02558"),
    ("boxul", "\x02518"),
    ("boxur", "\x02514"),
    ("boxv", "\x02502"),
    ("boxvH", "\x0256A"),
    ("boxvL", "\x02561"),
    ("boxvR", "\x0255E"),
    ("boxvh", "\x0253C"),
    ("boxvl", "\x02524"),
    ("boxvr", "\x0251C"),
    ("bprime", "\x02035"),
    ("breve", "\x002D8"),
    ("brvbar", "\x000A6"),
    ("bscr", "\x1D4B7"),
    ("bsemi", "\x0204F"),
    ("bsim", "\x0223D"),
    ("bsime", "\x022CD"),
    ("bsol", "\x0005C"),
    ("bsolb", "\x029C5"),
    ("bsolhsub", "\x027C8"),
    ("bull", "\x02022"),
    ("bullet", "\x02022"),
    ("bump", "\x0224E"),
    ("bumpE", "\x02AAE"),
    ("bumpe", "\x0224F"),
    ("bumpeq", "\x0224F"),
    ("cacute", "\x00107"),
    ("cap", "\x02229"),
    ("capand", "\x02A44"),
    ("capbrcup", "\x02A49") ]

{-# NOINLINE reftab22 #-}
reftab22 :: [(Text,Text)]
reftab22 =
  [ ("capcap", "\x02A4B"),
    ("capcup", "\x02A47"),
    ("capdot", "\x02A40"),
    ("caps", "\x02229\x0FE00"),
    ("caret", "\x02041"),
    ("caron", "\x002C7"),
    ("ccaps", "\x02A4D"),
    ("ccaron", "\x0010D"),
    ("ccedil", "\x000E7"),
    ("ccirc", "\x00109"),
    ("ccups", "\x02A4C"),
    ("ccupssm", "\x02A50"),
    ("cdot", "\x0010B"),
    ("cedil", "\x000B8"),
    ("cemptyv", "\x029B2"),
    ("cent", "\x000A2"),
    ("centerdot", "\x000B7"),
    ("cfr", "\x1D520"),
    ("chcy", "\x00447"),
    ("check", "\x02713"),
    ("checkmark", "\x02713"),
    ("chi", "\x003C7"),
    ("cir", "\x025CB"),
    ("cirE", "\x029C3"),
    ("circ", "\x002C6"),
    ("circeq", "\x02257"),
    ("circlearrowleft", "\x021BA"),
    ("circlearrowright", "\x021BB"),
    ("circledR", "\x000AE"),
    ("circledS", "\x024C8"),
    ("circledast", "\x0229B"),
    ("circledcirc", "\x0229A"),
    ("circleddash", "\x0229D"),
    ("cire", "\x02257"),
    ("cirfnint", "\x02A10"),
    ("cirmid", "\x02AEF"),
    ("cirscir", "\x029C2") ]

{-# NOINLINE reftab23 #-}
reftab23 :: [(Text,Text)]
reftab23 =
  [ ("clubs", "\x02663"),
    ("clubsuit", "\x02663"),
    ("colon", "\x0003A"),
    ("colone", "\x02254"),
    ("coloneq", "\x02254"),
    ("comma", "\x0002C"),
    ("commat", "\x00040"),
    ("comp", "\x02201"),
    ("compfn", "\x02218"),
    ("complement", "\x02201"),
    ("complexes", "\x02102"),
    ("cong", "\x02245"),
    ("congdot", "\x02A6D"),
    ("conint", "\x0222E"),
    ("copf", "\x1D554"),
    ("coprod", "\x02210"),
    ("copy", "\x000A9"),
    ("copysr", "\x02117"),
    ("crarr", "\x021B5"),
    ("cross", "\x02717"),
    ("cscr", "\x1D4B8"),
    ("csub", "\x02ACF"),
    ("csube", "\x02AD1"),
    ("csup", "\x02AD0"),
    ("csupe", "\x02AD2"),
    ("ctdot", "\x022EF"),
    ("cudarrl", "\x02938"),
    ("cudarrr", "\x02935"),
    ("cuepr", "\x022DE"),
    ("cuesc", "\x022DF"),
    ("cularr", "\x021B6"),
    ("cularrp", "\x0293D"),
    ("cup", "\x0222A"),
    ("cupbrcap", "\x02A48"),
    ("cupcap", "\x02A46"),
    ("cupcup", "\x02A4A"),
    ("cupdot", "\x0228D") ]

{-# NOINLINE reftab24 #-}
reftab24 :: [(Text,Text)]
reftab24 =
  [ ("cupor", "\x02A45"),
    ("cups", "\x0222A\x0FE00"),
    ("curarr", "\x021B7"),
    ("curarrm", "\x0293C"),
    ("curlyeqprec", "\x022DE"),
    ("curlyeqsucc", "\x022DF"),
    ("curlyvee", "\x022CE"),
    ("curlywedge", "\x022CF"),
    ("curren", "\x000A4"),
    ("curvearrowleft", "\x021B6"),
    ("curvearrowright", "\x021B7"),
    ("cuvee", "\x022CE"),
    ("cuwed", "\x022CF"),
    ("cwconint", "\x02232"),
    ("cwint", "\x02231"),
    ("cylcty", "\x0232D"),
    ("dArr", "\x021D3"),
    ("dHar", "\x02965"),
    ("dagger", "\x02020"),
    ("daleth", "\x02138"),
    ("darr", "\x02193"),
    ("dash", "\x02010"),
    ("dashv", "\x022A3"),
    ("dbkarow", "\x0290F"),
    ("dblac", "\x002DD"),
    ("dcaron", "\x0010F"),
    ("dcy", "\x00434"),
    ("dd", "\x02146"),
    ("ddagger", "\x02021"),
    ("ddarr", "\x021CA"),
    ("ddotseq", "\x02A77"),
    ("deg", "\x000B0"),
    ("delta", "\x003B4"),
    ("demptyv", "\x029B1"),
    ("dfisht", "\x0297F"),
    ("dfr", "\x1D521"),
    ("dharl", "\x021C3") ]

{-# NOINLINE reftab25 #-}
reftab25 :: [(Text,Text)]
reftab25 =
  [ ("dharr", "\x021C2"),
    ("diam", "\x022C4"),
    ("diamond", "\x022C4"),
    ("diamondsuit", "\x02666"),
    ("diams", "\x02666"),
    ("die", "\x000A8"),
    ("digamma", "\x003DD"),
    ("disin", "\x022F2"),
    ("div", "\x000F7"),
    ("divide", "\x000F7"),
    ("divideontimes", "\x022C7"),
    ("divonx", "\x022C7"),
    ("djcy", "\x00452"),
    ("dlcorn", "\x0231E"),
    ("dlcrop", "\x0230D"),
    ("dollar", "\x00024"),
    ("dopf", "\x1D555"),
    ("dot", "\x002D9"),
    ("doteq", "\x02250"),
    ("doteqdot", "\x02251"),
    ("dotminus", "\x02238"),
    ("dotplus", "\x02214"),
    ("dotsquare", "\x022A1"),
    ("doublebarwedge", "\x02306"),
    ("downarrow", "\x02193"),
    ("downdownarrows", "\x021CA"),
    ("downharpoonleft", "\x021C3"),
    ("downharpoonright", "\x021C2"),
    ("drbkarow", "\x02910"),
    ("drcorn", "\x0231F"),
    ("drcrop", "\x0230C"),
    ("dscr", "\x1D4B9"),
    ("dscy", "\x00455"),
    ("dsol", "\x029F6"),
    ("dstrok", "\x00111"),
    ("dtdot", "\x022F1"),
    ("dtri", "\x025BF") ]

{-# NOINLINE reftab26 #-}
reftab26 :: [(Text,Text)]
reftab26 =
  [ ("dtrif", "\x025BE"),
    ("duarr", "\x021F5"),
    ("duhar", "\x0296F"),
    ("dwangle", "\x029A6"),
    ("dzcy", "\x0045F"),
    ("dzigrarr", "\x027FF"),
    ("eDDot", "\x02A77"),
    ("eDot", "\x02251"),
    ("eacute", "\x000E9"),
    ("easter", "\x02A6E"),
    ("ecaron", "\x0011B"),
    ("ecir", "\x02256"),
    ("ecirc", "\x000EA"),
    ("ecolon", "\x02255"),
    ("ecy", "\x0044D"),
    ("edot", "\x00117"),
    ("ee", "\x02147"),
    ("efDot", "\x02252"),
    ("efr", "\x1D522"),
    ("eg", "\x02A9A"),
    ("egrave", "\x000E8"),
    ("egs", "\x02A96"),
    ("egsdot", "\x02A98"),
    ("el", "\x02A99"),
    ("elinters", "\x023E7"),
    ("ell", "\x02113"),
    ("els", "\x02A95"),
    ("elsdot", "\x02A97"),
    ("emacr", "\x00113"),
    ("empty", "\x02205"),
    ("emptyset", "\x02205"),
    ("emptyv", "\x02205"),
    ("emsp", "\x02003"),
    ("emsp13", "\x02004"),
    ("emsp14", "\x02005"),
    ("eng", "\x0014B"),
    ("ensp", "\x02002") ]

{-# NOINLINE reftab27 #-}
reftab27 :: [(Text,Text)]
reftab27 =
  [ ("eogon", "\x00119"),
    ("eopf", "\x1D556"),
    ("epar", "\x022D5"),
    ("eparsl", "\x029E3"),
    ("eplus", "\x02A71"),
    ("epsi", "\x003B5"),
    ("epsilon", "\x003B5"),
    ("epsiv", "\x003F5"),
    ("eqcirc", "\x02256"),
    ("eqcolon", "\x02255"),
    ("eqsim", "\x02242"),
    ("eqslantgtr", "\x02A96"),
    ("eqslantless", "\x02A95"),
    ("equals", "\x0003D"),
    ("equest", "\x0225F"),
    ("equiv", "\x02261"),
    ("equivDD", "\x02A78"),
    ("eqvparsl", "\x029E5"),
    ("erDot", "\x02253"),
    ("erarr", "\x02971"),
    ("escr", "\x0212F"),
    ("esdot", "\x02250"),
    ("esim", "\x02242"),
    ("eta", "\x003B7"),
    ("eth", "\x000F0"),
    ("euml", "\x000EB"),
    ("euro", "\x020AC"),
    ("excl", "\x00021"),
    ("exist", "\x02203"),
    ("expectation", "\x02130"),
    ("exponentiale", "\x02147"),
    ("fallingdotseq", "\x02252"),
    ("fcy", "\x00444"),
    ("female", "\x02640"),
    ("ffilig", "\x0FB03"),
    ("fflig", "\x0FB00"),
    ("ffllig", "\x0FB04") ]

{-# NOINLINE reftab28 #-}
reftab28 :: [(Text,Text)]
reftab28 =
  [ ("ffr", "\x1D523"),
    ("filig", "\x0FB01"),
    ("fjlig", "\x00066\x0006A"),
    ("flat", "\x0266D"),
    ("fllig", "\x0FB02"),
    ("fltns", "\x025B1"),
    ("fnof", "\x00192"),
    ("fopf", "\x1D557"),
    ("forall", "\x02200"),
    ("fork", "\x022D4"),
    ("forkv", "\x02AD9"),
    ("fpartint", "\x02A0D"),
    ("frac12", "\x000BD"),
    ("frac13", "\x02153"),
    ("frac14", "\x000BC"),
    ("frac15", "\x02155"),
    ("frac16", "\x02159"),
    ("frac18", "\x0215B"),
    ("frac23", "\x02154"),
    ("frac25", "\x02156"),
    ("frac34", "\x000BE"),
    ("frac35", "\x02157"),
    ("frac38", "\x0215C"),
    ("frac45", "\x02158"),
    ("frac56", "\x0215A"),
    ("frac58", "\x0215D"),
    ("frac78", "\x0215E"),
    ("frasl", "\x02044"),
    ("frown", "\x02322"),
    ("fscr", "\x1D4BB"),
    ("gE", "\x02267"),
    ("gEl", "\x02A8C"),
    ("gacute", "\x001F5"),
    ("gamma", "\x003B3"),
    ("gammad", "\x003DD"),
    ("gap", "\x02A86"),
    ("gbreve", "\x0011F") ]

{-# NOINLINE reftab29 #-}
reftab29 :: [(Text,Text)]
reftab29 =
  [ ("gcirc", "\x0011D"),
    ("gcy", "\x00433"),
    ("gdot", "\x00121"),
    ("ge", "\x02265"),
    ("gel", "\x022DB"),
    ("geq", "\x02265"),
    ("geqq", "\x02267"),
    ("geqslant", "\x02A7E"),
    ("ges", "\x02A7E"),
    ("gescc", "\x02AA9"),
    ("gesdot", "\x02A80"),
    ("gesdoto", "\x02A82"),
    ("gesdotol", "\x02A84"),
    ("gesl", "\x022DB\x0FE00"),
    ("gesles", "\x02A94"),
    ("gfr", "\x1D524"),
    ("gg", "\x0226B"),
    ("ggg", "\x022D9"),
    ("gimel", "\x02137"),
    ("gjcy", "\x00453"),
    ("gl", "\x02277"),
    ("glE", "\x02A92"),
    ("gla", "\x02AA5"),
    ("glj", "\x02AA4"),
    ("gnE", "\x02269"),
    ("gnap", "\x02A8A"),
    ("gnapprox", "\x02A8A"),
    ("gne", "\x02A88"),
    ("gneq", "\x02A88"),
    ("gneqq", "\x02269"),
    ("gnsim", "\x022E7"),
    ("gopf", "\x1D558"),
    ("grave", "\x00060"),
    ("gscr", "\x0210A"),
    ("gsim", "\x02273"),
    ("gsime", "\x02A8E"),
    ("gsiml", "\x02A90") ]

{-# NOINLINE reftab30 #-}
reftab30 :: [(Text,Text)]
reftab30 =
  [ ("gt", "\x0003E"),
    ("gtcc", "\x02AA7"),
    ("gtcir", "\x02A7A"),
    ("gtdot", "\x022D7"),
    ("gtlPar", "\x02995"),
    ("gtquest", "\x02A7C"),
    ("gtrapprox", "\x02A86"),
    ("gtrarr", "\x02978"),
    ("gtrdot", "\x022D7"),
    ("gtreqless", "\x022DB"),
    ("gtreqqless", "\x02A8C"),
    ("gtrless", "\x02277"),
    ("gtrsim", "\x02273"),
    ("gvertneqq", "\x02269\x0FE00"),
    ("gvnE", "\x02269\x0FE00"),
    ("hArr", "\x021D4"),
    ("hairsp", "\x0200A"),
    ("half", "\x000BD"),
    ("hamilt", "\x0210B"),
    ("hardcy", "\x0044A"),
    ("harr", "\x02194"),
    ("harrcir", "\x02948"),
    ("harrw", "\x021AD"),
    ("hbar", "\x0210F"),
    ("hcirc", "\x00125"),
    ("hearts", "\x02665"),
    ("heartsuit", "\x02665"),
    ("hellip", "\x02026"),
    ("hercon", "\x022B9"),
    ("hfr", "\x1D525"),
    ("hksearow", "\x02925"),
    ("hkswarow", "\x02926"),
    ("hoarr", "\x021FF"),
    ("homtht", "\x0223B"),
    ("hookleftarrow", "\x021A9"),
    ("hookrightarrow", "\x021AA"),
    ("hopf", "\x1D559") ]

{-# NOINLINE reftab31 #-}
reftab31 :: [(Text,Text)]
reftab31 =
  [ ("horbar", "\x02015"),
    ("hscr", "\x1D4BD"),
    ("hslash", "\x0210F"),
    ("hstrok", "\x00127"),
    ("hybull", "\x02043"),
    ("hyphen", "\x02010"),
    ("iacute", "\x000ED"),
    ("ic", "\x02063"),
    ("icirc", "\x000EE"),
    ("icy", "\x00438"),
    ("iecy", "\x00435"),
    ("iexcl", "\x000A1"),
    ("iff", "\x021D4"),
    ("ifr", "\x1D526"),
    ("igrave", "\x000EC"),
    ("ii", "\x02148"),
    ("iiiint", "\x02A0C"),
    ("iiint", "\x0222D"),
    ("iinfin", "\x029DC"),
    ("iiota", "\x02129"),
    ("ijlig", "\x00133"),
    ("imacr", "\x0012B"),
    ("image", "\x02111"),
    ("imagline", "\x02110"),
    ("imagpart", "\x02111"),
    ("imath", "\x00131"),
    ("imof", "\x022B7"),
    ("imped", "\x001B5"),
    ("in", "\x02208"),
    ("incare", "\x02105"),
    ("infin", "\x0221E"),
    ("infintie", "\x029DD"),
    ("inodot", "\x00131"),
    ("int", "\x0222B"),
    ("intcal", "\x022BA"),
    ("integers", "\x02124"),
    ("intercal", "\x022BA") ]

{-# NOINLINE reftab32 #-}
reftab32 :: [(Text,Text)]
reftab32 =
  [ ("intlarhk", "\x02A17"),
    ("intprod", "\x02A3C"),
    ("iocy", "\x00451"),
    ("iogon", "\x0012F"),
    ("iopf", "\x1D55A"),
    ("iota", "\x003B9"),
    ("iprod", "\x02A3C"),
    ("iquest", "\x000BF"),
    ("iscr", "\x1D4BE"),
    ("isin", "\x02208"),
    ("isinE", "\x022F9"),
    ("isindot", "\x022F5"),
    ("isins", "\x022F4"),
    ("isinsv", "\x022F3"),
    ("isinv", "\x02208"),
    ("it", "\x02062"),
    ("itilde", "\x00129"),
    ("iukcy", "\x00456"),
    ("iuml", "\x000EF"),
    ("jcirc", "\x00135"),
    ("jcy", "\x00439"),
    ("jfr", "\x1D527"),
    ("jmath", "\x00237"),
    ("jopf", "\x1D55B"),
    ("jscr", "\x1D4BF"),
    ("jsercy", "\x00458"),
    ("jukcy", "\x00454"),
    ("kappa", "\x003BA"),
    ("kappav", "\x003F0"),
    ("kcedil", "\x00137"),
    ("kcy", "\x0043A"),
    ("kfr", "\x1D528"),
    ("kgreen", "\x00138"),
    ("khcy", "\x00445"),
    ("kjcy", "\x0045C"),
    ("kopf", "\x1D55C"),
    ("kscr", "\x1D4C0") ]

{-# NOINLINE reftab33 #-}
reftab33 :: [(Text,Text)]
reftab33 =
  [ ("lAarr", "\x021DA"),
    ("lArr", "\x021D0"),
    ("lAtail", "\x0291B"),
    ("lBarr", "\x0290E"),
    ("lE", "\x02266"),
    ("lEg", "\x02A8B"),
    ("lHar", "\x02962"),
    ("lacute", "\x0013A"),
    ("laemptyv", "\x029B4"),
    ("lagran", "\x02112"),
    ("lambda", "\x003BB"),
    ("lang", "\x027E8"),
    ("langd", "\x02991"),
    ("langle", "\x027E8"),
    ("lap", "\x02A85"),
    ("laquo", "\x000AB"),
    ("larr", "\x02190"),
    ("larrb", "\x021E4"),
    ("larrbfs", "\x0291F"),
    ("larrfs", "\x0291D"),
    ("larrhk", "\x021A9"),
    ("larrlp", "\x021AB"),
    ("larrpl", "\x02939"),
    ("larrsim", "\x02973"),
    ("larrtl", "\x021A2"),
    ("lat", "\x02AAB"),
    ("latail", "\x02919"),
    ("late", "\x02AAD"),
    ("lates", "\x02AAD\x0FE00"),
    ("lbarr", "\x0290C"),
    ("lbbrk", "\x02772"),
    ("lbrace", "\x0007B"),
    ("lbrack", "\x0005B"),
    ("lbrke", "\x0298B"),
    ("lbrksld", "\x0298F"),
    ("lbrkslu", "\x0298D"),
    ("lcaron", "\x0013E") ]

{-# NOINLINE reftab34 #-}
reftab34 :: [(Text,Text)]
reftab34 =
  [ ("lcedil", "\x0013C"),
    ("lceil", "\x02308"),
    ("lcub", "\x0007B"),
    ("lcy", "\x0043B"),
    ("ldca", "\x02936"),
    ("ldquo", "\x0201C"),
    ("ldquor", "\x0201E"),
    ("ldrdhar", "\x02967"),
    ("ldrushar", "\x0294B"),
    ("ldsh", "\x021B2"),
    ("le", "\x02264"),
    ("leftarrow", "\x02190"),
    ("leftarrowtail", "\x021A2"),
    ("leftharpoondown", "\x021BD"),
    ("leftharpoonup", "\x021BC"),
    ("leftleftarrows", "\x021C7"),
    ("leftrightarrow", "\x02194"),
    ("leftrightarrows", "\x021C6"),
    ("leftrightharpoons", "\x021CB"),
    ("leftrightsquigarrow", "\x021AD"),
    ("leftthreetimes", "\x022CB"),
    ("leg", "\x022DA"),
    ("leq", "\x02264"),
    ("leqq", "\x02266"),
    ("leqslant", "\x02A7D"),
    ("les", "\x02A7D"),
    ("lescc", "\x02AA8"),
    ("lesdot", "\x02A7F"),
    ("lesdoto", "\x02A81"),
    ("lesdotor", "\x02A83"),
    ("lesg", "\x022DA\x0FE00"),
    ("lesges", "\x02A93"),
    ("lessapprox", "\x02A85"),
    ("lessdot", "\x022D6"),
    ("lesseqgtr", "\x022DA"),
    ("lesseqqgtr", "\x02A8B"),
    ("lessgtr", "\x02276") ]

{-# NOINLINE reftab35 #-}
reftab35 :: [(Text,Text)]
reftab35 =
  [ ("lesssim", "\x02272"),
    ("lfisht", "\x0297C"),
    ("lfloor", "\x0230A"),
    ("lfr", "\x1D529"),
    ("lg", "\x02276"),
    ("lgE", "\x02A91"),
    ("lhard", "\x021BD"),
    ("lharu", "\x021BC"),
    ("lharul", "\x0296A"),
    ("lhblk", "\x02584"),
    ("ljcy", "\x00459"),
    ("ll", "\x0226A"),
    ("llarr", "\x021C7"),
    ("llcorner", "\x0231E"),
    ("llhard", "\x0296B"),
    ("lltri", "\x025FA"),
    ("lmidot", "\x00140"),
    ("lmoust", "\x023B0"),
    ("lmoustache", "\x023B0"),
    ("lnE", "\x02268"),
    ("lnap", "\x02A89"),
    ("lnapprox", "\x02A89"),
    ("lne", "\x02A87"),
    ("lneq", "\x02A87"),
    ("lneqq", "\x02268"),
    ("lnsim", "\x022E6"),
    ("loang", "\x027EC"),
    ("loarr", "\x021FD"),
    ("lobrk", "\x027E6"),
    ("longleftarrow", "\x027F5"),
    ("longleftrightarrow", "\x027F7"),
    ("longmapsto", "\x027FC"),
    ("longrightarrow", "\x027F6"),
    ("looparrowleft", "\x021AB"),
    ("looparrowright", "\x021AC"),
    ("lopar", "\x02985"),
    ("lopf", "\x1D55D") ]

{-# NOINLINE reftab36 #-}
reftab36 :: [(Text,Text)]
reftab36 =
  [ ("loplus", "\x02A2D"),
    ("lotimes", "\x02A34"),
    ("lowast", "\x02217"),
    ("lowbar", "\x0005F"),
    ("loz", "\x025CA"),
    ("lozenge", "\x025CA"),
    ("lozf", "\x029EB"),
    ("lpar", "\x00028"),
    ("lparlt", "\x02993"),
    ("lrarr", "\x021C6"),
    ("lrcorner", "\x0231F"),
    ("lrhar", "\x021CB"),
    ("lrhard", "\x0296D"),
    ("lrm", "\x0200E"),
    ("lrtri", "\x022BF"),
    ("lsaquo", "\x02039"),
    ("lscr", "\x1D4C1"),
    ("lsh", "\x021B0"),
    ("lsim", "\x02272"),
    ("lsime", "\x02A8D"),
    ("lsimg", "\x02A8F"),
    ("lsqb", "\x0005B"),
    ("lsquo", "\x02018"),
    ("lsquor", "\x0201A"),
    ("lstrok", "\x00142"),
    ("lt", "\x0003C"),
    ("ltcc", "\x02AA6"),
    ("ltcir", "\x02A79"),
    ("ltdot", "\x022D6"),
    ("lthree", "\x022CB"),
    ("ltimes", "\x022C9"),
    ("ltlarr", "\x02976"),
    ("ltquest", "\x02A7B"),
    ("ltrPar", "\x02996"),
    ("ltri", "\x025C3"),
    ("ltrie", "\x022B4"),
    ("ltrif", "\x025C2") ]

{-# NOINLINE reftab37 #-}
reftab37 :: [(Text,Text)]
reftab37 =
  [ ("lurdshar", "\x0294A"),
    ("luruhar", "\x02966"),
    ("lvertneqq", "\x02268\x0FE00"),
    ("lvnE", "\x02268\x0FE00"),
    ("mDDot", "\x0223A"),
    ("macr", "\x000AF"),
    ("male", "\x02642"),
    ("malt", "\x02720"),
    ("maltese", "\x02720"),
    ("map", "\x021A6"),
    ("mapsto", "\x021A6"),
    ("mapstodown", "\x021A7"),
    ("mapstoleft", "\x021A4"),
    ("mapstoup", "\x021A5"),
    ("marker", "\x025AE"),
    ("mcomma", "\x02A29"),
    ("mcy", "\x0043C"),
    ("mdash", "\x02014"),
    ("measuredangle", "\x02221"),
    ("mfr", "\x1D52A"),
    ("mho", "\x02127"),
    ("micro", "\x000B5"),
    ("mid", "\x02223"),
    ("midast", "\x0002A"),
    ("midcir", "\x02AF0"),
    ("middot", "\x000B7"),
    ("minus", "\x02212"),
    ("minusb", "\x0229F"),
    ("minusd", "\x02238"),
    ("minusdu", "\x02A2A"),
    ("mlcp", "\x02ADB"),
    ("mldr", "\x02026"),
    ("mnplus", "\x02213"),
    ("models", "\x022A7"),
    ("mopf", "\x1D55E"),
    ("mp", "\x02213"),
    ("mscr", "\x1D4C2") ]

{-# NOINLINE reftab38 #-}
reftab38 :: [(Text,Text)]
reftab38 =
  [ ("mstpos", "\x0223E"),
    ("mu", "\x003BC"),
    ("multimap", "\x022B8"),
    ("mumap", "\x022B8"),
    ("nGg", "\x022D9\x00338"),
    ("nGt", "\x0226B\x020D2"),
    ("nGtv", "\x0226B\x00338"),
    ("nLeftarrow", "\x021CD"),
    ("nLeftrightarrow", "\x021CE"),
    ("nLl", "\x022D8\x00338"),
    ("nLt", "\x0226A\x020D2"),
    ("nLtv", "\x0226A\x00338"),
    ("nRightarrow", "\x021CF"),
    ("nVDash", "\x022AF"),
    ("nVdash", "\x022AE"),
    ("nabla", "\x02207"),
    ("nacute", "\x00144"),
    ("nang", "\x02220\x020D2"),
    ("nap", "\x02249"),
    ("napE", "\x02A70\x00338"),
    ("napid", "\x0224B\x00338"),
    ("napos", "\x00149"),
    ("napprox", "\x02249"),
    ("natur", "\x0266E"),
    ("natural", "\x0266E"),
    ("naturals", "\x02115"),
    ("nbsp", "\x000A0"),
    ("nbump", "\x0224E\x00338"),
    ("nbumpe", "\x0224F\x00338"),
    ("ncap", "\x02A43"),
    ("ncaron", "\x00148"),
    ("ncedil", "\x00146"),
    ("ncong", "\x02247"),
    ("ncongdot", "\x02A6D\x00338"),
    ("ncup", "\x02A42"),
    ("ncy", "\x0043D"),
    ("ndash", "\x02013") ]

{-# NOINLINE reftab39 #-}
reftab39 :: [(Text,Text)]
reftab39 =
  [ ("ne", "\x02260"),
    ("neArr", "\x021D7"),
    ("nearhk", "\x02924"),
    ("nearr", "\x02197"),
    ("nearrow", "\x02197"),
    ("nedot", "\x02250\x00338"),
    ("nequiv", "\x02262"),
    ("nesear", "\x02928"),
    ("nesim", "\x02242\x00338"),
    ("nexist", "\x02204"),
    ("nexists", "\x02204"),
    ("nfr", "\x1D52B"),
    ("ngE", "\x02267\x00338"),
    ("nge", "\x02271"),
    ("ngeq", "\x02271"),
    ("ngeqq", "\x02267\x00338"),
    ("ngeqslant", "\x02A7E\x00338"),
    ("nges", "\x02A7E\x00338"),
    ("ngsim", "\x02275"),
    ("ngt", "\x0226F"),
    ("ngtr", "\x0226F"),
    ("nhArr", "\x021CE"),
    ("nharr", "\x021AE"),
    ("nhpar", "\x02AF2"),
    ("ni", "\x0220B"),
    ("nis", "\x022FC"),
    ("nisd", "\x022FA"),
    ("niv", "\x0220B"),
    ("njcy", "\x0045A"),
    ("nlArr", "\x021CD"),
    ("nlE", "\x02266\x00338"),
    ("nlarr", "\x0219A"),
    ("nldr", "\x02025"),
    ("nle", "\x02270"),
    ("nleftarrow", "\x0219A"),
    ("nleftrightarrow", "\x021AE"),
    ("nleq", "\x02270") ]

{-# NOINLINE reftab40 #-}
reftab40 :: [(Text,Text)]
reftab40 =
  [ ("nleqq", "\x02266\x00338"),
    ("nleqslant", "\x02A7D\x00338"),
    ("nles", "\x02A7D\x00338"),
    ("nless", "\x0226E"),
    ("nlsim", "\x02274"),
    ("nlt", "\x0226E"),
    ("nltri", "\x022EA"),
    ("nltrie", "\x022EC"),
    ("nmid", "\x02224"),
    ("nopf", "\x1D55F"),
    ("not", "\x000AC"),
    ("notin", "\x02209"),
    ("notinE", "\x022F9\x00338"),
    ("notindot", "\x022F5\x00338"),
    ("notinva", "\x02209"),
    ("notinvb", "\x022F7"),
    ("notinvc", "\x022F6"),
    ("notni", "\x0220C"),
    ("notniva", "\x0220C"),
    ("notnivb", "\x022FE"),
    ("notnivc", "\x022FD"),
    ("npar", "\x02226"),
    ("nparallel", "\x02226"),
    ("nparsl", "\x02AFD\x020E5"),
    ("npart", "\x02202\x00338"),
    ("npolint", "\x02A14"),
    ("npr", "\x02280"),
    ("nprcue", "\x022E0"),
    ("npre", "\x02AAF\x00338"),
    ("nprec", "\x02280"),
    ("npreceq", "\x02AAF\x00338"),
    ("nrArr", "\x021CF"),
    ("nrarr", "\x0219B"),
    ("nrarrc", "\x02933\x00338"),
    ("nrarrw", "\x0219D\x00338"),
    ("nrightarrow", "\x0219B"),
    ("nrtri", "\x022EB") ]

{-# NOINLINE reftab41 #-}
reftab41 :: [(Text,Text)]
reftab41 =
  [ ("nrtrie", "\x022ED"),
    ("nsc", "\x02281"),
    ("nsccue", "\x022E1"),
    ("nsce", "\x02AB0\x00338"),
    ("nscr", "\x1D4C3"),
    ("nshortmid", "\x02224"),
    ("nshortparallel", "\x02226"),
    ("nsim", "\x02241"),
    ("nsime", "\x02244"),
    ("nsimeq", "\x02244"),
    ("nsmid", "\x02224"),
    ("nspar", "\x02226"),
    ("nsqsube", "\x022E2"),
    ("nsqsupe", "\x022E3"),
    ("nsub", "\x02284"),
    ("nsubE", "\x02AC5\x00338"),
    ("nsube", "\x02288"),
    ("nsubset", "\x02282\x020D2"),
    ("nsubseteq", "\x02288"),
    ("nsubseteqq", "\x02AC5\x00338"),
    ("nsucc", "\x02281"),
    ("nsucceq", "\x02AB0\x00338"),
    ("nsup", "\x02285"),
    ("nsupE", "\x02AC6\x00338"),
    ("nsupe", "\x02289"),
    ("nsupset", "\x02283\x020D2"),
    ("nsupseteq", "\x02289"),
    ("nsupseteqq", "\x02AC6\x00338"),
    ("ntgl", "\x02279"),
    ("ntilde", "\x000F1"),
    ("ntlg", "\x02278"),
    ("ntriangleleft", "\x022EA"),
    ("ntrianglelefteq", "\x022EC"),
    ("ntriangleright", "\x022EB"),
    ("ntrianglerighteq", "\x022ED"),
    ("nu", "\x003BD"),
    ("num", "\x00023") ]

{-# NOINLINE reftab42 #-}
reftab42 :: [(Text,Text)]
reftab42 =
  [ ("numero", "\x02116"),
    ("numsp", "\x02007"),
    ("nvDash", "\x022AD"),
    ("nvHarr", "\x02904"),
    ("nvap", "\x0224D\x020D2"),
    ("nvdash", "\x022AC"),
    ("nvge", "\x02265\x020D2"),
    ("nvgt", "\x0003E\x020D2"),
    ("nvinfin", "\x029DE"),
    ("nvlArr", "\x02902"),
    ("nvle", "\x02264\x020D2"),
    ("nvlt", "\x0003C\x020D2"),
    ("nvltrie", "\x022B4\x020D2"),
    ("nvrArr", "\x02903"),
    ("nvrtrie", "\x022B5\x020D2"),
    ("nvsim", "\x0223C\x020D2"),
    ("nwArr", "\x021D6"),
    ("nwarhk", "\x02923"),
    ("nwarr", "\x02196"),
    ("nwarrow", "\x02196"),
    ("nwnear", "\x02927"),
    ("oS", "\x024C8"),
    ("oacute", "\x000F3"),
    ("oast", "\x0229B"),
    ("ocir", "\x0229A"),
    ("ocirc", "\x000F4"),
    ("ocy", "\x0043E"),
    ("odash", "\x0229D"),
    ("odblac", "\x00151"),
    ("odiv", "\x02A38"),
    ("odot", "\x02299"),
    ("odsold", "\x029BC"),
    ("oelig", "\x00153"),
    ("ofcir", "\x029BF"),
    ("ofr", "\x1D52C"),
    ("ogon", "\x002DB"),
    ("ograve", "\x000F2") ]

{-# NOINLINE reftab43 #-}
reftab43 :: [(Text,Text)]
reftab43 =
  [ ("ogt", "\x029C1"),
    ("ohbar", "\x029B5"),
    ("ohm", "\x003A9"),
    ("oint", "\x0222E"),
    ("olarr", "\x021BA"),
    ("olcir", "\x029BE"),
    ("olcross", "\x029BB"),
    ("oline", "\x0203E"),
    ("olt", "\x029C0"),
    ("omacr", "\x0014D"),
    ("omega", "\x003C9"),
    ("omicron", "\x003BF"),
    ("omid", "\x029B6"),
    ("ominus", "\x02296"),
    ("oopf", "\x1D560"),
    ("opar", "\x029B7"),
    ("operp", "\x029B9"),
    ("oplus", "\x02295"),
    ("or", "\x02228"),
    ("orarr", "\x021BB"),
    ("ord", "\x02A5D"),
    ("order", "\x02134"),
    ("orderof", "\x02134"),
    ("ordf", "\x000AA"),
    ("ordm", "\x000BA"),
    ("origof", "\x022B6"),
    ("oror", "\x02A56"),
    ("orslope", "\x02A57"),
    ("orv", "\x02A5B"),
    ("oscr", "\x02134"),
    ("oslash", "\x000F8"),
    ("osol", "\x02298"),
    ("otilde", "\x000F5"),
    ("otimes", "\x02297"),
    ("otimesas", "\x02A36"),
    ("ouml", "\x000F6"),
    ("ovbar", "\x0233D") ]

{-# NOINLINE reftab44 #-}
reftab44 :: [(Text,Text)]
reftab44 =
  [ ("par", "\x02225"),
    ("para", "\x000B6"),
    ("parallel", "\x02225"),
    ("parsim", "\x02AF3"),
    ("parsl", "\x02AFD"),
    ("part", "\x02202"),
    ("pcy", "\x0043F"),
    ("percnt", "\x00025"),
    ("period", "\x0002E"),
    ("permil", "\x02030"),
    ("perp", "\x022A5"),
    ("pertenk", "\x02031"),
    ("pfr", "\x1D52D"),
    ("phi", "\x003C6"),
    ("phiv", "\x003D5"),
    ("phmmat", "\x02133"),
    ("phone", "\x0260E"),
    ("pi", "\x003C0"),
    ("pitchfork", "\x022D4"),
    ("piv", "\x003D6"),
    ("planck", "\x0210F"),
    ("planckh", "\x0210E"),
    ("plankv", "\x0210F"),
    ("plus", "\x0002B"),
    ("plusacir", "\x02A23"),
    ("plusb", "\x0229E"),
    ("pluscir", "\x02A22"),
    ("plusdo", "\x02214"),
    ("plusdu", "\x02A25"),
    ("pluse", "\x02A72"),
    ("plusmn", "\x000B1"),
    ("plussim", "\x02A26"),
    ("plustwo", "\x02A27"),
    ("pm", "\x000B1"),
    ("pointint", "\x02A15"),
    ("popf", "\x1D561"),
    ("pound", "\x000A3") ]

{-# NOINLINE reftab45 #-}
reftab45 :: [(Text,Text)]
reftab45 =
  [ ("pr", "\x0227A"),
    ("prE", "\x02AB3"),
    ("prap", "\x02AB7"),
    ("prcue", "\x0227C"),
    ("pre", "\x02AAF"),
    ("prec", "\x0227A"),
    ("precapprox", "\x02AB7"),
    ("preccurlyeq", "\x0227C"),
    ("preceq", "\x02AAF"),
    ("precnapprox", "\x02AB9"),
    ("precneqq", "\x02AB5"),
    ("precnsim", "\x022E8"),
    ("precsim", "\x0227E"),
    ("prime", "\x02032"),
    ("primes", "\x02119"),
    ("prnE", "\x02AB5"),
    ("prnap", "\x02AB9"),
    ("prnsim", "\x022E8"),
    ("prod", "\x0220F"),
    ("profalar", "\x0232E"),
    ("profline", "\x02312"),
    ("profsurf", "\x02313"),
    ("prop", "\x0221D"),
    ("propto", "\x0221D"),
    ("prsim", "\x0227E"),
    ("prurel", "\x022B0"),
    ("pscr", "\x1D4C5"),
    ("psi", "\x003C8"),
    ("puncsp", "\x02008"),
    ("qfr", "\x1D52E"),
    ("qint", "\x02A0C"),
    ("qopf", "\x1D562"),
    ("qprime", "\x02057"),
    ("qscr", "\x1D4C6"),
    ("quaternions", "\x0210D"),
    ("quatint", "\x02A16"),
    ("quest", "\x0003F") ]

{-# NOINLINE reftab46 #-}
reftab46 :: [(Text,Text)]
reftab46 =
  [ ("questeq", "\x0225F"),
    ("quot", "\x00022"),
    ("rAarr", "\x021DB"),
    ("rArr", "\x021D2"),
    ("rAtail", "\x0291C"),
    ("rBarr", "\x0290F"),
    ("rHar", "\x02964"),
    ("race", "\x0223D\x00331"),
    ("racute", "\x00155"),
    ("radic", "\x0221A"),
    ("raemptyv", "\x029B3"),
    ("rang", "\x027E9"),
    ("rangd", "\x02992"),
    ("range", "\x029A5"),
    ("rangle", "\x027E9"),
    ("raquo", "\x000BB"),
    ("rarr", "\x02192"),
    ("rarrap", "\x02975"),
    ("rarrb", "\x021E5"),
    ("rarrbfs", "\x02920"),
    ("rarrc", "\x02933"),
    ("rarrfs", "\x0291E"),
    ("rarrhk", "\x021AA"),
    ("rarrlp", "\x021AC"),
    ("rarrpl", "\x02945"),
    ("rarrsim", "\x02974"),
    ("rarrtl", "\x021A3"),
    ("rarrw", "\x0219D"),
    ("ratail", "\x0291A"),
    ("ratio", "\x02236"),
    ("rationals", "\x0211A"),
    ("rbarr", "\x0290D"),
    ("rbbrk", "\x02773"),
    ("rbrace", "\x0007D"),
    ("rbrack", "\x0005D"),
    ("rbrke", "\x0298C"),
    ("rbrksld", "\x0298E") ]

{-# NOINLINE reftab47 #-}
reftab47 :: [(Text,Text)]
reftab47 =
  [ ("rbrkslu", "\x02990"),
    ("rcaron", "\x00159"),
    ("rcedil", "\x00157"),
    ("rceil", "\x02309"),
    ("rcub", "\x0007D"),
    ("rcy", "\x00440"),
    ("rdca", "\x02937"),
    ("rdldhar", "\x02969"),
    ("rdquo", "\x0201D"),
    ("rdquor", "\x0201D"),
    ("rdsh", "\x021B3"),
    ("real", "\x0211C"),
    ("realine", "\x0211B"),
    ("realpart", "\x0211C"),
    ("reals", "\x0211D"),
    ("rect", "\x025AD"),
    ("reg", "\x000AE"),
    ("rfisht", "\x0297D"),
    ("rfloor", "\x0230B"),
    ("rfr", "\x1D52F"),
    ("rhard", "\x021C1"),
    ("rharu", "\x021C0"),
    ("rharul", "\x0296C"),
    ("rho", "\x003C1"),
    ("rhov", "\x003F1"),
    ("rightarrow", "\x02192"),
    ("rightarrowtail", "\x021A3"),
    ("rightharpoondown", "\x021C1"),
    ("rightharpoonup", "\x021C0"),
    ("rightleftarrows", "\x021C4"),
    ("rightleftharpoons", "\x021CC"),
    ("rightrightarrows", "\x021C9"),
    ("rightsquigarrow", "\x0219D"),
    ("rightthreetimes", "\x022CC"),
    ("ring", "\x002DA"),
    ("risingdotseq", "\x02253"),
    ("rlarr", "\x021C4") ]

{-# NOINLINE reftab48 #-}
reftab48 :: [(Text,Text)]
reftab48 =
  [ ("rlhar", "\x021CC"),
    ("rlm", "\x0200F"),
    ("rmoust", "\x023B1"),
    ("rmoustache", "\x023B1"),
    ("rnmid", "\x02AEE"),
    ("roang", "\x027ED"),
    ("roarr", "\x021FE"),
    ("robrk", "\x027E7"),
    ("ropar", "\x02986"),
    ("ropf", "\x1D563"),
    ("roplus", "\x02A2E"),
    ("rotimes", "\x02A35"),
    ("rpar", "\x00029"),
    ("rpargt", "\x02994"),
    ("rppolint", "\x02A12"),
    ("rrarr", "\x021C9"),
    ("rsaquo", "\x0203A"),
    ("rscr", "\x1D4C7"),
    ("rsh", "\x021B1"),
    ("rsqb", "\x0005D"),
    ("rsquo", "\x02019"),
    ("rsquor", "\x02019"),
    ("rthree", "\x022CC"),
    ("rtimes", "\x022CA"),
    ("rtri", "\x025B9"),
    ("rtrie", "\x022B5"),
    ("rtrif", "\x025B8"),
    ("rtriltri", "\x029CE"),
    ("ruluhar", "\x02968"),
    ("rx", "\x0211E"),
    ("sacute", "\x0015B"),
    ("sbquo", "\x0201A"),
    ("sc", "\x0227B"),
    ("scE", "\x02AB4"),
    ("scap", "\x02AB8"),
    ("scaron", "\x00161"),
    ("sccue", "\x0227D") ]

{-# NOINLINE reftab49 #-}
reftab49 :: [(Text,Text)]
reftab49 =
  [ ("sce", "\x02AB0"),
    ("scedil", "\x0015F"),
    ("scirc", "\x0015D"),
    ("scnE", "\x02AB6"),
    ("scnap", "\x02ABA"),
    ("scnsim", "\x022E9"),
    ("scpolint", "\x02A13"),
    ("scsim", "\x0227F"),
    ("scy", "\x00441"),
    ("sdot", "\x022C5"),
    ("sdotb", "\x022A1"),
    ("sdote", "\x02A66"),
    ("seArr", "\x021D8"),
    ("searhk", "\x02925"),
    ("searr", "\x02198"),
    ("searrow", "\x02198"),
    ("sect", "\x000A7"),
    ("semi", "\x0003B"),
    ("seswar", "\x02929"),
    ("setminus", "\x02216"),
    ("setmn", "\x02216"),
    ("sext", "\x02736"),
    ("sfr", "\x1D530"),
    ("sfrown", "\x02322"),
    ("sharp", "\x0266F"),
    ("shchcy", "\x00449"),
    ("shcy", "\x00448"),
    ("shortmid", "\x02223"),
    ("shortparallel", "\x02225"),
    ("shy", "\x000AD"),
    ("sigma", "\x003C3"),
    ("sigmaf", "\x003C2"),
    ("sigmav", "\x003C2"),
    ("sim", "\x0223C"),
    ("simdot", "\x02A6A"),
    ("sime", "\x02243"),
    ("simeq", "\x02243") ]

{-# NOINLINE reftab50 #-}
reftab50 :: [(Text,Text)]
reftab50 =
  [ ("simg", "\x02A9E"),
    ("simgE", "\x02AA0"),
    ("siml", "\x02A9D"),
    ("simlE", "\x02A9F"),
    ("simne", "\x02246"),
    ("simplus", "\x02A24"),
    ("simrarr", "\x02972"),
    ("slarr", "\x02190"),
    ("smallsetminus", "\x02216"),
    ("smashp", "\x02A33"),
    ("smeparsl", "\x029E4"),
    ("smid", "\x02223"),
    ("smile", "\x02323"),
    ("smt", "\x02AAA"),
    ("smte", "\x02AAC"),
    ("smtes", "\x02AAC\x0FE00"),
    ("softcy", "\x0044C"),
    ("sol", "\x0002F"),
    ("solb", "\x029C4"),
    ("solbar", "\x0233F"),
    ("sopf", "\x1D564"),
    ("spades", "\x02660"),
    ("spadesuit", "\x02660"),
    ("spar", "\x02225"),
    ("sqcap", "\x02293"),
    ("sqcaps", "\x02293\x0FE00"),
    ("sqcup", "\x02294"),
    ("sqcups", "\x02294\x0FE00"),
    ("sqsub", "\x0228F"),
    ("sqsube", "\x02291"),
    ("sqsubset", "\x0228F"),
    ("sqsubseteq", "\x02291"),
    ("sqsup", "\x02290"),
    ("sqsupe", "\x02292"),
    ("sqsupset", "\x02290"),
    ("sqsupseteq", "\x02292"),
    ("squ", "\x025A1") ]

{-# NOINLINE reftab51 #-}
reftab51 :: [(Text,Text)]
reftab51 =
  [ ("square", "\x025A1"),
    ("squarf", "\x025AA"),
    ("squf", "\x025AA"),
    ("srarr", "\x02192"),
    ("sscr", "\x1D4C8"),
    ("ssetmn", "\x02216"),
    ("ssmile", "\x02323"),
    ("sstarf", "\x022C6"),
    ("star", "\x02606"),
    ("starf", "\x02605"),
    ("straightepsilon", "\x003F5"),
    ("straightphi", "\x003D5"),
    ("strns", "\x000AF"),
    ("sub", "\x02282"),
    ("subE", "\x02AC5"),
    ("subdot", "\x02ABD"),
    ("sube", "\x02286"),
    ("subedot", "\x02AC3"),
    ("submult", "\x02AC1"),
    ("subnE", "\x02ACB"),
    ("subne", "\x0228A"),
    ("subplus", "\x02ABF"),
    ("subrarr", "\x02979"),
    ("subset", "\x02282"),
    ("subseteq", "\x02286"),
    ("subseteqq", "\x02AC5"),
    ("subsetneq", "\x0228A"),
    ("subsetneqq", "\x02ACB"),
    ("subsim", "\x02AC7"),
    ("subsub", "\x02AD5"),
    ("subsup", "\x02AD3"),
    ("succ", "\x0227B"),
    ("succapprox", "\x02AB8"),
    ("succcurlyeq", "\x0227D"),
    ("succeq", "\x02AB0"),
    ("succnapprox", "\x02ABA"),
    ("succneqq", "\x02AB6") ]

{-# NOINLINE reftab52 #-}
reftab52 :: [(Text,Text)]
reftab52 =
  [ ("succnsim", "\x022E9"),
    ("succsim", "\x0227F"),
    ("sum", "\x02211"),
    ("sung", "\x0266A"),
    ("sup", "\x02283"),
    ("sup1", "\x000B9"),
    ("sup2", "\x000B2"),
    ("sup3", "\x000B3"),
    ("supE", "\x02AC6"),
    ("supdot", "\x02ABE"),
    ("supdsub", "\x02AD8"),
    ("supe", "\x02287"),
    ("supedot", "\x02AC4"),
    ("suphsol", "\x027C9"),
    ("suphsub", "\x02AD7"),
    ("suplarr", "\x0297B"),
    ("supmult", "\x02AC2"),
    ("supnE", "\x02ACC"),
    ("supne", "\x0228B"),
    ("supplus", "\x02AC0"),
    ("supset", "\x02283"),
    ("supseteq", "\x02287"),
    ("supseteqq", "\x02AC6"),
    ("supsetneq", "\x0228B"),
    ("supsetneqq", "\x02ACC"),
    ("supsim", "\x02AC8"),
    ("supsub", "\x02AD4"),
    ("supsup", "\x02AD6"),
    ("swArr", "\x021D9"),
    ("swarhk", "\x02926"),
    ("swarr", "\x02199"),
    ("swarrow", "\x02199"),
    ("swnwar", "\x0292A"),
    ("szlig", "\x000DF"),
    ("target", "\x02316"),
    ("tau", "\x003C4"),
    ("tbrk", "\x023B4") ]

{-# NOINLINE reftab53 #-}
reftab53 :: [(Text,Text)]
reftab53 =
  [ ("tcaron", "\x00165"),
    ("tcedil", "\x00163"),
    ("tcy", "\x00442"),
    ("tdot", "\x020DB"),
    ("telrec", "\x02315"),
    ("tfr", "\x1D531"),
    ("there4", "\x02234"),
    ("therefore", "\x02234"),
    ("theta", "\x003B8"),
    ("thetasym", "\x003D1"),
    ("thetav", "\x003D1"),
    ("thickapprox", "\x02248"),
    ("thicksim", "\x0223C"),
    ("thinsp", "\x02009"),
    ("thkap", "\x02248"),
    ("thksim", "\x0223C"),
    ("thorn", "\x000FE"),
    ("tilde", "\x002DC"),
    ("times", "\x000D7"),
    ("timesb", "\x022A0"),
    ("timesbar", "\x02A31"),
    ("timesd", "\x02A30"),
    ("tint", "\x0222D"),
    ("toea", "\x02928"),
    ("top", "\x022A4"),
    ("topbot", "\x02336"),
    ("topcir", "\x02AF1"),
    ("topf", "\x1D565"),
    ("topfork", "\x02ADA"),
    ("tosa", "\x02929"),
    ("tprime", "\x02034"),
    ("trade", "\x02122"),
    ("triangle", "\x025B5"),
    ("triangledown", "\x025BF"),
    ("triangleleft", "\x025C3"),
    ("trianglelefteq", "\x022B4"),
    ("triangleq", "\x0225C") ]

{-# NOINLINE reftab54 #-}
reftab54 :: [(Text,Text)]
reftab54 =
  [ ("triangleright", "\x025B9"),
    ("trianglerighteq", "\x022B5"),
    ("tridot", "\x025EC"),
    ("trie", "\x0225C"),
    ("triminus", "\x02A3A"),
    ("triplus", "\x02A39"),
    ("trisb", "\x029CD"),
    ("tritime", "\x02A3B"),
    ("trpezium", "\x023E2"),
    ("tscr", "\x1D4C9"),
    ("tscy", "\x00446"),
    ("tshcy", "\x0045B"),
    ("tstrok", "\x00167"),
    ("twixt", "\x0226C"),
    ("twoheadleftarrow", "\x0219E"),
    ("twoheadrightarrow", "\x021A0"),
    ("uArr", "\x021D1"),
    ("uHar", "\x02963"),
    ("uacute", "\x000FA"),
    ("uarr", "\x02191"),
    ("ubrcy", "\x0045E"),
    ("ubreve", "\x0016D"),
    ("ucirc", "\x000FB"),
    ("ucy", "\x00443"),
    ("udarr", "\x021C5"),
    ("udblac", "\x00171"),
    ("udhar", "\x0296E"),
    ("ufisht", "\x0297E"),
    ("ufr", "\x1D532"),
    ("ugrave", "\x000F9"),
    ("uharl", "\x021BF"),
    ("uharr", "\x021BE"),
    ("uhblk", "\x02580"),
    ("ulcorn", "\x0231C"),
    ("ulcorner", "\x0231C"),
    ("ulcrop", "\x0230F"),
    ("ultri", "\x025F8") ]

{-# NOINLINE reftab55 #-}
reftab55 :: [(Text,Text)]
reftab55 =
  [ ("umacr", "\x0016B"),
    ("uml", "\x000A8"),
    ("uogon", "\x00173"),
    ("uopf", "\x1D566"),
    ("uparrow", "\x02191"),
    ("updownarrow", "\x02195"),
    ("upharpoonleft", "\x021BF"),
    ("upharpoonright", "\x021BE"),
    ("uplus", "\x0228E"),
    ("upsi", "\x003C5"),
    ("upsih", "\x003D2"),
    ("upsilon", "\x003C5"),
    ("upuparrows", "\x021C8"),
    ("urcorn", "\x0231D"),
    ("urcorner", "\x0231D"),
    ("urcrop", "\x0230E"),
    ("uring", "\x0016F"),
    ("urtri", "\x025F9"),
    ("uscr", "\x1D4CA"),
    ("utdot", "\x022F0"),
    ("utilde", "\x00169"),
    ("utri", "\x025B5"),
    ("utrif", "\x025B4"),
    ("uuarr", "\x021C8"),
    ("uuml", "\x000FC"),
    ("uwangle", "\x029A7"),
    ("vArr", "\x021D5"),
    ("vBar", "\x02AE8"),
    ("vBarv", "\x02AE9"),
    ("vDash", "\x022A8"),
    ("vangrt", "\x0299C"),
    ("varepsilon", "\x003F5"),
    ("varkappa", "\x003F0"),
    ("varnothing", "\x02205"),
    ("varphi", "\x003D5"),
    ("varpi", "\x003D6"),
    ("varpropto", "\x0221D") ]

{-# NOINLINE reftab56 #-}
reftab56 :: [(Text,Text)]
reftab56 =
  [ ("varr", "\x02195"),
    ("varrho", "\x003F1"),
    ("varsigma", "\x003C2"),
    ("varsubsetneq", "\x0228A\x0FE00"),
    ("varsubsetneqq", "\x02ACB\x0FE00"),
    ("varsupsetneq", "\x0228B\x0FE00"),
    ("varsupsetneqq", "\x02ACC\x0FE00"),
    ("vartheta", "\x003D1"),
    ("vartriangleleft", "\x022B2"),
    ("vartriangleright", "\x022B3"),
    ("vcy", "\x00432"),
    ("vdash", "\x022A2"),
    ("vee", "\x02228"),
    ("veebar", "\x022BB"),
    ("veeeq", "\x0225A"),
    ("vellip", "\x022EE"),
    ("verbar", "\x0007C"),
    ("vert", "\x0007C"),
    ("vfr", "\x1D533"),
    ("vltri", "\x022B2"),
    ("vnsub", "\x02282\x020D2"),
    ("vnsup", "\x02283\x020D2"),
    ("vopf", "\x1D567"),
    ("vprop", "\x0221D"),
    ("vrtri", "\x022B3"),
    ("vscr", "\x1D4CB"),
    ("vsubnE", "\x02ACB\x0FE00"),
    ("vsubne", "\x0228A\x0FE00"),
    ("vsupnE", "\x02ACC\x0FE00"),
    ("vsupne", "\x0228B\x0FE00"),
    ("vzigzag", "\x0299A"),
    ("wcirc", "\x00175"),
    ("wedbar", "\x02A5F"),
    ("wedge", "\x02227"),
    ("wedgeq", "\x02259"),
    ("weierp", "\x02118"),
    ("wfr", "\x1D534") ]

{-# NOINLINE reftab57 #-}
reftab57 :: [(Text,Text)]
reftab57 =
  [ ("wopf", "\x1D568"),
    ("wp", "\x02118"),
    ("wr", "\x02240"),
    ("wreath", "\x02240"),
    ("wscr", "\x1D4CC"),
    ("xcap", "\x022C2"),
    ("xcirc", "\x025EF"),
    ("xcup", "\x022C3"),
    ("xdtri", "\x025BD"),
    ("xfr", "\x1D535"),
    ("xhArr", "\x027FA"),
    ("xharr", "\x027F7"),
    ("xi", "\x003BE"),
    ("xlArr", "\x027F8"),
    ("xlarr", "\x027F5"),
    ("xmap", "\x027FC"),
    ("xnis", "\x022FB"),
    ("xodot", "\x02A00"),
    ("xopf", "\x1D569"),
    ("xoplus", "\x02A01"),
    ("xotime", "\x02A02"),
    ("xrArr", "\x027F9"),
    ("xrarr", "\x027F6"),
    ("xscr", "\x1D4CD"),
    ("xsqcup", "\x02A06"),
    ("xuplus", "\x02A04"),
    ("xutri", "\x025B3"),
    ("xvee", "\x022C1"),
    ("xwedge", "\x022C0"),
    ("yacute", "\x000FD"),
    ("yacy", "\x0044F"),
    ("ycirc", "\x00177"),
    ("ycy", "\x0044B"),
    ("yen", "\x000A5"),
    ("yfr", "\x1D536"),
    ("yicy", "\x00457"),
    ("yopf", "\x1D56A") ]

{-# NOINLINE reftab58 #-}
reftab58 :: [(Text,Text)]
reftab58 =
  [ ("yscr", "\x1D4CE"),
    ("yucy", "\x0044E"),
    ("yuml", "\x000FF"),
    ("zacute", "\x0017A"),
    ("zcaron", "\x0017E"),
    ("zcy", "\x00437"),
    ("zdot", "\x0017C"),
    ("zeetrf", "\x02128"),
    ("zeta", "\x003B6"),
    ("zfr", "\x1D537"),
    ("zhcy", "\x00436"),
    ("zigrarr", "\x021DD"),
    ("zopf", "\x1D56B"),
    ("zscr", "\x1D4CF"),
    ("zwj", "\x0200D"),
    ("zwnj", "\x0200C") ]

