#include <grape.h>
#include <alberta.h>

#include <unistd.h>
#include <getopt.h>

#if FE_DIM == 2
typedef HMESH2D GRAPE_MESH;
#define Grape_Mesh HMesh2d
#else
typedef HMESH3D GRAPE_MESH;
#define Grape_Mesh HMesh3d
#endif

extern GRAPE_MESH *setup_grape_mesh(MESH *mesh, char *name);
extern void grape_ini_f_data_d(GRAPE_MESH *, DOF_REAL_D_VEC *);
extern void grape_ini_f_data(GRAPE_MESH *, DOF_REAL_VEC *);

static  SCENE *last_scene = nil, *first_scene = nil;

static void sc_add(MESH *mesh, DOF_REAL_VEC *drv, DOF_REAL_D_VEC *drdv)
{
  FUNCNAME("sc_add");
  const char   *name = nil;
  SCENE        *scene;
  GRAPE_MESH *grape_mesh;

  TEST_EXIT(mesh,"no mesh ???\n");
 
  TEST_EXIT(FE_DIM==mesh->dim,"Mesh dimension must be %d!\n", FE_DIM);
 
  if (drv) name = drv->name;
  else if (drdv) name = drdv->name;
  else name = mesh->name;

  scene = (SCENE *)GRAPE(Scene,"new-instance")(name);
  if (!first_scene) first_scene = scene;
  if (last_scene) last_scene->next_scene = scene;
  last_scene = scene;

  grape_mesh = setup_grape_mesh(mesh, (char *) mesh->name);
  if (drdv)
    grape_ini_f_data_d(grape_mesh, drdv);
  else if (drv)
    grape_ini_f_data(grape_mesh, drv);
  scene->object = (G_SCENE_OBJECT *) grape_mesh;

  return;
}

static void sc_graph(SCENE *sc)
{
  if (sc)
  {
    MANAGER       *mgr;
#ifdef GRID_MODE
    GRAPHICDEVICE *grdev;

    grdev = (GRAPHICDEVICE *)GRAPE(GraphicDevice,"get-stddev")();
    grdev->clear();
    if (grdev && (grdev->grid_patch != G_GRID)) {
      GRAPE(grdev,"grid-patch")(G_GRID);
    }
#endif
    mgr = (MANAGER *)GRAPE(Manager,"get-stdmgr")();
    GRAPE(mgr,"handle")(sc);
  }
}

static void print_help(const char *funcName, FILE *f, int status)
{
  fprintf(f,
"Usage: %s [-p PATH] [OPTIONS]\n"						\
"          -m  MESH [-s DRV] [-v DRDV] [[-m MESH1] [-s DRV1] [-v DRDV1] ...]\n"	\
"\n"										\
"Example:\n"									\
"  %s --mesh=mymesh -s temperature --vector velocity\n"				\
"    where \"mymesh\", \"temperature\" and \"velocity\" are file-names.\n"	\
"\n"										\
"If a long option shows an argument as mandatory, then it is mandatory\n"	\
"for the equivalent short option also.  Similarly for optional arguments.\n"	\
"\n"										\
"The order of the options _is_ significant in the following cases:\n"		\
"`-p PATH' alters the search path for all following data-files.\n"		\
"`-m MESH' specifies a new mesh for all following DRVs and DRDVs (see below)\n"	\
"\n"										\
"Options:\n"									\
"  -m, --mesh=MESH\n"								\
"            The file-name of an ALBERTA-mesh gnereated by the ALBERTA\n"		\
"            library routines `write_mesh()' or `write_mesh_xdr()'\n"		\
"            `-x' and `-b' options below.\n"					\
"            This option is mandatory and may not be omitted. This option\n"	\
"            may be specified multiple times. All following dof-vectors\n"	\
"            given by the `-s' and `-v' options must belong to the most\n"	\
"            recently specified mesh.\n"					\
"  -b, --binary\n"								\
"            Expect MESH, DRV and DRDV to contain data in host dependent\n"	\
"            byte-order, generated by `write_SOMETHING()' routines of the\n"	\
"            ALBERTA library (SOMETHING is `mesh', `dof_real_vec' etc.\n"	\
"  -x, --xdr\n"									\
"            This is the default and just mentioned here for completeness.\n"	\
"            Expect MESH, DRV and DRDV to contain data in network\n"		\
"            byte-order, generated by `write_SOMETHING_xdr()' routines\n"	\
"            of the ALBERTA library. Per convention this means big-endian\n"	\
"            byte-order.\n"							\
"  -s, --scalar=DRV\n"								\
"            Load the data-file DRV which must contain a DOF_REAL_VEC\n"	\
"            dumped to disk by `write_dof_real_vec[_xdr]()'.\n"			\
"            This option may be specified multiple times. The DOF_REAL_VECs\n"	\
"            must belong to the most recently specified mesh.\n"		\
"            See `-m' and `-b' above.\n"					\
"  -v, --vector=DRDV\n"								\
"            Load the data-file DRDV which must contain a DOF_REAL_D_VEC\n"	\
"            dumped to disk by `write_dof_real_d_vec[_xdr]()'.\n"		\
"            This option may be specified multiple times. The vector\n"		\
"            must belong to the most recently specified mesh.\n"		\
"            See `-m' and `-b' above.\n"					\
"  -p, --path=PATH\n"								\
"            Specify a path prefix for all following data-files. This option\n"	\
"            may be specified multiple times. PATH is supposed to be the\n"	\
"            directory containing all data-files specified by the following\n"	\
"            `-m', `-s' and `-v' options.\n"					\
"  -h, --help\n"								\
"            Print this help.\n",
	  funcName, funcName);
  exit(status);
}

static const char *filename(const char *path, const char *fn)
{
  static char  name[1024];
 
  if (!fn) return(nil);
 
  if (path == nil || path[0] == '\0')
  {
    sprintf(name, "./%s", fn);
  }
  else
  {
    const char *cp = path;
    while (*cp)
      cp++;
    cp--;
    if (*cp == '/')
      sprintf(name, "%s%s", path, fn);
    else
      sprintf(name, "%s/%s", path, fn);
  }
  return((const char *) name);
}

static struct option long_options[] = {
  {"help",   0, 0, 'h' },
  {"binary", 0, 0, 'b' },
  {"xdr",    0, 0, 'x' },
  {"path",   1, 0, 'p' },
  {"mesh",   1, 0, 'm' },
  {"scalar", 1, 0, 's' },
  {"vector", 1, 0, 'v' },
  { NULL,    0, 0, '\0' }
};

int main(int argc, char **argv)
{
  FUNCNAME("main");
  MESH           *mesh = nil;
  const char     *mesh_path = nil;
  DOF_REAL_VEC   *drv = nil;
  DOF_REAL_D_VEC *drdv = nil;
  const char     *fn, *path = nil;
  REAL           dummy;
  int            c, option_index;

  MESH *(*rm)(const char *, REAL *,
	      NODE_PROJECTION *(*)(MESH *, MACRO_EL *, int));

  DOF_REAL_VEC   *(*rdrv)(const char *, MESH *, FE_SPACE *);
  DOF_REAL_D_VEC *(*rdrdv)(const char *, MESH *, FE_SPACE *);

  rm = read_mesh_xdr;
  rdrv = read_dof_real_vec_xdr;
  rdrdv = read_dof_real_d_vec_xdr;

  while (1) {
    c = getopt_long(argc, argv, "bhm:p:s:v:x", long_options, &option_index);
    if (c == -1)
      break;
    switch (c) {
    case 'b':
      rm = read_mesh;
      rdrv = read_dof_real_vec;
      rdrdv = read_dof_real_d_vec;
      break;
    case 'h':
      print_help(argv[0], stdout, 0);
      break;
    case 'm':
      mesh_path = optarg;
      fn = filename(path, mesh_path);
      MSG("reading mesh `%s'\n", fn);
      mesh = (*rm)(fn, &dummy, nil);
      TEST_EXIT(mesh,"could not read mesh `%s'!\n", fn);
      break;
    case 'p':
      path = optarg;
      break;
    case 's':
      if (mesh_path == nil) {
	ERROR("a mesh has to be given _FIRST_!\n");
	print_help(argv[0], stderr, 1);
      }
      fn = filename(path, optarg);
      MSG("reading dof_real_vec `%s'\n", fn);
      drv = (*rdrv)(fn, mesh, nil);
      TEST_EXIT(drv,"could not read dof_real_vec `%s'!\n", fn);
      sc_add(mesh, drv, nil);
      break;
    case 'v':
      if (mesh_path == nil) {
	ERROR("a mesh has to be given _FIRST_!\n");
	print_help(argv[0], stderr, 1);
      }
      fn = filename(path, optarg);
      MSG("reading dof_real_d_vec `%s'\n", fn);
      drdv = (*rdrdv)(fn, mesh, nil);
      TEST_EXIT(drdv,"could not read dof_real_d_vec `%s'!\n", fn);
      sc_add(mesh, nil, drdv);
      break;
    case 'x':
      rm = read_mesh_xdr;
      rdrv = read_dof_real_vec_xdr;
      rdrdv = read_dof_real_d_vec_xdr;
      break;
    default:
      print_help(argv[0], stderr, 1);
    }
  }

  if (mesh_path == nil) {
    ERROR("a mesh has to be given!\n\n");
    print_help(argv[0], stderr, 1);
  }

  if (!drv && !drdv)
  {
    sc_add(mesh, nil, nil);
  }

  g_project_add("uif-gm");

  if (first_scene)
    sc_graph(first_scene);

  return(0);
}
