/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     traverse_r_2d.c                                                */
/*                                                                          */
/* description:                                                             */
/*           recursive mesh traversal - 2d routines:                        */
/*           fill_macro_info_2d(), fill_elinfo_2d()                         */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*   fill_macro_info_2d:                              		            */
/*   ----------------                                   		    */
/*   fill EL_INFO structure for a macro element         		    */
/*--------------------------------------------------------------------------*/

static void fill_macro_info_2d(MESH *mesh,
			       const MACRO_EL *mel, EL_INFO *elinfo)
{
  FUNCNAME("fill_macro_info_2d");
  EL       *nb;
  MACRO_EL *mnb;
  int      i, j, k;
  FLAGS    fill_opp_coords;

  elinfo->mesh     = mesh;
  elinfo->macro_el = mel;
  elinfo->el       = mel->el;
  elinfo->parent   = nil;
  elinfo->level    = 0;

  if (elinfo->fill_flag & FILL_COORDS) {

    for (i=0; i<N_VERTICES_2D; i++) {
      DEBUG_TEST_EXIT(mel->coord[i], "no mel->coord[%d]\n",i);

      for (j=0;j<DIM_OF_WORLD;j++) elinfo->coord[i][j] = mel->coord[i][j];
    }
  }

  if (elinfo->fill_flag & (FILL_OPP_COORDS | FILL_NEIGH)) {
    fill_opp_coords = (elinfo->fill_flag & FILL_OPP_COORDS);
    for (i = 0; i < N_NEIGH_2D; i++) 
    {
      if ((mnb = mel->neigh[i])) 
      {
	nb = elinfo->neigh[i]      = mel->neigh[i]->el;
	k  = elinfo->opp_vertex[i] = mel->opp_vertex[i];

	if (nb->child[0] && (k != 2))  /*---  make nb nearest element  ---*/
	{
	  if  (fill_opp_coords) 
	  {
	    if (nb->new_coord)
	    {
	      for (j = 0; j < DIM_OF_WORLD; j++)
		elinfo->opp_coord[i][j] = nb->new_coord[j];
	    }
	    else
	    {
	      for (j = 0; j < DIM_OF_WORLD; j++) 
		elinfo->opp_coord[i][j] = 
		  (mnb->coord[0][j] + mnb->coord[1][j]) / 2;
	    }
	  }
	  nb = elinfo->neigh[i]      = nb->child[1-k];
	  k  = elinfo->opp_vertex[i] = 2;
	}
	else {
	  if  (fill_opp_coords) {
	    for (j = 0; j < DIM_OF_WORLD; j++)
	      elinfo->opp_coord[i][j] = mnb->coord[k][j];
	  }
	}
      }
      else {
	elinfo->neigh[i] = nil;
      }
    }
  }

  if (elinfo->fill_flag & FILL_BOUND) {
    for (i = 0; i < N_VERTICES_2D; i++)
      elinfo->vertex_bound[i] = mel->vertex_bound[i];

    for (i=0;i<N_EDGES_2D; i++)
      elinfo->edge_bound[i] = mel->edge_bound[i];
  }

  if(elinfo->fill_flag & FILL_PROJECTION) {
    for(i = 0; i < N_EDGES_2D + 1; i++)
      elinfo->projections[i] = mel->projection[i];

    elinfo->active_projection = mel->projection[0];

    /* If we have a projection for the refinement edge, it takes precedence! */
    if(mel->projection[3])
      elinfo->active_projection = mel->projection[3];
  }

  return;
}


/*--------------------------------------------------------------------------*/
/*   fill_elinfo_2d:                                  		            */
/*   ------------                                       		    */
/*   fill EL_INFO structure for one child of an element   		    */
/*--------------------------------------------------------------------------*/

static void fill_elinfo_2d(int ichild, const EL_INFO *elinfo_old,
			   EL_INFO *elinfo)
{
  FUNCNAME("fill_elinfo_2d");
  int     j;
  EL      *nb;
  EL      *el = elinfo_old->el;
  FLAGS   fill_flag = elinfo_old->fill_flag;
  FLAGS   fill_opp_coords;

  DEBUG_TEST_EXIT(el->child[0], "no children?\n");
  elinfo->el = el->child[ichild];
  DEBUG_TEST_EXIT(elinfo->el, "missing child %d?\n", ichild);

  elinfo->macro_el  = elinfo_old->macro_el;
  elinfo->fill_flag = fill_flag;
  elinfo->mesh      = elinfo_old->mesh;
  elinfo->parent    = el;
  elinfo->level     = elinfo_old->level + 1;


  if (fill_flag & FILL_COORDS)
  {
    if (el->new_coord) {
      for (j = 0; j < DIM_OF_WORLD; j++)
	elinfo->coord[2][j] = el->new_coord[j];
    }
    else {
      for (j = 0; j < DIM_OF_WORLD; j++)
	elinfo->coord[2][j] = 
	  0.5 * (elinfo_old->coord[0][j] + elinfo_old->coord[1][j]);
    }

    if (ichild==0) {
      for (j = 0; j < DIM_OF_WORLD; j++) {
	elinfo->coord[0][j] = elinfo_old->coord[2][j];
	elinfo->coord[1][j] = elinfo_old->coord[0][j];
      }
    }
    else {
      for (j = 0; j < DIM_OF_WORLD; j++) {
	elinfo->coord[0][j] = elinfo_old->coord[1][j];
	elinfo->coord[1][j] = elinfo_old->coord[2][j];
      }
    }
  }



  if (fill_flag & (FILL_NEIGH | FILL_OPP_COORDS))
  {
    fill_opp_coords = (fill_flag & FILL_OPP_COORDS);
    if (ichild==0) 
    {
      if ((elinfo->neigh[2] = elinfo_old->neigh[1])) 
      {
	if (fill_opp_coords) 
	{
	  for (j=0; j<DIM_OF_WORLD; j++)
	    elinfo->opp_coord[2][j] = elinfo_old->opp_coord[1][j];
	}
      }
      elinfo->opp_vertex[2] = elinfo_old->opp_vertex[1];

      if (el->child[0]  &&  el->child[1]->child[0]  &&  
	  el->child[1]->child[0]) {
	elinfo->neigh[1] = el->child[1]->child[1];
	DEBUG_TEST_EXIT(elinfo->neigh[1],
	     "el->child[1]->child[0]!=nil, but el->child[1]->child[1]=nil\n");

	elinfo->opp_vertex[1] = 2;
	if (fill_opp_coords) 
	{
	  if (el->child[1]->new_coord)
	  {
	    for (j=0; j<DIM_OF_WORLD; j++)
	      elinfo->opp_coord[1][j] = el->child[1]->new_coord[j];
	  }
	  else
	  {
	    for (j=0; j<DIM_OF_WORLD; j++)
	      elinfo->opp_coord[1][j] = 
		0.5 * (elinfo_old->coord[1][j] + elinfo_old->coord[2][j]);
	  }
	}
      }
      else 
      {
	elinfo->neigh[1] = el->child[1];
	DEBUG_TEST_EXIT(elinfo->neigh[1],
		    "el->child[0] != nil, but el->child[1] = nil\n");

	elinfo->opp_vertex[1] = 0;
	if (fill_opp_coords) 
	{
	  for (j=0; j<DIM_OF_WORLD; j++)
	    elinfo->opp_coord[1][j] = elinfo_old->coord[1][j];
	}
      }

      if ((nb = elinfo_old->neigh[2]))
      {
	DEBUG_TEST(elinfo_old->opp_vertex[2] == 2,"invalid neighbour\n"); 
	DEBUG_TEST_EXIT(nb->child[0], "missing children?\n");
	nb = nb->child[1];
	DEBUG_TEST_EXIT(nb, "missing child[1]?\n");

	if (nb->child[0]) 
	{
	  elinfo->opp_vertex[0] = 2;
	  if (fill_opp_coords)
	  {
	    if (nb->new_coord)
	    {
	      for (j=0; j<DIM_OF_WORLD; j++)
		elinfo->opp_coord[0][j] = nb->new_coord[j];
	    }
	    else
	    {
	      for (j=0; j<DIM_OF_WORLD; j++)
	      {
		elinfo->opp_coord[0][j] =
		  0.5*(elinfo_old->opp_coord[2][j] + elinfo_old->coord[0][j]);
	      }
	    }
	  }    
	  nb = nb->child[0];
	}
	else 
	{
	  elinfo->opp_vertex[0] = 1;
	  if (fill_opp_coords) 
	  {
	    for (j=0; j<DIM_OF_WORLD; j++)
	      elinfo->opp_coord[0][j] = elinfo_old->opp_coord[2][j];
	  }
	}
      }
      elinfo->neigh[0] = nb;
    }
    else   /* ichild==1 */
    { 
      if ((elinfo->neigh[2] = elinfo_old->neigh[0]))
      {
	if (fill_opp_coords)
	{
	  for (j=0; j<DIM_OF_WORLD; j++)
	    elinfo->opp_coord[2][j] = elinfo_old->opp_coord[0][j];
	}
      }
      elinfo->opp_vertex[2] = elinfo_old->opp_vertex[0];

      if (el->child[0]->child[0])
      {
	elinfo->neigh[0] = el->child[0]->child[0];
	elinfo->opp_vertex[0] = 2;
	if (fill_opp_coords)
	{
	  if (el->child[0]->new_coord)
	  {
	    for (j=0; j<DIM_OF_WORLD; j++)
	      elinfo->opp_coord[0][j] = el->child[0]->new_coord[j];
	  }
	  else
	  {
	    for (j=0; j<DIM_OF_WORLD; j++)
	      elinfo->opp_coord[0][j] = 
		0.5 * (elinfo_old->coord[0][j] + elinfo_old->coord[2][j]);
	  }
	}
      }
      else 
      {
	elinfo->neigh[0] = el->child[0];
	elinfo->opp_vertex[0] = 1;
	if (fill_opp_coords)
	{
	  for (j=0; j<DIM_OF_WORLD; j++)
	    elinfo->opp_coord[0][j] = elinfo_old->coord[0][j];
	}
      }

      if ((nb = elinfo_old->neigh[2]))
      {
	DEBUG_TEST(elinfo_old->opp_vertex[2] == 2,"invalid neighbour\n"); 
	nb = nb->child[0];
	DEBUG_TEST(nb,"missing child?\n");
	if (nb->child[0])
	{
	  elinfo->opp_vertex[1] = 2;
	  if (fill_opp_coords) 
	  {
	    if (nb->new_coord)
	      for (j=0; j<DIM_OF_WORLD; j++)
		elinfo->opp_coord[1][j] = nb->new_coord[j];
	    else
	      for (j=0; j<DIM_OF_WORLD; j++)
		elinfo->opp_coord[1][j] =
		  0.5*(elinfo_old->opp_coord[2][j] + elinfo_old->coord[1][j]);
	  }
	  nb = nb->child[1];
	}
	else 
	{
	  elinfo->opp_vertex[1] = 0;
	  if (fill_opp_coords) {
	    for (j=0; j<DIM_OF_WORLD; j++)
	      elinfo->opp_coord[1][j] = elinfo_old->opp_coord[2][j];
	  }
	}
      }
      elinfo->neigh[1] = nb;
    }
  }

  if (fill_flag & FILL_PROJECTION) {
    elinfo->projections[0] = elinfo_old->projections[0];

    if(ichild == 0) {
      elinfo->projections[1] = elinfo_old->projections[3];
      elinfo->projections[2] = elinfo_old->projections[0];
      elinfo->projections[3] = elinfo_old->projections[2];
    }
    else {
      elinfo->projections[1] = elinfo_old->projections[0];
      elinfo->projections[2] = elinfo_old->projections[3];
      elinfo->projections[3] = elinfo_old->projections[1];
    }

    if(elinfo->projections[3])
      elinfo->active_projection = elinfo->projections[3];
    else
      elinfo->active_projection = elinfo->projections[0];
  }

  if (fill_flag & FILL_BOUND)
  {
    elinfo->vertex_bound[2] = elinfo_old->edge_bound[2];

    if (ichild==0) {
      elinfo->vertex_bound[0] = elinfo_old->vertex_bound[2];
      elinfo->vertex_bound[1] = elinfo_old->vertex_bound[0];
      elinfo->edge_bound[0] = elinfo_old->edge_bound[2];
      elinfo->edge_bound[1] = INTERIOR;
      elinfo->edge_bound[2] = elinfo_old->edge_bound[1];
    }
    else {
      elinfo->vertex_bound[0] = elinfo_old->vertex_bound[1];
      elinfo->vertex_bound[1] = elinfo_old->vertex_bound[2];
      elinfo->edge_bound[0] = INTERIOR;
      elinfo->edge_bound[1] = elinfo_old->edge_bound[2];
      elinfo->edge_bound[2] = elinfo_old->edge_bound[0];
    }
  }

  return;
}
