/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     wall_quad_fast.c                                               */
/*                                                                          */
/* description:  Fast (i.e. caching) quadrature over walls                  */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*             Claus-Justus Heine                                           */
/*             Abteilung fuer Angewandte Mathematik                         */
/*             Albert-Ludwigs-Universitaet Freiburg                         */
/*             Hermann-Herder-Str. 10                                       */
/*             D-79104 Freiburg im Breisgau, Germany                        */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003),                         */
/*         C.-J. Heine (1998-2003)                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#include "alberta.h"

static int fac(int dim)
{
  int res = 1;

  while (dim) res *= dim--;
  return res;
}

const WALL_QUAD_FAST *get_wall_quad_fast(const BAS_FCTS *bas_fcts, 
					 const QUAD *quad, U_CHAR init_flag)
{
  FUNCNAME("get_wall_quad_fast");
  static WALL_QUAD_FAST  *first = nil;
  WALL_QUAD_FAST   *quad_fast;
  int              qp, wall, no;
  const int        *perm;
  int              d, i, j, k, l, m, n_points, n_bas_fcts, dim, card_sn;
  REAL             **g_phi;
  REAL             (**g_grd_phi)[N_LAMBDA];
  REAL             (**g_D2_phi)[N_LAMBDA][N_LAMBDA];
  REAL             *lambda;
  void             *mem = nil;
  /*  ALBERTA_LOCAL_RWLOCK_DEF(fqfast_lock); */

  if (quad->dim + 1 != (dim = bas_fcts->dim)) {
    ERROR("quad->dim+1 == %d is not %d == bas_fcts->dim!\n", quad->dim+1,dim);
    return(nil);
  }

  if(dim == 1) {
    ERROR("Not implemented for 1D!\n");
    return(nil);
  }

  card_sn = fac(dim);

  /* ALBERTA_RDLOCK(&fqfast_lock); */
  for (quad_fast = first; quad_fast; quad_fast = quad_fast->next) {
    if (quad_fast->quad == quad  &&  quad_fast->bas_fcts == bas_fcts) {
      break;
    }
  }

  if (quad_fast  &&  ((quad_fast->init_flag & init_flag) == init_flag)) {
    /* ALBERTA_RWUNLOCK(&fqfast_lock); */
    return((const WALL_QUAD_FAST *) quad_fast);
  }

  /* ALBERTA_RWUNLOCK(&fqfast_lock); */

  /* ok, remainder either will modify the qfast structure, or even add
   * a new one. Get the write lock. Of course, as we loose the lock in
   * between, it might (and will) happen that there are duplicated
   * quad_fast structures. But these cases will be rare.
   */
  /* ALBERTA_WRLOCK(&fqfast_lock); */

  n_points = quad->n_points;
  n_bas_fcts = bas_fcts->n_bas_fcts;

  if (!quad_fast) {
    quad_fast = MEM_ALLOC(1, WALL_QUAD_FAST);
    quad_fast->quad = quad;
    quad_fast->n_points = quad->n_points;
    quad_fast->n_bas_fcts = bas_fcts->n_bas_fcts;
    quad_fast->w = quad->w;
    quad_fast->init_flag = 0;
    for (i = 0; i < CARD_SN_MAX; i++) {
      for (j = 0; j < N_WALLS_MAX; j++) {
	quad_fast->lambda[i][j]    = NULL;
	quad_fast->phi[i][j]       = NULL;
	quad_fast->grd_phi[i][j]   = NULL;
	quad_fast->D2_phi[i][j]    = NULL;
      }
    }
    for (no = 0; no < card_sn; no++) {
      for (wall = 0; wall < N_NEIGH(dim); wall++) {
	quad_fast->lambda[no][wall] = MEM_ALLOC(n_points, REAL_B);
      }
    }
    quad_fast->next = first;
    quad_fast->bas_fcts = (BAS_FCTS *) bas_fcts;
    first = quad_fast;

    /* Initialize the barycentric coordinates w.r.t. to every possible
     * permutation. Same way as done in jump_res2() in estimator.c
     */
    for (no = 0; no < card_sn; no++) {
      for (wall = 0; wall < N_NEIGH(dim); wall++) {
	perm = sorted_wall_indices(dim, wall, no);
	for (qp = 0; qp < n_points; qp++) {
	  lambda = (REAL *)quad_fast->lambda[no][wall][qp];
	  lambda[wall] = 0.0;
	  for (d = 0; d < dim; d++) {
	    lambda[perm[d]] = quad->lambda[qp][d];
	  }
	}
      }
    }
  }

  /* initialize values of basis functions at quadrature points */
  if (!quad_fast->phi[0][0]  &&  (init_flag & INIT_PHI)) {
    BAS_FCT    **phi = bas_fcts->phi;
  
    for (no = 0; no < card_sn; no++) {
      for (k = 0; k < N_NEIGH(dim); k++) {
	quad_fast->phi[no][k] = g_phi = MEM_ALLOC(n_points, REAL *);

	for (i = 0; i < n_points; i++) {
	  lambda = (REAL *)quad_fast->lambda[no][k][i];
	  g_phi[i] = MEM_ALLOC(n_bas_fcts, REAL);

	  for (j = 0; j < n_bas_fcts; j++) {
	    g_phi[i][j] = phi[j](lambda);
	  }
	}
      }
    }
    quad_fast->init_flag |= INIT_PHI;
  }

  if (!quad_fast->grd_phi[0][0]  && (init_flag & INIT_GRD_PHI)) {
    GRD_BAS_FCT  **grd_phi = bas_fcts->grd_phi;
    const REAL   *grd;
      
    for (no = 0; no < card_sn; no++) {
      for (k = 0; k < N_NEIGH(dim); k++) {
	g_grd_phi =
	  quad_fast->grd_phi[no][k]  = MEM_ALLOC(n_points, REAL_B *);


	for (i = 0; i < n_points; i++) { 
	  lambda = (REAL *)quad_fast->lambda[no][k][i];
	  g_grd_phi[i] = MEM_ALLOC(n_bas_fcts, REAL_B);
	  
	  for (j = 0; j < n_bas_fcts; j++) {
	    grd = grd_phi[j](lambda);
	    for (l = 0; l < dim+1; l++) {
	      g_grd_phi[i][j][l] = grd[l];
	    }	    
	  }
	}
      }
    }
    quad_fast->init_flag |= INIT_GRD_PHI;
  }

  if (!quad_fast->D2_phi[0][0]  && (init_flag & INIT_D2_PHI)) {
    D2_BAS_FCT   **D2_phi = bas_fcts->D2_phi;
    const REAL   (*D2)[N_LAMBDA];

    for (no = 0; no < card_sn; no++) {
      for (k = 0; k < N_NEIGH(dim); k++) {
	g_D2_phi = quad_fast->D2_phi[no][k] =
	  MEM_ALLOC(n_points, REAL_BB *);

	for (i = 0; i < n_points; i++) {
	  lambda = (REAL *)quad_fast->lambda[no][k][i];
	  g_D2_phi[i] = MEM_ALLOC(n_bas_fcts, REAL_BB);
	  mem = (void *) (g_D2_phi[i]+n_bas_fcts);

	  for (j = 0; j < n_bas_fcts; j++) {
	    D2 = D2_phi[j](lambda);
	    for (m = 0; m < dim+1; m++) {
	      for (l = 0; l < dim+1; l++) {
		g_D2_phi[i][j][m][l] = D2[m][l];
	      }
	    }
	  }
	}
      }
    }
    quad_fast->init_flag |= INIT_D2_PHI;
  }

  /* ALBERTA_RWUNLOCK(&fqfast_lock); */

  if ((quad_fast->init_flag & init_flag) != init_flag)
  {
    ERROR("could not initialize quad_fast, returning pointer to nil\n");
    return(nil);
  }

  return((const WALL_QUAD_FAST *) quad_fast);
}


