/*
 * ifimsg.c - interface information message parser
 * Copyright (C) 2014 Tetsumune KISO <t2mune@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include "nield.h"
#include "rtnetlink.h"

#define HIST_MAX 16

/* top of interface list and history */
static struct list_head lhead = {&lhead, &lhead};
static struct list_head hhead = {&hhead, &hhead};

/*
 * search an entry in an interface list
 */
static inline struct iflist_entry *search_iflist(int index)
{
    struct list_head *l;

    list_for_each(l, &lhead) {
        struct iflist_entry *e;

        e = list_entry(l, struct iflist_entry, list);
        if(e->index == index)
            return(e);
    }

    return(NULL);
}

/*
 * add an entry into an interface list
 */ 
static inline void add_iflist_entry(struct list_head *l)
{
    list_init(l);
    list_add(l, &lhead);
}

/*
 * [not in use]
 * delete an entry in an interface list
 */ 
static inline void del_iflist_entry(int index)
{
    struct list_head *l, *n;

    list_for_each_safe(l, n, &lhead) {
        struct iflist_entry *e;

        e = list_entry(l, struct iflist_entry, list);
        if(e->index == index) {
            list_del(l);
            free(e);
        }
    }
}

/*
 * move an entry in an interface list to an interface history
 */
static inline void move_iflist_entry(int index)
{
    struct list_head *l, *n;

    list_for_each_safe(l, n, &lhead) {
        struct iflist_entry *e;

        e = list_entry(l, struct iflist_entry, list);
        if(e->index == index) {
            list_move(l, &hhead);
            break;
        }
    }
}

/*
 * delete an old entry in an interface history
 */
static inline void del_ifhist_entry(void)
{
    struct list_head *l, *n;
    int i = 0;

    list_for_each_safe(l, n, &hhead) {
        struct iflist_entry *e;

        if(++i > HIST_MAX) {
            e = list_entry(l, struct iflist_entry, list);
            list_del(l);
            free(e);
            break;
        }
    }
}

/*
 * search an entry in an interface history
 */ 
static inline struct iflist_entry *search_ifhist(int index)
{
    struct list_head *l;

    list_for_each(l, &hhead) {
        struct iflist_entry *e;

        e = list_entry(l, struct iflist_entry, list);
        if(e->index == index)
            return(e);
    }

    return(NULL);
}

/*
 * convert an interface index to name in lists
 */
char *if_indextoname_from_lists(int index, char *name)
{
    /* from kernel */
    if(if_indextoname(index, name))
        return(name);

    /* from interface list */
    if(if_indextoname_from_iflist(index, name))
        return(name);

    /* from interface history */
    if(if_indextoname_from_ifhist(index, name))
        snprintf(name, IFNAMSIZ, "(unknown %d)", index);

    return(name);
}

/*
 * convert an interface index to name in an interface list
 */
char *if_indextoname_from_iflist(int index, char *name)
{
    struct iflist_entry *e = search_iflist(index);

    if(!e)
        return(NULL);

    strncpy(name, e->name, sizeof(e->name));

    return(name);
}

/*
 * convert an interface index to name in an inetrface history
 */ 
char *if_indextoname_from_ifhist(int index, char *name)
{
    struct iflist_entry *e = search_ifhist(index);

    if(!e)
        return(NULL);

    strncpy(name, e->name, sizeof(e->name));

    return(name);
}

/*
 * get an interface type in an interface list
 */ 
unsigned short get_type_from_iflist(int index)
{
    struct iflist_entry *e = search_iflist(index);

    if(!e)
        return(0);

    return(e->type);
}

/*
 * get an interface type in an interface history
 */
unsigned short get_type_from_ifhist(int index)
{
    struct iflist_entry *e = search_ifhist(index);

    if(!e)
        return(0);

    return(e->type);
}

/*
 * print an interface list
 */ 
void print_iflist(void)
{
    struct list_head *l;
    FILE *iflist;
    char flags_list[MAX_STR_SIZE];

    iflist = fopen(IFLIST_FILE, "w");
    if(iflist == NULL) {
        rec_log("error: %s: can't open iflist file(%s)", __func__, IFLIST_FILE);
        return;
    }
    fprintf(iflist, "\n");
    fprintf(iflist, "*********************************************************************\n");
    fprintf(iflist, "[ interface list ]\n");

    list_for_each(l, &lhead) {
        struct iflist_entry *e;

        e = list_entry(l, struct iflist_entry, list);

        strncpy(flags_list, "", sizeof(flags_list));
        convert_iff_flags(e->flags, flags_list, sizeof(flags_list));

        fprintf(iflist, "%s[%d]: address %s\n", e->name, e->index, e->addr);
        fprintf(iflist, "%s[%d]: broadcast %s\n", e->name, e->index, e->brd);
        fprintf(iflist, "%s[%d]: flags %s\n", e->name, e->index, flags_list);
        fprintf(iflist, "%s[%d]: type %s\n",
            e->name, e->index, convert_arphrd_type(e->type));
        fprintf(iflist, "%s[%d]: vlan %hu\n", e->name, e->index, e->vid);
        fprintf(iflist, "%s[%d]: mtu %hu\n", e->name, e->index, e->mtu);
        fprintf(iflist, "%s[%d]: kind %s\n",
            e->name, e->index, strlen(e->kind) ? e->kind : "no");
        fprintf(iflist, "%s[%d]: master %s[%d]\n",
            e->name, e->index,
            strlen(e->name_master) ? e->name_master : "none" , e->index_master);
        fprintf(iflist, "%s[%d]: bridge-attached %s\n",
            e->name, e->index, e->br_attached ? "yes" : "no");
    }

    fclose(iflist);
}

/*
 * print an interface history
 */ 
void print_ifhist(void)
{
    struct list_head *l;
    FILE *ifhist;
    char flags_list[MAX_STR_SIZE];

    ifhist = fopen(IFHIST_FILE, "w");
    if(ifhist == NULL) {
        rec_log("error: %s: can't open ifhist file(%s)", __func__, IFHIST_FILE);
        return;
    }
    fprintf(ifhist, "*********************************************************************\n");
    fprintf(ifhist, "[ interface history ]\n");

    list_for_each(l, &hhead) {
        struct iflist_entry *e;

        e = list_entry(l, struct iflist_entry, list);

        strncpy(flags_list, "", sizeof(flags_list));
        convert_iff_flags(e->flags, flags_list, sizeof(flags_list));

        fprintf(ifhist, "%s[%d]: address %s\n", e->name, e->index, e->addr);
        fprintf(ifhist, "%s[%d]: broadcast %s\n", e->name, e->index, e->brd);
        fprintf(ifhist, "%s[%d]: flags %s\n", e->name, e->index, flags_list);
        fprintf(ifhist, "%s[%d]: type %s\n",
            e->name, e->index, convert_arphrd_type(e->type));
        fprintf(ifhist, "%s[%d]: vlan %d\n", e->name, e->index, e->vid);
        fprintf(ifhist, "%s[%d]: mtu %d\n", e->name, e->index, e->mtu);
        fprintf(ifhist, "%s[%d]: kind %s\n",
            e->name, e->index,
            strlen(e->kind) ? e->kind : "no");
        fprintf(ifhist, "%s[%d]: master %s[%d]\n",
            e->name, e->index,
            strlen(e->name_master) ? e->name_master : "none", e->index_master);
        fprintf(ifhist, "%s[%d]: bridge-attached %s\n",
            e->name, e->index,
            e->br_attached ? "yes" : "no");
    }

    fclose(ifhist);
}

/*
 * create an interface list
 */ 
int create_iflist(struct msghdr *msg)
{
    struct nlmsghdr *nlh;
    int nlh_len;
    struct ifinfomsg *ifim;
    int ifim_len;
    struct rtattr *ifla[__IFLA_MAX];
    struct iflist_entry *ifle;
    int log_opts = get_log_opts();

    /* get netlink message header */
    nlh = msg->msg_iov->iov_base;
    nlh_len = msg->msg_iov->iov_len;

    /* parse netlink message header */
    for( ; NLMSG_OK(nlh, nlh_len); nlh = NLMSG_NEXT(nlh, nlh_len)) {
        /* whether netlink message header ends or not */
        if(nlh->nlmsg_type == NLMSG_DONE)
            return(1);

        /* debug nlmsghdr */
        if(log_opts & L_DEBUG)
            debug_nlmsg(0, nlh);

        /* get ifinfomsg */
        ifim_len = NLMSG_PAYLOAD(nlh, 0);
        if(ifim_len < sizeof(*ifim)) {
            rec_log("error: %s: ifinfomsg: length too short", __func__);
            continue;
        }
        ifim = (struct ifinfomsg *)NLMSG_DATA(nlh);

        /* parse interface infomation attributes */
        parse_ifinfo(ifla, nlh);

        /* debug ifinfomsg */
        if(log_opts & L_DEBUG)
            debug_ifimsg(0, ifim, ifla, ifim_len);

        /* check bridge interface */
        if(ifim->ifi_family == PF_BRIDGE) {
            ifle = search_iflist(ifim->ifi_index);
            if(ifle)
                ifle->br_attached = 1;
        }

        /* check protocol family(PF_UNSPEC only) */
        if(ifim->ifi_family != PF_UNSPEC)
            continue;

        /* create interface list */
        ifle = (struct iflist_entry *)malloc(sizeof(struct iflist_entry));
        if(ifle == NULL) {
            rec_log("error: %s: malloc(): failed", __func__);
            return(-1);
        }
        memset(ifle, 0, sizeof(struct iflist_entry));
        list_init(&(ifle->list));

        ifle->index = ifim->ifi_index;
        ifle->flags = ifim->ifi_flags;
        ifle->type = ifim->ifi_type;

        /* get interface name */
        if(ifla[IFLA_IFNAME])
            if(parse_ifla_ifname(NULL, NULL, ifla[IFLA_IFNAME], ifle)) {
                free(ifle);
                continue;
            }

        /* get interface address */
        if(ifla[IFLA_ADDRESS])
            if(parse_ifla_address(NULL, NULL, ifla[IFLA_ADDRESS], ifle)) {
                free(ifle);
                continue;
            }

        /* get broadcast address */
        if(ifla[IFLA_BROADCAST])
            if(parse_ifla_broadcast(NULL, NULL, ifla[IFLA_BROADCAST], ifle)) {
                free(ifle);
                continue;
            }

#if HAVE_DECL_IFLA_LINKINFO
        /* get interface information */
        if(ifla[IFLA_LINKINFO])
            if(parse_ifla_linkinfo(NULL, NULL, ifla[IFLA_LINKINFO], ifle)) {
                free(ifle);
                continue;
            }
#endif

        /* get interface MTU */
        if(ifla[IFLA_MTU])
            if(parse_ifla_mtu(NULL, NULL, ifla[IFLA_MTU], ifle)) {
                free(ifle);
                continue;
            }

        /* get master interface */
        if(ifla[IFLA_MASTER])
            if(parse_ifla_master(NULL, NULL, ifla[IFLA_MASTER], ifle)) {
                free(ifle);
                continue;
            }

        /* add interface list */
        list_add(&(ifle->list), &lhead);
    }

    return(0);
}

/*
 * parse interface information message
 */
int parse_ifimsg(struct nlmsghdr *nlh)
{
    struct ifinfomsg *ifim;
    int ifim_len;
    struct rtattr *ifla[__IFLA_MAX];
    struct iflist_entry *ifle_tmp, *ifle;
    char msg[MAX_MSG_SIZE] = "";
    char *mp = msg;
    int log_opts = get_log_opts();

    /* debug nlmsghdr */
    if(log_opts & L_DEBUG)
        debug_nlmsg(0, nlh);

    /* get ifinfomsg */
    ifim_len = NLMSG_PAYLOAD(nlh, 0);
    if(ifim_len < sizeof(*ifim)) {
        rec_log("error: %s: ifinfomsg: length too short", __func__);
        return(1);
    }
    ifim = (struct ifinfomsg *)NLMSG_DATA(nlh);

    /* parse interface infomation message attributes */
    parse_ifinfo(ifla, nlh);

    /* debug ifinfomsg */
    if(log_opts & L_DEBUG)
        debug_ifimsg(0, ifim, ifla, ifim_len);

    /* create new interface list entry */
    ifle_tmp = malloc(sizeof(struct iflist_entry));
    if(!ifle_tmp) {
        rec_log("error: %s: malloc() failed", __func__);
        return(1);
    }
    memset(ifle_tmp, 0, sizeof(struct iflist_entry));
    list_init(&(ifle_tmp->list));

    ifle_tmp->index = ifim->ifi_index;
    ifle_tmp->flags = ifim->ifi_flags;
    ifle_tmp->type = ifim->ifi_type;

    /* get interface name */
    if(ifla[IFLA_IFNAME])
        if(parse_ifla_ifname(msg, &mp, ifla[IFLA_IFNAME], ifle_tmp)) {
            free(ifle_tmp);
            return(1);
        }

    /* get physical interface */
    if(ifla[IFLA_LINK])
        if(parse_ifla_link(msg, &mp, ifla[IFLA_LINK], ifle_tmp)) {
            free(ifle_tmp);
            return(1);
        }

    /* get interface address */
    if(ifla[IFLA_ADDRESS])
        if(parse_ifla_address(msg, &mp, ifla[IFLA_ADDRESS], ifle_tmp)) {
            free(ifle_tmp);
            return(1);
        }

    /* get broadcast address */
    if(ifla[IFLA_BROADCAST])
        if(parse_ifla_broadcast(msg, &mp, ifla[IFLA_BROADCAST], ifle_tmp)) {
            free(ifle_tmp);
            return(1);
        }

    /* get interface MTU */
    if(ifla[IFLA_MTU])
        if(parse_ifla_mtu(msg, &mp, ifla[IFLA_MTU], ifle_tmp)) {
            free(ifle_tmp);
            return(1);
        }

#if HAVE_DECL_IFLA_LINKINFO
    /* get interface information */
    if(ifla[IFLA_LINKINFO])
        if(parse_ifla_linkinfo(msg, &mp, ifla[IFLA_LINKINFO], ifle_tmp)) {
            free(ifle_tmp);
            return(1);
        }
#endif

    /* get master interface */
    if(ifla[IFLA_MASTER])
        if(parse_ifla_master(msg, &mp, ifla[IFLA_MASTER], ifle_tmp)) {
            free(ifle_tmp);
            return(1);
        }

    /* check RTM message(only RTM_NEWLINK or RTMDELLINK) */
    if((nlh->nlmsg_type != RTM_NEWLINK) && (nlh->nlmsg_type != RTM_DELLINK)) {
        rec_log("error: %s: unkonwn nlmsg_type: %d", __func__, nlh->nlmsg_type);
        free(ifle_tmp);
        return(0);
    }

    /* search interface list entry */
    ifle = search_iflist(ifle_tmp->index);

    /* check protocol family(PF_BRIDGE only) & nlmsg_type */
    if(ifim->ifi_family == PF_BRIDGE) {
        /* workaround: "ovs-vsctl add-br" command */
        if(ifle_tmp->index != ifle_tmp->index_master) {
            if(nlh->nlmsg_type == RTM_NEWLINK) {
                if(ifle && !ifle->br_attached) {
                    ifle->br_attached = 1;
                    ifle->index_master = ifle_tmp->index_master;
                    rec_log("interface %s attached to bridge %s",
                        ifle->name, ifle_tmp->name_master);
                    strncpy(ifle->name_master, ifle_tmp->name_master, IFNAMSIZ);
                }
            } else if(nlh->nlmsg_type == RTM_DELLINK) {
                if(ifle && ifle->br_attached) {
                    ifle->br_attached = 0;
                    rec_log("interface %s detached from bridge %s",
                        ifle->name, ifle->name_master);
                    strncpy(ifle->name_master, "", IFNAMSIZ);
                }
            }
        }
    }

    /* check protocol family(PF_UNSPEC only) */
    if(ifim->ifi_family != PF_UNSPEC) {
        free(ifle_tmp);
        return(0);
    }

    /* check nlmsg_type */
    if(nlh->nlmsg_type == RTM_NEWLINK)
        parse_rtm_newlink(msg, ifle, ifle_tmp, ifla);
    else if(nlh->nlmsg_type == RTM_DELLINK)
        parse_rtm_dellink(msg, ifle, ifle_tmp);

    return(0);
}

/*
 * parse RTM_NEWLINK
 */
void parse_rtm_newlink(char *msg, struct iflist_entry *ifle,
    struct iflist_entry *ifle_tmp, struct rtattr *ifla[])
{
    if(ifle) {
        /* check bonding interface */
        if(!(ifle->flags & IFF_SLAVE) && (ifle_tmp->flags & IFF_SLAVE)) {
            rec_log("interface %s attached to bonding %s",
                ifle_tmp->name, ifle_tmp->name_master);
            strncpy(ifle->name_master, ifle_tmp->name_master, sizeof(ifle->name_master));
        } else if((ifle->flags & IFF_SLAVE) && !(ifle_tmp->flags & IFF_SLAVE)) {
            rec_log("interface %s detached from bonding %s",
                ifle_tmp->name, ifle->name_master);
            strncpy(ifle->name_master, "", sizeof(ifle->name_master));
        }

        /* check administrative status */
        if((ifle->flags & IFF_UP) && !(ifle_tmp->flags & IFF_UP))
            rec_log("interface %s state changed to disabled",
                ifle_tmp->name);
        else if(!(ifle->flags & IFF_UP) && (ifle_tmp->flags & IFF_UP))
            rec_log("interface %s state changed to enabled",
                ifle_tmp->name);

        /* check operational status */
        if((ifle->flags & IFF_RUNNING) && !(ifle_tmp->flags & IFF_RUNNING))
            rec_log("interface %s state changed to down",
                ifle_tmp->name);
        else if(!(ifle->flags & IFF_RUNNING) && (ifle_tmp->flags & IFF_RUNNING))
            rec_log("interface %s state changed to up",
                ifle_tmp->name);

        /* check promiscuous status */
        if((ifle->flags & IFF_PROMISC) && !(ifle_tmp->flags & IFF_PROMISC))
            rec_log("interface %s left promiscuous mode",
                ifle_tmp->name);
        else if(!(ifle->flags & IFF_PROMISC) && (ifle_tmp->flags & IFF_PROMISC))
            rec_log("interface %s entered promiscuous mode",
                ifle_tmp->name);

        ifle->flags = ifle_tmp->flags;

        /* check interface name */
        if(ifla[IFLA_IFNAME]) {
            if(strncmp(ifle->name, ifle_tmp->name, IFNAMSIZ)) {
                rec_log("interface name changed from %s to %s",
                    ifle->name, ifle_tmp->name);
                strncpy(ifle->name, ifle_tmp->name, IFNAMSIZ);
            }
        }

        /* check interface address */
        if(ifla[IFLA_ADDRESS]) {
            if(memcmp(ifle->addr, ifle_tmp->addr, sizeof(ifle->addr))) {
                switch(ifle_tmp->type) {
                    case ARPHRD_TUNNEL:
                    case ARPHRD_IPGRE:
                    case ARPHRD_SIT:
                    case ARPHRD_TUNNEL6:
#ifdef ARPHRD_IP6GRE
                    case ARPHRD_IP6GRE:
#endif
                        rec_log("interface %s local address changed from %s to %s",
                            ifle->name, ifle->addr, ifle_tmp->addr);
                        break;
                    default:
                        rec_log("interface %s link layer address changed from %s to %s",
                            ifle->name, ifle->addr, ifle_tmp->addr);
                }
                memcpy(ifle->addr, ifle_tmp->addr, sizeof(ifle->addr));
            }
        }

        /* check broadcast address */
        if(ifla[IFLA_BROADCAST]) {
            if(memcmp(ifle->brd, ifle_tmp->brd, sizeof(ifle->brd))) {
                switch(ifle_tmp->type) {
                    case ARPHRD_TUNNEL:
                    case ARPHRD_IPGRE:
                    case ARPHRD_SIT:
                    case ARPHRD_TUNNEL6:
#ifdef ARPHRD_IP6GRE
                    case ARPHRD_IP6GRE:
#endif
                        rec_log("interface %s remote address changed from %s to %s",
                            ifle->name, ifle->brd, ifle_tmp->brd);
                        break;
                }
                memcpy(ifle->brd, ifle_tmp->brd, sizeof(ifle->brd));
            }
        }

        ifle->type = ifle_tmp->type;

        /* check interface MTU */
        if(ifla[IFLA_MTU]) {
            if(ifle->mtu != ifle_tmp->mtu) {
                rec_log("interface %s mtu changed from %d to %d",
                    ifle->name, ifle->mtu, ifle_tmp->mtu);
                ifle->mtu = ifle_tmp->mtu;
            }
        } 

        /* check interface vlan id */
        if(ifle->vid != ifle_tmp->vid) {
            rec_log("interface %s vlan id changed from %hu to %hu",
                ifle->name, ifle->vid, ifle_tmp->vid);
            ifle->vid = ifle_tmp->vid;
        }

        /* check master interface */
        if(ifle->index_master != ifle_tmp->index_master) {
            ifle->index_master = ifle_tmp->index_master;
        }

        free(ifle_tmp);
    } else {
        /* add interface list entry*/
        add_iflist_entry(&(ifle_tmp->list));

        /* check interface state */
        char state[MAX_STR_SIZE] = "";

        /* check administrative state */
        (ifle_tmp->flags & IFF_UP) ?
            strcpy(state, "enabled,") : strcpy(state, "disabled,");

        /* check operational state */
        (ifle_tmp->flags & IFF_RUNNING) ?
            strcat(state, "linkup") : strcat(state, "linkdown");

        rec_log("interface added: %sstate=%s", msg, state);
    }
}

/*
 * parse RTM_DEL_LINK
 */
void parse_rtm_dellink(char *msg, struct iflist_entry *ifle, struct iflist_entry *ifle_tmp)
{
    /* move entry from interface list to interface history */
    move_iflist_entry(ifle_tmp->index);
    del_ifhist_entry();

    /* check interface state */
    char state[MAX_STR_SIZE] = "";

    /* check administrative state */
    (ifle_tmp->flags & IFF_UP) ?
        strcpy(state, "enabled,") : strcpy(state, "disabled,");

    /* check operational state */
    (ifle_tmp->flags & IFF_RUNNING) ?
        strcat(state, "linkup") : strcat(state, "linkdown");

    rec_log("interface deleted: %sstate=%s", msg, state);
    free(ifle_tmp);
}

/*
 * psrse attribute IFLA_IFNAME
 */
int parse_ifla_ifname(char *msg, char **mp, struct rtattr *ifla, struct iflist_entry *ifle)
{
    if(!RTA_PAYLOAD(ifla)) {
        rec_log("error: %s: ifindex %d: no payload",
            __func__, ifle->index);
        return(1);
    } else if(RTA_PAYLOAD(ifla) > sizeof(ifle->name)) {
        rec_log("error: %s: ifindex %d: payload too long",
            __func__, ifle->index);
        return(1);
    }
    strncpy(ifle->name, RTA_DATA(ifla), sizeof(ifle->name));

    if(msg)
        *mp = add_log(msg, *mp, "name=%s ", ifle->name);

    return(0);
}

/*
 * parse attribute IFLA_LINK
 */
int parse_ifla_link(char *msg, char **mp, struct rtattr *ifla, struct iflist_entry *ifle)
{
    int index;
    char name[IFNAMSIZ] = "";

    if(RTA_PAYLOAD(ifla) < sizeof(index)) {
        rec_log("error: %s: ifindex %d: payload too short",
            __func__, ifle->index);
        return(1);
    }
    index = *(int *)RTA_DATA(ifla);
    if(index && ifle->index != index) {
        if_indextoname_from_lists(index, name);

        *mp = add_log(msg, *mp, "link=%s ", name);
    }

    return(0);
}

/*
 * parse attribute IFLA_ADDRESS
 */
int parse_ifla_address(char *msg, char **mp, struct rtattr *ifla, struct iflist_entry *ifle)
{
    char addrstr[INET6_ADDRSTRLEN+1] = "";
    int res;

    res = inet_ntop_ifi(ifle->type, ifla, addrstr, sizeof(addrstr));
    if(res) {
        rec_log("error: %s: ifindex %d: %s",
            __func__, ifle->index,
            (res == 1) ? strerror(errno) : "payload too short");
        return(1);
    }
    memcpy(ifle->addr, addrstr, sizeof(addrstr));

    if(msg)
        switch(ifle->type) {
            case ARPHRD_TUNNEL:
            case ARPHRD_IPGRE:
            case ARPHRD_SIT:
            case ARPHRD_TUNNEL6:
#ifdef ARPHRD_IP6GRE
            case ARPHRD_IP6GRE:
#endif
                *mp = add_log(msg, *mp, "local=%s ", ifle->addr);
                break;
            default:
                *mp = add_log(msg, *mp, "lladdr=%s ", ifle->addr);
        }

    return(0);
}

/*
 * parse attribute IFLA_BROADCAST
 */
int parse_ifla_broadcast(char *msg, char **mp, struct rtattr *ifla, struct iflist_entry *ifle)
{
    char brdstr[INET6_ADDRSTRLEN+1] = "";
    int res;

    res = inet_ntop_ifi(ifle->type, ifla, brdstr, sizeof(brdstr));
    if(res) {
        rec_log("error: %s: ifindex %d: %s",
            __func__, ifle->index,
            (res == 1) ? strerror(errno) : "pyaload too short");
        return(1);
    }
    memcpy(ifle->brd, brdstr, sizeof(brdstr));

    if(msg)
        switch(ifle->type) {
            case ARPHRD_TUNNEL:
            case ARPHRD_IPGRE:
            case ARPHRD_SIT:
            case ARPHRD_TUNNEL6:
#ifdef ARPHRD_IP6GRE
            case ARPHRD_IP6GRE:
#endif
                *mp = add_log(msg, *mp, "remote=%s ", ifle->brd);
                break;
        }

    return(0);
}

/*
 * parse attribute IFLA_MTU
 */
int parse_ifla_mtu(char *msg, char **mp, struct rtattr *ifla, struct iflist_entry *ifle)
{
    if(RTA_PAYLOAD(ifla) < sizeof(ifle->mtu)) {
        rec_log("error: %s: ifindex %d: payload too short",
            __func__, ifle->index);
        return(1);
    }
    ifle->mtu = *(int *)RTA_DATA(ifla);

    if(msg)
        *mp = add_log(msg, *mp, "mtu=%d ", ifle->mtu);

    return(0);
}

#if HAVE_DECL_IFLA_LINKINFO
/*
 * parse attribute IFLA_LINKINFO
 */
int parse_ifla_linkinfo(char *msg, char **mp, struct rtattr *ifla, struct iflist_entry *ifle)
{
    struct rtattr *linkinfo[__IFLA_INFO_MAX];

    parse_linkinfo(linkinfo, ifla);

    if(linkinfo[IFLA_INFO_KIND])
        if(parse_ifla_info_kind(msg, mp, linkinfo[IFLA_INFO_KIND], ifle))
            return(1);

    if(linkinfo[IFLA_INFO_DATA])
        if(parse_ifla_info_data(msg, mp, linkinfo[IFLA_INFO_DATA], ifle))
            return(1);

    return(0);
}

/*
 * parse attribute IFLA_INFO_KIND
 */
int parse_ifla_info_kind(char *msg, char **mp, struct rtattr *linkinfo, struct iflist_entry *ifle)
{
    if(!RTA_PAYLOAD(linkinfo)) {
        rec_log("error: %s: ifindex %d: no payload",
            __func__, ifle->index);
        return(1);
    } if(RTA_PAYLOAD(linkinfo) > sizeof(ifle->kind)) {
        rec_log("error: %s: ifindex %d: payload too long",
            __func__, ifle->index);
        return(1);
    }
    strncpy(ifle->kind, (char *)RTA_DATA(linkinfo), sizeof(ifle->kind));

    if(msg)
        *mp = add_log(msg, *mp, "kind=%s ", ifle->kind);

    return(0);
}

/*
 * parse attribute IFLA_INFO_DATA
 */
int parse_ifla_info_data(char *msg, char **mp, struct rtattr *linkinfo, struct iflist_entry *ifle)
{
#if HAVE_DECL_IFLA_VLAN_UNSPEC
    if(!strncmp(ifle->kind, "vlan", sizeof(ifle->kind)))
        parse_ifla_vlan(msg, mp, linkinfo, ifle);
#endif

#if HAVE_DECL_IFLA_GRE_UNSPEC
    if(!strncmp(ifle->kind, "gre", sizeof(ifle->kind)))
        parse_ifla_gre(msg, mp, linkinfo, ifle);

    if(!strncmp(ifle->kind, "gretap", sizeof(ifle->kind)))
        parse_ifla_gre(msg, mp, linkinfo, ifle);
#endif

#if HAVE_DECL_IFLA_MACVLAN_UNSPEC
    if(!strncmp(ifle->kind, "macvlan", sizeof(ifle->kind)))
        parse_ifla_macvlan(msg, mp, linkinfo, ifle);

    if(!strncmp(ifle->kind, "macvtap", sizeof(ifle->kind)))
        parse_ifla_macvlan(msg, mp, linkinfo, ifle);
#endif

#if HAVE_DECL_IFLA_VXLAN_UNSPEC
    if(!strncmp(ifle->kind, "vxlan", sizeof(ifle->kind)))
        parse_ifla_vxlan(msg, mp, linkinfo, ifle);
#endif

    return(0);
}
#endif

#if HAVE_DECL_IFLA_VLAN_UNSPEC
/*
 * parse attributes IFLA_VLAN_*
 */
int parse_ifla_vlan(char *msg, char **mp, struct rtattr *linkinfo, struct iflist_entry *ifle)
{
    struct rtattr *vlan[__IFLA_VLAN_MAX];

    parse_vlan(vlan, linkinfo);

    if(vlan[IFLA_VLAN_ID])
        if(parse_ifla_vlan_id(msg, mp, vlan[IFLA_VLAN_ID], ifle))
            return(1);

    if(vlan[IFLA_VLAN_EGRESS_QOS])
        if(parse_ifla_vlan_egress_qos(msg, mp, vlan[IFLA_VLAN_EGRESS_QOS], ifle))
            return(1);

    if(vlan[IFLA_VLAN_INGRESS_QOS])
        if(parse_ifla_vlan_ingress_qos(msg, mp, vlan[IFLA_VLAN_INGRESS_QOS], ifle))
            return(1);

    return(0);
}

/*
 * parse attribute IFLA_VLAN_ID
 */
int parse_ifla_vlan_id(char *msg, char **mp, struct rtattr *vlan, struct iflist_entry *ifle)
{
    if(RTA_PAYLOAD(vlan) < sizeof(ifle->vid)) {
        rec_log("error: %s: ifindex %d: payload too short",
            __func__, ifle->index);
        return(1);
    }
    ifle->vid = *(unsigned short *)RTA_DATA(vlan);

    if(msg)
        *mp = add_log(msg, *mp, "vid=%hu ", ifle->vid);

    return(0);
}

/*
 * parse attribute IFLA_VLAN_EGRESS_QOS
 */
int parse_ifla_vlan_egress_qos(char *msg, char **mp, struct rtattr *vlan, struct iflist_entry *ifle)
{
    if(msg) {
        *mp = add_log(msg, *mp, "egress-qos-map(from:to)=");

        if(parse_vlan_qos_mapping(msg, mp, vlan, ifle))
            return(1);
    }

    return(0);
}

/*
 * parse attribute IFLA_VLAN_INGRESS_QOS
 */
int parse_ifla_vlan_ingress_qos(char *msg, char **mp, struct rtattr *vlan, struct iflist_entry *ifle)
{
    if(msg) {
        *mp = add_log(msg, *mp, "ingress-qos-map(from:to)=");

        if(parse_vlan_qos_mapping(msg, mp, vlan, ifle))
            return(1);
    }

    return(0);

}

/*
 * parse VLAN QoS informaton messages
 */
int parse_vlan_qos_mapping(char *msg, char **mp, struct rtattr *qos, struct iflist_entry *ifle)
{
    struct ifla_vlan_qos_mapping *map;
    int len = RTA_PAYLOAD(qos);

    *mp = add_log(msg, *mp, "(");

    for(qos = RTA_DATA(qos); RTA_OK(qos, len); qos = RTA_NEXT(qos, len)) {
        if(RTA_PAYLOAD(qos) < sizeof(*map)) {
            rec_log("error: %s: ifindex %d: payload too short",
                __func__, ifle->index);
            return(1);
        }
        map = (struct ifla_vlan_qos_mapping *)RTA_DATA(qos);

        *mp = add_log(msg, *mp, "%u:%u ", map->from, map->to);
    }

    --(*mp);
    *mp = add_log(msg, *mp, ") ");

    return(0);
}
#endif

#if HAVE_DECL_IFLA_GRE_UNSPEC
/*
 * parse attributes IFLA_GRE_*
 */
int parse_ifla_gre(char *msg, char **mp, struct rtattr *linkinfo, struct iflist_entry *ifle)
{
    struct rtattr *gre[__IFLA_GRE_MAX];

    parse_gre(gre, linkinfo);

    if(gre[IFLA_GRE_LOCAL])
        if(parse_ifla_gre_local(msg, mp, gre[IFLA_GRE_LOCAL], ifle))
            return(1);

    if(gre[IFLA_GRE_REMOTE])
        if(parse_ifla_gre_remote(msg, mp, gre[IFLA_GRE_REMOTE], ifle))
            return(1);

    return(0);
}

/*
 * parse attribute IFLA_GRE_LOCAL
 */
int parse_ifla_gre_local(char *msg, char **mp, struct rtattr *gre, struct iflist_entry *ifle)
{
    char addr[INET6_ADDRSTRLEN+1] = "";
    int res;

    if(ifle->type == ARPHRD_ETHER)
        res = inet_ntop_ifi(ARPHRD_IPGRE, gre, addr, sizeof(addr));
    else
        res = inet_ntop_ifi(ifle->type, gre, addr, sizeof(addr));

    if(res) {
        rec_log("error: %s: IFLA_GRE_LOCAL(ifindex %d): %s",
            __func__, ifle->index,
            (res == 1) ? strerror(errno) : "payload too short");
        return(1);
    }

    if(msg)
        *mp = add_log(msg, *mp, "local=%s ", addr);

    return(0);
}

/*
 * parse attribute IFLA_GRE_REMOTE
 */
int parse_ifla_gre_remote(char *msg, char **mp, struct rtattr *gre, struct iflist_entry *ifle)
{
    char addr[INET6_ADDRSTRLEN+1] = "";
    int res;

    if(ifle->type == ARPHRD_ETHER)
        res = inet_ntop_ifi(ARPHRD_IPGRE, gre, addr, sizeof(addr));
    else
        res = inet_ntop_ifi(ifle->type, gre, addr, sizeof(addr));

    if(res) {
        rec_log("error: %s: IFLA_GRE_REMOTE(ifindex %d): %s",
            __func__, ifle->index,
            (res == 1) ?strerror(errno) : "payload too short");
        return(1);
    }

    if(msg)
        *mp = add_log(msg, *mp, "remote=%s ", addr);

    return(0);
}
#endif

#if HAVE_DECL_IFLA_MACVLAN_UNSPEC
/*
 * parse IFLA_MACVLAN_*
 */
int parse_ifla_macvlan(char *msg, char **mp, struct rtattr *linkinfo, struct iflist_entry *ifle)
{
    struct rtattr *macvlan[__IFLA_MACVLAN_MAX];

    parse_macvlan(macvlan, linkinfo);

    if(macvlan[IFLA_MACVLAN_MODE])
        if(parse_ifla_macvlan_mode(msg, mp, macvlan[IFLA_MACVLAN_MODE], ifle))
            return(1);

    return(0);
}

/*
 * parse attribute IFLA_MACVLAN_MODE
 */
int parse_ifla_macvlan_mode(char *msg, char **mp, struct rtattr *macvlan,
    struct iflist_entry *ifle)
{
    if(RTA_PAYLOAD(macvlan) < sizeof(unsigned)) {
        rec_log("error: %s: ifindex %d: payload too short",
            __func__, ifle->index);
        return(1);
    }

    if(msg)
        *mp = add_log(msg, *mp, "mode=%s ",
            convert_macvlan_mode(*(unsigned *)RTA_DATA(macvlan), 0));

    return(0);
}
#endif

#if HAVE_DECL_IFLA_VXLAN_UNSPEC
/*
 * parse attributes IFLA_VLXAN_*
 */
int parse_ifla_vxlan(char *msg, char **mp, struct rtattr *linkinfo, struct iflist_entry *ifle)
{
    struct rtattr *vxlan[__IFLA_VXLAN_MAX];

    parse_vxlan(vxlan, linkinfo);

    if(vxlan[IFLA_VXLAN_ID])
        if(parse_ifla_vxlan_id(msg, mp, vxlan[IFLA_VXLAN_ID], ifle))
            return(1);

    if(vxlan[IFLA_VXLAN_LINK])
        if(parse_ifla_vxlan_link(msg, mp, vxlan[IFLA_VXLAN_LINK], ifle))
            return(1);

    if(vxlan[IFLA_VXLAN_LOCAL])
        if(parse_ifla_vxlan_local(msg, mp, vxlan[IFLA_VXLAN_LOCAL], ifle))
            return(1);

    if(vxlan[IFLA_VXLAN_GROUP])
        if(parse_ifla_vxlan_group(msg, mp, vxlan[IFLA_VXLAN_GROUP], ifle))
            return(1);

    return(0);
}

/*
 * parse attribute IFLA_VXLAN_ID
 */
int parse_ifla_vxlan_id(char *msg, char **mp, struct rtattr *vxlan, struct iflist_entry *ifle)
{
    if(RTA_PAYLOAD(vxlan) < sizeof(unsigned)) {
        rec_log("error: %s: ifindex %d: payload too short",
            __func__, ifle->index);
        return(1);
    }

    if(msg)
        *mp = add_log(msg, *mp, "vnid=%u ", *(unsigned *)RTA_DATA(vxlan));

    return(0);
}

/*
 * parse attribute IFLA_VXLAN_LINK
 */
int parse_ifla_vxlan_link(char *msg, char **mp, struct rtattr *vxlan, struct iflist_entry *ifle)
{
    char name[IFNAMSIZ];

    if(RTA_PAYLOAD(vxlan) < sizeof(int)) {
        rec_log("error: %s: ifindex %d: payload too short",
            __func__, ifle->index);
        return(1);
    }
    if_indextoname_from_lists(*(int *)RTA_DATA(vxlan), name);

    if(msg)
        *mp = add_log(msg, *mp, "link=%s ", name);

    return(0);
}

/*
 * parse attribute IFLA_VXLAN_LOCAL
 */
int parse_ifla_vxlan_local(char *msg, char **mp, struct rtattr *vxlan, struct iflist_entry *ifle)
{
    char addr[INET_ADDRSTRLEN+1] = "";
    int res;

    /* not inet_ntop_ifi */
    res = inet_ntop_ifa(AF_INET, vxlan, addr, sizeof(addr));
    if(res) {
        rec_log("error: %s: ifindex %d: payload too short",
            __func__, ifle->index);
        return(1);
    }

    if(msg)
        *mp = add_log(msg, *mp, "local=%s ", addr);

    return(0);
}

#if HAVE_DECL_IFLA_VXLAN_GROUP
/*
 * parse attribute IFLA_VXLAN_GROUP
 */
int parse_ifla_vxlan_group(char *msg, char **mp, struct rtattr *vxlan, struct iflist_entry *ifle)
{
    char addr[INET_ADDRSTRLEN+1] = "";
    int res;

    /* not inet_ntop_ifi */
    res = inet_ntop_ifa(AF_INET, vxlan, addr, sizeof(addr));
    if(res) {
        rec_log("error: %s: ifindex %d: payload too short",
            __func__, ifle->index);
        return(1);
    }

    if(msg)
        *mp = add_log(msg, *mp, "group=%s ", addr);

    return(0);
}
#endif
#endif

/*
 * parse attribute IFLA_MASTER
 */
int parse_ifla_master(char *msg, char **mp, struct rtattr *ifla, struct iflist_entry *ifle)
{
    if(RTA_PAYLOAD(ifla) < sizeof(ifle->index_master)) {
        rec_log("error: %s: IFLA_MASTER(ifindex %d): payload too short",
            __func__, ifle->index);
        return(1);
    }
    ifle->index_master = *(int *)RTA_DATA(ifla);
    if_indextoname_from_lists(ifle->index_master, ifle->name_master);

    if(msg)
        *mp = add_log(msg, *mp, "master=%s ", ifle->name_master);

    return(0);
}

/*
 * convert interface address from binary to text
 */ 
int inet_ntop_ifi(unsigned short type, struct rtattr *ifla, char *saddr, int slen)
{
    unsigned char *addr = RTA_DATA(ifla);
    int i, len = RTA_PAYLOAD(ifla);
    char *p = saddr;

    switch(type) {
        case ARPHRD_TUNNEL:
        case ARPHRD_IPGRE:
        case ARPHRD_SIT:
            if(len < 4)
                return(2);
            if(!inet_ntop(AF_INET, addr, saddr, slen))
                return(1);
            return(0);
        case ARPHRD_TUNNEL6:
#ifdef ARPHRD_IP6GRE
        case ARPHRD_IP6GRE:
#endif
            if(len < 16)
                return(2);
            if(!inet_ntop(AF_INET6, addr, saddr, slen))
                return(1);
            return(0);
    }

    for(i = 0; i < len; i++)
        if(p - saddr < slen)
            p += snprintf(p, slen - strlen(saddr), "%02x%s",
                addr[i], (i + 1 == len) ? "" : ":");

    return(0);
}

/*
 * debug interface information message
 */ 
void debug_ifimsg(int lev, struct ifinfomsg *ifim, struct rtattr *ifla[], int ifim_len)
{
    /* debug ifinfomsg */
    char ifname[IFNAMSIZ] = "";
    char flags_list[MAX_STR_SIZE] = "";

    if_indextoname_from_lists(ifim->ifi_index, ifname);
    convert_iff_flags(ifim->ifi_flags, flags_list, sizeof(flags_list));

    rec_dbg(lev, "*********************************************************************");
    rec_dbg(lev, "[ ifinfomsg(%d) ]",
        NLMSG_ALIGN(sizeof(struct ifinfomsg)));
    rec_dbg(lev, "    ifi_family(%d): %d(%s)",
        sizeof(ifim->ifi_family), ifim->ifi_family,
        convert_af_type(ifim->ifi_family));
    rec_dbg(lev, "    __ifi_pad(%d): %d",
        sizeof(ifim->__ifi_pad), ifim->__ifi_pad);
    rec_dbg(lev, "    ifi_type(%d): %hu(%s)",
        sizeof(ifim->ifi_type), ifim->ifi_type,
        convert_arphrd_type(ifim->ifi_type));
    rec_dbg(lev, "    ifi_index(%d): %d(%s)",
        sizeof(ifim->ifi_index), ifim->ifi_index, ifname);
    rec_dbg(lev, "    ifi_flags(%d): 0x%.8x(%s)",
        sizeof(ifim->ifi_flags), ifim->ifi_flags, flags_list);
    rec_dbg(lev, "    ifi_change(%d): %u",
        sizeof(ifim->ifi_change), ifim->ifi_change);

    /* debug interface link attributes */
    rec_dbg(lev, "*********************************************************************");
    rec_dbg(lev, "[ ifinfomsg attributes(%d) ]",
        NLMSG_ALIGN(ifim_len - NLMSG_ALIGN(sizeof(struct ifinfomsg))));

    if(ifla[IFLA_ADDRESS])
        debug_ifla_address(lev+1, ifim, ifla[IFLA_ADDRESS]);

    if(ifla[IFLA_BROADCAST])
        debug_ifla_broadcast(lev+1, ifim, ifla[IFLA_BROADCAST]);

    if(ifla[IFLA_IFNAME])
        debug_ifla_ifname(lev+1, ifla[IFLA_IFNAME]);

    if(ifla[IFLA_MTU])
        debug_ifla_mtu(lev+1, ifla[IFLA_MTU]);

    if(ifla[IFLA_LINK])
        debug_ifla_link(lev+1, ifla[IFLA_LINK]);

    if(ifla[IFLA_QDISC])
        debug_ifla_qdisc(lev+1, ifla[IFLA_QDISC]);

    if(ifla[IFLA_STATS])
        debug_ifla_stats(lev+1, ifla[IFLA_STATS]);

    if(ifla[IFLA_COST])
        debug_ifla_cost(lev+1, ifla[IFLA_COST]);

    if(ifla[IFLA_PRIORITY])
        debug_ifla_priority(lev+1, ifla[IFLA_PRIORITY]);

    if(ifla[IFLA_MASTER])
        debug_ifla_master(lev+1, ifla[IFLA_MASTER]);

    if(ifla[IFLA_WIRELESS])
        debug_ifla_wireless(lev+1, ifla[IFLA_WIRELESS]);

    if(ifla[IFLA_PROTINFO])
        debug_ifla_protoinfo(lev+1, ifla[IFLA_PROTINFO]);

    if(ifla[IFLA_TXQLEN])
        debug_ifla_txqlen(lev+1, ifla[IFLA_TXQLEN]);

    if(ifla[IFLA_MAP])
        debug_ifla_map(lev+1, ifla[IFLA_MAP]);

    if(ifla[IFLA_WEIGHT])
        debug_ifla_weight(lev+1, ifla[IFLA_WEIGHT]);

    if(ifla[IFLA_OPERSTATE])
        debug_ifla_operstate(lev+1, ifla[IFLA_OPERSTATE]);

    if(ifla[IFLA_LINKMODE])
        debug_ifla_linkmode(lev+1, ifla[IFLA_LINKMODE]);

#if HAVE_DECL_IFLA_LINKINFO
    if(ifla[IFLA_LINKINFO])
        debug_ifla_linkinfo(lev+1, ifim, ifla[IFLA_LINKINFO]);
#endif

#if HAVE_DECL_IFLA_NET_NS_PID
    if(ifla[IFLA_NET_NS_PID])
        debug_ifla_net_ns_pid(lev+1, ifla[IFLA_NET_NS_PID]);
#endif

#if HAVE_DECL_IFLA_IFALIAS
    if(ifla[IFLA_IFALIAS])
        debug_ifla_ifalias(lev+1, ifla[IFLA_IFALIAS]);
#endif

#if HAVE_DECL_IFLA_NUM_VF
    if(ifla[IFLA_NUM_VF])
        debug_ifla_num_vf(lev+1, ifla[IFLA_NUM_VF]);
#endif

#if HAVE_DECL_IFLA_VFINFO_LIST
    if(ifla[IFLA_VFINFO_LIST])
        debug_ifla_vfinfo_list(lev+1, ifla[IFLA_VFINFO_LIST]);
#endif

#if HAVE_DECL_IFLA_STATS64
    if(ifla[IFLA_STATS64])
        debug_ifla_stats64(lev+1, ifla[IFLA_STATS64]);
#endif

#if HAVE_DECL_IFLA_VF_PORTS
    if(ifla[IFLA_VF_PORTS])
        debug_ifla_vf_ports(lev+1, ifla[IFLA_VF_PORTS]);
#endif

#if HAVE_DECL_IFLA_PORT_SELF
    if(ifla[IFLA_PORT_SELF])
        debug_ifla_port_self(lev+1, ifla[IFLA_PORT_SELF]);
#endif

#if HAVE_DECL_IFLA_AF_SPEC
    if(ifla[IFLA_AF_SPEC])
        debug_ifla_af_spec(lev+1, ifla[IFLA_AF_SPEC]);
#endif

#if HAVE_DECL_IFLA_GROUP
    if(ifla[IFLA_GROUP])
        debug_ifla_group(lev+1, ifla[IFLA_GROUP]);
#endif

#if HAVE_DECL_IFLA_NET_NS_FD
    if(ifla[IFLA_NET_NS_FD])
        debug_ifla_net_ns_fd(lev+1, ifla[IFLA_NET_NS_FD]);
#endif

#if HAVE_DECL_IFLA_EXT_MASK
    if(ifla[IFLA_EXT_MASK])
        debug_ifla_ext_mask(lev+1, ifla[IFLA_EXT_MASK]);
#endif

#if HAVE_DECL_IFLA_PROMISCUITY
    if(ifla[IFLA_PROMISCUITY])
        debug_ifla_promiscuity(lev+1, ifla[IFLA_PROMISCUITY]);
#endif

#if HAVE_DECL_IFLA_NUM_TX_QUEUES
    if(ifla[IFLA_NUM_TX_QUEUES])
        debug_ifla_num_tx_queues(lev+1, ifla[IFLA_NUM_TX_QUEUES]);
#endif

#if HAVE_DECL_IFLA_NUM_RX_QUEUES
    if(ifla[IFLA_NUM_RX_QUEUES])
        debug_ifla_num_rx_queues(lev+1, ifla[IFLA_NUM_RX_QUEUES]);
#endif

    rec_dbg(lev, "");

    return;
}

/*
 * debug attribute IFLA_ADDRESS
 */ 
void debug_ifla_address(int lev, struct ifinfomsg *ifim, struct rtattr *ifla)
{
    char addr[INET6_ADDRSTRLEN+1] = "";
    int res;

    res = inet_ntop_ifi(ifim->ifi_type, ifla, addr, sizeof(addr));
    if(res) {
        rec_dbg(lev, "IFLA_ADDRESS(%hu): -- %s --",
            RTA_ALIGN(ifla->rta_len),
            (res == 1) ? strerror(errno) : "payload too short");
        return;
    }

    rec_dbg(lev, "IFLA_ADDRESS(%hu): %s", RTA_ALIGN(ifla->rta_len), addr);
}

/*
 * debug attribute IFLA_BROADCAST
 */ 
void debug_ifla_broadcast(int lev, struct ifinfomsg *ifim, struct rtattr *ifla)
{
    char brd[INET6_ADDRSTRLEN+1] = "";
    int res;

    res = inet_ntop_ifi(ifim->ifi_type, ifla, brd, sizeof(brd));
    if(res) {
        rec_dbg(lev, "IFLA_BROADCAST(%hu): -- %s --",
            RTA_ALIGN(ifla->rta_len),
            (res == 1) ? strerror(errno) : "payload too short");
        return;
    }

    rec_dbg(lev, "IFLA_BROADCAST(%hu): %s", RTA_ALIGN(ifla->rta_len), brd);
}

/*
 * debug attribute IFLA_IFNAME
 */ 
void debug_ifla_ifname(int lev, struct rtattr *ifla)
{
    char name[IFNAMSIZ] = "";

    if(!RTA_PAYLOAD(ifla)) {
        rec_dbg(lev, "IFLA_IFNAME(%hu): -- no payload --",
            RTA_ALIGN(ifla->rta_len));
        return;
    } else if(RTA_PAYLOAD(ifla) > sizeof(name)) {
        rec_dbg(lev, "IFLA_IFNAME(%hu): -- payload too long --",
            RTA_ALIGN(ifla->rta_len));
        return;
    }
    strncpy(name, (char *)RTA_DATA(ifla), sizeof(name));

    rec_dbg(lev, "IFLA_IFNAME(%hu): %s", RTA_ALIGN(ifla->rta_len), name);
}

/*
 * debug attribute IFLA_MTU
 */ 
void debug_ifla_mtu(int lev, struct rtattr *ifla)
{
    if(RTA_PAYLOAD(ifla) < sizeof(int)) {
        rec_dbg(lev, "IFLA_MTU(%hu): -- payload too short --",
            RTA_ALIGN(ifla->rta_len));
        return;
    }
    rec_dbg(lev, "IFLA_MTU(%hu): %d",
        RTA_ALIGN(ifla->rta_len), *(int *)RTA_DATA(ifla));
}

/*
 * debug attribute IFLA_LINK
 */ 
void debug_ifla_link(int lev, struct rtattr *ifla)
{
    int index;
    char name[IFNAMSIZ] = "";

    if(RTA_PAYLOAD(ifla) < sizeof(int)) {
        rec_dbg(lev, "IFLA_LINK(%hu): -- payload too short --",
            RTA_ALIGN(ifla->rta_len));
        return;
    }
    index = *(unsigned *)RTA_DATA(ifla);
    if_indextoname_from_lists(index, name);

    rec_dbg(lev, "IFLA_LINK(%hu): %u(%s)",
        RTA_ALIGN(ifla->rta_len), index, name);
}

/*
 * debug attribute IFLA_QDISC
 */
void debug_ifla_qdisc(int lev, struct rtattr *ifla)
{
    char qdisc[IFNAMSIZ] = "";

    if(!RTA_PAYLOAD(ifla))
        strncpy(qdisc, "-- no payload --", sizeof(qdisc));
    else if(RTA_PAYLOAD(ifla) > sizeof(qdisc))
        strncpy(qdisc, "-- payload too long --", sizeof(qdisc));
    else
        strncpy(qdisc, (char *)RTA_DATA(ifla), sizeof(qdisc));

    rec_dbg(lev, "IFLA_QDISC(%hu): %s", RTA_ALIGN(ifla->rta_len), qdisc);
}

/*
 * debug attribute IFLA_STATS
 */ 
void debug_ifla_stats(int lev, struct rtattr *ifla)
{
    struct rtnl_link_stats *stats;

    if(RTA_PAYLOAD(ifla) < sizeof(*stats)) {
        rec_dbg(lev, "    IFLA_STATS(%hu): -- payload too short --",
            RTA_ALIGN(ifla->rta_len));
        return;
    }
    stats = (struct rtnl_link_stats *)RTA_DATA(ifla);

    rec_dbg(lev, "IFLA_STATS(%hu):", RTA_ALIGN(ifla->rta_len));
    rec_dbg(lev, "    [ rtnl_link_stats(%d) ]", sizeof(*stats));
    rec_dbg(lev, "        rx_packets(%d): %u", sizeof(stats->rx_packets), stats->rx_packets);
    rec_dbg(lev, "        tx_packets(%d): %u", sizeof(stats->tx_packets), stats->tx_packets);
    rec_dbg(lev, "        rx_bytes(%d): %u", sizeof(stats->rx_bytes), stats->rx_bytes);
    rec_dbg(lev, "        tx_bytes(%d): %u", sizeof(stats->tx_bytes), stats->tx_bytes);
    rec_dbg(lev, "        rx_errors(%d): %u", sizeof(stats->rx_errors), stats->rx_errors);
    rec_dbg(lev, "        tx_errors(%d): %u", sizeof(stats->tx_errors), stats->tx_errors);
    rec_dbg(lev, "        rx_dropped(%d): %u", sizeof(stats->rx_dropped), stats->rx_dropped);
    rec_dbg(lev, "        tx_dropped(%d): %u", sizeof(stats->tx_dropped), stats->tx_dropped);
    rec_dbg(lev, "        multicast(%d): %u", sizeof(stats->multicast), stats->multicast);
    rec_dbg(lev, "        collisions(%d): %u", sizeof(stats->collisions), stats->collisions);
    rec_dbg(lev, "        rx_length_errors(%d): %u",
        sizeof(stats->rx_length_errors), stats->rx_length_errors);
    rec_dbg(lev, "        rx_over_errors(%d): %u",
        sizeof(stats->rx_over_errors), stats->rx_over_errors);
    rec_dbg(lev, "        rx_crc_errors(%d): %u",
        sizeof(stats->rx_crc_errors), stats->rx_crc_errors);
    rec_dbg(lev, "        rx_frame_errors(%d): %u",
        sizeof(stats->rx_frame_errors), stats->rx_frame_errors);
    rec_dbg(lev, "        rx_fifo_errors(%d): %u",
        sizeof(stats->rx_fifo_errors), stats->rx_fifo_errors);
    rec_dbg(lev, "        rx_missed_errors(%d): %u",
        sizeof(stats->rx_missed_errors), stats->rx_missed_errors);
    rec_dbg(lev, "        tx_aborted_errors(%d): %u",
        sizeof(stats->tx_aborted_errors), stats->tx_aborted_errors);
    rec_dbg(lev, "        tx_carrier_errors(%d): %u",
        sizeof(stats->tx_carrier_errors), stats->tx_carrier_errors);
    rec_dbg(lev, "        tx_fifo_errors(%d): %u",
        sizeof(stats->tx_fifo_errors), stats->tx_fifo_errors);
    rec_dbg(lev, "        tx_heartbeat_errors(%d): %u",
        sizeof(stats->tx_heartbeat_errors), stats->tx_heartbeat_errors);
    rec_dbg(lev, "        tx_window_errors(%d): %u",
        sizeof(stats->tx_window_errors), stats->tx_window_errors);
    rec_dbg(lev, "        rx_compressed(%d): %u",
        sizeof(stats->rx_compressed), stats->rx_compressed);
    rec_dbg(lev, "        tx_compressed(%d): %u",
        sizeof(stats->tx_compressed), stats->tx_compressed);
}

/*
 * debug attribute IFLA_COST
 */ 
void debug_ifla_cost(int lev, struct rtattr *ifla) {
    if(RTA_PAYLOAD(ifla) < sizeof(unsigned)) {
        rec_dbg(lev, "IFLA_COST(%hu): -- payload too short --",
            RTA_ALIGN(ifla->rta_len));
        return;
    }
    rec_dbg(lev, "IFLA_COST(%hu): %u",
        RTA_ALIGN(ifla->rta_len), *(unsigned *)RTA_DATA(ifla));
}

/*
 * debug attribute IFLA_PRIORITY
 */ 
void debug_ifla_priority(int lev, struct rtattr *ifla)
{
    if(RTA_PAYLOAD(ifla) < sizeof(unsigned)) {
        rec_dbg(lev, "IFLA_PRIORITY(%hu): -- payload too short --",
            RTA_ALIGN(ifla->rta_len));
        return;
    }
    rec_dbg(lev, "IFLA_PRIORITY(%hu): %u",
        RTA_ALIGN(ifla->rta_len), *(unsigned *)RTA_DATA(ifla));
}

/*
 * debug attribute IFLA_MASTER
 */ 
void debug_ifla_master(int lev, struct rtattr *ifla)
{
    int index;
    char name[IFNAMSIZ] = "";

    if(RTA_PAYLOAD(ifla) < sizeof(index)) {
        rec_dbg(lev, "IFLA_MASTER(%hu): -- payload too short --",
            RTA_ALIGN(ifla->rta_len));
        return;
    }
    index = *(unsigned *)RTA_DATA(ifla);
    if_indextoname_from_lists(index, name);

    rec_dbg(lev, "IFLA_MASTER(%hu): %d(%s)",
        RTA_ALIGN(ifla->rta_len), index, name);
}

/*
 * debug attribute IFLA_WIRELESS
 */ 
void debug_ifla_wireless(int lev, struct rtattr *ifla)
{
    rec_dbg(lev, "IFLA_WIRELESS(%hu): -- ignored --",
    RTA_ALIGN(ifla->rta_len));
}

/*
 * debug attribute IFLA_PROTOINFO
 */ 
void debug_ifla_protoinfo(int lev, struct rtattr *ifla)
{
    rec_dbg(lev, "IFLA_PROTINFO(%hu): -- ignored --",
        RTA_ALIGN(ifla->rta_len));
}

/*
 * debug attribute IFLA_TXQLEN
 */ 
void debug_ifla_txqlen(int lev, struct rtattr *ifla)
{
    if(RTA_PAYLOAD(ifla) < sizeof(unsigned)) {
        rec_dbg(lev, "IFLA_TXQLEN(%hu): -- payload too short --",
            RTA_ALIGN(ifla->rta_len));
        return;
    }
    rec_dbg(lev, "IFLA_TXQLEN(%hu): %u",
        RTA_ALIGN(ifla->rta_len), *(unsigned *)RTA_DATA(ifla));
}

/*
 * debug attribute IFLA_MAP
 */ 
void debug_ifla_map(int lev, struct rtattr *ifla)
{
    struct rtnl_link_ifmap *ifmap;

    if(RTA_PAYLOAD(ifla) < sizeof(*ifmap)) {
        rec_dbg(lev, "IFLA_MAP(%hu): -- payload too short --",
            RTA_ALIGN(ifla->rta_len));
        return;
    }
    ifmap = (struct rtnl_link_ifmap *)RTA_DATA(ifla);

    rec_dbg(lev, "IFLA_MAP(%hu):", RTA_ALIGN(ifla->rta_len));
    rec_dbg(lev, "    [ rtnl_link_ifmap(%d) ]", sizeof(*ifmap));
    rec_dbg(lev, "        mem_start(%d): 0x%016x",
        sizeof(ifmap->mem_start), ifmap->mem_start);
    rec_dbg(lev, "        mem_end(%d): 0x%016x",
        sizeof(ifmap->mem_end), ifmap->mem_end);
    rec_dbg(lev, "        base_addr(%d): 0x%016x",
        sizeof(ifmap->base_addr), ifmap->base_addr);
    rec_dbg(lev, "        irq(%d): %hu", sizeof(ifmap->irq), ifmap->irq);
    rec_dbg(lev, "        dma(%d): 0x%02x", sizeof(ifmap->dma), ifmap->dma);
    rec_dbg(lev, "        port(%d): 0x%02x", sizeof(ifmap->port), ifmap->port);
}

/*
 * debug attribute IFLA_WEIGHT
 */ 
void debug_ifla_weight(int lev, struct rtattr *ifla)
{
    if(RTA_PAYLOAD(ifla) < sizeof(unsigned)) {
        rec_dbg(lev, "IFLA_WEIGHT(%hu): -- payload too short --",
            RTA_ALIGN(ifla->rta_len));
        return;
    }
    rec_dbg(lev, "IFLA_WEIGHT(%hu): %u",
        RTA_ALIGN(ifla->rta_len), *(unsigned *)RTA_DATA(ifla));
}

/*
 * debug attribute IFLA_OPERSTATE
 */ 
void debug_ifla_operstate(int lev, struct rtattr *ifla)
{
    if(RTA_PAYLOAD(ifla) < sizeof(unsigned char)) {
        rec_dbg(lev, "IFLA_OPERSTATE(%hu): -- payload too short --",
            RTA_ALIGN(ifla->rta_len));
        return;
    }
    rec_dbg(lev, "IFLA_OPERSTATE(%hu): %d(%s)",
        RTA_ALIGN(ifla->rta_len), *(int *)RTA_DATA(ifla),
        convert_if_oper_state(*(int *)RTA_DATA(ifla)));
}

/*
 * debug attribute IFLA_LINKMODE
 */ 
void debug_ifla_linkmode(int lev, struct rtattr *ifla)
{
    if(RTA_PAYLOAD(ifla) < sizeof(unsigned char)) {
        rec_dbg(lev, "IFLA_LINKMODE(%hu): -- payload too short --",
            RTA_ALIGN(ifla->rta_len));
        return;
    }
    rec_dbg(lev, "IFLA_LINKMODE(%hu): %d(%s)",
        RTA_ALIGN(ifla->rta_len), *(int *)RTA_DATA(ifla),
        convert_if_link_mode(*(int *)RTA_DATA(ifla)));
}

#if HAVE_DECL_IFLA_LINKINFO
/*
 * debug attribute IFLA_LINKINFO
 *
 * IFLA_LINKINFO was committed in kernel 2.6.23.
 *
 * commit 38f7b870d4a6a5d3ec21557e849620cb7d032965
 * Author: Patrick McHardy <kaber@trash.net>
 * Date:   Wed Jun 13 14:03:51 2007 -0700
 *
 *     [RTNETLINK]: Link creation API
 */
void debug_ifla_linkinfo(int lev, struct ifinfomsg *ifim, struct rtattr *ifla)
{
    struct rtattr *linkinfo[__IFLA_INFO_MAX];
    char kind[MODULE_NAME_LEN] = "";

    parse_linkinfo(linkinfo, ifla);

    rec_dbg(lev, "IFLA_LINKINFO(%hu):", RTA_ALIGN(ifla->rta_len));

    if(linkinfo[IFLA_INFO_KIND])
        debug_ifla_info_kind(lev+1, linkinfo[IFLA_INFO_KIND], kind, sizeof(kind));

    if(linkinfo[IFLA_INFO_DATA])
        debug_ifla_info_data(lev+1, ifim, linkinfo[IFLA_INFO_DATA], kind, sizeof(kind));

    if(linkinfo[IFLA_INFO_XSTATS])
        debug_ifla_info_xstats(lev+1, linkinfo[IFLA_INFO_XSTATS]);
}

/*
 * debug attribute IFLA_INFO_KIND
 */
void debug_ifla_info_kind(int lev, struct rtattr *linkinfo, char *kind, int len)
{
    if(!RTA_PAYLOAD(linkinfo)) {
        rec_dbg(lev, "IFLA_INFO_KIND(%hu): -- payload too short --",
            RTA_ALIGN(linkinfo->rta_len));
        return;
    } else if(RTA_PAYLOAD(linkinfo) > len) {
        rec_dbg(lev, "IFLA_INFO_KIND(%hu): -- payload too long --",
            RTA_ALIGN(linkinfo->rta_len));
        return;
    }
    strncpy(kind, RTA_DATA(linkinfo), len);

    rec_dbg(lev, "IFLA_INFO_KIND(%hu): %s", RTA_ALIGN(linkinfo->rta_len), kind);
}

/*
 * debug attribute IFLA_INFO_DATA
 */
void debug_ifla_info_data(int lev, struct ifinfomsg *ifim, struct rtattr *linkinfo,
    char *kind, int len)
{
#if HAVE_DECL_IFLA_VLAN_UNSPEC
    if(!strncmp(kind, "vlan", len)) {
        debug_ifla_vlan(lev, linkinfo);
        return;
    }
#endif

#if HAVE_DECL_IFLA_GRE_UNSPEC
    if(!strncmp(kind, "gre", len)) {
        debug_ifla_gre(lev, ifim, linkinfo);
        return;
    }

    if(!strncmp(kind, "gretap", len)) {
        debug_ifla_gre(lev, ifim, linkinfo);
        return;
    }
#endif

#if HAVE_DECL_IFLA_MACVLAN_UNSPEC
    if(!strncmp(kind, "macvlan", len)) {
        debug_ifla_macvlan(lev, linkinfo);
        return;
    }

    if(!strncmp(kind, "macvtap", len)) {
        debug_ifla_macvlan(lev, linkinfo);
        return;
    }
#endif

#if HAVE_DECL_IFLA_VXLAN_UNSPEC
    if(!strncmp(kind, "vxlan", len)) { 
        debug_ifla_vxlan(lev, linkinfo);
        return;
    }
#endif

    rec_dbg(lev, "IFLA_INFO_DATA(%hu): -- ignored --",
        RTA_ALIGN(linkinfo->rta_len));
}

/*
 * debug attribute IFLA_INFO_XSTATS
 */
void debug_ifla_info_xstats(int lev, struct rtattr *linkinfo)
{
    rec_dbg(lev, "IFLA_INFO_XSTATS(%hu): -- ignored --",
        RTA_ALIGN(linkinfo->rta_len));
}
#endif

#if HAVE_DECL_IFLA_VLAN_UNSPEC
/*
 * debug VLAN interface information messages
 */
void debug_ifla_vlan(int lev, struct rtattr *linkinfo)
{
    struct rtattr *vlan[__IFLA_VLAN_MAX];

    rec_dbg(lev, "IFLA_INFO_DATA(%hu):", RTA_ALIGN(linkinfo->rta_len));

    parse_vlan(vlan, linkinfo);

    if(vlan[IFLA_VLAN_ID])
        debug_ifla_vlan_id(lev+1, vlan[IFLA_VLAN_ID]);

    if(vlan[IFLA_VLAN_FLAGS])
        debug_ifla_vlan_flags(lev+1, vlan[IFLA_VLAN_FLAGS]);

    if(vlan[IFLA_VLAN_EGRESS_QOS])
        debug_ifla_vlan_egress_qos(lev+1, vlan[IFLA_VLAN_EGRESS_QOS]);

    if(vlan[IFLA_VLAN_INGRESS_QOS])
        debug_ifla_vlan_ingress_qos(lev+1, vlan[IFLA_VLAN_INGRESS_QOS]);
}

/*
 * debug attribute IFLA_VLAN_ID
 */
void debug_ifla_vlan_id(int lev, struct rtattr *vlan)
{
    if(RTA_PAYLOAD(vlan) < sizeof(unsigned short)) {
        rec_dbg(lev, "IFLA_VLAN_ID(%hu): -- payload too short --",
            RTA_ALIGN(vlan->rta_len));
        return;
    }
    rec_dbg(lev, "IFLA_VLAN_ID(%hu): %hu",
        RTA_ALIGN(vlan->rta_len), *(unsigned short *)RTA_DATA(vlan));
}

/*
 * debug attribute IFLA_VLAN_FLAGS
 */
void debug_ifla_vlan_flags(int lev, struct rtattr *vlan)
{
    struct ifla_vlan_flags *flags;
    char flags_list[MAX_STR_SIZE] = "";

    if(RTA_PAYLOAD(vlan) < sizeof(*flags)) {
        rec_dbg(lev, "IFLA_VLAN_FLAGS(%hu): -- payload too short --",
            RTA_ALIGN(vlan->rta_len));
        return;
    }

    flags = (struct ifla_vlan_flags *)RTA_DATA(vlan);
    convert_vlan_flags(flags->flags, flags_list, sizeof(flags_list));

    rec_dbg(lev, "IFLA_VLAN_FLAGS(%hu):", RTA_ALIGN(vlan->rta_len));
    rec_dbg(lev, "    [ ifla_flags(%d) ]", sizeof(struct ifla_vlan_flags));
    rec_dbg(lev, "        flags(%d): 0x%.8x(%s)",
        sizeof(flags->flags), flags->flags, flags_list);
    rec_dbg(lev, "        mask(%d): 0x%.8x", sizeof(flags->mask), flags->mask);
}

/*
 * debug attribute IFLA_VLAN_EGRESS_QOS
 */
void debug_ifla_vlan_egress_qos(int lev, struct rtattr *vlan)
{
    rec_dbg(lev, "IFLA_VLAN_EGRESS_QOS(%hu):", RTA_ALIGN(vlan->rta_len));

    debug_vlan_qos_mapping(lev+1, vlan);
}

/*
 * debug attribute IFLA_VLAN_INGRESS_QOS
 */
void debug_ifla_vlan_ingress_qos(int lev, struct rtattr *vlan)
{
    struct rtattr *vlan_ingress_qos[__IFLA_VLAN_QOS_MAX];

    rec_dbg(lev, "IFLA_VLAN_INGRESS_QOS(%hu):", RTA_ALIGN(vlan->rta_len));

    debug_vlan_qos_mapping(lev+1, vlan_ingress_qos[IFLA_VLAN_QOS_MAPPING]);
}

/*
 * debug VLAN QoS informaton messages
 */
void debug_vlan_qos_mapping(int lev, struct rtattr *qos)
{
    struct ifla_vlan_qos_mapping *map;
    int len = RTA_PAYLOAD(qos);

    for(qos = RTA_DATA(qos); RTA_OK(qos, len); qos = RTA_NEXT(qos, len)) {
        if(RTA_PAYLOAD(qos) < sizeof(*map)) {
            rec_dbg(lev, "IFLA_VLAN_QOS_MAPPING(%hu): -- payload too short --",
                RTA_ALIGN(qos->rta_len));
            return;
        }
        map = (struct ifla_vlan_qos_mapping *)RTA_DATA(qos);

        rec_dbg(lev, "IFLA_VLAN_QOS_MAPPING(%hu):", RTA_ALIGN(qos->rta_len));
        rec_dbg(lev, "    [ ifla_vlan_qos_mapping(%d) ]", sizeof(*map));
        rec_dbg(lev, "        from(%d): %u", sizeof(map->from), map->from);
        rec_dbg(lev, "        to(%d): %u", sizeof(map->to), map->to);
    }
}
#endif

#if HAVE_DECL_IFLA_GRE_UNSPEC
/*
 * debug GRE interface information messages
 */
void debug_ifla_gre(int lev, struct ifinfomsg *ifim, struct rtattr *linkinfo)
{
    struct rtattr *gre[__IFLA_GRE_MAX];

    rec_dbg(lev, "IFLA_INFO_DATA(%hu):", RTA_ALIGN(linkinfo->rta_len));

    parse_gre(gre, linkinfo);

    if(gre[IFLA_GRE_LINK])
        debug_ifla_gre_link(lev+1, gre[IFLA_GRE_LINK]);

    if(gre[IFLA_GRE_IFLAGS])
        debug_ifla_gre_iflags(lev+1, gre[IFLA_GRE_IFLAGS]);

    if(gre[IFLA_GRE_OFLAGS])
        debug_ifla_gre_oflags(lev+1, gre[IFLA_GRE_OFLAGS]);

    if(gre[IFLA_GRE_IKEY])
        debug_ifla_gre_ikey(lev+1, gre[IFLA_GRE_IKEY]);

    if(gre[IFLA_GRE_OKEY])
        debug_ifla_gre_okey(lev+1, gre[IFLA_GRE_OKEY]);

    if(gre[IFLA_GRE_LOCAL])
        debug_ifla_gre_local(lev+1, ifim, gre[IFLA_GRE_LOCAL]);

    if(gre[IFLA_GRE_REMOTE])
        debug_ifla_gre_remote(lev+1, ifim, gre[IFLA_GRE_REMOTE]);

    if(gre[IFLA_GRE_TTL])
        debug_ifla_gre_ttl(lev+1, gre[IFLA_GRE_TTL]);

    if(gre[IFLA_GRE_TOS])
        debug_ifla_gre_tos(lev+1, gre[IFLA_GRE_TOS]);

    if(gre[IFLA_GRE_PMTUDISC])
        debug_ifla_gre_pmtudisc(lev+1, gre[IFLA_GRE_PMTUDISC]);
}

/*
 * debug attribute IFLA_GRE_LINK
 */
void debug_ifla_gre_link(int lev, struct rtattr *gre)
{
    int index;
    char name[IFNAMSIZ] = "";

    if(RTA_PAYLOAD(gre) < sizeof(index)) {
        rec_dbg(lev, "IFLA_GRE_LINK(%hu): -- payload too short --",
            RTA_ALIGN(gre->rta_len));
        return;
    }
    index = *(int *)RTA_DATA(gre);
    if_indextoname_from_lists(index, name);

    rec_dbg(lev, "IFLA_GRE_LINK(%hu): %u(%s)", RTA_ALIGN(gre->rta_len), index, name);
}

/*
 * debug attribute IFLA_GRE_IFLAGS
 */
void debug_ifla_gre_iflags(int lev, struct rtattr *gre)
{
    unsigned short flags;
    char flags_list[MAX_STR_SIZE] = "";

    if(RTA_PAYLOAD(gre) < sizeof(flags)) {
        rec_dbg(lev, "IFLA_GRE_IFLAGS(%hu): -- payload too short --",
            RTA_ALIGN(gre->rta_len));
        return;
    }
    flags = *(unsigned short *)RTA_DATA(gre);
    convert_gre_flags(flags, flags_list, sizeof(flags_list));

    rec_dbg(lev, "IFLA_GRE_IFLAGS(%hu): 0x%.4x(%s)",
        RTA_ALIGN(gre->rta_len), flags, flags_list);
}

/*
 * debug attribute IFLA_GRE_OFLAGS
 */
void debug_ifla_gre_oflags(int lev, struct rtattr *gre)
{
    unsigned short flags;
    char flags_list[MAX_STR_SIZE] = "";

    if(RTA_PAYLOAD(gre) < sizeof(flags)) {
        rec_dbg(lev, "IFLA_GRE_OFLAGS(%hu): -- payload too short --",
            RTA_ALIGN(gre->rta_len));
        return;
    }
    flags = *(unsigned short *)RTA_DATA(gre);
    convert_gre_flags(flags, flags_list, sizeof(flags_list));

    rec_dbg(lev, "IFLA_GRE_OFLAGS(%hu): 0x%.4x(%s)",
        RTA_ALIGN(gre->rta_len), flags, flags_list);
}

/*
 * debug attribute IFLA_GRE_IKEY
 */
void debug_ifla_gre_ikey(int lev, struct rtattr *gre)
{
    if(RTA_PAYLOAD(gre) < sizeof(unsigned)) {
        rec_dbg(lev, "IFLA_GRE_IKEY(%hu): -- payload too short --",
            RTA_ALIGN(gre->rta_len));
        return;
    }
    rec_dbg(lev, "IFLA_GRE_IKEY(%hu): 0x%.8x",
        RTA_ALIGN(gre->rta_len), *(unsigned *)RTA_DATA(gre));
}

/*
 * debug attribute IFLA_GRE_OKEY
 */
void debug_ifla_gre_okey(int lev, struct rtattr *gre)
{
    if(RTA_PAYLOAD(gre) < sizeof(unsigned)) {
        rec_dbg(lev, "IFLA_GRE_OKEY(%hu): -- payload too short --",
            RTA_ALIGN(gre->rta_len));
        return;
    }

    rec_dbg(lev, "IFLA_GRE_OKEY(%hu): 0x%.8x",
        RTA_ALIGN(gre->rta_len), *(unsigned *)RTA_DATA(gre));
}

/*
 * debug attribute IFLA_GRE_LOCAL
 */
void debug_ifla_gre_local(int lev, struct ifinfomsg *ifim, struct rtattr *gre)
{
    char addr[INET6_ADDRSTRLEN+1] = "";
    int res;

    if(ifim->ifi_type == ARPHRD_ETHER)
        res = inet_ntop_ifi(ARPHRD_IPGRE, gre, addr, sizeof(addr));
    else
        res = inet_ntop_ifi(ifim->ifi_type, gre, addr, sizeof(addr));

    if(res) {
        rec_dbg(lev, "IFLA_GRE_LOCAL(%hu): -- %s --",
            RTA_ALIGN(gre->rta_len),
            (res == 1) ? strerror(errno) : "payload too short");
        return;
    }

    rec_dbg(lev, "IFLA_GRE_LOCAL(%hu): %s", RTA_ALIGN(gre->rta_len), addr);
}

/*
 * debug attribute IFLA_GRE_REMOTE
 */
void debug_ifla_gre_remote(int lev, struct ifinfomsg *ifim, struct rtattr *gre)
{
    char addr[INET6_ADDRSTRLEN+1] = "";
    int res;

    if(ifim->ifi_type == ARPHRD_ETHER)
        res = inet_ntop_ifi(ARPHRD_IPGRE, gre, addr, sizeof(addr));
    else
        res = inet_ntop_ifi(ifim->ifi_type, gre, addr, sizeof(addr));

    if(res) {
        rec_dbg(lev, "IFLA_GRE_REMOTE(%hu): -- %s --",
            RTA_ALIGN(gre->rta_len),
            (res == 1) ? strerror(errno) : "payload too short");
        return;
    }

    rec_dbg(lev, "IFLA_GRE_REMOTE(%hu): %s", RTA_ALIGN(gre->rta_len), addr);
}

/*
 * debug attribute IFLA_GRE_TTL
 */
void debug_ifla_gre_ttl(int lev, struct rtattr *gre)
{
    if(RTA_PAYLOAD(gre) < sizeof(unsigned char)) {
        rec_dbg(lev, "IFLA_GRE_TTL(%hu): -- payload too short --",
            RTA_ALIGN(gre->rta_len));
        return;
    }
    rec_dbg(lev, "IFLA_GRE_TTL(%hu): %u",
        RTA_ALIGN(gre->rta_len), *(unsigned char *)RTA_DATA(gre));
}

/*
 * debug attribute IFLA_GRE_TOS
 */
void debug_ifla_gre_tos(int lev, struct rtattr *gre)
{
    if(RTA_PAYLOAD(gre) < sizeof(unsigned char)) {
        rec_dbg(lev, "IFLA_GRE_TOS(%hu): -- payload too short --",
            RTA_ALIGN(gre->rta_len));
        return;
    }
    rec_dbg(lev, "IFLA_GRE_TOS(%hu): %u",
        RTA_ALIGN(gre->rta_len), *(unsigned char *)RTA_DATA(gre));
}

/*
 * debug attribute IFLA_GRE_PMTUDISC
 */
void debug_ifla_gre_pmtudisc(int lev, struct rtattr *gre)
{
    if(RTA_PAYLOAD(gre) < sizeof(unsigned char)) {
        rec_dbg(lev, "IFLA_GRE_PMTUDISC(%hu): -- payload too short --",
            RTA_ALIGN(gre->rta_len));
        return;
    }
    rec_dbg(lev, "IFLA_GRE_PMTUDISC(%hu): %u",
        RTA_ALIGN(gre->rta_len), *(unsigned char *)RTA_DATA(gre));
}
#endif

#if HAVE_DECL_IFLA_MACVLAN_UNSPEC
/*
 * debug MACVLAN interface information messages
 */
void debug_ifla_macvlan(int lev, struct rtattr *linkinfo)
{
    struct rtattr *macvlan[__IFLA_MACVLAN_MAX];

    rec_dbg(lev, "IFLA_INFO_DATA(%hu):", RTA_ALIGN(linkinfo->rta_len));

    parse_macvlan(macvlan, linkinfo);

    if(macvlan[IFLA_MACVLAN_MODE])
        debug_ifla_macvlan_mode(lev+1, macvlan[IFLA_MACVLAN_MODE]);

#if HAVE_DECL_IFLA_MACVLAN_FLAGS
    if(macvlan[IFLA_MACVLAN_FLAGS])
        debug_ifla_macvlan_flags(lev+1, macvlan[IFLA_MACVLAN_FLAGS]);
#endif
}

/*
 * debug attribute IFLA_MACVLAN_MODE
 */
void debug_ifla_macvlan_mode(int lev, struct rtattr *macvlan)
{
    int mode;

    if(RTA_PAYLOAD(macvlan) < sizeof(unsigned)) {
        rec_dbg(lev, "IFLA_MACVLAN_MODE(%hu): -- payload too short --",
            RTA_ALIGN(macvlan->rta_len));
        return;
    }
    mode = *(unsigned *)RTA_DATA(macvlan);

    rec_dbg(lev, "IFLA_MACVLAN_MODE(%hu): %u(%s)",
        RTA_ALIGN(macvlan->rta_len), mode, convert_macvlan_mode(mode, 1));
}

#if HAVE_DECL_IFLA_MACVLAN_FLAGS
/*
 * debug attribute IFLA_MACVLAN_FLAGS
 */
void debug_ifla_macvlan_flags(int lev, struct rtattr *macvlan)
{
    rec_dbg(lev, "IFLA_MACVLAN_FLAGS(%hu): -- ignored --",
        RTA_ALIGN(macvlan->rta_len));
}
#endif
#endif

#if HAVE_DECL_IFLA_VXLAN_UNSPEC
/*
 * debug VXLAN interface information messages
 */
void debug_ifla_vxlan(int lev, struct rtattr *linkinfo)
{
    struct rtattr *vxlan[__IFLA_VXLAN_MAX];

    rec_dbg(lev, "IFLA_INFO_DATA(%hu):", RTA_ALIGN(linkinfo->rta_len));

    parse_vxlan(vxlan, linkinfo);

    if(vxlan[IFLA_VXLAN_ID])
        debug_ifla_vxlan_id(lev+1, vxlan[IFLA_VXLAN_ID]);

#if HAVE_DECL_IFLA_VXLAN_GROUP
    if(vxlan[IFLA_VXLAN_GROUP])
        debug_ifla_vxlan_group(lev+1, vxlan[IFLA_VXLAN_GROUP]);
#endif

    if(vxlan[IFLA_VXLAN_LINK])
        debug_ifla_vxlan_link(lev+1, vxlan[IFLA_VXLAN_LINK]);

    if(vxlan[IFLA_VXLAN_LOCAL])
        debug_ifla_vxlan_local(lev+1, vxlan[IFLA_VXLAN_LOCAL]);

    if(vxlan[IFLA_VXLAN_TTL])
        debug_ifla_vxlan_ttl(lev+1, vxlan[IFLA_VXLAN_TTL]);

    if(vxlan[IFLA_VXLAN_TOS])
        debug_ifla_vxlan_tos(lev+1, vxlan[IFLA_VXLAN_TOS]);

    if(vxlan[IFLA_VXLAN_LEARNING])
        debug_ifla_vxlan_learning(lev+1, vxlan[IFLA_VXLAN_LEARNING]);

    if(vxlan[IFLA_VXLAN_AGEING])
        debug_ifla_vxlan_ageing(lev+1, vxlan[IFLA_VXLAN_AGEING]);

    if(vxlan[IFLA_VXLAN_LIMIT])
        debug_ifla_vxlan_limit(lev+1, vxlan[IFLA_VXLAN_LIMIT]);

#if HAVE_DECL_IFLA_VXLAN_PORT_RANGE
    if(vxlan[IFLA_VXLAN_PORT_RANGE])
        debug_ifla_vxlan_port_range(lev+1, vxlan[IFLA_VXLAN_PORT_RANGE]);
#endif
}

/*
 * debug attribute IFLA_VXLAN_ID
 */
void debug_ifla_vxlan_id(int lev, struct rtattr *vxlan)
{
    if(RTA_PAYLOAD(vxlan) < sizeof(unsigned)) {
        rec_dbg(lev, "IFLA_VXLAN_ID(%hu): -- payload too short --",
            RTA_ALIGN(vxlan->rta_len));
        return;
    }
    rec_dbg(lev, "IFLA_VXLAN_ID(%hu): %u",
        RTA_ALIGN(vxlan->rta_len), *(unsigned *)RTA_DATA(vxlan));
}

#if HAVE_DECL_IFLA_VXLAN_GROUP
/*
 * debug attribute IFLA_VXLAN_GROUP
 */
void debug_ifla_vxlan_group(int lev, struct rtattr *vxlan)
{
    char addr[INET_ADDRSTRLEN+1] = "";
    int res;

    /* not inet_ntop_ifi */
    res = inet_ntop_ifa(AF_INET, vxlan, addr, sizeof(addr));
    if(res) {
        rec_dbg(lev, "IFLA_VXLAN_GROUP(%hu): -- %s --",
            RTA_ALIGN(vxlan->rta_len),
            (res == 1) ? strerror(errno) : "payload too short");
        return;
    }

    rec_dbg(lev, "IFLA_VXLAN_GROUP(%hu): %s",
        RTA_ALIGN(vxlan->rta_len), addr);
}
#endif

/*
 * debug attribute IFLA_VXLAN_LINK
 */
void debug_ifla_vxlan_link(int lev, struct rtattr *vxlan)
{
    char name[IFNAMSIZ] = "";

    if(RTA_PAYLOAD(vxlan) < sizeof(int)) {
        rec_dbg(lev, "IFLA_VXLAN_LINK(%hu): -- payload too short --",
            RTA_ALIGN(vxlan->rta_len));
        return;
    }
    if_indextoname_from_lists(*(int *)RTA_DATA(vxlan), name);

    rec_dbg(lev, "IFLA_VXLAN_LINK(%hu): %s",
        RTA_ALIGN(vxlan->rta_len), name);
}

/*
 * debug attribute IFLA_VXLAN_LOCAL
 */
void debug_ifla_vxlan_local(int lev, struct rtattr *vxlan)
{
    char addr[INET_ADDRSTRLEN+1] = "";
    int res;

    /* not inet_ntop_ifi */
    res = inet_ntop_ifa(AF_INET, vxlan, addr, sizeof(addr));
    if(res) {
        rec_dbg(lev, "IFLA_VXLAN_LOCAL(%hu): -- %s --",
            RTA_ALIGN(vxlan->rta_len),
            (res == 1) ? strerror(errno) : "payload too short");
        return;
    }

    rec_dbg(lev, "IFLA_VXLAN_LOCAL(%hu): %s",
        RTA_ALIGN(vxlan->rta_len), addr);
}

/*
 * debug attribute IFLA_VXLAN_TTL
 */
void debug_ifla_vxlan_ttl(int lev, struct rtattr *vxlan)
{
    if(RTA_PAYLOAD(vxlan) < sizeof(unsigned char)) {
        rec_dbg(lev, "IFLA_VXLAN_TTL(%hu): -- payload too short --",
            RTA_ALIGN(vxlan->rta_len));
        return;
    }
    rec_dbg(lev, "IFLA_VXLAN_TTL(%hu): %d",
        RTA_ALIGN(vxlan->rta_len), *(unsigned char *)RTA_DATA(vxlan));
}

/*
 * debug attribute IFLA_VXLAN_TOS
 */
void debug_ifla_vxlan_tos(int lev, struct rtattr *vxlan)
{
    if(RTA_PAYLOAD(vxlan) < sizeof(unsigned char)) {
        rec_dbg(lev, "IFLA_VXLAN_TOS(%hu): -- payload too short --",
            RTA_ALIGN(vxlan->rta_len));
        return;
    }
    rec_dbg(lev, "IFLA_VXLAN_TOS(%hu): %d",
        RTA_ALIGN(vxlan->rta_len), *(unsigned char *)RTA_DATA(vxlan));
}

/*
 * debug attribute IFLA_VXLAN_LEARNING
 */
void debug_ifla_vxlan_learning(int lev, struct rtattr *vxlan)
{
    if(RTA_PAYLOAD(vxlan) < sizeof(unsigned char)) {
        rec_dbg(lev, "IFLA_VXLAN_LEARNING(%hu): -- payload too short --",
            RTA_ALIGN(vxlan->rta_len));
        return;
    }
    rec_dbg(lev, "IFLA_VXLAN_LEARNING(%hu): %d",
        RTA_ALIGN(vxlan->rta_len), *(unsigned char *)RTA_DATA(vxlan));
}

/*
 * debug attribute IFLA_VXLAN_AGEING
 */
void debug_ifla_vxlan_ageing(int lev, struct rtattr *vxlan)
{
    if(RTA_PAYLOAD(vxlan) < sizeof(unsigned)) {
        rec_dbg(lev, "IFLA_VXLAN_AGEING(%hu): -- payload too short --",
            RTA_ALIGN(vxlan->rta_len));
        return;
    }
    rec_dbg(lev, "IFLA_VXLAN_AGEING(%hu): %u",
        RTA_ALIGN(vxlan->rta_len), *(unsigned *)RTA_DATA(vxlan));
}

/*
 * debug attribute IFLA_VXLAN_LIMIT
 */
void debug_ifla_vxlan_limit(int lev, struct rtattr *vxlan)
{
    if(RTA_PAYLOAD(vxlan) < sizeof(unsigned)) {
        rec_dbg(lev, "IFLA_VXLAN_LIMIT(%hu): -- payload too short --",
            RTA_ALIGN(vxlan->rta_len));
        return;
    }
    rec_dbg(lev, "IFLA_VXLAN_LIMIT(%hu): %u",
        RTA_ALIGN(vxlan->rta_len), *(unsigned *)RTA_DATA(vxlan));
}

#if HAVE_DECL_IFLA_VXLAN_PORT_RANGE
/*
 * debug attribute IFLA_VXLAN_PORT_RANGE
 */
void debug_ifla_vxlan_port_range(int lev, struct rtattr *vxlan)
{
    struct ifla_vxlan_port_range *range;

    if(RTA_PAYLOAD(vxlan) < sizeof(*range)) {
        rec_dbg(lev, "IFLA_VXLAN_PORT_RANGE(%hu): -- payload too short --",
            RTA_ALIGN(vxlan->rta_len));
        return;
    }
    range = (struct ifla_vxlan_port_range *)RTA_DATA(vxlan);

    rec_dbg(lev, "IFLA_VXLAN_PORT_RANGE(%hu):", RTA_ALIGN(vxlan->rta_len));
    rec_dbg(lev, "    [ ifla_vxlan_port_range(%d) ]", sizeof(*range));
    rec_dbg(lev, "        low(%d): %hu", sizeof(range->low), range->low);
    rec_dbg(lev, "        high(%d): %hu", sizeof(range->high), range->high);
}
#endif
#endif

#if HAVE_DECL_IFLA_NET_NS_PID
/*
 * debug attribute IFLA_NET_NS_PID
 */
void debug_ifla_net_ns_pid(int lev, struct rtattr *ifla)
{
    if(RTA_PAYLOAD(ifla) < sizeof(unsigned)) {
        rec_dbg(lev, "IFLA_NET_NS_PID(%hu): -- payload too short --",
            RTA_ALIGN(ifla->rta_len));
        return;
    }
    rec_dbg(lev, "IFLA_NET_NS_PID(%hu): %u",
        RTA_ALIGN(ifla->rta_len), *(unsigned *)RTA_DATA(ifla));
}
#endif

#if HAVE_DECL_IFLA_IFALIAS
/*
 * debug attribute IFLA_IFALIAS
 */ 
void debug_ifla_ifalias(int lev, struct rtattr *ifla)
{
    char alias[IFNAMSIZ] = "";

    if(!RTA_PAYLOAD(ifla))
        strncpy(alias, "-- no payload --", sizeof(alias));
    else if(RTA_PAYLOAD(ifla) > sizeof(alias))
        strncpy(alias, "-- payload too long --", sizeof(alias));
    else
        strncpy(alias, (char *)RTA_DATA(ifla), sizeof(alias));

    rec_dbg(lev, "IFLA_IFALIAS(%hu): %s", RTA_ALIGN(ifla->rta_len), alias);
}
#endif

#if HAVE_DECL_IFLA_NUM_VF
/*
 * debug attribute IFLA_NUM_VF
 */
void debug_ifla_num_vf(int lev, struct rtattr *ifla)
{
    if(RTA_PAYLOAD(ifla) < sizeof(unsigned)) {
        rec_dbg(lev, "IFLA_NUM_VF(%hu): -- payload too short --",
            RTA_ALIGN(ifla->rta_len));
        return;
    }
    rec_dbg(lev, "IFLA_NUM_VF(%hu): %u",
        RTA_ALIGN(ifla->rta_len), *(unsigned *)RTA_DATA(ifla));
}
#endif

#if HAVE_DECL_IFLA_VFINFO_LIST
/*
 * debug attribute IFLA_VFINFO_LIST
 */ 
void debug_ifla_vfinfo_list(int lev, struct rtattr *ifla)
{
    rec_dbg(lev, "IFLA_VFINFO_LIST(%hu): -- ignored --",
        RTA_ALIGN(ifla->rta_len));
}
#endif

#if HAVE_DECL_IFLA_STATS64
/*
 * debug attribute IFLA_STATS64
 */
void debug_ifla_stats64(int lev, struct rtattr *ifla)
{
    struct rtnl_link_stats64 *stats;

    if(RTA_PAYLOAD(ifla) < sizeof(*stats)) {
        rec_dbg(lev, "IFLA_STATS64(%hu): -- payload too short --",
            RTA_ALIGN(ifla->rta_len));
        return;
    }
    stats = (struct rtnl_link_stats64 *)RTA_DATA(ifla);

    rec_dbg(lev, "IFLA_STATS64(%hu):", RTA_ALIGN(ifla->rta_len));
    rec_dbg(lev, "    [ rtnl_link_stats64(%d) ]", sizeof(*stats));
    rec_dbg(lev, "        rx_packets(%d): %llu", sizeof(stats->rx_packets), stats->rx_packets);
    rec_dbg(lev, "        tx_packets(%d): %llu", sizeof(stats->tx_packets), stats->tx_packets);
    rec_dbg(lev, "        rx_bytes(%d): %llu", sizeof(stats->rx_bytes), stats->rx_bytes);
    rec_dbg(lev, "        tx_bytes(%d): %llu", sizeof(stats->tx_bytes), stats->tx_bytes);
    rec_dbg(lev, "        rx_errors(%d): %llu", sizeof(stats->rx_errors), stats->rx_errors);
    rec_dbg(lev, "        tx_errors(%d): %llu", sizeof(stats->tx_errors), stats->tx_errors);
    rec_dbg(lev, "        rx_dropped(%d): %llu", sizeof(stats->rx_dropped), stats->rx_dropped);
    rec_dbg(lev, "        tx_dropped(%d): %llu", sizeof(stats->tx_dropped), stats->tx_dropped);
    rec_dbg(lev, "        multicast(%d): %llu", sizeof(stats->multicast), stats->multicast);
    rec_dbg(lev, "        collisions(%d): %llu", sizeof(stats->collisions), stats->collisions);
    rec_dbg(lev, "        rx_length_errors(%d): %llu",
        sizeof(stats->rx_length_errors), stats->rx_length_errors);
    rec_dbg(lev, "        rx_over_errors(%d): %llu",
        sizeof(stats->rx_over_errors), stats->rx_over_errors);
    rec_dbg(lev, "        rx_crc_errors(%d): %llu",
        sizeof(stats->rx_crc_errors), stats->rx_crc_errors);
    rec_dbg(lev, "        rx_frame_errors(%d): %llu",
        sizeof(stats->rx_frame_errors), stats->rx_frame_errors);
    rec_dbg(lev, "        rx_fifo_errors(%d): %llu",
        sizeof(stats->rx_fifo_errors), stats->rx_fifo_errors);
    rec_dbg(lev, "        rx_missed_errors(%d): %llu",
        sizeof(stats->rx_missed_errors), stats->rx_missed_errors);
    rec_dbg(lev, "        tx_aborted_errors(%d): %llu",
        sizeof(stats->tx_aborted_errors), stats->tx_aborted_errors);
    rec_dbg(lev, "        tx_carrier_errors(%d): %llu",
        sizeof(stats->tx_carrier_errors), stats->tx_carrier_errors);
    rec_dbg(lev, "        tx_fifo_errors(%d): %llu",
        sizeof(stats->tx_fifo_errors), stats->tx_fifo_errors);
    rec_dbg(lev, "        tx_heartbeat_errors(%d): %llu",
        sizeof(stats->tx_heartbeat_errors), stats->tx_heartbeat_errors);
    rec_dbg(lev, "        tx_window_errors(%d): %llu",
        sizeof(stats->tx_window_errors), stats->tx_window_errors);
    rec_dbg(lev, "        rx_compressed(%d): %llu",
        sizeof(stats->rx_compressed), stats->rx_compressed);
    rec_dbg(lev, "        tx_compressed(%d): %llu",
        sizeof(stats->tx_compressed), stats->tx_compressed);
}
#endif

#if HAVE_DECL_IFLA_VF_PORTS
/*
 * debug attribute IFLA_VF_PORTS
 */
void debug_ifla_vf_ports(int lev, struct rtattr *ifla)
{
    rec_dbg(lev, "IFLA_VF_PORTS(%hu): -- ignored --",
        RTA_ALIGN(ifla->rta_len));
}
#endif

#if HAVE_DECL_IFLA_PORT_SELF
/*
 * debug attribute IFLA_PORT_SELF
 */
void debug_ifla_port_self(int lev, struct rtattr *ifla)
{
    rec_dbg(lev, "IFLA_PORT_SELF(%hu): -- ignored --",
        RTA_ALIGN(ifla->rta_len));
}
#endif

#if HAVE_DECL_IFLA_AF_SPEC
/*
 * debug attribute IFLA_AF_SPEC
 */
void debug_ifla_af_spec(int lev, struct rtattr *ifla)
{
    rec_dbg(lev, "IFLA_AF_SPEC(%hu): -- ignored --",
        RTA_ALIGN(ifla->rta_len));
}
#endif

#if HAVE_DECL_IFLA_GROUP
/*
 * debug attribute IFLA_GROUP
 */
void debug_ifla_group(int lev, struct rtattr *ifla)
{
    if(RTA_PAYLOAD(ifla) < sizeof(int)) {
        rec_dbg(lev, "IFLA_GROUP(%hu): -- payload too short --",
            RTA_ALIGN(ifla->rta_len));
        return;
    }
    rec_dbg(lev, "IFLA_GROUP(%hu): %d",
        RTA_ALIGN(ifla->rta_len), *(int *)RTA_DATA(ifla));
}
#endif

#if HAVE_DECL_IFLA_NET_NS_FD
/*
 * debug attribute IFLA_NET_NS_FD
 */
void debug_ifla_net_ns_fd(int lev, struct rtattr *ifla)
{
    rec_dbg(lev, "IFLA_NET_NS_FD(%hu): -- ignored --",
        RTA_ALIGN(ifla->rta_len));
}
#endif

#if HAVE_DECL_IFLA_EXT_MASK
/*
 * debug attribute IFLA_EXT_MASK
 */
void debug_ifla_ext_mask(int lev, struct rtattr *ifla)
{
    rec_dbg(lev, "IFLA_EXT_MASK(%hu): -- ignored --",
        RTA_ALIGN(ifla->rta_len));
}
#endif

#if HAVE_DECL_IFLA_PROMISCUITY
/*
 * debug attribute IFLA_PROMISCUITY
 */
void debug_ifla_promiscuity(int lev, struct rtattr *ifla)
{
    if(RTA_PAYLOAD(ifla) < sizeof(int)) {
        rec_dbg(lev, "IFLA_PROMISCUITY(%hu): -- payload too short --",
            RTA_ALIGN(ifla->rta_len));
        return;
    }
    rec_dbg(lev, "IFLA_PROMISCUITY(%hu): %d",
        RTA_ALIGN(ifla->rta_len), *(int *)RTA_DATA(ifla));
}
#endif

#if HAVE_DECL_IFLA_NUM_TX_QUEUES
/*
 * debug attribute IFLA_NUM_TX_QUEUES
 */
void debug_ifla_num_tx_queues(int lev, struct rtattr *ifla)
{
    if(RTA_PAYLOAD(ifla) < sizeof(int)) {
        rec_dbg(lev, "IFLA_NUM_TX_QUEUES(%hu): -- payload too short --",
            RTA_ALIGN(ifla->rta_len));
        return;
    }
    rec_dbg(lev, "IFLA_NUM_TX_QUEUES(%hu): %d",
        RTA_ALIGN(ifla->rta_len), *(int *)RTA_DATA(ifla));
}
#endif

#if HAVE_DECL_IFLA_NUM_RX_QUEUES
/*
 * debug attribute IFLA_NUM_RX_QUEUES
 */
void debug_ifla_num_rx_queues(int lev, struct rtattr *ifla)
{
    if(RTA_PAYLOAD(ifla) < sizeof(int)) {
        rec_dbg(lev, "IFLA_NUM_RX_QUEUES(%hu): -- payload too short --",
            RTA_ALIGN(ifla->rta_len));
        return;
    }
    rec_dbg(lev, "IFLA_NUM_RX_QUEUES(%hu): %d",
        RTA_ALIGN(ifla->rta_len), *(int *)RTA_DATA(ifla));
}
#endif

/*
 * convert address family from number to string
 */
const char *convert_af_type(int type)
{
#define _AF_TYPE(s) \
    if(type == AF_##s) \
        return #s;
    _AF_TYPE(UNSPEC);
    _AF_TYPE(LOCAL);
    _AF_TYPE(UNIX);
    _AF_TYPE(FILE);
    _AF_TYPE(INET);
    _AF_TYPE(AX25);
    _AF_TYPE(IPX);
    _AF_TYPE(APPLETALK);
    _AF_TYPE(NETROM);
    _AF_TYPE(BRIDGE);
    _AF_TYPE(ATMPVC);
    _AF_TYPE(X25);
    _AF_TYPE(INET6);
    _AF_TYPE(ROSE);
    _AF_TYPE(DECnet);
    _AF_TYPE(NETBEUI);
    _AF_TYPE(SECURITY);
    _AF_TYPE(KEY);
    _AF_TYPE(NETLINK);
    _AF_TYPE(ROUTE);
    _AF_TYPE(PACKET);
    _AF_TYPE(ASH);
    _AF_TYPE(ECONET);
    _AF_TYPE(ATMSVC);
#ifdef AF_RDS
    _AF_TYPE(RDS);
#endif
    _AF_TYPE(SNA);
    _AF_TYPE(IRDA);
    _AF_TYPE(PPPOX);
    _AF_TYPE(WANPIPE);
#ifdef AF_LLC
    _AF_TYPE(LLC);
#endif
#ifdef AF_CAN
    _AF_TYPE(CAN);
#endif
#ifdef AF_TIPC
    _AF_TYPE(TIPC);
#endif
    _AF_TYPE(BLUETOOTH);
#ifdef AF_IUCV
    _AF_TYPE(IUCV);
#endif
#ifdef AF_RXRPC
    _AF_TYPE(RXRPC);
#endif
#ifdef AF_ISDN
    _AF_TYPE(ISDN);
#endif
#ifdef AF_PHONET
    _AF_TYPE(PHONET);
#endif
#ifdef AF_IEEE802154
    _AF_TYPE(IEEE802154);
#endif
#ifdef AF_CAIF
    _AF_TYPE(CAIF);
#endif
#ifdef AF_ALG
    _AF_TYPE(ALG);
#endif
    _AF_TYPE(MAX);
#undef _AF_TYPE
    return("UNKNOWN");
}

/*
 * convert interafce type from number to string
 */ 
const char *convert_arphrd_type(int type)
{
#define _ARPHRD_TYPE(s) \
    if(type == ARPHRD_##s) \
        return(#s);
    _ARPHRD_TYPE(NETROM);
    _ARPHRD_TYPE(ETHER);
    _ARPHRD_TYPE(EETHER);
    _ARPHRD_TYPE(AX25);
    _ARPHRD_TYPE(PRONET);
    _ARPHRD_TYPE(CHAOS);
    _ARPHRD_TYPE(IEEE802);
    _ARPHRD_TYPE(ARCNET);
    _ARPHRD_TYPE(APPLETLK);
    _ARPHRD_TYPE(DLCI);
    _ARPHRD_TYPE(ATM);
    _ARPHRD_TYPE(METRICOM);
    _ARPHRD_TYPE(IEEE1394);
    _ARPHRD_TYPE(EUI64);
    _ARPHRD_TYPE(INFINIBAND);
    _ARPHRD_TYPE(SLIP);
    _ARPHRD_TYPE(CSLIP);
    _ARPHRD_TYPE(SLIP6);
    _ARPHRD_TYPE(CSLIP6);
    _ARPHRD_TYPE(RSRVD);
    _ARPHRD_TYPE(ADAPT);
    _ARPHRD_TYPE(ROSE);
    _ARPHRD_TYPE(X25);
    _ARPHRD_TYPE(HWX25);
#ifdef ARPHRD_CAN
    _ARPHRD_TYPE(CAN);
#endif
    _ARPHRD_TYPE(PPP);
    _ARPHRD_TYPE(CISCO);
    _ARPHRD_TYPE(HDLC);
    _ARPHRD_TYPE(LAPB);
    _ARPHRD_TYPE(DDCMP);
    _ARPHRD_TYPE(RAWHDLC);
    _ARPHRD_TYPE(TUNNEL);
    _ARPHRD_TYPE(TUNNEL6);
    _ARPHRD_TYPE(FRAD);
    _ARPHRD_TYPE(SKIP);
    _ARPHRD_TYPE(LOOPBACK);
    _ARPHRD_TYPE(LOCALTLK);
    _ARPHRD_TYPE(FDDI);
    _ARPHRD_TYPE(BIF);
    _ARPHRD_TYPE(SIT);
    _ARPHRD_TYPE(IPDDP);
    _ARPHRD_TYPE(IPGRE);
    _ARPHRD_TYPE(PIMREG);
    _ARPHRD_TYPE(HIPPI);
    _ARPHRD_TYPE(ASH);
    _ARPHRD_TYPE(ECONET);
    _ARPHRD_TYPE(IRDA);
    _ARPHRD_TYPE(FCPP);
    _ARPHRD_TYPE(FCAL);
    _ARPHRD_TYPE(FCPL);
    _ARPHRD_TYPE(FCFABRIC);
    _ARPHRD_TYPE(IEEE802_TR);
    _ARPHRD_TYPE(IEEE80211);
    _ARPHRD_TYPE(IEEE80211_PRISM);
    _ARPHRD_TYPE(IEEE80211_RADIOTAP);
#ifdef ARPHRD_IEEE802154
    _ARPHRD_TYPE(IEEE802154);
#endif
#ifdef ARPHRD_PHONET
    _ARPHRD_TYPE(PHONET);
#endif
#ifdef ARPHRD_PHONET_PIPE
    _ARPHRD_TYPE(PHONET_PIPE);
#endif
#ifdef ARPHRD_CAIF
    _ARPHRD_TYPE(CAIF);
#endif
#ifdef ARPHRD_IP6GRE
    _ARPHRD_TYPE(IP6GRE);
#endif
#undef _ARPHRD_TYPE
    return("UNKNOWN");
}

/*
 * convert interface flags from number to string
 */
void convert_iff_flags(int flags, char *flags_list, int len)
{
    if(!flags) {
        strncpy(flags_list, "NONE", len);
        return;
    }
#define _IFF_FLAGS(s) \
    if((flags & IFF_##s) && (len - strlen(flags_list) -1 > 0)) \
        (flags &= ~IFF_##s) ? \
            strncat(flags_list, #s ",", len - strlen(flags_list) - 1) : \
            strncat(flags_list, #s, len - strlen(flags_list) - 1);
    _IFF_FLAGS(UP);
    _IFF_FLAGS(BROADCAST);
    _IFF_FLAGS(DEBUG);
    _IFF_FLAGS(LOOPBACK);
    _IFF_FLAGS(POINTOPOINT);
    _IFF_FLAGS(NOTRAILERS);
    _IFF_FLAGS(RUNNING);
    _IFF_FLAGS(NOARP);
    _IFF_FLAGS(PROMISC);
    _IFF_FLAGS(ALLMULTI);
    _IFF_FLAGS(MASTER);
    _IFF_FLAGS(SLAVE);
    _IFF_FLAGS(MULTICAST);
    _IFF_FLAGS(PORTSEL);
    _IFF_FLAGS(AUTOMEDIA);
    _IFF_FLAGS(DYNAMIC);
    _IFF_FLAGS(LOWER_UP);
    _IFF_FLAGS(DORMANT);
#ifdef IFF_ECHO
    _IFF_FLAGS(ECHO);
#endif
#undef _IFF_FLAGS
    if(!strlen(flags_list))
        strncpy(flags_list, "UNKNOWN", len);
}

/*
 * convert interafce operational state from number to string
 */
const char *convert_if_oper_state(int state)
{
#define _IF_OPER_STATE(s) \
    if(state == IF_OPER_##s) \
        return(#s);
    _IF_OPER_STATE(UNKNOWN);
    _IF_OPER_STATE(NOTPRESENT);
    _IF_OPER_STATE(DOWN);
    _IF_OPER_STATE(LOWERLAYERDOWN);
    _IF_OPER_STATE(TESTING);
    _IF_OPER_STATE(DORMANT);
    _IF_OPER_STATE(UP);
#undef _IF_OPER_STATE
    return("UNKNOWN");
}

/*
 * convert interafce link mode from number to string
 */
const char *convert_if_link_mode(int mode)
{
#define _IFLA_LINKMODE(s) \
    if(mode == IF_LINK_MODE_##s) \
        return(#s);
    _IFLA_LINKMODE(DEFAULT);
    _IFLA_LINKMODE(DORMANT);
#undef _IFLA_LINKMODE
    return("UNKNOWN");
}

#if HAVE_DECL_IFLA_VLAN_FLAGS
/*
 * convert VLAN flags from number to string
 */
void convert_vlan_flags(int flags, char *flags_list, int len)
{
    if(!flags) {
        strncpy(flags_list, "NONE", len);
        return;
    }
#define _VLAN_FLAGS(s) \
    if((flags & VLAN_FLAG_##s) && (len - strlen(flags_list) - 1 > 0)) \
        (flags &= ~VLAN_FLAG_##s) ? \
            strncat(flags_list, #s ",", len - strlen(flags_list) - 1) : \
            strncat(flags_list, #s, len - strlen(flags_list) - 1);
#if HAVE_DECL_VLAN_FLAG_REORDER_HDR
    _VLAN_FLAGS(REORDER_HDR);
#endif
#if HAVE_DECL_VLAN_FLAG_GVRP
    _VLAN_FLAGS(GVRP);
#endif
#if HAVE_DECL_VLAN_FLAG_LOOSE_BINDING
    _VLAN_FLAGS(LOOSE_BINDING);
#endif
#undef _VLAN_FLAGS
    if(!strlen(flags_list))
        strncpy(flags_list, "UNKNOWN", len);
}
#endif

#if HAVE_DECL_IFLA_GRE_IFLAGS
/*
 * convert GRE flags from number to string
 */
void convert_gre_flags(int flags, char *flags_list, int len)
{
    if(!flags) {
        strncpy(flags_list, "NONE", len);
        return;
    }
#define _GRE_FLAGS(s) \
    if((flags & GRE_##s) && (len - strlen(flags_list) - 1 > 0)) \
        (flags &= ~GRE_##s) ? \
            strncat(flags_list, #s ",", len - strlen(flags_list) - 1) : \
            strncat(flags_list, #s, len - strlen(flags_list) - 1);
    _GRE_FLAGS(CSUM);
    _GRE_FLAGS(ROUTING);
    _GRE_FLAGS(KEY);
    _GRE_FLAGS(SEQ);
    _GRE_FLAGS(STRICT);
#undef _GRE_FLAGS
    if(!strlen(flags_list))
        strncpy(flags_list, "UNKNOWN", len);
}
#endif

#if HAVE_DECL_IFLA_MACVLAN_UNSPEC
/*
 * convert MACVLAN flags from number to string
 */
const char *convert_macvlan_mode(int mode, int debug)
{
#define _MACVLAN_MODE(s1, s2) \
    if(mode == MACVLAN_MODE_##s1) \
        return(debug ? #s1 : #s2);
    _MACVLAN_MODE(PRIVATE, private);
    _MACVLAN_MODE(VEPA, vepa);
    _MACVLAN_MODE(BRIDGE, bridge);
#if HAVE_DECL_MACVLAN_MODE_PASSTHRU
    _MACVLAN_MODE(PASSTHRU, passthru);
#endif
#undef _MACVLAN_MODE
    return(debug ? "UNKNOWN" : "unknown");
}
#endif
