// --------------------------------------------------------------------------
//                   OpenMS -- Open-Source Mass Spectrometry
// --------------------------------------------------------------------------
// Copyright The OpenMS Team -- Eberhard Karls University Tuebingen,
// ETH Zurich, and Freie Universitaet Berlin 2002-2015.
//
// This software is released under a three-clause BSD license:
//  * Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
//  * Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//  * Neither the name of any author or any participating institution
//    may be used to endorse or promote products derived from this software
//    without specific prior written permission.
// For a full list of authors, refer to the file AUTHORS.
// --------------------------------------------------------------------------
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL ANY OF THE AUTHORS OR THE CONTRIBUTING
// INSTITUTIONS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
// OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
// WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
// OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
// ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// --------------------------------------------------------------------------
// $Maintainer: Stephan Aiche $
// $Authors: Stephan Aiche $
// --------------------------------------------------------------------------


#ifndef OPENMS_ANALYSIS_MAPMATCHING_MAPALIGNMENTTRANSFORMER_H
#define OPENMS_ANALYSIS_MAPMATCHING_MAPALIGNMENTTRANSFORMER_H

#include <vector>
#include <OpenMS/config.h>

#include <OpenMS/KERNEL/MSExperiment.h>
#include <OpenMS/KERNEL/FeatureMap.h>

namespace OpenMS
{

  class TransformationDescription;
  class ConsensusMap;
  class PeptideIdentification;
  class ConsensusFeature;

  /**
   * @brief The MapAlignmentTransformer class
   */
  class OPENMS_DLLAPI MapAlignmentTransformer
  {

public:
    /// Applies the <i>given</i> transformations to peak maps
    static void transformPeakMaps(std::vector<MSExperiment<> > & maps, const std::vector<TransformationDescription> & given_trafos);

    /// Applies the <i>given</i> transformations to feature maps
    static void transformFeatureMaps(std::vector<FeatureMap > & maps, const std::vector<TransformationDescription> & given_trafos);

    /// Applies the <i>given</i> transformations to consensus maps
    static void transformConsensusMaps(std::vector<ConsensusMap> & maps, const std::vector<TransformationDescription> & given_trafos);

    /// Applies the <i>given</i> transformations to peptide identifications
    static void transformPeptideIdentifications(std::vector<std::vector<PeptideIdentification> > & maps, const std::vector<TransformationDescription> & given_trafos);


    /// Applies the <i>given</i> transformations to a single peak map
    static void transformSinglePeakMap(MSExperiment<> & msexp, const TransformationDescription & trafo);

    /// Applies the <i>given</i> transformations to a single feature map
    static void transformSingleFeatureMap(FeatureMap & fmap, const TransformationDescription & trafo);

    /// Applies the <i>given</i> transformations to a single consensus map
    static void transformSingleConsensusMap(ConsensusMap & cmap, const TransformationDescription & trafo);

    /// Applies the <i>given</i> transformations to a single peptide identification
    static void transformSinglePeptideIdentification(std::vector<PeptideIdentification> & pepids, const TransformationDescription & trafo);

private:

    /// apply a transformation to a feature
    static void applyToFeature_(Feature & feature, const TransformationDescription & trafo);

    /// apply a transformation to a basic feature
    static void applyToBaseFeature_(BaseFeature & feature, const TransformationDescription & trafo);

    /// apply a transformation to a consensus feature
    static void applyToConsensusFeature_(ConsensusFeature & feature, const TransformationDescription & trafo);

  };
} // namespace OpenMS

#endif // OPENMS_ANALYSIS_MAPMATCHING_MAPALIGNMENTTRANSFORMER_H
