
/*****************************************************************************
*
* Copyright (c) 2003-2016 by The University of Queensland
* http://www.uq.edu.au
*
* Primary Business: Queensland, Australia
* Licensed under the Apache License, version 2.0
* http://www.apache.org/licenses/LICENSE-2.0
*
* Development until 2012 by Earth Systems Science Computational Center (ESSCC)
* Development 2012-2013 by School of Earth Sciences
* Development from 2014 by Centre for Geoscience Computing (GeoComp)
*
*****************************************************************************/


#if !defined  escript_BinaryOp_20040315_H
#define escript_BinaryOp_20040315_H
#include "system_dep.h"

#include "DataTypes.h"
#include "DataConstant.h"
#include "DataTagged.h"
#include "DataExpanded.h"
#include "DataMaths.h"

/**
\file BinaryOp.h 
\brief Describes binary operations performed on instances of DataAbstract.

For operations on DataVector see DataMaths.h.
For operations on double* see LocalOps.h.
*/

namespace escript {
/**
   \brief
   Perform the given binary operation.
   \param left Input/Output - The left hand side.
   \param right Input - The right hand side.
   \param operation Input - The operation to perform.
*/
template <class BinaryFunction>
inline void binaryOp(DataTagged& left, const DataConstant& right, 
		     BinaryFunction operation)
{
//  binaryOp(left,right.getPointDataView(),operation);
  //
  // perform the operation on each tagged value
  const DataTagged::DataMapType& lookup=left.getTagLookup();
  DataTagged::DataMapType::const_iterator i;
  DataTagged::DataMapType::const_iterator lookupEnd=lookup.end();
  DataTypes::ValueType& leftVec=left.getVectorRW();
  const DataTypes::ShapeType& leftShape=left.getShape();
  const DataTypes::ShapeType& rightShape=right.getShape();
  double rvalue=right.getVectorRO()[0];		// for rank==0
  const DataTypes::ValueType& rightVec=right.getVectorRO();   // for rank>0
  if (right.getRank()==0) {
    for (i=lookup.begin();i!=lookupEnd;i++) {
      DataMaths::binaryOp(leftVec,leftShape,i->second,rvalue,operation);
    }
  } else {
    for (i=lookup.begin();i!=lookupEnd;i++) {
      DataMaths::binaryOp(leftVec, leftShape, i->second,rightVec,rightShape,0,operation);
    }
  }
  //
  // finally perform the operation on the default value
  if (right.getRank()==0) {
    DataMaths::binaryOp(leftVec,leftShape,left.getDefaultOffset(),rvalue,operation);
  } else {
    DataMaths::binaryOp(leftVec,leftShape,left.getDefaultOffset(),rightVec,rightShape,0,operation);
  }
}

/**
   \brief apply the binary op to each value in left and the single value right.

   The value in right will be assumed to begin at offset 0
*/
template <class BinaryFunction>
inline void binaryOp(DataTagged& left, const DataTypes::ValueType& right, 
		     const DataTypes::ShapeType& shape,
		     BinaryFunction operation)
{
  //
  // perform the operation on each tagged value
  const DataTagged::DataMapType& lookup=left.getTagLookup();
  DataTagged::DataMapType::const_iterator i;
  DataTagged::DataMapType::const_iterator lookupEnd=lookup.end();
  DataTypes::ValueType& lvec=left.getVectorRW();
  const DataTypes::ShapeType& lshape=left.getShape();
  if (DataTypes::getRank(shape)==0) {
    for (i=lookup.begin();i!=lookupEnd;i++) {
      DataMaths::binaryOp(lvec, lshape,i->second,right[0],operation);
    }
  } else {
    for (i=lookup.begin();i!=lookupEnd;i++) {
      DataMaths::binaryOp(lvec, lshape, i->second,right,shape,0,operation);
    }
  }
  //
  // finally perform the operation on the default value
  if (DataTypes::getRank(shape)==0) {
    DataMaths::binaryOp(lvec,lshape,left.getDefaultOffset(),right[0],operation);
  } else {
    DataMaths::binaryOp(lvec,lshape,left.getDefaultOffset(),right, shape,0,operation);
  }
}




template <class BinaryFunction>
inline void binaryOp(DataTagged& left, const DataTagged& right, 
		     BinaryFunction operation)
{
  using namespace DataMaths;

  int right_rank=right.getRank();
  //
  // Add the right hand tag keys which can't currently be found on the left
  const DataTagged::DataMapType& rightLookup=right.getTagLookup();
  DataTagged::DataMapType::const_iterator i;
  DataTagged::DataMapType::const_iterator rightLookupEnd=rightLookup.end();
  for (i=rightLookup.begin();i!=rightLookupEnd;i++) {
    //
    // If the left does not already have a value assigned to this tag,
    // add the right hand tag to the left hand tag list and assign
    // the left's default value.
    if (!left.isCurrentTag(i->first)) {
      left.addTag(i->first);
    }
  }
  DataTypes::ValueType& leftVec=left.getVectorRW();
  const DataTypes::ShapeType& leftShape=left.getShape();
  //
  // Perform the operation.
  const DataTagged::DataMapType& leftLookup=left.getTagLookup();
  DataTagged::DataMapType::const_iterator leftLookupEnd=leftLookup.end();
  for (i=leftLookup.begin();i!=leftLookupEnd;i++) {
    if (right_rank==0) {
       binaryOp(leftVec,leftShape,i->second, right.getDataByTagRO(i->first,0),operation);

    } else {	// rank>0
       binaryOp(leftVec,leftShape,left.getOffsetForTag(i->first),right.getVectorRO(), right.getShape(), right.getOffsetForTag(i->first), operation);
    }
  }
  //
  // finally perform the operation on the default value
  if (right_rank==0) {
     binaryOp(leftVec,leftShape, left.getDefaultOffset(), right.getVectorRO()[0],operation);
  } else {
     binaryOp(leftVec,leftShape, left.getDefaultOffset(), right.getVectorRO(), right.getShape(), right.getDefaultOffset(), operation);
  }
}

template <class BinaryFunction>
inline void binaryOp(DataConstant& left, const DataConstant& right, 
		     BinaryFunction operation)
{
  if (right.getRank()==0) {
    double r=right.getVectorRO()[0];
    DataMaths::binaryOp(left.getVectorRW(), left.getShape(),0, r,operation);
  } else {
    DataMaths::binaryOp(left.getVectorRW(), left.getShape(),0, right.getVectorRO(),right.getShape(),0,operation);
  }

}



template <class BinaryFunction>
inline void binaryOp(DataExpanded& left, const DataReady& right, 
		     BinaryFunction operation)
{
  int i,j;
  DataTypes::ValueType::size_type numDPPSample=left.getNumDPPSample();
  DataTypes::ValueType::size_type numSamples=left.getNumSamples();
  if (right.getRank()==0) {

    const DataTypes::ShapeType& leftShape=left.getShape();
    DataTypes::ValueType& leftVec=left.getVectorRW();
    //
    // This will call the double version of binaryOp
    #pragma omp parallel for private(i,j) schedule(static)
    for (i=0;i<numSamples;i++) {
      for (j=0;j<numDPPSample;j++) {
	DataMaths::binaryOp(leftVec,leftShape,left.getPointOffset(i,j), right.getVectorRO()[right.getPointOffset(i,j)]  ,operation);
      }
    }
  } else {
    #pragma omp parallel for private(i,j) schedule(static)
    for (i=0;i<numSamples;i++) {
      for (j=0;j<numDPPSample;j++) {
	DataMaths::binaryOp(left.getVectorRW(),left.getShape(),left.getPointOffset(i,j), right.getVectorRO(), right.getShape(),right.getPointOffset(i,j), operation);
      }
    }
  }
}


} // end of namespace

#endif
