## package RefDB::Prefs
## RefDB Preferences management module

## markus@mhoenicka.de 2002-12-27
## $Id: Prefs.pm,v 1.2 2003/09/16 21:06:22 mhoenicka Exp $

##   This program is free software; you can redistribute it and/or modify
##   it under the terms of the GNU General Public License as published by
##   the Free Software Foundation; either version 2 of the License, or
##   (at your option) any later version.
##   
##   This program is distributed in the hope that it will be useful,
##   but WITHOUT ANY WARRANTY; without even the implied warranty of
##   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##   GNU General Public License for more details.
   
##   You should have received a copy of the GNU General Public License
##   along with this program; if not, write to the Free Software
##   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

## Package main documentation

=head1 NAME

RefDB::Prefs - Perl extension for managing preferences of RefDB apps

=head1 SYNOPSIS

use RefDB::Prefs;

my %prefs;

my $prefs = RefDB::Prefs::->new("/usr/local/etc/myapprc", "/home/user/myapprc");


=head1 DESCRIPTION

RefDB::Prefs allows to read the contents of global and personal config files into a hash. Personal settings override global settings.

=head1 FEEDBACK

Send bug reports, questions, and comments to the refdb-users mailing list at:

refdb-users@lists.sourceforge.net

For list information and archives, please visit:

http://lists.sourceforge.net/lists/listinfo/refdb-users


=head1 AUTHOR

Markus Hoenicka, markus@mhoenicka.de

=head1 SEE ALSO

This module is part of the RefDB package, a reference manager and bibliography tool for markup languages. Please visit http://refdb.sourceforge.net for further information.

=cut

package RefDB::Prefs;

use strict;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

$VERSION = "1.2";

##********************************************************************
## read_prefs(): reads config file settings into a hash
## Arguments: path of global config file
##            path of local config file
## Returns: number of name/value pairs read, or 0 if none found
##********************************************************************
sub new($$) {
    my ($class, $globalprefs, $localprefs) = @_;
    my $self = {};

    my $havesome = 0;

    ## the config files are simply whitespace-separated name/value pairs,
    ## one pair per line. A '#' denotes a comment. As the values may
    ## contain spaces, we have to join all tokens except the key after
    ## splitting the string

    ## read global config file
    if (defined($globalprefs) && length($globalprefs)) {
	if (open PREF, "< $globalprefs") {
	    while (<PREF>) {
		unless ($_ =~ m/^\#/) {
		    my @fields = split /\s/, $_;
		    my $key = shift @fields;
		    $self->{$key} = join " ", @fields;
		    $havesome++;
		}
	    }
	    close PREF;
	}
    }

    ## read personal config file. First try visible, then hidden
    if (defined($localprefs) && length($localprefs)) {
	if (open PREF, "< $localprefs") {
	    while (<PREF>) {
		unless ($_ =~ m/^\#/) {
		    my @fields = split /\s/, $_;
		    my $key = shift @fields;
		    $self->{$key} = join " ", @fields;
		    $havesome++;
		}
	    }
	    close PREF;
	}
	else {
	    $localprefs =~ s%(.*/)([^/]*)$%$1.$2%;

	    if (open PREF, "< $localprefs") {
		while (<PREF>) {
		    unless ($_ =~ m/^\#/) {
			my @fields = split /\s/, $_;
			my $key = shift @fields;
			$self->{$key} = join " ", @fields;
			$havesome++;
		    }
		}
		close PREF;
	    }
	}
    }

    bless $self, $class;
    return $self;
} ## end of new


1;
__END__
