/* Copyright (c) 2004-2015. The SimGrid Team.
 * All rights reserved.                                                     */

/* This program is free software; you can redistribute it and/or modify it
 * under the terms of the license (GNU LGPL) which comes with this package. */

#ifndef SURF_MODEL_H_
#define SURF_MODEL_H_

#include <xbt.h>
#include <string>
#include <vector>
#include <memory>
#include <boost/function.hpp>
#include <boost/intrusive/list.hpp>
#include "surf/trace_mgr.h"
#include "xbt/lib.h"
#include "surf/surf_routing.h"
#include "simgrid/platf_interface.h"
#include "surf/surf.h"
#include "surf/surf_private.h"
#include "internal_config.h"

#ifdef LIBSIGC
#include <sigc++/sigc++.h>
#define surf_callback(arg1, ...)  sigc::signal<arg1,__VA_ARGS__>
#define surf_callback_connect(callback, fun_ptr) callback.connect(sigc::ptr_fun(fun_ptr))
#define surf_callback_emit(callback, ...) callback.emit(__VA_ARGS__)
#else
#include <boost/signals2.hpp>
#define surf_callback(arg1, ...)  boost::signals2::signal<arg1(__VA_ARGS__)>
#define surf_callback_connect(callback, fun_ptr) callback.connect(fun_ptr)
#define surf_callback_emit(callback, ...) callback(__VA_ARGS__)
#endif

#ifdef _MSC_VER
#pragma warning( disable : 4251)
// 4251: needs to have dll-interface to be used by clients of class
#endif

extern XBT_PRIVATE tmgr_history_t history;
#define NO_MAX_DURATION -1.0

using namespace std;

/*********
 * Utils *
 *********/

/* user-visible parameters */
extern XBT_PRIVATE double sg_tcp_gamma;
extern XBT_PRIVATE double sg_sender_gap;
extern XBT_PRIVATE double sg_latency_factor;
extern XBT_PRIVATE double sg_bandwidth_factor;
extern XBT_PRIVATE double sg_weight_S_parameter;
extern XBT_PRIVATE int sg_network_crosstraffic;
extern XBT_PRIVATE xbt_dynar_t surf_path;

extern "C" {
XBT_PUBLIC(double) surf_get_clock(void);
}

extern XBT_PRIVATE double sg_sender_gap;

extern XBT_PRIVATE surf_callback(void, void) surfExitCallbacks;

int XBT_PRIVATE __surf_is_absolute_file_path(const char *file_path);

/***********
 * Classes *
 ***********/

enum heap_action_type{
  LATENCY = 100,
  MAX_DURATION,
  NORMAL,
  NOTSET
};

/*********
 * Trace *
 *********/
/* For the trace and trace:connect tag (store their content till the end of the parsing) */
XBT_PUBLIC_DATA(xbt_dict_t) traces_set_list;
XBT_PUBLIC_DATA(xbt_dict_t) trace_connect_list_host_avail;
XBT_PUBLIC_DATA(xbt_dict_t) trace_connect_list_power;
XBT_PUBLIC_DATA(xbt_dict_t) trace_connect_list_link_avail;
XBT_PUBLIC_DATA(xbt_dict_t) trace_connect_list_bandwidth;
XBT_PUBLIC_DATA(xbt_dict_t) trace_connect_list_latency;

/**********
 * Action *
 **********/
XBT_PRIVATE void surf_action_lmm_update_index_heap(void *action, int i);

/** @ingroup SURF_interface
 * @brief SURF action interface class
 * @details An action is an event generated by a resource (e.g.: a communication for the network)
 */
XBT_PUBLIC_CLASS Action {
public:
  boost::intrusive::list_member_hook<> action_hook;
  boost::intrusive::list_member_hook<> action_lmm_hook;
  typedef boost::intrusive::member_hook<
    Action, boost::intrusive::list_member_hook<>, &Action::action_hook> ActionOptions;
  typedef boost::intrusive::list<Action, ActionOptions> ActionList;
private:
  /**
   * @brief Common initializations for the constructors
   */
  void initialize(Model *model, double cost, bool failed,
                  lmm_variable_t var = NULL);

public:
  /**
   * @brief Action constructor
   *
   * @param model The Model associated to this Action
   * @param cost The cost of the Action
   * @param failed If the action is impossible (e.g.: execute something on a switched off host)
   */
  Action(Model *model, double cost, bool failed);

  /**
   * @brief Action constructor
   *
   * @param model The Model associated to this Action
   * @param cost The cost of the Action
   * @param failed If the action is impossible (e.g.: execute something on a switched off host)
   * @param var The lmm variable associated to this Action if it is part of a LMM component
   */
  Action(Model *model, double cost, bool failed, lmm_variable_t var);

  /** @brief Destructor */
  virtual ~Action();

  /** @brief Mark that the action is now finished */
  void finish();

  /** @brief Get the [state](\ref e_surf_action_state_t) of the current Action */
  e_surf_action_state_t getState(); /**< get the state*/
  /** @brief Set the [state](\ref e_surf_action_state_t) of the current Action */
  virtual void setState(e_surf_action_state_t state);

  /** @brief Get the bound of the current Action */
  double getBound();
  /** @brief Set the bound of the current Action */
  void setBound(double bound);

  /** @brief Get the start time of the current action */
  double getStartTime();
  /** @brief Get the finish time of the current action */
  double getFinishTime();

  /** @brief Get the user data associated to the current action */
  void *getData() {return p_data;}
  /** @brief Set the user data associated to the current action */
  void setData(void* data);

  /** @brief Get the cost of the current action */
  double getCost() {return m_cost;}
  /** @brief Set the cost of the current action */
  void setCost(double cost) {m_cost = cost;}

  /** @brief Update the maximum duration of the current action
   *  @param delta Amount to remove from the MaxDuration */
  void updateMaxDuration(double delta) {double_update(&m_maxDuration, delta,sg_surf_precision);}

  /** @brief Update the remaining time of the current action
   *  @param delta Amount to remove from the remaining time */
  void updateRemains(double delta) {double_update(&m_remains, delta, sg_maxmin_precision*sg_surf_precision);}

  /** @brief Set the remaining time of the current action */
  void setRemains(double value) {m_remains = value;}
  /** @brief Get the remaining time of the current action after updating the resource */
  virtual double getRemains();
  /** @brief Get the remaining time of the current action without updating the resource */
  double getRemainsNoUpdate();

  /** @brief Set the finish time of the current action */
  void setFinishTime(double value) {m_finish = value;}

  /**@brief Add a reference to the current action (refcounting) */
  void ref();
  /** @brief Unref that action (and destroy it if refcount reaches 0)
   *  @return true if the action was destroyed and false if someone still has references on it
   */
  virtual int unref();

  /** @brief Cancel the current Action if running */
  virtual void cancel();

  /** @brief Suspend the current Action */
  virtual void suspend();

  /** @brief Resume the current Action */
  virtual void resume();

  /** @brief Returns true if the current action is running */
  virtual bool isSuspended();

  /** @brief Get the maximum duration of the current action */
  double getMaxDuration() {return m_maxDuration;}
  /** @brief Set the maximum duration of the current Action */
  virtual void setMaxDuration(double duration);

  /** @brief Get the tracing category associated to the current action */
  char *getCategory() {return p_category;}
  /** @brief Set the tracing category of the current Action */
  void setCategory(const char *category);

  /** @brief Get the priority of the current Action */
  double getPriority() {return m_priority;};
  /** @brief Set the priority of the current Action */
  virtual void setPriority(double priority);

  /** @brief Get the state set in which the action is */
  ActionList* getStateSet() {return p_stateSet;};

  s_xbt_swag_hookup_t p_stateHookup;

  Model *getModel() {return p_model;}

protected:
  ActionList* p_stateSet;
  double m_priority; /**< priority (1.0 by default) */
  int    m_refcount;
  double m_remains; /**< How much of that cost remains to be done in the currently running task */
  double m_maxDuration; /*< max_duration (may fluctuate until the task is completed) */
  double m_finish; /**< finish time : this is modified during the run and fluctuates until the task is completed */

private:
  double m_start; /**< start time  */
  char *p_category;               /**< tracing category for categorized resource utilization monitoring */

  #ifdef HAVE_LATENCY_BOUND_TRACKING
  int m_latencyLimited;               /**< Set to 1 if is limited by latency, 0 otherwise */
  #endif
  double    m_cost;
  Model *p_model;
  void *p_data; /**< for your convenience */

  /* LMM */
public:
  virtual void updateRemainingLazy(double now);
  void heapInsert(xbt_heap_t heap, double key, enum heap_action_type hat);
  void heapRemove(xbt_heap_t heap);
  void heapUpdate(xbt_heap_t heap, double key, enum heap_action_type hat);
  void updateIndexHeap(int i);
  lmm_variable_t getVariable() {return p_variable;}
  double getLastUpdate() {return m_lastUpdate;}
  void refreshLastUpdate() {m_lastUpdate = surf_get_clock();}
  enum heap_action_type getHat() {return m_hat;}
  bool is_linked() {return action_lmm_hook.is_linked();}
  void gapRemove();

protected:
  lmm_variable_t p_variable;
  double m_lastValue;
  double m_lastUpdate;
  int m_suspended;
  int m_indexHeap;
  enum heap_action_type m_hat;
};

typedef Action::ActionList ActionList;

typedef boost::intrusive::member_hook<
  Action, boost::intrusive::list_member_hook<>, &Action::action_lmm_hook> ActionLmmOptions;
typedef boost::intrusive::list<Action, ActionLmmOptions> ActionLmmList;
typedef ActionLmmList* ActionLmmListPtr;

/*********
 * Model *
 *********/
XBT_PUBLIC_DATA(xbt_dynar_t) model_list;

/** @ingroup SURF_interface
 * @brief SURF model interface class
 * @details A model is an object which handle the interactions between its Resources and its Actions
 */
XBT_PUBLIC_CLASS Model {
public:
  /** @brief Constructor */
  Model();
  /** @brief Destructor */
  virtual ~Model();

  virtual void addTraces() =0;

  /**
   * @brief Get the set of [actions](@ref Action) in *ready* state
   *
   * @return The set of [actions](@ref Action) in *ready* state
   */
  virtual ActionList* getReadyActionSet() {return p_readyActionSet;}

  /**
   * @brief Get the set of [actions](@ref Action) in *running* state
   *
   * @return The set of [actions](@ref Action) in *running* state
   */
  virtual ActionList* getRunningActionSet() {return p_runningActionSet;}

  /**
   * @brief Get the set of [actions](@ref Action) in *failed* state
   *
   * @return The set of [actions](@ref Action) in *failed* state
   */
  virtual ActionList* getFailedActionSet() {return p_failedActionSet;}

  /**
   * @brief Get the set of [actions](@ref Action) in *done* state
   *
   * @return The set of [actions](@ref Action) in *done* state
   */
  virtual ActionList* getDoneActionSet() {return p_doneActionSet;}

  /**
   * @brief Get the set of modified [actions](@ref Action)
   *
   * @return The set of modified [actions](@ref Action)
   */
  virtual ActionLmmListPtr getModifiedSet() {return p_modifiedSet;}

  /**
   * @brief Get the maxmin system of the current Model
   *
   * @return The maxmin system of the current Model
   */
  lmm_system_t getMaxminSystem() {return p_maxminSystem;}

  /**
   * @brief Get the update mechanism of the current Model
   * @see e_UM_t
   *
   * @return [description]
   */
  e_UM_t getUpdateMechanism() {return p_updateMechanism;}

  /**
   * @brief Get Action heap
   * @details [TODO]
   *
   * @return The Action heap
   */
  xbt_heap_t getActionHeap() {return p_actionHeap;}

  /**
   * @brief share the resources
   * @details Share the resources between the actions
   *
   * @param now The current time of the simulation
   * @return The delta of time till the next action will finish
   */
  virtual double shareResources(double now);
  virtual double shareResourcesLazy(double now);
  virtual double shareResourcesFull(double now);
  double shareResourcesMaxMin(ActionList* running_actions,
                                      lmm_system_t sys,
                                      void (*solve) (lmm_system_t));

  /**
   * @brief Update state of actions
   * @details Update action to the current time
   *
   * @param now The current time of the simulation
   * @param delta The delta of time since the last update
   */
  virtual void updateActionsState(double now, double delta);
  virtual void updateActionsStateLazy(double now, double delta);
  virtual void updateActionsStateFull(double now, double delta);

  /** @brief Returns whether this model have an idempotent shareResource()
   *
   * The only model that is not is NS3: computing the next timestamp moves the model up to that point,
   * so we need to call it only when the next timestamp of other sources is computed.
   */
  virtual bool shareResourcesIsIdempotent()=0;

protected:
  ActionLmmListPtr p_modifiedSet;
  lmm_system_t p_maxminSystem;
  e_UM_t p_updateMechanism = UM_UNDEFINED;
  int m_selectiveUpdate;
  xbt_heap_t p_actionHeap;

private:
  ActionList* p_readyActionSet; /**< Actions in state SURF_ACTION_READY */
  ActionList* p_runningActionSet; /**< Actions in state SURF_ACTION_RUNNING */
  ActionList* p_failedActionSet; /**< Actions in state SURF_ACTION_FAILED */
  ActionList* p_doneActionSet; /**< Actions in state SURF_ACTION_DONE */
};

/************
 * Resource *
 ************/

/** @ingroup SURF_interface
 * @brief Resource which have a metric handled by a maxmin system
 */
typedef struct {
  double scale;             /**< The scale of the metric */
  double peak;              /**< The peak of the metric */
  tmgr_trace_event_t event; /**< The associated trace event associated to the metric */
} s_surf_metric_t;

/** @ingroup SURF_interface
 * @brief SURF resource interface class
 * @details A resource represent an element of a component (e.g.: a link for the network)
 */
XBT_PUBLIC_CLASS Resource {
public:
  /**
   * @brief Resource constructor
   */
  Resource();

  /**
   * @brief Resource constructor
   *
   * @param model Model associated to this Resource
   * @param name The name of the Resource
   * @param props Dictionary of properties associated to this Resource
   */
  Resource(Model *model, const char *name, xbt_dict_t props);

  /**
   * @brief Resource constructor
   *
   * @param model Model associated to this Resource
   * @param name The name of the Resource
   * @param props Dictionary of properties associated to this Resource
   * @param constraint The lmm constraint associated to this Resource if it is part of a LMM component
   */
  Resource(Model *model, const char *name, xbt_dict_t props, lmm_constraint_t constraint);
  /**
   * @brief Resource constructor
   *
   * @param model Model associated to this Resource
   * @param name The name of the Resource
   * @param props Dictionary of properties associated to this Resource
   * @param stateInit the initial state of the Resource
   */
  Resource(Model *model, const char *name, xbt_dict_t props, e_surf_resource_state_t stateInit);

  /**
   * @brief Resource destructor
   */
  virtual ~Resource();

  /**
   * @brief Get the Model of the current Resource
   *
   * @return The Model of the current Resource
   */
  Model *getModel();

  /**
   * @brief Get the name of the current Resource
   *
   * @return The name of the current Resource
   */
  const char *getName();

  /**
   * @brief Get the properties of the current Resource
   *
   * @return The properties of the current Resource
   */
  virtual xbt_dict_t getProperties();

  /**
   * @brief Update the state of the current Resource
   * @details [TODO]
   *
   * @param event_type [TODO]
   * @param value [TODO]
   * @param date [TODO]
   */
  virtual void updateState(tmgr_trace_event_t event_type, double value, double date)=0;

  /**
   * @brief Check if the current Resource is used
   * @return true if the current Resource is used, false otherwise
   */
  virtual bool isUsed()=0;

  /**
   * @brief Check if the current Resource is active
   *
   * @return true if the current Resource is active, false otherwise
   */
  bool isOn();

  /**
   * @brief Turn on the current Resource
   */
  void turnOn();

  /**
   * @brief Turn off the current Resource
   */
  void turnOff();

  /**
   * @brief Get the [state](\ref e_surf_resource_state_t) of the current Resource
   *
   * @return The state of the currenrt Resource
   */
  virtual e_surf_resource_state_t getState();

  /**
   * @brief Set the [state](\ref e_surf_resource_state_t) of the current Resource
   *
   * @param state The new state of the current Resource
   */
  virtual void setState(e_surf_resource_state_t state);

private:
  const char *p_name;
  xbt_dict_t p_properties;
  Model *p_model;
  bool m_running;
  e_surf_resource_state_t m_stateCurrent;

  /* LMM */
public:
  /**
   * @brief Get the lmm constraint associated to this Resource if it is part of a LMM component
   *
   * @return The lmm constraint associated to this Resource
   */
  lmm_constraint_t getConstraint();
private:
  lmm_constraint_t p_constraint;
};

#endif /* SURF_MODEL_H_ */
