﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/networkmanager/NetworkManagerRequest.h>
#include <aws/networkmanager/NetworkManager_EXPORTS.h>
#include <aws/networkmanager/model/Bandwidth.h>
#include <aws/networkmanager/model/Tag.h>

#include <utility>

namespace Aws {
namespace NetworkManager {
namespace Model {

/**
 */
class CreateLinkRequest : public NetworkManagerRequest {
 public:
  AWS_NETWORKMANAGER_API CreateLinkRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateLink"; }

  AWS_NETWORKMANAGER_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The ID of the global network.</p>
   */
  inline const Aws::String& GetGlobalNetworkId() const { return m_globalNetworkId; }
  inline bool GlobalNetworkIdHasBeenSet() const { return m_globalNetworkIdHasBeenSet; }
  template <typename GlobalNetworkIdT = Aws::String>
  void SetGlobalNetworkId(GlobalNetworkIdT&& value) {
    m_globalNetworkIdHasBeenSet = true;
    m_globalNetworkId = std::forward<GlobalNetworkIdT>(value);
  }
  template <typename GlobalNetworkIdT = Aws::String>
  CreateLinkRequest& WithGlobalNetworkId(GlobalNetworkIdT&& value) {
    SetGlobalNetworkId(std::forward<GlobalNetworkIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description of the link.</p> <p>Constraints: Maximum length of 256
   * characters.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateLinkRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of the link.</p> <p>Constraints: Maximum length of 128 characters.
   * Cannot include the following characters: | \ ^</p>
   */
  inline const Aws::String& GetType() const { return m_type; }
  inline bool TypeHasBeenSet() const { return m_typeHasBeenSet; }
  template <typename TypeT = Aws::String>
  void SetType(TypeT&& value) {
    m_typeHasBeenSet = true;
    m_type = std::forward<TypeT>(value);
  }
  template <typename TypeT = Aws::String>
  CreateLinkRequest& WithType(TypeT&& value) {
    SetType(std::forward<TypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The upload speed and download speed in Mbps. </p>
   */
  inline const Bandwidth& GetBandwidth() const { return m_bandwidth; }
  inline bool BandwidthHasBeenSet() const { return m_bandwidthHasBeenSet; }
  template <typename BandwidthT = Bandwidth>
  void SetBandwidth(BandwidthT&& value) {
    m_bandwidthHasBeenSet = true;
    m_bandwidth = std::forward<BandwidthT>(value);
  }
  template <typename BandwidthT = Bandwidth>
  CreateLinkRequest& WithBandwidth(BandwidthT&& value) {
    SetBandwidth(std::forward<BandwidthT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The provider of the link.</p> <p>Constraints: Maximum length of 128
   * characters. Cannot include the following characters: | \ ^</p>
   */
  inline const Aws::String& GetProvider() const { return m_provider; }
  inline bool ProviderHasBeenSet() const { return m_providerHasBeenSet; }
  template <typename ProviderT = Aws::String>
  void SetProvider(ProviderT&& value) {
    m_providerHasBeenSet = true;
    m_provider = std::forward<ProviderT>(value);
  }
  template <typename ProviderT = Aws::String>
  CreateLinkRequest& WithProvider(ProviderT&& value) {
    SetProvider(std::forward<ProviderT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the site.</p>
   */
  inline const Aws::String& GetSiteId() const { return m_siteId; }
  inline bool SiteIdHasBeenSet() const { return m_siteIdHasBeenSet; }
  template <typename SiteIdT = Aws::String>
  void SetSiteId(SiteIdT&& value) {
    m_siteIdHasBeenSet = true;
    m_siteId = std::forward<SiteIdT>(value);
  }
  template <typename SiteIdT = Aws::String>
  CreateLinkRequest& WithSiteId(SiteIdT&& value) {
    SetSiteId(std::forward<SiteIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags to apply to the resource during creation.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateLinkRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateLinkRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_globalNetworkId;
  bool m_globalNetworkIdHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  Aws::String m_type;
  bool m_typeHasBeenSet = false;

  Bandwidth m_bandwidth;
  bool m_bandwidthHasBeenSet = false;

  Aws::String m_provider;
  bool m_providerHasBeenSet = false;

  Aws::String m_siteId;
  bool m_siteIdHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace NetworkManager
}  // namespace Aws
