/* sphereAlign.h  14 Feb 2008 TKS

  declarations for a spherical image alignment program built on
  the autopano-sift-c code base.

  copyright (c) 2008 TKSharpless.  Use freely under GPL license terms.

  N.B. include Panorama.h (or AutoPanoSift.h), if used, before this file

*/
/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public
 *  License along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifndef _SPHEREALIGN_H
#define _SPHEREALIGN_H


#ifdef __cplusplus
extern "C" {
#endif

////#include "pano13\panorama.h"
#ifndef PANORAMA_H
enum
{                               // Enumerates for Image.format
    _rectilinear = 0,
    _panorama = 1,
    _fisheye_circ = 2,
    _fisheye_ff = 3,
    _equirectangular = 4,
    _spherical_cp = 5,
    _spherical_tp = 6,
    _mirror = 7,
    _orthographic = 8,
    _cubic = 9,
    _stereographic = 10,
    _mercator = 11,
    _trans_mercator = 12,
    _trans_panorama = 13,
    _sinusoidal = 14,
    _lambert    = 15,
    _lambertazimuthal  = 16,
    _albersequalareaconic = 17,
    _millercylindrical = 18,
};
#endif	//ndef PANORAMA_H

#ifndef Pi
  #define Pi 3.1415926535897932384626433832795
  #define DEG2RAD( x ) ((x) * Pi/180.0 )
  #define RAD2DEG( x ) ((x) * 180.0 / Pi )
#endif

/* A minimal 'universal' desriptor for images in memory
*/
typedef struct MemImageDescr {
	int ID;				// search key
	void * data;		// -> primary array, 0 if unallocated
	int width, height;	// raster size
	int spp, bps;		// pixel size
  // pixel type flags	 
	char isSigned,	 // integer types only
		 isFloat,	 // HALF or float or double
		 isPacked,	 // samples of each pixel adjacent
		 alphaPos;	 // 0: none, else 1-origin index of alpha channel
					 // if < 0, alpha is in the image at pNext
  // pointer to another image
	struct MemImageDescr * pNext;  // 0 if none
} MemImageDescr, *pMemImageDescr;

/* MemImageDescr methods
*/
pMemImageDescr MemImageDescr_new0();
pMemImageDescr MemImageDescr_delete();
// create with data from a PT Image
//pMemImageDescr MemImageDescr_newImage( int ID, Image * pi );

		/** database records **/

typedef struct LensInfo {
	char * name;
	double FLmm;
	int radialFn;   // 0: ?? 1: tan(a) 2: 2sin(a/2) 3: sin(a) 4: (a)
	double Klens;	// projection function parameter
	double c[3];	// correction polynomial coeffs (x = r^2/FL^2)
} LensInfo, * pLensInfo;

typedef struct CameraInfo {
	char * name;
	int PT_fmt;		// codes defined in panorama.h
	int widpix, hgtpix;
	double widmm, hgtmm;
} CameraInfo, * pCameraInfo;


/* Projection data for input images

  In all fields except PT_fmt, zero unambiguously means "unspecified".

  Strictly speaking an image projection is given by an instance of
  mounting a given lens on a given camera, as mechanical alignment 
  may change a little.  So there should be one of these per "mount".
  
  To accomodate panoramic cameras (and computed panoramas, anamorphic 
  lenses, cheesy cameras with nonsquare pixels, etc) there are separate 
  horizontal and vertical projection functions, both of which must be 
  specified in all cases, even if they are the same. 

  We distinguish projection functions of real lens from projection 
  functions generated by other means (gears, computer, ...).  Real
  lens functions are subject to empirical correction, the others are
  assumed to be ideal.  The primary means of making this distinction is 
  whether a physical focal length and sensor size are given, or not; but
  some code just tests whether the lens parameters, hKlens and vKlens are 
  nonzero or zero.

  For ideal projections, a FL in pixels must be specified -- it is the 
  scale factor that relates pixel position to angle (think of it as the
  radius of the panosphere).

  The elliptically symmetric formats, rectilinear, equal-area, spherical, 
  orthographic, stereographic and mirror-ball could in principle have been
  projected by a lens; and actual lenses exist for all but stereographic.
  (equal area, orthographic and spherical are ideal "fisheye" projections;
  most real fisheye lenses approximate the equal-area.)  They are easy to 
  interconvert via a one-dimensional remapping of the radius.
  
  Assymetric formats may have a lens projection on one axis only.  For 
  example, a rotating slit camera with a rectilinear lens makes quasi-
  cylindrical images, and with a fisheye lens makes quasi-equirectangular
  ones.
 
  Klens parametrizes the fisheye lens projections, via the formula 
  radius/FL = Klens * sin( angle / Klens ).  The nominal values are 2 for
  the common equal-area type, 1 for orthographic and 100 for equal-angle 
  (true spherical).  Someday it may parametrize rectilinear and/or mirror 
  ball lenses, too; for now set it to 1 for those lens types. 
    
  If Klens is nonzero the correction polynomial also applies, but will have 
  no effect if all 3 coefficients are zero.  
*/

typedef struct CamLens {
// camera parameters
  int widpix, hgtpix;		// raster dimensions (rqd)
  double widmm, hgtmm;		// sensor size (optional)
  int    PT_fmt;	// codes defined in panorama.h, no typedef
  int	lensAxis;	// 0: neither 1: X, 2: Y 3: both
// lens parameters
  double FLmm;			// (0 for ideal projections)
  double Klens;			// parameter of the radius-to-angle fn
  double (*R2A)( double R, double K ); // radius-to-angle
  double (*A2R)( double A, double K ); // angle-to-radius
  double rCorr[3];		// correction polynomial coeffs
  double Rref;			// "unit" radius scales correction poly
// pixel <=> angle scale factors (usually computed from above)
  double hFLpix, vFLpix;	// pixels per radian,> 0 (both rq'd)
// "mount" parameters
  double hCpix, vCpix;		// lens center coordinates
  double Rcrop;             // largest valid normalized radius
} CamLens, * pCamLens;

/* CamLens methods
  R stands for radius normalized by focal length
  fmt is a Panotools format code
  flmm should be 0 for ideal projections, else > 0
*/
// create empty
pCamLens CamLens_new0();
// create with minimal info
pCamLens CamLens_new1( int wid, int hgt, int fmt, double flpix, double flmm );
// create with database info
pCamLens CamLens_new( pCameraInfo pc, pLensInfo pl );
//  change projection type
int CamLens_setProjection( pCamLens self, int fmt );
// normalized radius <=> angle in radians
double CamLens_AofR( pCamLens self, double R );
double CamLens_RofA( pCamLens self, double A );
// max legal normalized radius
double CamLens_Rmax( pCamLens self );	// get Rcrop = largest valid R
// get normalized radius of a point in pixel coords
double CamLens_getR( CamLens *p, double x, double y );


/* convert pixel coordinates between elliptically symmetric projections

   Each projection is specified by a CamLens.  Only points in 
   the Rmax circles of 'source' projection can be converted.

   For speed, tabulates the radial remapping function.

*/
typedef struct saRemap {
// source origin & scale
  double sX0, sY0;	// center
  double sFx, sFy;  // FLs
// dest origin & scale
  double dX0, dY0;	// center
  double dFx, dFy;	// FLs
 // function tables
  int npts;
  double *xval;	// source radius
  double *yval; // dest radius
  double *Cxy;	// spline for y(x)
  double *Cyx;  // spline for x(y) (0 if not valid)
} saRemap;

saRemap *	saRemap_new0( void );
void		saRemap_delete( saRemap *p );

/* set up a mapping
   returns NULL if it can't be reaalized
*/
saRemap * saRemap_new( CamLens * ps, CamLens * pd );

/* convert pixel coordinates
  if the mapping is undefined at the given input point,
  return 0 with output = (0,0)
  else return 1 
*/
int saRemap_fwd ( saRemap *p,
				  double xsrc, double ysrc,
				  double *xdest, double *ydest
				 );
int saRemap_inv ( saRemap *p,
				  double xdest, double ydest,
				  double *xsrc, double *ysrc
				 );

/* minimal image info for APSC++
  source is project file 'i' lines
  (LoadProjectImages() reads this)
*/
typedef struct DIinfo {
  // 'i' line input
	char name[262];
	int width, height;
	int format;
	double hfov;
	double yaw, pitch, roll;
} DIinfo, * pDIinfo;

// create empty
pDIinfo DIinfo_new0();
// delete
void DIinfo_delete( pDIinfo self );


#ifdef __cplusplus
}
#endif

#endif	// ndef _SPHEREALIGN_H
