#include <cstdio>
#include <iostream>

#include <QString>
#include <QSqlError>

#include "libmythui/schemawizard.h"
#include "libmythbase/compat.h"
#include "libmythbase/dbutil.h"
#include "libmythbase/mythconfig.h"
#include "libmythbase/mythcorecontext.h"
#include "libmythbase/mythdb.h"
#include "libmythbase/mythdbcheck.h"
#include "libmythbase/mythlogging.h"
#include "libmythbase/mythversion.h"

#include "cardutil.h"
#include "dbcheck.h"
#include "mythvideoprofile.h"
#include "recordinginfo.h"
#include "recordingprofile.h"
#include "recordingrule.h"
#include "videodbcheck.h" // for 1267

// TODO convert all dates to UTC

#define MINIMUM_DBMS_VERSION 5,0,15

const QString currentDatabaseVersion = MYTH_DATABASE_VERSION;

static bool doUpgradeTVDatabaseSchema(void);
static bool tryUpgradeTVDatabaseSchema(bool upgradeAllowed, bool upgradeIfNoUI, bool informSystemd);

#if CONFIG_SYSTEMD_NOTIFY
#include <systemd/sd-daemon.h>
static inline void db_sd_notify(const char *str)
{
    sd_notifyf(0, "STATUS=Database update: %s", str);
}
#else
static inline void db_sd_notify(const char */*str*/) {};
#endif

/** \defgroup db_schema MythTV Database Schema

\section db_intro Introduction

\verbatim
The schema contains the following tables:
\endverbatim

\htmlonly
<table>
<tr><td>capturecard                <td>pk(cardid)
<tr><td>cardinput                  <td>pk(cardinputid)
<tr><td>channel                    <td>pk(chanid) k(channum,sourceid)
<tr><td>channelgroup               <td>pk(id)
<tr><td>channelgroupnames          <td>pk(grpid)
<tr><td>codecparams                <td>pk(profile,name)
<tr><td>conflictresolutionany      <td>k(prefertitle) k(disliketitle)
<tr><td>conflictresolutionoverride <td>k(chanid,starttime) k(endtime)
<tr><td>conflictresolutionsingle   <td>k(preferchanid,preferstarttime) k(preferendtime)
<tr><td>credits                    <td>uk(chanid,starttime,person,role) k(person,role)
<tr><td>dtv_multiplex              <td>pk(mplexid)
<tr><td>dtv_privatetypes
<tr><td>dvb_signal_quality         <td>pk(id) k(sampletime,cardid)
<tr><td>housekeeping               <td>pk(tag)
<tr><td>jobqueue                   <td>pk(id) uk(chanid,starttime,type,inserttime)
<tr><td>jumppoints                 <td>pk(destination,hostname)
<tr><td>keybindings                <td>pk(context,action,hostname)
<tr><td>keyword                    <td>uk(phrase,searchtype)
<tr><td>mythlog                    <td>pk(logid)
<tr><td>oldprogram                 <td>pk(oldtitle)
<tr><td>oldrecorded                <td>k(endtime) k(title) k(seriesid) k(programid)
                                       pk(station,starttime,title)
<tr><td>people                     <td>pk(person) uk(name)
<tr><td>pidcache                   <td>
<tr><td>profilegroups              <td>pk(id) uk(name,hostname)
<tr><td>program                    <td>k(endtime) k(title_pronounce) k(seriesid)
                                       k(programid,starttime) k(chanid,starttime,endtime)
                                       k(title,subtitle,starttime)
<tr><td>programgenres              <td>pk(chanid,starttime,relevance)
<tr><td>programrating              <td>uk(chanid,starttime,system,rating)
                                       k(starttime,system)
<tr><td>recgrouppassword           <td>pk(recgroup) uk(recgroup)
<tr><td>record                     <td>pk(recordid) k(chanid,starttime) k(title)
                                       k(seriesid) k(programid)
<tr><td>recorded                   <td>k(endtime) k(seriesid) k(programid)
<tr><td>recordedmarkup             <td>pk(chanid,starttime,mark,type)
<tr><td>recordingprofiles          <td>pk(id)
<tr><td>recordoverride             <td>
<tr><td>settings                   <td>k(value,hostname)
<tr><td>videosource                <td>pk(sourceid) uk(name)
<tr><td>displayprofilegroups       <td>pk(name, host), uk(profileid)
<tr><td>displayprofiles            <td>pk(profileid),
</table>
\endhtmlonly

Where pk means PRIMARY KEY, uk means "UNIQUE KEY", and k means "KEY".
BUSQ refers to the Big Ugly SQL Query in scheduler.cpp

\section capturecard_table Capture Card Table (capturecard)
This table describes the attributes of a capture card used by MythTV.

The primary key 'cardid' is automatically generated by the database.

The 'videodevice' field normally points to a device used to capture
the video, but for DVB where multiple devices are used it is simply
a number indicating which set of devices to use. The 'audiodevice'
is only used by framegrabber cards, encoding cards like the Hauppauge
PVR line encode the audio as part of the video stream, while DTV
cards like the various ATSC and DVB cards receive the audio as part
of the video stream. The same applies to the 'vbidevice', PVR and
DTV cards have captions encoded in the digital stream, while
framegrabber cards use this device to provide caption and other
data to MythTV.

The 'audioratelimit' field is used to set the audio capture rate
when using the 'audiodevice' field. This is mostly useful for
V4L cards that have troublesome on-board audio, such as some
cards supported by the bttv driver.


The 'cardtype' is an important field for all cards as it guides the
interpretation of the rest of the fields, it has several possible
values: "V4L", "MJPEG", "DVB", "MPEG", "FIREWIRE", and "GO7007".
"V4L" indicates a V4L compatible device, this could be a
true V4L device or something like a Firewire camera with the
"firewire->v4l" driver.
"MJPEG" describes a Matrox G200 or DC10 card which captures
individual frames as JPEG images.
"DVB" can describe any card supported by the Linux DVB
drivers, including the pcHDTV and other ATSC cards.
The "MPEG" type describes a device using the ivtv drivers such as
the Hauppauge PVR[150,250,350,500]s +Freestyle cards, the Yuan MPEG
series, the Avermedia M179, and the Adaptec VideOh! AVC-2410.
The "FIREWIRE" card type does not describe Firewire cameras or
capture cards but instead describes the Firewire port required by
law in the USA on cable boxes for subscribers who receive HDTV
channels and request the Firewire port feature. This port lets
you communicate with your cable box without needing to build an
IR transmitter; in some cases, it also allows you to digitally
capture video.
The "GO7007" card type describes a USB MPEG-4 encoder such
as the Plextor ConvertX.

The 'hostname' field is another important field for all cards
as it specifies which backend the capture card is connected to.

The 'signal_timeout' and 'channel_timeout' indicate in
milliseconds how long it should take to get a signal and
channel lock respectively.

The 'dvb_swfilter', 'dvb_sat_type', 'dvb_wait_for_seqstart',
'skipbtaudio', 'dvb_on_demand', and 'dvb_diseqc_type' columns
are all "DVB" specific configuration parameters.

Both 'dvb_recordts' and 'dvb_hw_decoder' are unused, and
will be dropped in future versions of MythTV.

The 'firewire_port', 'firewire_node', 'firewire_speed',
'firewire_model', 'firewire_connection' are all "FIREWIRE" specific
configuration parameters. The first three define the connection,
and function much like 'videodevice' does for other capture "cards".
The 'firewire_model' describes the cable box model, for example
"DCT-6000" describes a box that communicates using the same protocol
as the Motorola DCT-6000. The 'firewire_connection' field describes
the overall communication protocol, i.e. are we using "Broadcast"
or "Point-to-Point" communication.

\section dtv_multiplex_table Digital Television Multiplex Table (dtv_multiplex)
This table contains the information needed to tune to a particular
frequency on a particular input.

The primary key, 'mplexid', is automatically generated by the database.
The most important field is the 'frequency' field which contains the
transmission frequency in Hertz. The 'sourceid' which tells us which
card and on what input of that card this multiplex applies to according
to the \ref videosource_table. Along with a channel number 'sourceid'
also uniquely identifies a channel in the \ref channel_table. The
'modulation' field tells us whether the modulation mode is "qpsk",
"qam_16", "qam_32", "qam_64", "qam_128", "qam_256", "8vsb", "16vsb",
or "auto". Finally, the 'sistandard' field is the last field which
applies to both DVB and ATSC, it contains on of four strings, "dvb",
"atsc", "mpeg" or "opencable".

When describing a DVB Multiplex, the networkid, and transportid are
used along with the serviceid in the \ref channel_table to tune to
a particular MPEG stream in DVBChannel.

When describing an ATSC Multiplex, the \ref channel_table contains
the major and minor channels used to tune to a particular stream.

Other fields are used by different DVBx implementations to tune to a
a particular type of service and MythTV just passes them along to the
drivers to take care of.



\section dtv_privatetypes_table Digital Television Private Types Table (dtv_privatetypes)
This table contains free form data pertaining to DVB services, it is
like a settings table for DVB.



\section videosource_table Video Source Table (videosource)
This table tells us about a particular input on a video capture card.

The primary key, 'sourceid', is automatically generated by the database.
The 'name' field gives the input name as passed to the video capture card
driver. The 'xmltvgrabber' field tells us where we get our listings from.
The 'userid' and 'password' fields are optionally used to log in to a
listings service, with the 'lineupid' being used to select a particular
lineup available from the listings service. Finally, the 'freqtable'
tells us which channel number to frequency mapping to use with
analog channels, us, us-cable, pal, etc.



\section channel_table Channel Table (channel)
The primary key, 'chanid' is generated by multiplying the sourceid times
1000 and adding the channel number to create a unique chanid.
See also ChannelUtil::CreateChanId() .

The 'channum' field contains the channel number to be input to jump
to a channel.

The 'sourceid' field contains the \ref videosource_table key, which
specifies which listings service to use and which input on which video
capture card to find this channel.

The 'callsign' field contains the government mandated license number
of the channel's broadcaster if this is an over-the-air channel.

The 'name' field provides the alphanumeric name of this channel for
use in the %MythTV user interface.

The 'icon' field provides the path to an optional channel icon for this
channel.

The 'freqid' field provides the channel number to tune to, as
differentiated from the 'channum' used in the %MythTV user interface.
When the 'mplexid' is set it is used instead. And the 'finetune' field
is used to adjust the frequency provided by the combination of 'freqid'
and 'tvformat', this can be used when there is carrier drift in a cable
system.

The 'xmltvid' field is used to identify this channel to the listings
provider.

The 'tmoffset' field is used to apply an offset (in minutes) from the listings
provided by the provider to a new time in the MythTV program guide database.
This is very handy when the listings provider has listings which are offset
by a few hours on individual channels with the rest of them being correct.

The 'recpriority' field is used tell the scheduler from which of two
otherwise equivalent programs on two different channels should be
preferred, a higher number means this channel is more preferred.

The 'tvformat' is one of "ntsc", "ntsc-jp", "pal", "atsc", etc and is used
for various purposes, such as translating a channel number to a frequency,
setting the recording frame size, etc.

'commfree' tells us whether this is a commercial free channel, such as
those on the BBC and CBC networks. On commercial free channels we
do not need to run the commercial detector.

'commmethod' tells us which commercial flagger to use on the channel, the
default is to use the global setting

'visible' tells us whether we should show this channel in the channel
guide.

'outputfilters' contains a list of filters that should be applied
when playing recordings from this channel.

The 'contrast' 'brightness', 'colour', and 'hue' fields are used by frame
grabbing recorders to adjust the picture before encoding each frame.
And, 'videofilters' contains a list of filters that should be applied
when recording from channel using a frame grabbing recorder.

For digital television, the 'mplexid' field contains the
\ref dtv_multiplex_table key for determining additional digital tuning
parameters. The 'serviceid' field contains the MPEG program number when
describing a DVB service. The 'useonairguide' field is used by DVB to
determine whether to use the on-air-guide data from this channel to
update the listings.

\section program_table Program Entry Table (program)
'category_type' holds one of these exact four strings:
"movie", "series", "sports" or "tvshow".

'airdate' is a string representing the year of release for movies and
may have no meaning for other types of shows.

'stars' is a floating point number from 0.0 to 1.0. On a four star scale,
1.0 would be four stars, 0.75 would be three stars and so on.

'originalairdate' if provided is the date when a show was, or will be,
first televised.

'previouslyshown' is a field created by MythTV to try to determine if
a showing is more than 14 days after its original air date or if the
show was marked as a repeat and did not have a date for the first
airing. If this is "0" it usually means that this is a brand new show
or a rebroadcast within the first two weeks.

'programid' is the Tribune Media Service database record identifier
for each program description. In general, these start with a two
letter prefix, MV, EP, SP or SH that equate to the 'category_type'.
For most, the last four digits are "0000" except EP where the last
four digits are the episode number in the series. Note that these
are generated by TMS and not the show's producers but they are
usually in the same order as the original air dates for the episodes.

\section oldrecorded_table Old Recorded Table (oldrecorded)

oldrecorded imposes the restriction of one entry per
title, starttime and callsign.  The scheduler relies on this when
restoring any previous status for programs that aren't currently
recording and for catching reactivation requests.

The duplicate field is used to indicate if this record should be used
to check for duplicates in the BUSQ

\section oldfind_table Old Find Table (oldfind)

If a matching entry exists in
the oldfind table, the program is considered a duplicate regardless of
the duplicate setting in recorded and oldrecorded.  oldfind is an
imperfect optimization to avoid greatly increasing the time needed for
the big scheduler query.

But the real reason for oldfind is that a search
rule may not be able to match by title. For instance, a FindWeekly
"Bill Murray (People Search)" needs to know that "Meatballs" is a
duplicate for this week if "Stripes" has already been recorded. Oldfind
tracks if the rule (recordid) has recorded in the current time
interval (findid). When a recording has a findid, ForgetHistory()
and DeleteHistory() update oldfind while updating oldrecorded.

\section recorded_table Recorded Table (recorded

The duplicate field is used to indicate if this record should be used
to check for duplicates in the BUSQ
 */

/**
 *  \brief Called from outside dbcheck.cpp to update the schema.
 *
 *   If the "DBSchemaVer" property equals the currentDatabase version this
 *   returns true immediately. If not, we try to do a database backup,
 *   if interactive prompt the user for permission to do the upgrade,
 *   and if permitted or non-interactive with a successful backup
 *   do the actual update.
 *
 *   If the program running this function is killed while
 *   this is running then the schema may be corrupted.
 *
 *  \return false on failure, error, or if the user selected "Exit."
 */
bool UpgradeTVDatabaseSchema(const bool upgradeAllowed,
                             const bool upgradeIfNoUI,
                             [[maybe_unused]] const bool informSystemd)
{
#ifdef IGNORE_SCHEMA_VER_MISMATCH
    return true;
#endif
    // Suppress DB messages and turn of the settings cache,
    // These are likely to confuse the users and the code, respectively.
    GetMythDB()->SetSuppressDBMessages(true);
    gCoreContext->ActivateSettingsCache(false);

    // Get the schema upgrade lock
    MSqlQuery query(MSqlQuery::InitCon());
    bool locked = DBUtil::TryLockSchema(query, 1);
    for (uint i = 0; i < 2*60 && !locked; i++)
    {
        if (informSystemd)
            db_sd_notify("waiting for lock");
        LOG(VB_GENERAL, LOG_INFO, "Waiting for database schema upgrade lock");
        locked = DBUtil::TryLockSchema(query, 1);
        if (locked)
            LOG(VB_GENERAL, LOG_INFO, "Got schema upgrade lock");
    }
    if (!locked)
    {
        LOG(VB_GENERAL, LOG_INFO, "Failed to get schema upgrade lock");
        if (informSystemd)
            db_sd_notify("failed to get schema upgrade lock");
        GetMythDB()->SetSuppressDBMessages(false);
        gCoreContext->ActivateSettingsCache(true);
        return false;
    }

    bool success = tryUpgradeTVDatabaseSchema(upgradeAllowed, upgradeIfNoUI, informSystemd);

    // On any exit we want to re-enable the DB messages so errors
    // are reported and we want to make sure the setting cache is
    // enabled for good performance and we must unlock the schema
    // lock.
    if (informSystemd)
        db_sd_notify(success ? "success" : "failed");
    GetMythDB()->SetSuppressDBMessages(false);
    gCoreContext->ActivateSettingsCache(true);
    DBUtil::UnlockSchema(query);
    return success;
}

static bool tryUpgradeTVDatabaseSchema(bool upgradeAllowed, bool upgradeIfNoUI, bool informSystemd)
{
    // Fix bad upgrade
    // In commit 08a7b58a73 the schema was upgraded to 1385 in fixes/35
    // This should not be done in a fixes branch.
    QString ver = gCoreContext->GetSetting("DbSchemaVer");
    if (upgradeAllowed && ver == "1385" && currentDatabaseVersion == "1384") {
        gCoreContext->SaveSettingOnHost("DbSchemaVer","1384", nullptr);
        LOG(VB_GENERAL, LOG_WARNING, "Downgrading Database from version 1385 to 1384.");
    }

    // Determine if an upgrade is needed
    SchemaUpgradeWizard* schema_wizard = SchemaUpgradeWizard::Get(
        "DBSchemaVer", "MythTV", currentDatabaseVersion);
    if (schema_wizard->Compare() == 0) // DB schema is what we need it to be..
        return true;

    if (!upgradeAllowed)
        LOG(VB_GENERAL, LOG_WARNING, "Not allowed to upgrade the database.");

    if (informSystemd)
        db_sd_notify("waiting for user input");
    // Pop up messages, questions, warnings, etc.
    switch (schema_wizard->PromptForUpgrade(
                "TV", upgradeAllowed, upgradeIfNoUI, MINIMUM_DBMS_VERSION))
    {
        case MYTH_SCHEMA_USE_EXISTING:
            return true;
        case MYTH_SCHEMA_ERROR:
        case MYTH_SCHEMA_EXIT:
            return false;
        case MYTH_SCHEMA_UPGRADE:
            break;
    }

    LOG(VB_GENERAL, LOG_DEBUG, QString("Newest MythTV Schema Version : %1")
        .arg(currentDatabaseVersion));

    // Upgrade the schema
    if (informSystemd)
        db_sd_notify("upgrading database");
    if (!doUpgradeTVDatabaseSchema())
    {
        LOG(VB_GENERAL, LOG_ERR, "Database schema upgrade failed.");
        return false;
    }

    LOG(VB_GENERAL, LOG_INFO, "Database schema upgrade complete.");

    return true;
}

/** \fn doUpgradeTVDatabaseSchema(void)
 *  \brief This is called by UpgradeTVDatabaseSchema() to actually upgrade
 *         the schema to what MythTV expects.
 *
 *   This function Initializes the database if the "DBSchemaVer" property
 *   does not exist in the database. Otherwise it upgrades the schema
 *   from the one in that property to the currentDatabaseVersion, by
 *   running the SQL commands needed for each upgrade between the two
 *   versions. This will only upgrade the schema, so bad things may
 *   happen if you upgrade and then downgrade MythTV across DB Schema
 *   versions.
 *
 */
static bool doUpgradeTVDatabaseSchema(void)
{
    QString dbver = gCoreContext->GetSetting("DBSchemaVer");
    if (dbver == currentDatabaseVersion)
    {
        return true;
    }

    // Don't rely on this, please specify these when creating the database.
    {
        MSqlQuery query(MSqlQuery::InitCon());
        if (!query.exec(QString("ALTER DATABASE %1 DEFAULT "
                                "CHARACTER SET utf8 COLLATE utf8_general_ci;")
                        .arg(GetMythDB()->GetDatabaseName())))
        {
            MythDB::DBError("UpgradeTVDatabaseSchema -- alter charset", query);
        }
    }

    if (DBUtil::IsNewDatabase())
    {
        if (!InitializeMythSchema())
            return false;
        dbver = gCoreContext->GetSetting("DBSchemaVer");
    }

    if (dbver.isEmpty() || dbver.toInt() <  1027)
    {
        LOG(VB_GENERAL, LOG_ERR, "Unrecognized database schema version. "
                                 "Unable to upgrade database.");
        return false;
    }
    if (dbver.toInt() <  1348)
    {
        LOG(VB_GENERAL, LOG_ERR, "Your database version is too old to upgrade "
                                 "with this version of MythTV. You will need "
                                 "to use mythtv-setup from MythTV 29 to 34 "
                                 "to upgrade your database before "
                                 "upgrading to this version of MythTV.");
        return false;
    }

    if (dbver == "1348")
    {
        DBUpdates updates {
            "update capturecard "
            "    set videodevice=left(videodevice, "
            "                         locate('-', videodevice)-1) "
            "    where cardtype='HDHOMERUN' "
            "          and videodevice like '%-%'"
        };
        if (!performActualUpdate("MythTV", "DBSchemaVer",
                                 updates, "1349", dbver))
            return false;
    }

    if (dbver == "1349")
    {
        DBUpdates updates {
            // Incorrect DB update removed
        };
        if (!performActualUpdate("MythTV", "DBSchemaVer",
                                 updates, "1350", dbver))
            return false;
    }

    if (dbver == "1350")
    {
        DBUpdates updates {
            "ALTER TABLE videosource ADD COLUMN bouquet_id INT DEFAULT 0;",
            "ALTER TABLE videosource ADD COLUMN region_id INT DEFAULT 0;"
        };
        if (!performActualUpdate("MythTV", "DBSchemaVer",
                                 updates, "1351", dbver))
            return false;
    }

    if (dbver == "1351")
    {
        DBUpdates updates {
            "ALTER TABLE videosource MODIFY bouquet_id INT UNSIGNED;",
            "ALTER TABLE videosource MODIFY region_id INT UNSIGNED;",
            "ALTER TABLE channel ADD COLUMN service_type INT UNSIGNED DEFAULT 0 AFTER serviceid;"
        };
        if (!performActualUpdate("MythTV", "DBSchemaVer",
                                 updates, "1352", dbver))
            return false;
    }

    if (dbver == "1352")
    {
        DBUpdates updates {
            "ALTER TABLE capturecard MODIFY schedgroup TINYINT(1) DEFAULT 1 NOT NULL"
        };
        if (!performActualUpdate("MythTV", "DBSchemaVer",
                                 updates, "1353", dbver))
            return false;
    }

    if (dbver == "1353")
    {
        DBUpdates updates {
            "ALTER TABLE channel ADD COLUMN deleted TIMESTAMP NULL"
        };
        if (!performActualUpdate("MythTV", "DBSchemaVer",
                                 updates, "1354", dbver))
            return false;
    }

    if (dbver == "1354")
    {
        DBUpdates updates {
            "ALTER TABLE videosource ADD COLUMN scanfrequency INT UNSIGNED DEFAULT 0;"
        };
        if (!performActualUpdate("MythTV", "DBSchemaVer",
                                 updates, "1355", dbver))
            return false;
    }

    if (dbver == "1355")
    {
        DBUpdates updates {
            "UPDATE capturecard "
            "SET displayname = CONCAT('Input ', cardid) "
            "WHERE displayname = ''"
        };
        if (!performActualUpdate("MythTV", "DBSchemaVer",
                                 updates, "1356", dbver))
            return false;
    }

    if (dbver == "1356")
    {
        DBUpdates updates {
            "REPLACE INTO recordfilter (filterid, description, clause, "
            "                          newruledefault) "
            "  VALUES (12, 'Priority channel', 'channel.recpriority > 0', 0)"
        };
        if (!performActualUpdate("MythTV", "DBSchemaVer",
                                 updates, "1357", dbver))
            return false;
    }

    if (dbver == "1357")
    {
        // convert old VideoDisplayProfile settings to new format
        MythVideoProfileItem temp;
        std::vector<MythVideoProfileItem> profiles;

        MSqlQuery query(MSqlQuery::InitCon());
        query.prepare("SELECT profileid, value, data FROM displayprofiles "
                      "ORDER BY profileid");

        for (;;)
        {
            if (!query.exec())
                break;

            uint currentprofile = 0;
            while (query.next())
            {
                if (query.value(0).toUInt() != currentprofile)
                {
                    if (currentprofile)
                    {
                        temp.SetProfileID(currentprofile);
                        profiles.push_back(temp);
                    }
                    temp.Clear();
                    currentprofile = query.value(0).toUInt();
                }
                temp.Set(query.value(1).toString(), query.value(2).toString());
            }

            if (currentprofile)
            {
                temp.SetProfileID(currentprofile);
                profiles.push_back(temp);
            }

            for (const MythVideoProfileItem& profile : std::as_const(profiles))
            {
                QString newdecoder;
                QString newrender;
                QString newdeint0;
                QString newdeint1;

                QString olddecoder = profile.Get("pref_decoder");
                QString oldrender  = profile.Get("pref_videorenderer");
                QString olddeint0  = profile.Get("pref_deint0");
                QString olddeint1  = profile.Get("pref_deint1");

                if (oldrender == "xv-blit")
                {
                    newdecoder = "ffmpeg";
                    newrender  = "opengl-yv12";
                }
                if (olddecoder == "openmax" || oldrender == "openmax")
                {
                    newdecoder = "mmal-dec";
                    newrender  = "opengl-yv12";
                }
                if ((olddecoder == "mediacodec") || (olddecoder == "nvdec") ||
                    (olddecoder == "vda") || (olddecoder == "vaapi2") ||
                    (olddecoder == "vaapi" && oldrender == "openglvaapi") ||
                    (olddecoder == "vdpau" && oldrender == "vdpau"))
                {
                    if (oldrender != "opengl-hw")
                        newrender = "opengl-hw";
                }
                if (olddecoder == "vda")
                    newdecoder = "vtb";
                if (olddecoder == "vaapi2")
                    newdecoder = "vaapi";

                auto UpdateDeinterlacer = [](const QString &Olddeint, QString &Newdeint, const QString &Decoder)
                {
                    if (Olddeint.isEmpty())
                    {
                        Newdeint = "none";
                    }
                    else if (Olddeint == "none" ||
                             Olddeint.contains(DEINT_QUALITY_SHADER) ||
                             Olddeint.contains(DEINT_QUALITY_DRIVER) ||
                             Olddeint.contains(DEINT_QUALITY_LOW) ||
                             Olddeint.contains(DEINT_QUALITY_MEDIUM) ||
                             Olddeint.contains(DEINT_QUALITY_HIGH))
                    {
                        return;
                    }

                    QStringList newsettings;
                    bool driver = (Decoder != "ffmpeg") &&
                        (Olddeint.contains("vaapi") || Olddeint.contains("vdpau") ||
                         Olddeint.contains("nvdec"));
                    if (driver)
                        newsettings << DEINT_QUALITY_DRIVER;
                    if (Olddeint.contains("opengl") || driver)
                        newsettings << DEINT_QUALITY_SHADER;

                    if (Olddeint.contains("greedy") || Olddeint.contains("yadif") ||
                        Olddeint.contains("kernel") || Olddeint.contains("advanced") ||
                        Olddeint.contains("compensated") || Olddeint.contains("adaptive"))
                    {
                        newsettings << DEINT_QUALITY_HIGH;
                    }
                    else if (Olddeint.contains("bob") || Olddeint.contains("onefield") ||
                             Olddeint.contains("linedouble"))
                    {
                        newsettings << DEINT_QUALITY_LOW;
                    }
                    else
                    {
                        newsettings << DEINT_QUALITY_MEDIUM;
                    }
                    Newdeint = newsettings.join(":");
                };

                QString decoder = newdecoder.isEmpty() ? olddecoder : newdecoder;
                UpdateDeinterlacer(olddeint0, newdeint0, decoder);
                UpdateDeinterlacer(olddeint1, newdeint1, decoder);

                auto UpdateData = [](uint ProfileID, const QString &Value, const QString &Data)
                {
                    MSqlQuery update(MSqlQuery::InitCon());
                    update.prepare(
                        "UPDATE displayprofiles SET data = :DATA "
                        "WHERE profileid = :PROFILEID AND value = :VALUE");
                    update.bindValue(":PROFILEID", ProfileID);
                    update.bindValue(":VALUE",     Value);
                    update.bindValue(":DATA",      Data);
                    if (!update.exec())
                        LOG(VB_GENERAL, LOG_ERR,
                            QString("Error updating display profile id %1").arg(ProfileID));
                };

                uint id = profile.GetProfileID();
                if (!newdecoder.isEmpty())
                    UpdateData(id, "pref_decoder", newdecoder);
                if (!newrender.isEmpty())
                    UpdateData(id, "pref_videorenderer", newrender);
                if (!newdeint0.isEmpty())
                    UpdateData(id, "pref_deint0", newdeint0);
                if (!newdeint1.isEmpty())
                    UpdateData(id, "pref_deint1", newdeint1);
            }
            break;
        }

        // remove old studio levels keybinding
        DBUpdates updates {
            "DELETE FROM keybindings WHERE action='TOGGLESTUDIOLEVELS'"
        };

        if (!performActualUpdate("MythTV", "DBSchemaVer",
                                 updates, "1358", dbver))
            return false;
    }

    if (dbver == "1358")
    {
        DBUpdates updates {
            // Allow videosouce.userid to be NULL as originally intended.
            "ALTER TABLE videosource "
            "  CHANGE COLUMN userid userid VARCHAR(128) NULL DEFAULT NULL",
            // And fix any leftover, empty values.
            "UPDATE videosource "
            "  SET userid = NULL "
            "  WHERE userid = ''",
            // Remove potential clear text credentials no longer usable
            "UPDATE videosource "
            "  SET userid = NULL, password = NULL "
            "  WHERE xmltvgrabber IN ('schedulesdirect1', 'datadirect')"
        };
        if (!performActualUpdate("MythTV", "DBSchemaVer",
                                 updates, "1359", dbver))
            return false;
    }

    if (dbver == "1359")
    {
        // XineramaMonitorAspectRatio was previously ignored for single screen
        // setups but now acts as an override for the display aspect ratio.
        // 0.0 indicates 'Auto' - which should be the default.
        DBUpdates updates {
            "UPDATE settings SET data='0.0' WHERE value='XineramaMonitorAspectRatio'"
        };
        if (!performActualUpdate("MythTV", "DBSchemaVer",
                                 updates, "1360", dbver))
            return false;
    }

    if (dbver == "1360")
    {
        // missed in 1357 - convert old vdpau and openglvaapi renderers to opengl
        // convert ancient quartz-blit to opengl as well
        MythVideoProfileItem temp;
        std::vector<MythVideoProfileItem> profiles;

        MSqlQuery query(MSqlQuery::InitCon());
        query.prepare("SELECT profileid, value, data FROM displayprofiles "
                      "ORDER BY profileid");

        // coverity[unreachable] False positive.
        for (;;)
        {
            if (!query.exec())
                break;

            uint currentprofile = 0;
            while (query.next())
            {
                if (query.value(0).toUInt() != currentprofile)
                {
                    if (currentprofile)
                    {
                        temp.SetProfileID(currentprofile);
                        profiles.push_back(temp);
                    }
                    temp.Clear();
                    currentprofile = query.value(0).toUInt();
                }
                temp.Set(query.value(1).toString(), query.value(2).toString());
            }

            if (currentprofile)
            {
                temp.SetProfileID(currentprofile);
                profiles.push_back(temp);
            }

            for (const MythVideoProfileItem& profile : std::as_const(profiles))
            {
                // the old deinterlacers will have been converted already
                QString oldrender  = profile.Get("pref_videorenderer");
                if (oldrender == "quartz-blit" || oldrender == "openglvaapi" ||
                    oldrender == "vdpau")
                {
                    auto UpdateData = [](uint ProfileID, const QString &Value, const QString &Data)
                    {
                        MSqlQuery update(MSqlQuery::InitCon());
                        update.prepare(
                            "UPDATE displayprofiles SET data = :DATA "
                            "WHERE profileid = :PROFILEID AND value = :VALUE");
                        update.bindValue(":PROFILEID", ProfileID);
                        update.bindValue(":VALUE",     Value);
                        update.bindValue(":DATA",      Data);
                        if (!update.exec())
                            LOG(VB_GENERAL, LOG_ERR,
                                QString("Error updating display profile id %1").arg(ProfileID));
                    };

                    uint id = profile.GetProfileID();
                    UpdateData(id, "pref_decoder", "ffmpeg");
                    UpdateData(id, "pref_videorenderer", "opengl-yv12");
                }
            }
            break;
        }

        if (!UpdateDBVersionNumber("MythTV", "DBSchemaVer", "1361", dbver))
            return false;
    }
    if (dbver == "1361")
    {
        DBUpdates updates {
            "ALTER TABLE program CHANGE COLUMN videoprop videoprop "
            "    SET('WIDESCREEN', 'HDTV', 'MPEG2', 'AVC', 'HEVC') NOT NULL; ",
            "ALTER TABLE recordedprogram CHANGE COLUMN videoprop videoprop "
            "    SET('WIDESCREEN', 'HDTV', 'MPEG2', 'AVC', 'HEVC', "
            "        '720', '1080', '4K', '3DTV', 'PROGRESSIVE', "
            "        'DAMAGED') NOT NULL; ",
        };
        if (!performActualUpdate("MythTV", "DBSchemaVer",
                                 updates, "1362", dbver))
            return false;
    }

    if (dbver == "1362")
    {
        MSqlQuery select(MSqlQuery::InitCon());
        select.prepare(
            QString("select index_name from information_schema.statistics "
            "where table_schema = '%1' "
            "and table_name = 'recordedartwork' "
            "and seq_in_index = 1 "
            "and column_name = 'inetref'")
            .arg(GetMythDB()->GetDatabaseName()));

        if (!select.exec())
        {
            MythDB::DBError("Unable to retrieve index values.", select);
            return false;
        }

        DBUpdates updates {
            "CREATE INDEX recordedartwork_ix1 ON recordedartwork (inetref); "
        };

        // do not create index if already done.
        if (select.size() > 0) {
            updates.clear();
        }

        if (!performActualUpdate("MythTV", "DBSchemaVer",
                                 updates, "1363", dbver))
            return false;
    }

    if (dbver == "1363")
    {
        MSqlQuery query(MSqlQuery::InitCon());

        // insert a new profile group for Sat>IP
        query.prepare("INSERT INTO profilegroups SET name = 'Sat>IP Recorder', "
                       "cardtype = 'SATIP', is_default = 1;");
        if (!query.exec())
        {
            MythDB::DBError("Unable to insert Sat>IP profilegroup.", query);
            return false;
        }

        // get the id of the new profile group
        int groupid = query.lastInsertId().toInt();

        // insert the recording profiles
        query.prepare("INSERT INTO recordingprofiles SET name = \"Default\", profilegroup = :GROUPID;");
        query.bindValue(":GROUPID", groupid);
        if (!query.exec())
        {
            MythDB::DBError("Unable to insert 'Default' recordingprofile.", query);
            return false;
        }

        query.prepare("INSERT INTO recordingprofiles SET name = \"Live TV\", profilegroup = :GROUPID;");
        query.bindValue(":GROUPID", groupid);
        if (!query.exec())
        {
            MythDB::DBError("Unable to insert 'Live TV' recordingprofile.", query);
            return false;
        }

        query.prepare("INSERT INTO recordingprofiles SET name = \"High Quality\", profilegroup = :GROUPID;");
        query.bindValue(":GROUPID", groupid);
        if (!query.exec())
        {
            MythDB::DBError("Unable to insert 'High Quality' recordingprofile.", query);
            return false;
        }

        query.prepare("INSERT INTO recordingprofiles SET name = \"Low Quality\", profilegroup = :GROUPID;");
        query.bindValue(":GROUPID", groupid);
        if (!query.exec())
        {
            MythDB::DBError("Unable to insert 'Low Quality' recordingprofile.", query);
            return false;
        }

        if (!UpdateDBVersionNumber("MythTV", "DBSchemaVer", "1364", dbver))
            return false;
    }

    if (dbver == "1364")
    {
        // Set depmethod to none for all manual, recording rules.
        DBUpdates updates {
            "UPDATE record SET dupmethod = 1 WHERE search = 5"
        };
        if (!performActualUpdate("MythTV", "DBSchemaVer",
                                 updates, "1365", dbver))
            return false;
    }

    if (dbver == "1365")
    {
        DBUpdates updates {
            "ALTER TABLE channelscan_channel ADD COLUMN service_type INT UNSIGNED NOT NULL DEFAULT 0;"
        };
        if (!performActualUpdate("MythTV", "DBSchemaVer",
                                 updates, "1366", dbver))
            return false;
    }

    if (dbver == "1366")
    {
        DBUpdates updates {
            "ALTER TABLE channelscan_dtv_multiplex ADD COLUMN signal_strength INT NOT NULL DEFAULT 0;"
        };
        if (!performActualUpdate("MythTV", "DBSchemaVer",
                                 updates, "1367", dbver))
            return false;
    }

    if (dbver == "1367")
    {
        DBUpdates updates {
            "ALTER TABLE videosource ADD COLUMN lcnoffset INT UNSIGNED DEFAULT 0;"
        };
        if (!performActualUpdate("MythTV", "DBSchemaVer",
                                 updates, "1368", dbver))
            return false;
    }
    if (dbver == "1368")
    {
        DBUpdates updates {
            // NOLINTNEXTLINE(bugprone-suspicious-missing-comma)
            "ALTER TABLE credits ADD COLUMN priority "
            "    TINYINT UNSIGNED DEFAULT 0;",
            "ALTER TABLE credits ADD COLUMN roleid "
            "    MEDIUMINT UNSIGNED DEFAULT 0;",
            "ALTER TABLE credits drop key chanid, "
            "     add unique key `chanid` "
            "          (chanid, starttime, person, role, roleid);"
            "ALTER TABLE recordedcredits ADD COLUMN priority "
            "    TINYINT UNSIGNED DEFAULT 0;",
            "ALTER TABLE recordedcredits ADD COLUMN roleid "
            "    MEDIUMINT UNSIGNED DEFAULT 0;",
            "ALTER TABLE recordedcredits drop key chanid, "
            "     add unique key `chanid` "
            "          (chanid, starttime, person, role, roleid);"
            "CREATE TABLE roles (roleid MEDIUMINT UNSIGNED NOT NULL AUTO_INCREMENT,"
            "  `name` varchar(128) CHARACTER SET utf8 COLLATE utf8_bin"
            "    NOT NULL DEFAULT '',"
            "  PRIMARY KEY (roleid),"
            "  UNIQUE KEY `name` (`name`)"
            ") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
        };
        if (!performActualUpdate("MythTV", "DBSchemaVer",
                                 updates, "1369", dbver))
            return false;
    }
    if (dbver == "1369")
    {
        DBUpdates updates {
            "ALTER TABLE programrating MODIFY COLUMN `system` "
            "    varchar(128);",
            "ALTER TABLE programrating MODIFY COLUMN rating "
            "    varchar(128);",
            "ALTER TABLE recordedrating MODIFY COLUMN `system` "
            "    varchar(128);",
            "ALTER TABLE recordedrating MODIFY COLUMN rating "
            "    varchar(128);",
        };
        if (!performActualUpdate("MythTV", "DBSchemaVer",
                                 updates, "1370", dbver))
            return false;
    }

    if (dbver == "1370")
    {
        // Migrate users from ttvdb.py to ttvdb4.py
        DBUpdates updates {
            "UPDATE settings SET data=REPLACE(data, 'ttvdb.py', 'ttvdb4.py') "
             "WHERE value='TelevisionGrabber'"
        };
        if (!performActualUpdate("MythTV", "DBSchemaVer",
                                 updates, "1371", dbver))
            return false;
    }

    if (dbver == "1371")
    {
        // Recording extender tables are now created later.

        if (!DBUtil::CheckTableColumnExists(QString("record"), QString("autoextend")))
        {
            // If that worked, modify existing tables.
            DBUpdates updates = getRecordingExtenderDbInfo(0);
            if (!performActualUpdate("MythTV", "DBSchemaVer",
                                     updates, "1372", dbver))
                return false;
        }
        else
        {
            if (!UpdateDBVersionNumber("MythTV", "DBSchemaVer", "1372", dbver))
                return false;
        }
    }

    if (dbver == "1372")
    {
        if (!DBUtil::CheckTableColumnExists(QString("recorded"), QString("lastplay")))
        {
            DBUpdates updates {
                "ALTER TABLE recorded ADD COLUMN lastplay "
                "    TINYINT UNSIGNED DEFAULT 0 AFTER bookmark;",
            };
            if (!performActualUpdate("MythTV", "DBSchemaVer",
                                     updates, "1373", dbver))
                return false;
        }
        else
        {
            if (!UpdateDBVersionNumber("MythTV", "DBSchemaVer", "1373", dbver))
                return false;
        }
    }

    if (dbver == "1373")
    {
        DBUpdates updates {
            // adjust inetref prefixes to new grabber script
            "UPDATE record SET inetref=REPLACE(inetref, 'ttvdb.py', 'ttvdb4.py');",
            "UPDATE recorded SET inetref=REPLACE(inetref, 'ttvdb.py', 'ttvdb4.py');",
            "UPDATE oldrecorded SET inetref=REPLACE(inetref, 'ttvdb.py', 'ttvdb4.py');",
            "UPDATE videometadata SET inetref=REPLACE(inetref, 'ttvdb.py', 'ttvdb4.py');",
            "UPDATE program SET inetref=REPLACE(inetref, 'ttvdb.py', 'ttvdb4.py');",
            "UPDATE recordedprogram SET inetref=REPLACE(inetref, 'ttvdb.py', 'ttvdb4.py');",
            "UPDATE recordedartwork SET inetref=REPLACE(inetref, 'ttvdb.py', 'ttvdb4.py');"
        };
        if (!performActualUpdate("MythTV", "DBSchemaVer",
                                 updates, "1374", dbver))
            return false;
    }

    if (dbver == "1374")
    {
        // Recording extender tables are now created later.
        DBUpdates updates {};
        if (!performActualUpdate("MythTV", "DBSchemaVer",
                                 updates, "1375", dbver))
            return false;
    }

    if (dbver == "1375")
    {
        // Delete any existing recording extender tables.
        DBUpdates updates = getRecordingExtenderDbInfo(-1);
        if (!performUpdateSeries("MythTV", updates))
            return false;

        // Now (re)create them.
        updates = getRecordingExtenderDbInfo(1);
        if (!performUpdateSeries("MythTV", updates))
            return false;

        // Add new tv listing name ->api name mappings for college
        // basketball.
        updates = getRecordingExtenderDbInfo(2);
        if (!performActualUpdate("MythTV", "DBSchemaVer",
                                 updates, "1376", dbver))
            return false;
    }

    if (dbver == "1376")
    {
        DBUpdates updates {
            "DROP TABLE IF EXISTS `logging`;",
            "UPDATE settings SET data='0' WHERE value='LogEnabled'; ",   // Keeps MythWeb happy
        };
        if (!performActualUpdate("MythTV", "DBSchemaVer",
                                 updates, "1377", dbver))
            return false;
    }

    if (dbver == "1377")
    {
        // Change reverted, but the version number can't be reused.
        DBUpdates updates {};
        if (!performActualUpdate("MythTV", "DBSchemaVer",
                                 updates, "1378", dbver))
            return false;
    }

    if (dbver == "1378")
    {
        DBUpdates updates {
            "CREATE INDEX dir_id ON gallery_files (dir_id);"
        };
        if (!performActualUpdate("MythTV", "DBSchemaVer",
                                 updates, "1379", dbver))
            return false;
    }

    if (dbver == "1379")
    {
        DBUpdates updates {
            "ALTER TABLE channelscan_channel ADD COLUMN logical_channel INT UNSIGNED NOT NULL DEFAULT 0 AFTER service_type;"
            "ALTER TABLE channelscan_channel ADD COLUMN simulcast_channel INT UNSIGNED NOT NULL DEFAULT 0 AFTER logical_channel;"
        };
        if (!performActualUpdate("MythTV", "DBSchemaVer",
                                 updates, "1380", dbver))
            return false;
    }

    if (dbver == "1380")
    {
        DBUpdates updates {
            "ALTER TABLE filemarkup ADD INDEX (type);"
        } ;
        if (!performActualUpdate("MythTV", "DBSchemaVer",
                                 updates, "1381", dbver))
            return false;
    }

    if (dbver == "1381")
    {
        DBUpdates updates {
            "ALTER TABLE iptv_channel ADD INDEX (chanid);"
            "ALTER TABLE channelgroup ADD INDEX (chanid);"
        } ;
        if (!performActualUpdate("MythTV", "DBSchemaVer",
                                 updates, "1382", dbver))
            return false;
    }

    if (dbver == "1382")
    {
        DBUpdates updates {
            "ALTER TABLE iptv_channel MODIFY iptvid INT UNSIGNED;"
        } ;
        if (!performActualUpdate("MythTV", "DBSchemaVer",
                                 updates, "1383", dbver))
            return false;
    }

    if (dbver == "1383")
    {
        DBUpdates updates {
            "ALTER TABLE iptv_channel MODIFY iptvid INT UNSIGNED AUTO_INCREMENT;"
        } ;
        if (!performActualUpdate("MythTV", "DBSchemaVer",
                                 updates, "1384", dbver))
            return false;
    }

    return true;
}

/**
 * command to get the the initial database layout from an empty database:
 *
 * mysqldump \
 *     --compact --skip-opt --create-options --no-tablespaces \
 *     --ignore-table=mythconverg.schemalock mythconverg | \
 *   sed -e '/^\(SET\|INS\).*;$/d;/^\/\*!40101.*$/d;/^\/\*M!999999.*$/d;' | \
 *   sed -e 's/utf8mb./utf8/' | \
 *   sed -e 's/^.*[^;]$/"&"/;s/^).*;$/"&",/'
 *
 * command to get the initial data:
 *
 * mysqldump \
 *     --compact --skip-opt --no-tablespaces -t mythconverg | \
 *   sed -e '/^\/\*M!999999.*$/d;s/^.*$/"&",/' -e 's#\\#\\\\#g'
 *
 * don't forget to delete host specific data
 *
 */
bool InitializeMythSchema(void)
{
    MSqlQuery query(MSqlQuery::InitCon());
    query.prepare("SHOW TABLES;");

    // check for > 1 table here since the schemalock table should exist
    if (query.exec() && query.isActive() && query.size() > 1)
    {
        QString msg = QString(
            "Told to create a NEW database schema, but the database\n"
            "already has %1 tables.\n"
            "If you are sure this is a good MythTV database, verify\n"
            "that the settings table has the DBSchemaVer variable.\n")
            .arg(query.size() - 1);
        LOG(VB_GENERAL, LOG_ERR, msg);
        return false;
    }

    LOG(VB_GENERAL, LOG_NOTICE,
        "Inserting MythTV initial database information.");

// Don't try to moderinze string literals in this section.  This lets
// us cut-n-paste the output of mysqldump into the code without
// generating any warning messages.
// NOLINTBEGIN(modernize-raw-string-literal)

    DBUpdates updates {
"CREATE TABLE `bdbookmark` ("
"  `serialid` varchar(40) NOT NULL DEFAULT '',"
"  `name` varchar(128) DEFAULT NULL,"
"  `bdstate` varchar(4096) NOT NULL DEFAULT '',"
"  `timestamp` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,"
"  PRIMARY KEY (`serialid`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `capturecard` ("
"  `cardid` int unsigned NOT NULL AUTO_INCREMENT,"
"  `parentid` int unsigned NOT NULL DEFAULT '0',"
"  `videodevice` varchar(128) DEFAULT NULL,"
"  `audiodevice` varchar(128) DEFAULT NULL,"
"  `vbidevice` varchar(128) DEFAULT NULL,"
"  `cardtype` varchar(32) DEFAULT 'V4L',"
"  `defaultinput` varchar(32) DEFAULT 'Television',"
"  `audioratelimit` int DEFAULT NULL,"
"  `hostname` varchar(64) DEFAULT NULL,"
"  `dvb_swfilter` int DEFAULT '0',"
"  `dvb_sat_type` int NOT NULL DEFAULT '0',"
"  `dvb_wait_for_seqstart` int NOT NULL DEFAULT '1',"
"  `skipbtaudio` tinyint(1) DEFAULT '0',"
"  `dvb_on_demand` tinyint NOT NULL DEFAULT '0',"
"  `dvb_diseqc_type` smallint DEFAULT NULL,"
"  `firewire_speed` int unsigned NOT NULL DEFAULT '0',"
"  `firewire_model` varchar(32) DEFAULT NULL,"
"  `firewire_connection` int unsigned NOT NULL DEFAULT '0',"
"  `signal_timeout` int NOT NULL DEFAULT '1000',"
"  `channel_timeout` int NOT NULL DEFAULT '3000',"
"  `dvb_tuning_delay` int unsigned NOT NULL DEFAULT '0',"
"  `contrast` int NOT NULL DEFAULT '0',"
"  `brightness` int NOT NULL DEFAULT '0',"
"  `colour` int NOT NULL DEFAULT '0',"
"  `hue` int NOT NULL DEFAULT '0',"
"  `diseqcid` int unsigned DEFAULT NULL,"
"  `dvb_eitscan` tinyint(1) NOT NULL DEFAULT '1',"
"  `inputname` varchar(32) NOT NULL DEFAULT 'None',"
"  `sourceid` int unsigned NOT NULL DEFAULT '0',"
"  `externalcommand` varchar(128) DEFAULT NULL,"
"  `changer_device` varchar(128) DEFAULT NULL,"
"  `changer_model` varchar(128) DEFAULT NULL,"
"  `tunechan` varchar(10) DEFAULT NULL,"
"  `startchan` varchar(10) DEFAULT NULL,"
"  `displayname` varchar(64) NOT NULL DEFAULT '',"
"  `dishnet_eit` tinyint(1) NOT NULL DEFAULT '0',"
"  `recpriority` int NOT NULL DEFAULT '0',"
"  `quicktune` tinyint NOT NULL DEFAULT '0',"
"  `schedorder` int unsigned NOT NULL DEFAULT '1',"
"  `livetvorder` int unsigned NOT NULL DEFAULT '1',"
"  `reclimit` int unsigned NOT NULL DEFAULT '1',"
"  `schedgroup` tinyint(1) NOT NULL DEFAULT '1',"
"  PRIMARY KEY (`cardid`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `cardinput` ("
"  `cardinputid` int unsigned NOT NULL AUTO_INCREMENT,"
"  `cardid` int unsigned NOT NULL DEFAULT '0',"
"  `sourceid` int unsigned NOT NULL DEFAULT '0',"
"  `inputname` varchar(32) NOT NULL DEFAULT '',"
"  `externalcommand` varchar(128) DEFAULT NULL,"
"  `changer_device` varchar(128) DEFAULT NULL,"
"  `changer_model` varchar(128) DEFAULT NULL,"
"  `tunechan` varchar(10) DEFAULT NULL,"
"  `startchan` varchar(10) DEFAULT NULL,"
"  `displayname` varchar(64) NOT NULL DEFAULT '',"
"  `dishnet_eit` tinyint(1) NOT NULL DEFAULT '0',"
"  `recpriority` int NOT NULL DEFAULT '0',"
"  `quicktune` tinyint NOT NULL DEFAULT '0',"
"  `schedorder` int unsigned NOT NULL DEFAULT '0',"
"  `livetvorder` int unsigned NOT NULL DEFAULT '0',"
"  PRIMARY KEY (`cardinputid`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `channel` ("
"  `chanid` int unsigned NOT NULL DEFAULT '0',"
"  `channum` varchar(10) NOT NULL DEFAULT '',"
"  `freqid` varchar(10) DEFAULT NULL,"
"  `sourceid` int unsigned DEFAULT NULL,"
"  `callsign` varchar(20) NOT NULL DEFAULT '',"
"  `name` varchar(64) NOT NULL DEFAULT '',"
"  `icon` varchar(255) NOT NULL DEFAULT '',"
"  `finetune` int DEFAULT NULL,"
"  `videofilters` varchar(255) NOT NULL DEFAULT '',"
"  `xmltvid` varchar(255) NOT NULL DEFAULT '',"
"  `recpriority` int NOT NULL DEFAULT '0',"
"  `contrast` int DEFAULT '32768',"
"  `brightness` int DEFAULT '32768',"
"  `colour` int DEFAULT '32768',"
"  `hue` int DEFAULT '32768',"
"  `tvformat` varchar(10) NOT NULL DEFAULT 'Default',"
"  `visible` tinyint(1) NOT NULL DEFAULT '1',"
"  `outputfilters` varchar(255) NOT NULL DEFAULT '',"
"  `useonairguide` tinyint(1) DEFAULT '0',"
"  `mplexid` smallint DEFAULT NULL,"
"  `serviceid` mediumint unsigned DEFAULT NULL,"
"  `service_type` int unsigned DEFAULT '0',"
"  `tmoffset` int NOT NULL DEFAULT '0',"
"  `atsc_major_chan` int unsigned NOT NULL DEFAULT '0',"
"  `atsc_minor_chan` int unsigned NOT NULL DEFAULT '0',"
"  `last_record` datetime NOT NULL,"
"  `default_authority` varchar(32) NOT NULL DEFAULT '',"
"  `commmethod` int NOT NULL DEFAULT '-1',"
"  `iptvid` smallint unsigned DEFAULT NULL,"
"  `deleted` timestamp NULL DEFAULT NULL,"
"  PRIMARY KEY (`chanid`),"
"  KEY `channel_src` (`channum`,`sourceid`),"
"  KEY `sourceid` (`sourceid`,`xmltvid`,`chanid`),"
"  KEY `visible` (`visible`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `channelgroup` ("
"  `id` int unsigned NOT NULL AUTO_INCREMENT,"
"  `chanid` int unsigned NOT NULL DEFAULT '0',"
"  `grpid` int NOT NULL DEFAULT '1',"
"  PRIMARY KEY (`id`),"
"  KEY `chanid` (`chanid`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `channelgroupnames` ("
"  `grpid` int unsigned NOT NULL AUTO_INCREMENT,"
"  `name` varchar(64) NOT NULL DEFAULT '0',"
"  PRIMARY KEY (`grpid`)"
") ENGINE=MyISAM AUTO_INCREMENT=2 DEFAULT CHARSET=utf8;",
"CREATE TABLE `channelscan` ("
"  `scanid` int unsigned NOT NULL AUTO_INCREMENT,"
"  `cardid` int unsigned NOT NULL,"
"  `sourceid` int unsigned NOT NULL,"
"  `processed` tinyint unsigned NOT NULL,"
"  `scandate` datetime NOT NULL,"
"  PRIMARY KEY (`scanid`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `channelscan_channel` ("
"  `transportid` int unsigned NOT NULL,"
"  `scanid` int unsigned NOT NULL,"
"  `mplex_id` smallint NOT NULL,"
"  `source_id` int unsigned NOT NULL,"
"  `channel_id` int unsigned NOT NULL DEFAULT '0',"
"  `callsign` varchar(20) NOT NULL DEFAULT '',"
"  `service_name` varchar(64) NOT NULL DEFAULT '',"
"  `chan_num` varchar(10) NOT NULL DEFAULT '',"
"  `service_id` mediumint unsigned NOT NULL DEFAULT '0',"
"  `atsc_major_channel` int unsigned NOT NULL DEFAULT '0',"
"  `atsc_minor_channel` int unsigned NOT NULL DEFAULT '0',"
"  `use_on_air_guide` tinyint(1) NOT NULL DEFAULT '0',"
"  `hidden` tinyint(1) NOT NULL DEFAULT '0',"
"  `hidden_in_guide` tinyint(1) NOT NULL DEFAULT '0',"
"  `freqid` varchar(10) NOT NULL DEFAULT '',"
"  `icon` varchar(255) NOT NULL DEFAULT '',"
"  `tvformat` varchar(10) NOT NULL DEFAULT 'Default',"
"  `xmltvid` varchar(64) NOT NULL DEFAULT '',"
"  `pat_tsid` int unsigned NOT NULL DEFAULT '0',"
"  `vct_tsid` int unsigned NOT NULL DEFAULT '0',"
"  `vct_chan_tsid` int unsigned NOT NULL DEFAULT '0',"
"  `sdt_tsid` int unsigned NOT NULL DEFAULT '0',"
"  `orig_netid` int unsigned NOT NULL DEFAULT '0',"
"  `netid` int unsigned NOT NULL DEFAULT '0',"
"  `si_standard` varchar(10) NOT NULL,"
"  `in_channels_conf` tinyint unsigned NOT NULL DEFAULT '0',"
"  `in_pat` tinyint unsigned NOT NULL DEFAULT '0',"
"  `in_pmt` tinyint unsigned NOT NULL DEFAULT '0',"
"  `in_vct` tinyint unsigned NOT NULL DEFAULT '0',"
"  `in_nit` tinyint unsigned NOT NULL DEFAULT '0',"
"  `in_sdt` tinyint unsigned NOT NULL DEFAULT '0',"
"  `is_encrypted` tinyint unsigned NOT NULL DEFAULT '0',"
"  `is_data_service` tinyint unsigned NOT NULL DEFAULT '0',"
"  `is_audio_service` tinyint unsigned NOT NULL DEFAULT '0',"
"  `is_opencable` tinyint unsigned NOT NULL DEFAULT '0',"
"  `could_be_opencable` tinyint unsigned NOT NULL DEFAULT '0',"
"  `decryption_status` smallint unsigned NOT NULL DEFAULT '0',"
"  `default_authority` varchar(32) NOT NULL DEFAULT '',"
"  `service_type` int unsigned NOT NULL DEFAULT '0',"
"  `logical_channel` int unsigned NOT NULL DEFAULT '0',"
"  `simulcast_channel` int unsigned NOT NULL DEFAULT '0'"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `channelscan_dtv_multiplex` ("
"  `transportid` int unsigned NOT NULL AUTO_INCREMENT,"
"  `scanid` int unsigned NOT NULL,"
"  `mplexid` smallint unsigned NOT NULL,"
"  `frequency` bigint unsigned NOT NULL,"
"  `inversion` char(1) NOT NULL DEFAULT 'a',"
"  `symbolrate` bigint unsigned NOT NULL DEFAULT '0',"
"  `fec` varchar(10) NOT NULL DEFAULT 'auto',"
"  `polarity` char(1) NOT NULL DEFAULT '',"
"  `hp_code_rate` varchar(10) NOT NULL DEFAULT 'auto',"
"  `mod_sys` varchar(10) DEFAULT NULL,"
"  `rolloff` varchar(4) DEFAULT NULL,"
"  `lp_code_rate` varchar(10) NOT NULL DEFAULT 'auto',"
"  `modulation` varchar(10) NOT NULL DEFAULT 'auto',"
"  `transmission_mode` char(1) NOT NULL DEFAULT 'a',"
"  `guard_interval` varchar(10) NOT NULL DEFAULT 'auto',"
"  `hierarchy` varchar(10) NOT NULL DEFAULT 'auto',"
"  `bandwidth` char(1) NOT NULL DEFAULT 'a',"
"  `sistandard` varchar(10) NOT NULL,"
"  `tuner_type` smallint unsigned NOT NULL,"
"  `default_authority` varchar(32) NOT NULL DEFAULT '',"
"  `signal_strength` int NOT NULL DEFAULT '0',"
"  PRIMARY KEY (`transportid`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `codecparams` ("
"  `profile` int unsigned NOT NULL DEFAULT '0',"
"  `name` varchar(128) NOT NULL DEFAULT '',"
"  `value` varchar(128) DEFAULT NULL,"
"  PRIMARY KEY (`profile`,`name`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `credits` ("
"  `person` mediumint unsigned NOT NULL DEFAULT '0',"
"  `chanid` int unsigned NOT NULL DEFAULT '0',"
"  `starttime` datetime NOT NULL DEFAULT '0000-00-00 00:00:00',"
"  `role` set('actor','director','producer','executive_producer','writer','guest_star','host','adapter','presenter','commentator','guest') NOT NULL DEFAULT '',"
"  `priority` tinyint unsigned DEFAULT '0',"
"  `roleid` mediumint unsigned DEFAULT '0',"
"  UNIQUE KEY `chanid` (`chanid`,`starttime`,`person`,`role`,`roleid`),"
"  KEY `person` (`person`,`role`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `customexample` ("
"  `rulename` varchar(64) NOT NULL,"
"  `fromclause` varchar(10000) NOT NULL DEFAULT '',"
"  `whereclause` varchar(10000) NOT NULL DEFAULT '',"
"  `search` tinyint NOT NULL DEFAULT '0',"
"  PRIMARY KEY (`rulename`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `diseqc_config` ("
"  `cardinputid` int unsigned NOT NULL,"
"  `diseqcid` int unsigned NOT NULL,"
"  `value` varchar(16) NOT NULL DEFAULT '',"
"  KEY `id` (`cardinputid`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `diseqc_tree` ("
"  `diseqcid` int unsigned NOT NULL AUTO_INCREMENT,"
"  `parentid` int unsigned DEFAULT NULL,"
"  `ordinal` tinyint unsigned NOT NULL,"
"  `type` varchar(16) NOT NULL DEFAULT '',"
"  `subtype` varchar(16) NOT NULL DEFAULT '',"
"  `description` varchar(32) NOT NULL DEFAULT '',"
"  `switch_ports` tinyint unsigned NOT NULL DEFAULT '0',"
"  `rotor_hi_speed` float NOT NULL DEFAULT '0',"
"  `rotor_lo_speed` float NOT NULL DEFAULT '0',"
"  `rotor_positions` varchar(255) NOT NULL DEFAULT '',"
"  `lnb_lof_switch` int NOT NULL DEFAULT '0',"
"  `lnb_lof_hi` int NOT NULL DEFAULT '0',"
"  `lnb_lof_lo` int NOT NULL DEFAULT '0',"
"  `cmd_repeat` int NOT NULL DEFAULT '1',"
"  `lnb_pol_inv` tinyint NOT NULL DEFAULT '0',"
"  `address` tinyint unsigned NOT NULL DEFAULT '0',"
"  `scr_userband` int unsigned NOT NULL DEFAULT '0',"
"  `scr_frequency` int unsigned NOT NULL DEFAULT '1400',"
"  `scr_pin` int NOT NULL DEFAULT '-1',"
"  PRIMARY KEY (`diseqcid`),"
"  KEY `parentid` (`parentid`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `displayprofilegroups` ("
"  `name` varchar(128) NOT NULL,"
"  `hostname` varchar(64) NOT NULL,"
"  `profilegroupid` int unsigned NOT NULL AUTO_INCREMENT,"
"  PRIMARY KEY (`name`,`hostname`),"
"  UNIQUE KEY `profilegroupid` (`profilegroupid`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `displayprofiles` ("
"  `profilegroupid` int unsigned NOT NULL,"
"  `profileid` int unsigned NOT NULL AUTO_INCREMENT,"
"  `value` varchar(128) NOT NULL,"
"  `data` varchar(255) NOT NULL DEFAULT '',"
"  KEY `profilegroupid` (`profilegroupid`),"
"  KEY `profileid` (`profileid`,`value`),"
"  KEY `profileid_2` (`profileid`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `dtv_multiplex` ("
"  `mplexid` smallint NOT NULL AUTO_INCREMENT,"
"  `sourceid` smallint DEFAULT NULL,"
"  `transportid` int DEFAULT NULL,"
"  `networkid` int DEFAULT NULL,"
"  `frequency` int DEFAULT NULL,"
"  `inversion` char(1) DEFAULT 'a',"
"  `symbolrate` int DEFAULT NULL,"
"  `fec` varchar(10) DEFAULT 'auto',"
"  `polarity` char(1) DEFAULT NULL,"
"  `modulation` varchar(10) DEFAULT 'auto',"
"  `bandwidth` char(1) DEFAULT 'a',"
"  `lp_code_rate` varchar(10) DEFAULT 'auto',"
"  `transmission_mode` char(1) DEFAULT 'a',"
"  `guard_interval` varchar(10) DEFAULT 'auto',"
"  `visible` smallint NOT NULL DEFAULT '0',"
"  `constellation` varchar(10) DEFAULT 'auto',"
"  `hierarchy` varchar(10) DEFAULT 'auto',"
"  `hp_code_rate` varchar(10) DEFAULT 'auto',"
"  `mod_sys` varchar(10) DEFAULT NULL,"
"  `rolloff` varchar(4) DEFAULT NULL,"
"  `sistandard` varchar(10) DEFAULT 'dvb',"
"  `serviceversion` smallint DEFAULT '33',"
"  `updatetimestamp` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,"
"  `default_authority` varchar(32) NOT NULL DEFAULT '',"
"  PRIMARY KEY (`mplexid`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `dtv_privatetypes` ("
"  `sitype` varchar(4) NOT NULL DEFAULT '',"
"  `networkid` int NOT NULL DEFAULT '0',"
"  `private_type` varchar(20) NOT NULL DEFAULT '',"
"  `private_value` varchar(100) NOT NULL DEFAULT ''"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `dvdbookmark` ("
"  `serialid` varchar(16) NOT NULL DEFAULT '',"
"  `name` varchar(32) DEFAULT NULL,"
"  `title` smallint NOT NULL DEFAULT '0',"
"  `audionum` tinyint NOT NULL DEFAULT '-1',"
"  `subtitlenum` tinyint NOT NULL DEFAULT '-1',"
"  `framenum` bigint NOT NULL DEFAULT '0',"
"  `timestamp` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,"
"  `dvdstate` varchar(1024) NOT NULL DEFAULT '',"
"  PRIMARY KEY (`serialid`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `dvdinput` ("
"  `intid` int unsigned NOT NULL,"
"  `hsize` int unsigned DEFAULT NULL,"
"  `vsize` int unsigned DEFAULT NULL,"
"  `ar_num` int unsigned DEFAULT NULL,"
"  `ar_denom` int unsigned DEFAULT NULL,"
"  `fr_code` int unsigned DEFAULT NULL,"
"  `letterbox` tinyint(1) DEFAULT NULL,"
"  `v_format` varchar(16) DEFAULT NULL,"
"  PRIMARY KEY (`intid`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `dvdtranscode` ("
"  `intid` int NOT NULL AUTO_INCREMENT,"
"  `input` int unsigned DEFAULT NULL,"
"  `name` varchar(128) NOT NULL,"
"  `sync_mode` int unsigned DEFAULT NULL,"
"  `use_yv12` tinyint(1) DEFAULT NULL,"
"  `cliptop` int DEFAULT NULL,"
"  `clipbottom` int DEFAULT NULL,"
"  `clipleft` int DEFAULT NULL,"
"  `clipright` int DEFAULT NULL,"
"  `f_resize_h` int DEFAULT NULL,"
"  `f_resize_w` int DEFAULT NULL,"
"  `hq_resize_h` int DEFAULT NULL,"
"  `hq_resize_w` int DEFAULT NULL,"
"  `grow_h` int DEFAULT NULL,"
"  `grow_w` int DEFAULT NULL,"
"  `clip2top` int DEFAULT NULL,"
"  `clip2bottom` int DEFAULT NULL,"
"  `clip2left` int DEFAULT NULL,"
"  `clip2right` int DEFAULT NULL,"
"  `codec` varchar(128) NOT NULL,"
"  `codec_param` varchar(128) DEFAULT NULL,"
"  `bitrate` int DEFAULT NULL,"
"  `a_sample_r` int DEFAULT NULL,"
"  `a_bitrate` int DEFAULT NULL,"
"  `two_pass` tinyint(1) DEFAULT NULL,"
"  `tc_param` varchar(128) DEFAULT NULL,"
"  PRIMARY KEY (`intid`)"
") ENGINE=MyISAM AUTO_INCREMENT=12 DEFAULT CHARSET=utf8;",
"CREATE TABLE `eit_cache` ("
"  `chanid` int NOT NULL,"
"  `eventid` int unsigned NOT NULL DEFAULT '0',"
"  `tableid` tinyint unsigned NOT NULL,"
"  `version` tinyint unsigned NOT NULL,"
"  `endtime` int unsigned NOT NULL,"
"  `status` tinyint NOT NULL DEFAULT '0',"
"  PRIMARY KEY (`chanid`,`eventid`,`status`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `filemarkup` ("
"  `filename` text NOT NULL,"
"  `mark` mediumint unsigned NOT NULL DEFAULT '0',"
"  `offset` bigint unsigned DEFAULT NULL,"
"  `type` tinyint NOT NULL DEFAULT '0',"
"  KEY `filename` (`filename`(255)),"
"  KEY `type` (`type`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `gallery_directories` ("
"  `dir_id` int NOT NULL AUTO_INCREMENT,"
"  `filename` varchar(255) NOT NULL,"
"  `name` varchar(255) NOT NULL,"
"  `path` varchar(255) NOT NULL,"
"  `parent_id` int NOT NULL,"
"  `dir_count` int NOT NULL DEFAULT '0',"
"  `file_count` int NOT NULL DEFAULT '0',"
"  `hidden` tinyint(1) NOT NULL DEFAULT '0',"
"  PRIMARY KEY (`dir_id`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `gallery_files` ("
"  `file_id` int NOT NULL AUTO_INCREMENT,"
"  `filename` varchar(255) NOT NULL,"
"  `name` varchar(255) NOT NULL,"
"  `path` varchar(255) NOT NULL,"
"  `dir_id` int NOT NULL DEFAULT '0',"
"  `type` int NOT NULL DEFAULT '0',"
"  `modtime` int NOT NULL DEFAULT '0',"
"  `size` int NOT NULL DEFAULT '0',"
"  `extension` varchar(255) NOT NULL,"
"  `angle` int NOT NULL DEFAULT '0',"
"  `date` int NOT NULL DEFAULT '0',"
"  `zoom` int NOT NULL DEFAULT '0',"
"  `hidden` tinyint(1) NOT NULL DEFAULT '0',"
"  `orientation` int NOT NULL DEFAULT '0',"
"  PRIMARY KEY (`file_id`),"
"  KEY `dir_id` (`dir_id`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `housekeeping` ("
"  `tag` varchar(64) NOT NULL,"
"  `hostname` varchar(64) DEFAULT NULL,"
"  `lastrun` datetime DEFAULT NULL,"
"  `lastsuccess` datetime DEFAULT NULL,"
"  UNIQUE KEY `task` (`tag`,`hostname`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `inputgroup` ("
"  `cardinputid` int unsigned NOT NULL,"
"  `inputgroupid` int unsigned NOT NULL,"
"  `inputgroupname` varchar(128) DEFAULT NULL"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `internetcontent` ("
"  `name` varchar(255) NOT NULL,"
"  `thumbnail` varchar(255) DEFAULT NULL,"
"  `type` smallint NOT NULL,"
"  `author` varchar(128) NOT NULL,"
"  `description` text NOT NULL,"
"  `commandline` text NOT NULL,"
"  `version` double NOT NULL,"
"  `updated` datetime NOT NULL DEFAULT '0000-00-00 00:00:00',"
"  `search` tinyint(1) NOT NULL,"
"  `tree` tinyint(1) NOT NULL,"
"  `podcast` tinyint(1) NOT NULL,"
"  `download` tinyint(1) NOT NULL,"
"  `host` varchar(128) DEFAULT NULL"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `internetcontentarticles` ("
"  `feedtitle` varchar(255) NOT NULL,"
"  `path` text NOT NULL,"
"  `paththumb` text NOT NULL,"
"  `title` varchar(255) NOT NULL,"
"  `subtitle` varchar(255) NOT NULL,"
"  `season` smallint NOT NULL DEFAULT '0',"
"  `episode` smallint NOT NULL DEFAULT '0',"
"  `description` text NOT NULL,"
"  `url` text NOT NULL,"
"  `type` smallint NOT NULL,"
"  `thumbnail` text NOT NULL,"
"  `mediaURL` text NOT NULL,"
"  `author` varchar(255) NOT NULL,"
"  `date` datetime NOT NULL DEFAULT '0000-00-00 00:00:00',"
"  `time` int NOT NULL,"
"  `rating` varchar(255) NOT NULL,"
"  `filesize` bigint NOT NULL,"
"  `player` varchar(255) NOT NULL,"
"  `playerargs` text NOT NULL,"
"  `download` varchar(255) NOT NULL,"
"  `downloadargs` text NOT NULL,"
"  `width` smallint NOT NULL,"
"  `height` smallint NOT NULL,"
"  `language` varchar(128) NOT NULL,"
"  `podcast` tinyint(1) NOT NULL,"
"  `downloadable` tinyint(1) NOT NULL,"
"  `customhtml` tinyint(1) NOT NULL,"
"  `countries` varchar(255) NOT NULL"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `inuseprograms` ("
"  `chanid` int unsigned NOT NULL DEFAULT '0',"
"  `starttime` datetime NOT NULL DEFAULT '0000-00-00 00:00:00',"
"  `recusage` varchar(128) NOT NULL DEFAULT '',"
"  `lastupdatetime` datetime NOT NULL DEFAULT '0000-00-00 00:00:00',"
"  `hostname` varchar(64) NOT NULL DEFAULT '',"
"  `rechost` varchar(64) NOT NULL,"
"  `recdir` varchar(255) NOT NULL DEFAULT '',"
"  KEY `chanid` (`chanid`,`starttime`),"
"  KEY `recusage` (`recusage`,`lastupdatetime`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `iptv_channel` ("
"  `iptvid` smallint unsigned NOT NULL AUTO_INCREMENT,"
"  `chanid` int unsigned NOT NULL,"
"  `url` text NOT NULL,"
"  `type` set('data','rfc2733-1','rfc2733-2','rfc5109-1','rfc5109-2','smpte2022-1','smpte2022-2') DEFAULT NULL,"
"  `bitrate` int unsigned NOT NULL,"
"  PRIMARY KEY (`iptvid`),"
"  KEY `chanid` (`chanid`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `jobqueue` ("
"  `id` int NOT NULL AUTO_INCREMENT,"
"  `chanid` int NOT NULL DEFAULT '0',"
"  `starttime` datetime NOT NULL DEFAULT '0000-00-00 00:00:00',"
"  `inserttime` datetime NOT NULL DEFAULT '0000-00-00 00:00:00',"
"  `type` int NOT NULL DEFAULT '0',"
"  `cmds` int NOT NULL DEFAULT '0',"
"  `flags` int NOT NULL DEFAULT '0',"
"  `status` int NOT NULL DEFAULT '0',"
"  `statustime` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,"
"  `hostname` varchar(64) NOT NULL DEFAULT '',"
"  `args` blob NOT NULL,"
"  `comment` varchar(128) NOT NULL DEFAULT '',"
"  `schedruntime` datetime NOT NULL DEFAULT '2007-01-01 00:00:00',"
"  PRIMARY KEY (`id`),"
"  UNIQUE KEY `chanid` (`chanid`,`starttime`,`type`,`inserttime`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `jumppoints` ("
"  `destination` varchar(128) NOT NULL DEFAULT '',"
"  `description` varchar(255) DEFAULT NULL,"
"  `keylist` varchar(128) DEFAULT NULL,"
"  `hostname` varchar(64) NOT NULL DEFAULT '',"
"  PRIMARY KEY (`destination`,`hostname`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `keybindings` ("
"  `context` varchar(32) NOT NULL DEFAULT '',"
"  `action` varchar(32) NOT NULL DEFAULT '',"
"  `description` varchar(255) DEFAULT NULL,"
"  `keylist` varchar(128) DEFAULT NULL,"
"  `hostname` varchar(64) NOT NULL DEFAULT '',"
"  PRIMARY KEY (`context`,`action`,`hostname`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `keyword` ("
"  `phrase` varchar(128) CHARACTER SET utf8 COLLATE utf8_bin NOT NULL DEFAULT '',"
"  `searchtype` int unsigned NOT NULL DEFAULT '3',"
"  UNIQUE KEY `phrase` (`phrase`,`searchtype`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `livestream` ("
"  `id` int unsigned NOT NULL AUTO_INCREMENT,"
"  `width` int unsigned NOT NULL,"
"  `height` int unsigned NOT NULL,"
"  `bitrate` int unsigned NOT NULL,"
"  `audiobitrate` int unsigned NOT NULL,"
"  `samplerate` int unsigned NOT NULL,"
"  `audioonlybitrate` int unsigned NOT NULL,"
"  `segmentsize` int unsigned NOT NULL DEFAULT '10',"
"  `maxsegments` int unsigned NOT NULL DEFAULT '0',"
"  `startsegment` int unsigned NOT NULL DEFAULT '0',"
"  `currentsegment` int unsigned NOT NULL DEFAULT '0',"
"  `segmentcount` int unsigned NOT NULL DEFAULT '0',"
"  `percentcomplete` int unsigned NOT NULL DEFAULT '0',"
"  `created` datetime NOT NULL,"
"  `lastmodified` datetime NOT NULL,"
"  `relativeurl` varchar(512) NOT NULL,"
"  `fullurl` varchar(1024) NOT NULL,"
"  `status` int unsigned NOT NULL DEFAULT '0',"
"  `statusmessage` varchar(256) NOT NULL,"
"  `sourcefile` varchar(512) NOT NULL,"
"  `sourcehost` varchar(64) NOT NULL,"
"  `sourcewidth` int unsigned NOT NULL DEFAULT '0',"
"  `sourceheight` int unsigned NOT NULL DEFAULT '0',"
"  `outdir` varchar(256) NOT NULL,"
"  `outbase` varchar(128) NOT NULL,"
"  PRIMARY KEY (`id`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `mythlog` ("
"  `logid` int unsigned NOT NULL AUTO_INCREMENT,"
"  `module` varchar(32) NOT NULL DEFAULT '',"
"  `priority` int NOT NULL DEFAULT '0',"
"  `acknowledged` tinyint(1) DEFAULT '0',"
"  `logdate` datetime DEFAULT NULL,"
"  `host` varchar(128) DEFAULT NULL,"
"  `message` varchar(255) NOT NULL DEFAULT '',"
"  `details` varchar(16000) NOT NULL DEFAULT '',"
"  PRIMARY KEY (`logid`),"
"  KEY `module` (`module`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `oldfind` ("
"  `recordid` int NOT NULL DEFAULT '0',"
"  `findid` int NOT NULL DEFAULT '0',"
"  PRIMARY KEY (`recordid`,`findid`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `oldprogram` ("
"  `oldtitle` varchar(128) CHARACTER SET utf8 COLLATE utf8_bin NOT NULL DEFAULT '',"
"  `airdate` datetime NOT NULL DEFAULT '0000-00-00 00:00:00',"
"  PRIMARY KEY (`oldtitle`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `oldrecorded` ("
"  `chanid` int unsigned NOT NULL DEFAULT '0',"
"  `starttime` datetime NOT NULL DEFAULT '0000-00-00 00:00:00',"
"  `endtime` datetime NOT NULL DEFAULT '0000-00-00 00:00:00',"
"  `title` varchar(128) NOT NULL DEFAULT '',"
"  `subtitle` varchar(128) NOT NULL DEFAULT '',"
"  `description` varchar(16000) NOT NULL DEFAULT '',"
"  `season` smallint NOT NULL,"
"  `episode` smallint NOT NULL,"
"  `category` varchar(64) NOT NULL DEFAULT '',"
"  `seriesid` varchar(64) DEFAULT NULL,"
"  `programid` varchar(64) DEFAULT NULL,"
"  `inetref` varchar(40) NOT NULL,"
"  `findid` int NOT NULL DEFAULT '0',"
"  `recordid` int NOT NULL DEFAULT '0',"
"  `station` varchar(20) NOT NULL DEFAULT '',"
"  `rectype` int unsigned NOT NULL DEFAULT '0',"
"  `duplicate` tinyint(1) NOT NULL DEFAULT '0',"
"  `recstatus` int NOT NULL DEFAULT '0',"
"  `reactivate` smallint NOT NULL DEFAULT '0',"
"  `generic` tinyint(1) NOT NULL,"
"  `future` tinyint(1) NOT NULL DEFAULT '0',"
"  PRIMARY KEY (`station`,`starttime`,`title`),"
"  KEY `endtime` (`endtime`),"
"  KEY `title` (`title`),"
"  KEY `seriesid` (`seriesid`),"
"  KEY `programid` (`programid`),"
"  KEY `recordid` (`recordid`),"
"  KEY `recstatus` (`recstatus`,`programid`,`seriesid`),"
"  KEY `recstatus_2` (`recstatus`,`title`,`subtitle`),"
"  KEY `future` (`future`),"
"  KEY `chanid` (`chanid`,`starttime`),"
"  KEY `subtitle` (`subtitle`),"
"  KEY `description` (`description`(255))"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `people` ("
"  `person` mediumint unsigned NOT NULL AUTO_INCREMENT,"
"  `name` varchar(128) CHARACTER SET utf8 COLLATE utf8_bin NOT NULL DEFAULT '',"
"  PRIMARY KEY (`person`),"
"  UNIQUE KEY `name` (`name`(41))"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `pidcache` ("
"  `chanid` smallint NOT NULL DEFAULT '0',"
"  `pid` int NOT NULL DEFAULT '-1',"
"  `tableid` int NOT NULL DEFAULT '-1',"
"  KEY `chanid` (`chanid`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `playgroup` ("
"  `name` varchar(32) NOT NULL DEFAULT '',"
"  `titlematch` varchar(255) NOT NULL DEFAULT '',"
"  `skipahead` int NOT NULL DEFAULT '0',"
"  `skipback` int NOT NULL DEFAULT '0',"
"  `timestretch` int NOT NULL DEFAULT '0',"
"  `jump` int NOT NULL DEFAULT '0',"
"  PRIMARY KEY (`name`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `powerpriority` ("
"  `priorityname` varchar(64) CHARACTER SET utf8 COLLATE utf8_bin NOT NULL,"
"  `recpriority` int NOT NULL DEFAULT '0',"
"  `selectclause` varchar(16000) NOT NULL DEFAULT '',"
"  PRIMARY KEY (`priorityname`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `profilegroups` ("
"  `id` int unsigned NOT NULL AUTO_INCREMENT,"
"  `name` varchar(128) DEFAULT NULL,"
"  `cardtype` varchar(32) NOT NULL DEFAULT 'V4L',"
"  `is_default` int DEFAULT '0',"
"  `hostname` varchar(64) DEFAULT NULL,"
"  PRIMARY KEY (`id`),"
"  UNIQUE KEY `name` (`name`,`hostname`),"
"  KEY `cardtype` (`cardtype`)"
") ENGINE=MyISAM AUTO_INCREMENT=20 DEFAULT CHARSET=utf8;",
"CREATE TABLE `program` ("
"  `chanid` int unsigned NOT NULL DEFAULT '0',"
"  `starttime` datetime NOT NULL DEFAULT '0000-00-00 00:00:00',"
"  `endtime` datetime NOT NULL DEFAULT '0000-00-00 00:00:00',"
"  `title` varchar(128) NOT NULL DEFAULT '',"
"  `subtitle` varchar(128) NOT NULL DEFAULT '',"
"  `description` varchar(16000) NOT NULL DEFAULT '',"
"  `category` varchar(64) NOT NULL DEFAULT '',"
"  `category_type` varchar(64) NOT NULL DEFAULT '',"
"  `airdate` year NOT NULL DEFAULT '0000',"
"  `stars` float NOT NULL DEFAULT '0',"
"  `previouslyshown` tinyint NOT NULL DEFAULT '0',"
"  `title_pronounce` varchar(128) NOT NULL DEFAULT '',"
"  `stereo` tinyint(1) NOT NULL DEFAULT '0',"
"  `subtitled` tinyint(1) NOT NULL DEFAULT '0',"
"  `hdtv` tinyint(1) NOT NULL DEFAULT '0',"
"  `closecaptioned` tinyint(1) NOT NULL DEFAULT '0',"
"  `partnumber` int NOT NULL DEFAULT '0',"
"  `parttotal` int NOT NULL DEFAULT '0',"
"  `seriesid` varchar(64) NOT NULL DEFAULT '',"
"  `originalairdate` date DEFAULT NULL,"
"  `showtype` varchar(30) NOT NULL DEFAULT '',"
"  `colorcode` varchar(20) NOT NULL DEFAULT '',"
"  `syndicatedepisodenumber` varchar(20) NOT NULL DEFAULT '',"
"  `programid` varchar(64) NOT NULL DEFAULT '',"
"  `manualid` int unsigned NOT NULL DEFAULT '0',"
"  `generic` tinyint(1) DEFAULT '0',"
"  `listingsource` int NOT NULL DEFAULT '0',"
"  `first` tinyint(1) NOT NULL DEFAULT '0',"
"  `last` tinyint(1) NOT NULL DEFAULT '0',"
"  `audioprop` set('STEREO','MONO','SURROUND','DOLBY','HARDHEAR','VISUALIMPAIR') NOT NULL,"
"  `subtitletypes` set('HARDHEAR','NORMAL','ONSCREEN','SIGNED') NOT NULL,"
"  `videoprop` set('WIDESCREEN','HDTV','MPEG2','AVC','HEVC') NOT NULL,"
"  `inetref` varchar(40) DEFAULT '',"
"  `season` int NOT NULL DEFAULT '0',"
"  `episode` int NOT NULL DEFAULT '0',"
"  `totalepisodes` int NOT NULL DEFAULT '0',"
"  PRIMARY KEY (`chanid`,`starttime`,`manualid`),"
"  KEY `endtime` (`endtime`),"
"  KEY `title_pronounce` (`title_pronounce`),"
"  KEY `seriesid` (`seriesid`),"
"  KEY `id_start_end` (`chanid`,`starttime`,`endtime`),"
"  KEY `program_manualid` (`manualid`),"
"  KEY `previouslyshown` (`previouslyshown`),"
"  KEY `programid` (`programid`,`starttime`),"
"  KEY `starttime` (`starttime`),"
"  KEY `subtitle` (`subtitle`),"
"  KEY `description` (`description`(255)),"
"  KEY `title_subtitle_start` (`title`,`subtitle`,`starttime`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `programgenres` ("
"  `chanid` int unsigned NOT NULL DEFAULT '0',"
"  `starttime` datetime NOT NULL DEFAULT '0000-00-00 00:00:00',"
"  `relevance` char(1) NOT NULL DEFAULT '',"
"  `genre` varchar(30) DEFAULT NULL,"
"  PRIMARY KEY (`chanid`,`starttime`,`relevance`),"
"  KEY `genre` (`genre`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `programrating` ("
"  `chanid` int unsigned NOT NULL DEFAULT '0',"
"  `starttime` datetime NOT NULL DEFAULT '0000-00-00 00:00:00',"
"  `system` varchar(128) DEFAULT NULL,"
"  `rating` varchar(128) DEFAULT NULL,"
"  UNIQUE KEY `chanid` (`chanid`,`starttime`,`system`,`rating`),"
"  KEY `starttime` (`starttime`,`system`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `recgrouppassword` ("
"  `recgroup` varchar(32) CHARACTER SET utf8 COLLATE utf8_bin NOT NULL DEFAULT '',"
"  `password` varchar(10) NOT NULL DEFAULT '',"
"  PRIMARY KEY (`recgroup`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `recgroups` ("
"  `recgroupid` smallint NOT NULL AUTO_INCREMENT,"
"  `recgroup` varchar(64) NOT NULL DEFAULT '',"
"  `displayname` varchar(64) NOT NULL DEFAULT '',"
"  `password` varchar(40) NOT NULL DEFAULT '',"
"  `special` tinyint(1) NOT NULL DEFAULT '0',"
"  PRIMARY KEY (`recgroupid`),"
"  UNIQUE KEY `recgroup` (`recgroup`)"
") ENGINE=MyISAM AUTO_INCREMENT=4 DEFAULT CHARSET=utf8;",
"CREATE TABLE `record` ("
"  `recordid` int unsigned NOT NULL AUTO_INCREMENT,"
"  `type` int unsigned NOT NULL DEFAULT '0',"
"  `chanid` int unsigned DEFAULT NULL,"
"  `starttime` time DEFAULT NULL,"
"  `startdate` date DEFAULT NULL,"
"  `endtime` time DEFAULT NULL,"
"  `enddate` date DEFAULT NULL,"
"  `title` varchar(128) NOT NULL DEFAULT '',"
"  `subtitle` varchar(128) NOT NULL DEFAULT '',"
"  `description` varchar(16000) NOT NULL DEFAULT '',"
"  `season` smallint NOT NULL,"
"  `episode` smallint NOT NULL,"
"  `category` varchar(64) NOT NULL DEFAULT '',"
"  `profile` varchar(128) NOT NULL DEFAULT 'Default',"
"  `recpriority` int NOT NULL DEFAULT '0',"
"  `autoexpire` int NOT NULL DEFAULT '0',"
"  `maxepisodes` int NOT NULL DEFAULT '0',"
"  `maxnewest` int NOT NULL DEFAULT '0',"
"  `startoffset` int NOT NULL DEFAULT '0',"
"  `endoffset` int NOT NULL DEFAULT '0',"
"  `recgroup` varchar(32) NOT NULL DEFAULT 'Default',"
"  `dupmethod` int NOT NULL DEFAULT '6',"
"  `dupin` int NOT NULL DEFAULT '15',"
"  `station` varchar(20) NOT NULL DEFAULT '',"
"  `seriesid` varchar(64) DEFAULT NULL,"
"  `programid` varchar(64) DEFAULT NULL,"
"  `inetref` varchar(40) NOT NULL,"
"  `search` int unsigned NOT NULL DEFAULT '0',"
"  `autotranscode` tinyint(1) NOT NULL DEFAULT '0',"
"  `autocommflag` tinyint(1) NOT NULL DEFAULT '0',"
"  `autouserjob1` tinyint(1) NOT NULL DEFAULT '0',"
"  `autouserjob2` tinyint(1) NOT NULL DEFAULT '0',"
"  `autouserjob3` tinyint(1) NOT NULL DEFAULT '0',"
"  `autouserjob4` tinyint(1) NOT NULL DEFAULT '0',"
"  `autometadata` tinyint(1) NOT NULL DEFAULT '0',"
"  `findday` tinyint NOT NULL DEFAULT '0',"
"  `findtime` time NOT NULL DEFAULT '00:00:00',"
"  `findid` int NOT NULL DEFAULT '0',"
"  `inactive` tinyint(1) NOT NULL DEFAULT '0',"
"  `parentid` int NOT NULL DEFAULT '0',"
"  `transcoder` int NOT NULL DEFAULT '0',"
"  `playgroup` varchar(32) NOT NULL DEFAULT 'Default',"
"  `prefinput` int NOT NULL DEFAULT '0',"
"  `next_record` datetime DEFAULT NULL,"
"  `last_record` datetime DEFAULT NULL,"
"  `last_delete` datetime DEFAULT NULL,"
"  `storagegroup` varchar(32) NOT NULL DEFAULT 'Default',"
"  `avg_delay` int NOT NULL DEFAULT '100',"
"  `filter` int unsigned NOT NULL DEFAULT '0',"
"  `recgroupid` smallint NOT NULL DEFAULT '1',"
"  `autoextend` tinyint unsigned DEFAULT '0',"
"  PRIMARY KEY (`recordid`),"
"  UNIQUE KEY `chanid` (`chanid`,`starttime`,`startdate`,`title`,`type`),"
"  KEY `title` (`title`),"
"  KEY `seriesid` (`seriesid`),"
"  KEY `programid` (`programid`),"
"  KEY `maxepisodes` (`maxepisodes`),"
"  KEY `search` (`search`),"
"  KEY `type` (`type`),"
"  KEY `recgroupid` (`recgroupid`)"
") ENGINE=MyISAM AUTO_INCREMENT=2 DEFAULT CHARSET=utf8;",
"CREATE TABLE `recorded` ("
"  `chanid` int unsigned NOT NULL DEFAULT '0',"
"  `starttime` datetime NOT NULL DEFAULT '0000-00-00 00:00:00',"
"  `endtime` datetime NOT NULL DEFAULT '0000-00-00 00:00:00',"
"  `title` varchar(128) NOT NULL DEFAULT '',"
"  `subtitle` varchar(128) NOT NULL DEFAULT '',"
"  `description` varchar(16000) NOT NULL DEFAULT '',"
"  `season` smallint NOT NULL,"
"  `episode` smallint NOT NULL,"
"  `category` varchar(64) NOT NULL DEFAULT '',"
"  `hostname` varchar(64) NOT NULL DEFAULT '',"
"  `bookmark` tinyint(1) NOT NULL DEFAULT '0',"
"  `lastplay` tinyint unsigned DEFAULT '0',"
"  `editing` int unsigned NOT NULL DEFAULT '0',"
"  `cutlist` tinyint(1) NOT NULL DEFAULT '0',"
"  `autoexpire` int NOT NULL DEFAULT '0',"
"  `commflagged` int unsigned NOT NULL DEFAULT '0',"
"  `recgroup` varchar(32) NOT NULL DEFAULT 'Default',"
"  `recordid` int DEFAULT NULL,"
"  `seriesid` varchar(64) DEFAULT NULL,"
"  `programid` varchar(64) DEFAULT NULL,"
"  `inetref` varchar(40) NOT NULL,"
"  `lastmodified` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,"
"  `filesize` bigint NOT NULL DEFAULT '0',"
"  `stars` float NOT NULL DEFAULT '0',"
"  `previouslyshown` tinyint(1) DEFAULT '0',"
"  `originalairdate` date DEFAULT NULL,"
"  `preserve` tinyint(1) NOT NULL DEFAULT '0',"
"  `findid` int NOT NULL DEFAULT '0',"
"  `deletepending` tinyint(1) NOT NULL DEFAULT '0',"
"  `transcoder` int NOT NULL DEFAULT '0',"
"  `timestretch` float NOT NULL DEFAULT '1',"
"  `recpriority` int NOT NULL DEFAULT '0',"
"  `basename` varchar(255) NOT NULL,"
"  `progstart` datetime NOT NULL DEFAULT '0000-00-00 00:00:00',"
"  `progend` datetime NOT NULL DEFAULT '0000-00-00 00:00:00',"
"  `playgroup` varchar(32) NOT NULL DEFAULT 'Default',"
"  `profile` varchar(32) NOT NULL DEFAULT '',"
"  `duplicate` tinyint(1) NOT NULL DEFAULT '0',"
"  `transcoded` tinyint(1) NOT NULL DEFAULT '0',"
"  `watched` tinyint NOT NULL DEFAULT '0',"
"  `storagegroup` varchar(32) NOT NULL DEFAULT 'Default',"
"  `bookmarkupdate` timestamp NOT NULL DEFAULT '0000-00-00 00:00:00',"
"  `recgroupid` smallint NOT NULL DEFAULT '1',"
"  `recordedid` int unsigned NOT NULL AUTO_INCREMENT,"
"  `inputname` varchar(32) DEFAULT NULL,"
"  PRIMARY KEY (`recordedid`),"
"  UNIQUE KEY `chanid` (`chanid`,`starttime`),"
"  KEY `endtime` (`endtime`),"
"  KEY `seriesid` (`seriesid`),"
"  KEY `programid` (`programid`),"
"  KEY `title` (`title`),"
"  KEY `recordid` (`recordid`),"
"  KEY `deletepending` (`deletepending`,`lastmodified`),"
"  KEY `recgroup` (`recgroup`,`endtime`),"
"  KEY `recgroupid` (`recgroupid`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `recordedartwork` ("
"  `inetref` varchar(255) NOT NULL,"
"  `season` smallint NOT NULL,"
"  `host` text NOT NULL,"
"  `coverart` text NOT NULL,"
"  `fanart` text NOT NULL,"
"  `banner` text NOT NULL,"
"  KEY `recordedartwork_ix1` (`inetref`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `recordedcredits` ("
"  `person` mediumint unsigned NOT NULL DEFAULT '0',"
"  `chanid` int unsigned NOT NULL DEFAULT '0',"
"  `starttime` datetime NOT NULL DEFAULT '0000-00-00 00:00:00',"
"  `role` set('actor','director','producer','executive_producer','writer','guest_star','host','adapter','presenter','commentator','guest') NOT NULL DEFAULT '',"
"  `priority` tinyint unsigned DEFAULT '0',"
"  `roleid` mediumint unsigned DEFAULT '0',"
"  UNIQUE KEY `chanid` (`chanid`,`starttime`,`person`,`role`,`roleid`),"
"  KEY `person` (`person`,`role`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `recordedfile` ("
"  `basename` varchar(128) NOT NULL DEFAULT '',"
"  `filesize` bigint NOT NULL DEFAULT '0',"
"  `width` smallint unsigned NOT NULL DEFAULT '0',"
"  `height` smallint unsigned NOT NULL DEFAULT '0',"
"  `fps` float(6,3) NOT NULL DEFAULT '0.000',"
"  `aspect` float(8,6) NOT NULL DEFAULT '0.000000',"
"  `audio_sample_rate` smallint unsigned NOT NULL DEFAULT '0',"
"  `audio_channels` tinyint unsigned NOT NULL DEFAULT '0',"
"  `audio_codec` varchar(255) NOT NULL DEFAULT '',"
"  `video_codec` varchar(255) NOT NULL DEFAULT '',"
"  `comment` varchar(255) NOT NULL DEFAULT '',"
"  `hostname` varchar(64) NOT NULL,"
"  `storagegroup` varchar(32) NOT NULL,"
"  `id` int NOT NULL AUTO_INCREMENT,"
"  `recordedid` int unsigned NOT NULL,"
"  `container` varchar(255) NOT NULL DEFAULT '',"
"  `total_bitrate` mediumint unsigned NOT NULL DEFAULT '0',"
"  `video_avg_bitrate` mediumint unsigned NOT NULL DEFAULT '0',"
"  `video_max_bitrate` mediumint unsigned NOT NULL DEFAULT '0',"
"  `audio_avg_bitrate` mediumint unsigned NOT NULL DEFAULT '0',"
"  `audio_max_bitrate` mediumint unsigned NOT NULL DEFAULT '0',"
"  PRIMARY KEY (`id`),"
"  UNIQUE KEY `recordedid` (`recordedid`),"
"  KEY `basename` (`basename`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `recordedmarkup` ("
"  `chanid` int unsigned NOT NULL DEFAULT '0',"
"  `starttime` datetime NOT NULL DEFAULT '0000-00-00 00:00:00',"
"  `mark` mediumint unsigned NOT NULL DEFAULT '0',"
"  `type` tinyint NOT NULL DEFAULT '0',"
"  `data` int unsigned DEFAULT NULL,"
"  PRIMARY KEY (`chanid`,`starttime`,`type`,`mark`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `recordedprogram` ("
"  `chanid` int unsigned NOT NULL DEFAULT '0',"
"  `starttime` datetime NOT NULL DEFAULT '0000-00-00 00:00:00',"
"  `endtime` datetime NOT NULL DEFAULT '0000-00-00 00:00:00',"
"  `title` varchar(128) NOT NULL DEFAULT '',"
"  `subtitle` varchar(128) NOT NULL DEFAULT '',"
"  `description` varchar(16000) NOT NULL DEFAULT '',"
"  `category` varchar(64) NOT NULL DEFAULT '',"
"  `category_type` varchar(64) NOT NULL DEFAULT '',"
"  `airdate` year NOT NULL DEFAULT '0000',"
"  `stars` float unsigned NOT NULL DEFAULT '0',"
"  `previouslyshown` tinyint NOT NULL DEFAULT '0',"
"  `title_pronounce` varchar(128) NOT NULL DEFAULT '',"
"  `stereo` tinyint(1) NOT NULL DEFAULT '0',"
"  `subtitled` tinyint(1) NOT NULL DEFAULT '0',"
"  `hdtv` tinyint(1) NOT NULL DEFAULT '0',"
"  `closecaptioned` tinyint(1) NOT NULL DEFAULT '0',"
"  `partnumber` int NOT NULL DEFAULT '0',"
"  `parttotal` int NOT NULL DEFAULT '0',"
"  `seriesid` varchar(64) DEFAULT NULL,"
"  `originalairdate` date DEFAULT NULL,"
"  `showtype` varchar(30) NOT NULL DEFAULT '',"
"  `colorcode` varchar(20) NOT NULL DEFAULT '',"
"  `syndicatedepisodenumber` varchar(20) NOT NULL DEFAULT '',"
"  `programid` varchar(64) DEFAULT NULL,"
"  `manualid` int unsigned NOT NULL DEFAULT '0',"
"  `generic` tinyint(1) DEFAULT '0',"
"  `listingsource` int NOT NULL DEFAULT '0',"
"  `first` tinyint(1) NOT NULL DEFAULT '0',"
"  `last` tinyint(1) NOT NULL DEFAULT '0',"
"  `audioprop` set('STEREO','MONO','SURROUND','DOLBY','HARDHEAR','VISUALIMPAIR') NOT NULL,"
"  `subtitletypes` set('HARDHEAR','NORMAL','ONSCREEN','SIGNED') NOT NULL,"
"  `videoprop` set('WIDESCREEN','HDTV','MPEG2','AVC','HEVC','720','1080','4K','3DTV','PROGRESSIVE','DAMAGED') NOT NULL,"
"  `inetref` varchar(40) DEFAULT '',"
"  `season` int NOT NULL DEFAULT '0',"
"  `episode` int NOT NULL DEFAULT '0',"
"  `totalepisodes` int NOT NULL DEFAULT '0',"
"  PRIMARY KEY (`chanid`,`starttime`,`manualid`),"
"  KEY `endtime` (`endtime`),"
"  KEY `title` (`title`),"
"  KEY `title_pronounce` (`title_pronounce`),"
"  KEY `seriesid` (`seriesid`),"
"  KEY `programid` (`programid`),"
"  KEY `id_start_end` (`chanid`,`starttime`,`endtime`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `recordedrating` ("
"  `chanid` int unsigned NOT NULL DEFAULT '0',"
"  `starttime` datetime NOT NULL DEFAULT '0000-00-00 00:00:00',"
"  `system` varchar(128) DEFAULT NULL,"
"  `rating` varchar(128) DEFAULT NULL,"
"  UNIQUE KEY `chanid` (`chanid`,`starttime`,`system`,`rating`),"
"  KEY `starttime` (`starttime`,`system`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `recordedseek` ("
"  `chanid` int unsigned NOT NULL DEFAULT '0',"
"  `starttime` datetime NOT NULL DEFAULT '0000-00-00 00:00:00',"
"  `mark` mediumint unsigned NOT NULL DEFAULT '0',"
"  `offset` bigint unsigned NOT NULL,"
"  `type` tinyint NOT NULL DEFAULT '0',"
"  PRIMARY KEY (`chanid`,`starttime`,`type`,`mark`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `recordfilter` ("
"  `filterid` int unsigned NOT NULL,"
"  `description` varchar(64) DEFAULT NULL,"
"  `clause` varchar(256) DEFAULT NULL,"
"  `newruledefault` tinyint(1) DEFAULT '0',"
"  PRIMARY KEY (`filterid`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `recordingprofiles` ("
"  `id` int unsigned NOT NULL AUTO_INCREMENT,"
"  `name` varchar(128) DEFAULT NULL,"
"  `videocodec` varchar(128) DEFAULT NULL,"
"  `audiocodec` varchar(128) DEFAULT NULL,"
"  `profilegroup` int unsigned NOT NULL DEFAULT '0',"
"  PRIMARY KEY (`id`),"
"  KEY `profilegroup` (`profilegroup`)"
") ENGINE=MyISAM AUTO_INCREMENT=78 DEFAULT CHARSET=utf8;",
"CREATE TABLE `recordmatch` ("
"  `recordid` int unsigned NOT NULL,"
"  `chanid` int unsigned NOT NULL,"
"  `starttime` datetime NOT NULL,"
"  `manualid` int unsigned NOT NULL,"
"  `oldrecduplicate` tinyint(1) DEFAULT NULL,"
"  `recduplicate` tinyint(1) DEFAULT NULL,"
"  `findduplicate` tinyint(1) DEFAULT NULL,"
"  `oldrecstatus` int DEFAULT NULL,"
"  `findid` int NOT NULL DEFAULT '0',"
"  UNIQUE KEY `recordid` (`recordid`,`chanid`,`starttime`),"
"  KEY `chanid` (`chanid`,`starttime`,`manualid`),"
"  KEY `recordid_2` (`recordid`,`findid`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `roles` ("
"  `roleid` mediumint unsigned NOT NULL AUTO_INCREMENT,"
"  `name` varchar(128) CHARACTER SET utf8 COLLATE utf8_bin NOT NULL DEFAULT '',"
"  PRIMARY KEY (`roleid`),"
"  UNIQUE KEY `name` (`name`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `scannerfile` ("
"  `fileid` bigint unsigned NOT NULL AUTO_INCREMENT,"
"  `filesize` bigint unsigned NOT NULL DEFAULT '0',"
"  `filehash` varchar(64) NOT NULL DEFAULT '',"
"  `added` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,"
"  PRIMARY KEY (`fileid`),"
"  UNIQUE KEY `filehash` (`filehash`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `scannerpath` ("
"  `fileid` bigint unsigned NOT NULL,"
"  `hostname` varchar(64) NOT NULL DEFAULT 'localhost',"
"  `storagegroup` varchar(32) NOT NULL DEFAULT 'Default',"
"  `filename` varchar(255) NOT NULL DEFAULT '',"
"  PRIMARY KEY (`fileid`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `settings` ("
"  `value` varchar(128) NOT NULL DEFAULT '',"
"  `data` varchar(16000) NOT NULL DEFAULT '',"
"  `hostname` varchar(64) DEFAULT NULL,"
"  KEY `value` (`value`,`hostname`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `sportsapi` ("
"  `id` int unsigned NOT NULL,"
"  `provider` tinyint unsigned DEFAULT '0',"
"  `name` varchar(128) NOT NULL,"
"  `key1` varchar(64) NOT NULL,"
"  `key2` varchar(64) NOT NULL,"
"  PRIMARY KEY (`id`),"
"  UNIQUE KEY `provider` (`provider`,`key1`(25),`key2`(50))"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `sportscleanup` ("
"  `id` int unsigned NOT NULL AUTO_INCREMENT,"
"  `provider` tinyint unsigned DEFAULT '0',"
"  `weight` int unsigned NOT NULL,"
"  `key1` varchar(256) NOT NULL,"
"  `name` varchar(256) NOT NULL,"
"  `pattern` varchar(256) NOT NULL,"
"  `nth` tinyint unsigned NOT NULL,"
"  `replacement` varchar(128) NOT NULL,"
"  PRIMARY KEY (`id`)"
") ENGINE=MyISAM AUTO_INCREMENT=32 DEFAULT CHARSET=utf8;",
"CREATE TABLE `sportslisting` ("
"  `id` int unsigned NOT NULL AUTO_INCREMENT,"
"  `api` int unsigned NOT NULL,"
"  `title` varchar(128) NOT NULL,"
"  PRIMARY KEY (`id`)"
") ENGINE=MyISAM AUTO_INCREMENT=96 DEFAULT CHARSET=utf8;",
"CREATE TABLE `storagegroup` ("
"  `id` int NOT NULL AUTO_INCREMENT,"
"  `groupname` varchar(32) NOT NULL,"
"  `hostname` varchar(64) NOT NULL DEFAULT '',"
"  `dirname` varchar(235) CHARACTER SET utf8 COLLATE utf8_bin NOT NULL DEFAULT '',"
"  PRIMARY KEY (`id`),"
"  UNIQUE KEY `grouphostdir` (`groupname`,`hostname`,`dirname`),"
"  KEY `hostname` (`hostname`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `tvchain` ("
"  `chanid` int unsigned NOT NULL DEFAULT '0',"
"  `starttime` datetime NOT NULL DEFAULT '0000-00-00 00:00:00',"
"  `chainid` varchar(128) NOT NULL DEFAULT '',"
"  `chainpos` int NOT NULL DEFAULT '0',"
"  `discontinuity` tinyint(1) NOT NULL DEFAULT '0',"
"  `watching` int NOT NULL DEFAULT '0',"
"  `hostprefix` varchar(128) NOT NULL DEFAULT '',"
"  `cardtype` varchar(32) NOT NULL DEFAULT 'V4L',"
"  `input` varchar(32) NOT NULL DEFAULT '',"
"  `channame` varchar(32) NOT NULL DEFAULT '',"
"  `endtime` datetime NOT NULL DEFAULT '0000-00-00 00:00:00',"
"  PRIMARY KEY (`chanid`,`starttime`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `tvosdmenu` ("
"  `osdcategory` varchar(32) NOT NULL,"
"  `livetv` tinyint NOT NULL DEFAULT '0',"
"  `recorded` tinyint NOT NULL DEFAULT '0',"
"  `video` tinyint NOT NULL DEFAULT '0',"
"  `dvd` tinyint NOT NULL DEFAULT '0',"
"  `description` varchar(32) NOT NULL,"
"  PRIMARY KEY (`osdcategory`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `upnpmedia` ("
"  `intid` int unsigned NOT NULL DEFAULT '0',"
"  `class` varchar(64) NOT NULL DEFAULT '',"
"  `itemtype` varchar(128) NOT NULL DEFAULT '',"
"  `parentid` int unsigned NOT NULL DEFAULT '0',"
"  `itemproperties` varchar(255) NOT NULL DEFAULT '',"
"  `filepath` varchar(512) NOT NULL DEFAULT '',"
"  `title` varchar(255) NOT NULL DEFAULT '',"
"  `filename` varchar(512) NOT NULL DEFAULT '',"
"  `coverart` varchar(512) NOT NULL DEFAULT '',"
"  PRIMARY KEY (`intid`),"
"  KEY `class` (`class`),"
"  KEY `filepath` (`filepath`(333)),"
"  KEY `parentid` (`parentid`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `user_permissions` ("
"  `userid` int unsigned NOT NULL,"
"  `permission` varchar(128) NOT NULL DEFAULT '',"
"  PRIMARY KEY (`userid`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `user_sessions` ("
"  `sessiontoken` varchar(40) NOT NULL DEFAULT '',"
"  `userid` int unsigned NOT NULL,"
"  `client` varchar(128) NOT NULL,"
"  `created` datetime NOT NULL,"
"  `lastactive` datetime NOT NULL,"
"  `expires` datetime NOT NULL,"
"  PRIMARY KEY (`sessiontoken`),"
"  UNIQUE KEY `userid_client` (`userid`,`client`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `users` ("
"  `userid` int unsigned NOT NULL AUTO_INCREMENT,"
"  `username` varchar(128) NOT NULL DEFAULT '',"
"  `password_digest` varchar(32) NOT NULL DEFAULT '',"
"  `lastlogin` datetime NOT NULL DEFAULT '0000-00-00 00:00:00',"
"  PRIMARY KEY (`userid`),"
"  KEY `username` (`username`)"
") ENGINE=MyISAM AUTO_INCREMENT=2 DEFAULT CHARSET=utf8;",
"CREATE TABLE `videocast` ("
"  `intid` int unsigned NOT NULL AUTO_INCREMENT,"
"  `cast` varchar(128) NOT NULL,"
"  PRIMARY KEY (`intid`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `videocategory` ("
"  `intid` int unsigned NOT NULL AUTO_INCREMENT,"
"  `category` varchar(128) NOT NULL,"
"  PRIMARY KEY (`intid`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `videocollection` ("
"  `intid` int unsigned NOT NULL AUTO_INCREMENT,"
"  `title` varchar(256) NOT NULL,"
"  `contenttype` set('MOVIE','TELEVISION','ADULT','MUSICVIDEO','HOMEVIDEO') NOT NULL DEFAULT '',"
"  `plot` text,"
"  `network` varchar(128) DEFAULT NULL,"
"  `collectionref` varchar(128) NOT NULL,"
"  `certification` varchar(128) DEFAULT NULL,"
"  `genre` varchar(128) DEFAULT '',"
"  `releasedate` date DEFAULT NULL,"
"  `language` varchar(10) DEFAULT NULL,"
"  `status` varchar(64) DEFAULT NULL,"
"  `rating` float DEFAULT '0',"
"  `ratingcount` int DEFAULT '0',"
"  `runtime` smallint unsigned DEFAULT '0',"
"  `banner` text,"
"  `fanart` text,"
"  `coverart` text,"
"  PRIMARY KEY (`intid`),"
"  KEY `title` (`title`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `videocountry` ("
"  `intid` int unsigned NOT NULL AUTO_INCREMENT,"
"  `country` varchar(128) NOT NULL,"
"  PRIMARY KEY (`intid`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `videogenre` ("
"  `intid` int unsigned NOT NULL AUTO_INCREMENT,"
"  `genre` varchar(128) NOT NULL,"
"  PRIMARY KEY (`intid`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `videometadata` ("
"  `intid` int unsigned NOT NULL AUTO_INCREMENT,"
"  `title` varchar(128) NOT NULL,"
"  `subtitle` text NOT NULL,"
"  `tagline` varchar(255) DEFAULT NULL,"
"  `director` varchar(128) NOT NULL,"
"  `studio` varchar(128) DEFAULT NULL,"
"  `plot` text,"
"  `rating` varchar(128) NOT NULL,"
"  `inetref` varchar(255) NOT NULL,"
"  `collectionref` int NOT NULL DEFAULT '-1',"
"  `homepage` text NOT NULL,"
"  `year` int unsigned NOT NULL,"
"  `releasedate` date NOT NULL,"
"  `userrating` float NOT NULL,"
"  `length` int unsigned NOT NULL,"
"  `playcount` int NOT NULL DEFAULT '0',"
"  `season` smallint unsigned NOT NULL DEFAULT '0',"
"  `episode` smallint unsigned NOT NULL DEFAULT '0',"
"  `showlevel` int unsigned NOT NULL,"
"  `filename` text NOT NULL,"
"  `hash` varchar(128) NOT NULL,"
"  `coverfile` text NOT NULL,"
"  `childid` int NOT NULL DEFAULT '-1',"
"  `browse` tinyint(1) NOT NULL DEFAULT '1',"
"  `watched` tinyint(1) NOT NULL DEFAULT '0',"
"  `processed` tinyint(1) NOT NULL DEFAULT '0',"
"  `playcommand` varchar(255) DEFAULT NULL,"
"  `category` int unsigned NOT NULL DEFAULT '0',"
"  `trailer` text,"
"  `host` text NOT NULL,"
"  `screenshot` text,"
"  `banner` text,"
"  `fanart` text,"
"  `insertdate` timestamp NULL DEFAULT CURRENT_TIMESTAMP,"
"  `contenttype` set('MOVIE','TELEVISION','ADULT','MUSICVIDEO','HOMEVIDEO') NOT NULL DEFAULT '',"
"  PRIMARY KEY (`intid`),"
"  KEY `director` (`director`),"
"  KEY `title` (`title`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `videometadatacast` ("
"  `idvideo` int unsigned NOT NULL,"
"  `idcast` int unsigned NOT NULL,"
"  UNIQUE KEY `idvideo` (`idvideo`,`idcast`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `videometadatacountry` ("
"  `idvideo` int unsigned NOT NULL,"
"  `idcountry` int unsigned NOT NULL,"
"  UNIQUE KEY `idvideo_2` (`idvideo`,`idcountry`),"
"  KEY `idvideo` (`idvideo`),"
"  KEY `idcountry` (`idcountry`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `videometadatagenre` ("
"  `idvideo` int unsigned NOT NULL,"
"  `idgenre` int unsigned NOT NULL,"
"  UNIQUE KEY `idvideo_2` (`idvideo`,`idgenre`),"
"  KEY `idvideo` (`idvideo`),"
"  KEY `idgenre` (`idgenre`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `videopart` ("
"  `fileid` bigint unsigned NOT NULL,"
"  `videoid` int unsigned NOT NULL,"
"  `order` smallint unsigned NOT NULL DEFAULT '1',"
"  PRIMARY KEY (`videoid`,`order`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `videopathinfo` ("
"  `intid` int unsigned NOT NULL AUTO_INCREMENT,"
"  `path` text,"
"  `contenttype` set('MOVIE','TELEVISION','ADULT','MUSICVIDEO','HOMEVIDEO') NOT NULL DEFAULT '',"
"  `collectionref` int DEFAULT '0',"
"  `recurse` tinyint(1) DEFAULT '0',"
"  PRIMARY KEY (`intid`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `videosource` ("
"  `sourceid` int unsigned NOT NULL AUTO_INCREMENT,"
"  `name` varchar(128) NOT NULL DEFAULT '',"
"  `xmltvgrabber` varchar(128) DEFAULT NULL,"
"  `userid` varchar(128) DEFAULT NULL,"
"  `freqtable` varchar(16) NOT NULL DEFAULT 'default',"
"  `lineupid` varchar(64) DEFAULT NULL,"
"  `password` varchar(64) DEFAULT NULL,"
"  `useeit` smallint NOT NULL DEFAULT '0',"
"  `configpath` varchar(4096) DEFAULT NULL,"
"  `dvb_nit_id` int DEFAULT '-1',"
"  `bouquet_id` int unsigned DEFAULT NULL,"
"  `region_id` int unsigned DEFAULT NULL,"
"  `scanfrequency` int unsigned DEFAULT '0',"
"  `lcnoffset` int unsigned DEFAULT '0',"
"  PRIMARY KEY (`sourceid`),"
"  UNIQUE KEY `name` (`name`)"
") ENGINE=MyISAM DEFAULT CHARSET=utf8;",
"CREATE TABLE `videotypes` ("
"  `intid` int unsigned NOT NULL AUTO_INCREMENT,"
"  `extension` varchar(128) NOT NULL,"
"  `playcommand` varchar(255) NOT NULL,"
"  `f_ignore` tinyint(1) DEFAULT NULL,"
"  `use_default` tinyint(1) DEFAULT NULL,"
"  PRIMARY KEY (`intid`)"
") ENGINE=MyISAM AUTO_INCREMENT=33 DEFAULT CHARSET=utf8;",

"INSERT INTO `channelgroupnames` VALUES (1,'Favorites');",
"INSERT INTO `customexample` VALUES ('New Flix','','program.category_type = \\'movie\\' AND program.airdate >= \\n     YEAR(DATE_SUB(NOW(), INTERVAL 1 YEAR)) \\nAND program.stars > 0.5 ',1);",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',9018,'channel_numbers','131');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',9018,'guide_fixup','2');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',256,'guide_fixup','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',257,'guide_fixup','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',256,'tv_types','1,150,134,133');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',257,'tv_types','1,150,134,133');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4100,'sdt_mapping','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4101,'sdt_mapping','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4102,'sdt_mapping','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4103,'sdt_mapping','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4104,'sdt_mapping','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4105,'sdt_mapping','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4106,'sdt_mapping','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4107,'sdt_mapping','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4097,'sdt_mapping','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4098,'sdt_mapping','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4100,'tv_types','1,145,154');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4101,'tv_types','1,145,154');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4102,'tv_types','1,145,154');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4103,'tv_types','1,145,154');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4104,'tv_types','1,145,154');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4105,'tv_types','1,145,154');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4106,'tv_types','1,145,154');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4107,'tv_types','1,145,154');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4097,'tv_types','1,145,154');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4098,'tv_types','1,145,154');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4100,'guide_fixup','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4101,'guide_fixup','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4102,'guide_fixup','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4103,'guide_fixup','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4104,'guide_fixup','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4105,'guide_fixup','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4106,'guide_fixup','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4107,'guide_fixup','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4096,'guide_fixup','5');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4097,'guide_fixup','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4098,'guide_fixup','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',94,'tv_types','1,128');",
"INSERT INTO `dtv_privatetypes` VALUES ('atsc',1793,'guide_fixup','3');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',40999,'guide_fixup','4');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',70,'force_guide_present','yes');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',70,'guide_ranges','80,80,96,96');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4112,'channel_numbers','131');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4115,'channel_numbers','131');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4116,'channel_numbers','131');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',12802,'channel_numbers','131');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',12803,'channel_numbers','131');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',12829,'channel_numbers','131');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',40999,'parse_subtitle_list','1070,1308,1041,1306,1307,1030,1016,1131,1068,1069');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4096,'guide_fixup','5');",
"INSERT INTO `dvdinput` VALUES (1,720,480,16,9,1,1,'ntsc');",
"INSERT INTO `dvdinput` VALUES (2,720,480,16,9,1,0,'ntsc');",
"INSERT INTO `dvdinput` VALUES (3,720,480,4,3,1,1,'ntsc');",
"INSERT INTO `dvdinput` VALUES (4,720,480,4,3,1,0,'ntsc');",
"INSERT INTO `dvdinput` VALUES (5,720,576,16,9,3,1,'pal');",
"INSERT INTO `dvdinput` VALUES (6,720,576,16,9,3,0,'pal');",
"INSERT INTO `dvdinput` VALUES (7,720,576,4,3,3,1,'pal');",
"INSERT INTO `dvdinput` VALUES (8,720,576,4,3,3,0,'pal');",
"INSERT INTO `dvdtranscode` VALUES (1,1,'Good',2,1,16,16,0,0,2,0,0,0,0,0,32,32,8,8,'divx5',NULL,1618,NULL,NULL,0,NULL);",
"INSERT INTO `dvdtranscode` VALUES (2,2,'Excellent',2,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,'divx5',NULL,0,NULL,NULL,1,NULL);",
"INSERT INTO `dvdtranscode` VALUES (3,2,'Good',2,1,0,0,8,8,0,0,0,0,0,0,0,0,0,0,'divx5',NULL,1618,NULL,NULL,0,NULL);",
"INSERT INTO `dvdtranscode` VALUES (4,2,'Medium',2,1,0,0,8,8,5,5,0,0,0,0,0,0,0,0,'divx5',NULL,1200,NULL,NULL,0,NULL);",
"INSERT INTO `dvdtranscode` VALUES (5,3,'Good',2,1,0,0,0,0,0,0,0,0,2,0,80,80,8,8,'divx5',NULL,0,NULL,NULL,0,NULL);",
"INSERT INTO `dvdtranscode` VALUES (6,4,'Excellent',2,1,0,0,0,0,0,0,0,0,2,0,0,0,0,0,'divx5',NULL,0,NULL,NULL,1,NULL);",
"INSERT INTO `dvdtranscode` VALUES (7,4,'Good',2,1,0,0,8,8,0,2,0,0,0,0,0,0,0,0,'divx5',NULL,1618,NULL,NULL,0,NULL);",
"INSERT INTO `dvdtranscode` VALUES (8,5,'Good',1,1,16,16,0,0,5,0,0,0,0,0,40,40,8,8,'divx5',NULL,1618,NULL,NULL,0,NULL);",
"INSERT INTO `dvdtranscode` VALUES (9,6,'Good',1,1,0,0,16,16,5,0,0,0,0,0,0,0,0,0,'divx5',NULL,1618,NULL,NULL,0,NULL);",
"INSERT INTO `dvdtranscode` VALUES (10,7,'Good',1,1,0,0,0,0,1,0,0,0,0,0,76,76,8,8,'divx5',NULL,1618,NULL,NULL,0,NULL);",
"INSERT INTO `dvdtranscode` VALUES (11,8,'Good',1,1,0,0,0,0,1,0,0,0,0,0,0,0,0,0,'divx5',NULL,1618,NULL,NULL,0,NULL);",
"INSERT INTO `playgroup` VALUES ('Default','',30,5,100,0);",
"INSERT INTO `profilegroups` VALUES (1,'Software Encoders (v4l based)','V4L',1,NULL);",
"INSERT INTO `profilegroups` VALUES (2,'IVTV MPEG-2 Encoders','MPEG',1,NULL);",
"INSERT INTO `profilegroups` VALUES (3,'Hardware MJPEG Encoders (Matrox G200-TV, Miro DC10, etc)','MJPEG',1,NULL);",
"INSERT INTO `profilegroups` VALUES (4,'Hardware HDTV','HDTV',1,NULL);",
"INSERT INTO `profilegroups` VALUES (5,'Hardware DVB Encoders','DVB',1,NULL);",
"INSERT INTO `profilegroups` VALUES (6,'Transcoders','TRANSCODE',1,NULL);",
"INSERT INTO `profilegroups` VALUES (7,'FireWire Input','FIREWIRE',1,NULL);",
"INSERT INTO `profilegroups` VALUES (8,'USB Mpeg-4 Encoder (Plextor ConvertX, etc)','GO7007',1,NULL);",
"INSERT INTO `profilegroups` VALUES (14,'Import Recorder','IMPORT',1,NULL);",
"INSERT INTO `profilegroups` VALUES (10,'Freebox Input','FREEBOX',1,NULL);",
"INSERT INTO `profilegroups` VALUES (11,'HDHomeRun Recorders','HDHOMERUN',1,NULL);",
"INSERT INTO `profilegroups` VALUES (12,'CRC IP Recorders','CRC_IP',1,NULL);",
"INSERT INTO `profilegroups` VALUES (13,'HD-PVR Recorders','HDPVR',1,NULL);",
"INSERT INTO `profilegroups` VALUES (15,'ASI Recorder (DVEO)','ASI',1,NULL);",
"INSERT INTO `profilegroups` VALUES (16,'OCUR Recorder (CableLabs)','OCUR',1,NULL);",
"INSERT INTO `profilegroups` VALUES (17,'Ceton Recorder','CETON',1,NULL);",
"INSERT INTO `profilegroups` VALUES (18,'VBox Recorder','VBOX',1,NULL);",
"INSERT INTO `profilegroups` VALUES (19,'Sat>IP Recorder','SATIP',1,NULL);",
"INSERT INTO `recgroups` VALUES (1,'Default','','',1);",
"INSERT INTO `recgroups` VALUES (2,'LiveTV','','',1);",
"INSERT INTO `recgroups` VALUES (3,'Deleted','','',1);",
"INSERT INTO `record` VALUES (1,11,0,'21:57:44','2012-08-11','21:57:44','2012-08-11','Default (Template)','','',0,0,'Default','Default',0,0,0,0,0,0,'Default',6,15,'','','','',0,0,1,0,0,0,0,1,-1,'00:00:00',735091,0,0,0,'Default',0,NULL,NULL,NULL,'Default',100,0,1,0);",
"INSERT INTO `recordfilter` VALUES (0,'New episode','program.previouslyshown = 0',0);",
"INSERT INTO `recordfilter` VALUES (1,'Identifiable episode','program.generic = 0',0);",
"INSERT INTO `recordfilter` VALUES (2,'First showing','program.first > 0',0);",
"INSERT INTO `recordfilter` VALUES (3,'Prime time','HOUR(CONVERT_TZ(program.starttime, \\'Etc/UTC\\', \\'SYSTEM\\')) >= 19 AND HOUR(CONVERT_TZ(program.starttime, \\'Etc/UTC\\', \\'SYSTEM\\')) < 22',0);",
"INSERT INTO `recordfilter` VALUES (4,'Commercial free','channel.commmethod = -2',0);",
"INSERT INTO `recordfilter` VALUES (5,'High definition','program.hdtv > 0',0);",
"INSERT INTO `recordfilter` VALUES (6,'This episode','(RECTABLE.programid <> \\'\\' AND program.programid = RECTABLE.programid) OR (RECTABLE.programid = \\'\\' AND program.subtitle = RECTABLE.subtitle AND program.description = RECTABLE.description)',0);",
"INSERT INTO `recordfilter` VALUES (7,'This series','(RECTABLE.seriesid <> \\'\\' AND program.seriesid = RECTABLE.seriesid)',0);",
"INSERT INTO `recordfilter` VALUES (8,'This time','ABS(TIMESTAMPDIFF(MINUTE, CONVERT_TZ(  ADDTIME(RECTABLE.startdate, RECTABLE.starttime), \\'Etc/UTC\\', \\'SYSTEM\\'),   CONVERT_TZ(program.starttime, \\'Etc/UTC\\', \\'SYSTEM\\'))) MOD 1440   NOT BETWEEN 11 AND 1429',0);",
"INSERT INTO `recordfilter` VALUES (9,'This day and time','ABS(TIMESTAMPDIFF(MINUTE, CONVERT_TZ(  ADDTIME(RECTABLE.startdate, RECTABLE.starttime), \\'Etc/UTC\\', \\'SYSTEM\\'),   CONVERT_TZ(program.starttime, \\'Etc/UTC\\', \\'SYSTEM\\'))) MOD 10080   NOT BETWEEN 11 AND 10069',0);",
"INSERT INTO `recordfilter` VALUES (10,'This channel','channel.callsign = RECTABLE.station',0);",
"INSERT INTO `recordfilter` VALUES (11,'No episodes','program.category_type <> \\'series\\'',0);",
"INSERT INTO `recordfilter` VALUES (12,'Priority channel','channel.recpriority > 0',0);",
"INSERT INTO `recordingprofiles` VALUES (1,'Default',NULL,NULL,1);",
"INSERT INTO `recordingprofiles` VALUES (2,'Live TV',NULL,NULL,1);",
"INSERT INTO `recordingprofiles` VALUES (3,'High Quality',NULL,NULL,1);",
"INSERT INTO `recordingprofiles` VALUES (4,'Low Quality',NULL,NULL,1);",
"INSERT INTO `recordingprofiles` VALUES (5,'Default',NULL,NULL,2);",
"INSERT INTO `recordingprofiles` VALUES (6,'Live TV',NULL,NULL,2);",
"INSERT INTO `recordingprofiles` VALUES (7,'High Quality',NULL,NULL,2);",
"INSERT INTO `recordingprofiles` VALUES (8,'Low Quality',NULL,NULL,2);",
"INSERT INTO `recordingprofiles` VALUES (9,'Default',NULL,NULL,3);",
"INSERT INTO `recordingprofiles` VALUES (10,'Live TV',NULL,NULL,3);",
"INSERT INTO `recordingprofiles` VALUES (11,'High Quality',NULL,NULL,3);",
"INSERT INTO `recordingprofiles` VALUES (12,'Low Quality',NULL,NULL,3);",
"INSERT INTO `recordingprofiles` VALUES (13,'Default',NULL,NULL,4);",
"INSERT INTO `recordingprofiles` VALUES (14,'Live TV',NULL,NULL,4);",
"INSERT INTO `recordingprofiles` VALUES (15,'High Quality',NULL,NULL,4);",
"INSERT INTO `recordingprofiles` VALUES (16,'Low Quality',NULL,NULL,4);",
"INSERT INTO `recordingprofiles` VALUES (17,'Default',NULL,NULL,5);",
"INSERT INTO `recordingprofiles` VALUES (18,'Live TV',NULL,NULL,5);",
"INSERT INTO `recordingprofiles` VALUES (19,'High Quality',NULL,NULL,5);",
"INSERT INTO `recordingprofiles` VALUES (20,'Low Quality',NULL,NULL,5);",
"INSERT INTO `recordingprofiles` VALUES (21,'RTjpeg/MPEG4',NULL,NULL,6);",
"INSERT INTO `recordingprofiles` VALUES (22,'MPEG2',NULL,NULL,6);",
"INSERT INTO `recordingprofiles` VALUES (23,'Default',NULL,NULL,8);",
"INSERT INTO `recordingprofiles` VALUES (24,'Live TV',NULL,NULL,8);",
"INSERT INTO `recordingprofiles` VALUES (25,'High Quality',NULL,NULL,8);",
"INSERT INTO `recordingprofiles` VALUES (26,'Low Quality',NULL,NULL,8);",
"INSERT INTO `recordingprofiles` VALUES (27,'High Quality',NULL,NULL,6);",
"INSERT INTO `recordingprofiles` VALUES (28,'Medium Quality',NULL,NULL,6);",
"INSERT INTO `recordingprofiles` VALUES (29,'Low Quality',NULL,NULL,6);",
"INSERT INTO `recordingprofiles` VALUES (30,'Default',NULL,NULL,10);",
"INSERT INTO `recordingprofiles` VALUES (31,'Live TV',NULL,NULL,10);",
"INSERT INTO `recordingprofiles` VALUES (32,'High Quality',NULL,NULL,10);",
"INSERT INTO `recordingprofiles` VALUES (33,'Low Quality',NULL,NULL,10);",
"INSERT INTO `recordingprofiles` VALUES (34,'Default',NULL,NULL,11);",
"INSERT INTO `recordingprofiles` VALUES (35,'Live TV',NULL,NULL,11);",
"INSERT INTO `recordingprofiles` VALUES (36,'High Quality',NULL,NULL,11);",
"INSERT INTO `recordingprofiles` VALUES (37,'Low Quality',NULL,NULL,11);",
"INSERT INTO `recordingprofiles` VALUES (38,'Default',NULL,NULL,12);",
"INSERT INTO `recordingprofiles` VALUES (39,'Live TV',NULL,NULL,12);",
"INSERT INTO `recordingprofiles` VALUES (40,'High Quality',NULL,NULL,12);",
"INSERT INTO `recordingprofiles` VALUES (41,'Low Quality',NULL,NULL,12);",
"INSERT INTO `recordingprofiles` VALUES (42,'Default',NULL,NULL,7);",
"INSERT INTO `recordingprofiles` VALUES (43,'Live TV',NULL,NULL,7);",
"INSERT INTO `recordingprofiles` VALUES (44,'High Quality',NULL,NULL,7);",
"INSERT INTO `recordingprofiles` VALUES (45,'Low Quality',NULL,NULL,7);",
"INSERT INTO `recordingprofiles` VALUES (46,'Default',NULL,NULL,9);",
"INSERT INTO `recordingprofiles` VALUES (47,'Live TV',NULL,NULL,9);",
"INSERT INTO `recordingprofiles` VALUES (48,'High Quality',NULL,NULL,9);",
"INSERT INTO `recordingprofiles` VALUES (49,'Low Quality',NULL,NULL,9);",
"INSERT INTO `recordingprofiles` VALUES (50,'Default',NULL,NULL,13);",
"INSERT INTO `recordingprofiles` VALUES (51,'Live TV',NULL,NULL,13);",
"INSERT INTO `recordingprofiles` VALUES (52,'High Quality',NULL,NULL,13);",
"INSERT INTO `recordingprofiles` VALUES (53,'Low Quality',NULL,NULL,13);",
"INSERT INTO `recordingprofiles` VALUES (54,'Default',NULL,NULL,14);",
"INSERT INTO `recordingprofiles` VALUES (55,'Live TV',NULL,NULL,14);",
"INSERT INTO `recordingprofiles` VALUES (56,'High Quality',NULL,NULL,14);",
"INSERT INTO `recordingprofiles` VALUES (57,'Low Quality',NULL,NULL,14);",
"INSERT INTO `recordingprofiles` VALUES (58,'Default',NULL,NULL,15);",
"INSERT INTO `recordingprofiles` VALUES (59,'Live TV',NULL,NULL,15);",
"INSERT INTO `recordingprofiles` VALUES (60,'High Quality',NULL,NULL,15);",
"INSERT INTO `recordingprofiles` VALUES (61,'Low Quality',NULL,NULL,15);",
"INSERT INTO `recordingprofiles` VALUES (62,'Default',NULL,NULL,16);",
"INSERT INTO `recordingprofiles` VALUES (63,'Live TV',NULL,NULL,16);",
"INSERT INTO `recordingprofiles` VALUES (64,'High Quality',NULL,NULL,16);",
"INSERT INTO `recordingprofiles` VALUES (65,'Low Quality',NULL,NULL,16);",
"INSERT INTO `recordingprofiles` VALUES (66,'Default',NULL,NULL,17);",
"INSERT INTO `recordingprofiles` VALUES (67,'Live TV',NULL,NULL,17);",
"INSERT INTO `recordingprofiles` VALUES (68,'High Quality',NULL,NULL,17);",
"INSERT INTO `recordingprofiles` VALUES (69,'Low Quality',NULL,NULL,17);",
"INSERT INTO `recordingprofiles` VALUES (70,'Default',NULL,NULL,18);",
"INSERT INTO `recordingprofiles` VALUES (71,'Live TV',NULL,NULL,18);",
"INSERT INTO `recordingprofiles` VALUES (72,'High Quality',NULL,NULL,18);",
"INSERT INTO `recordingprofiles` VALUES (73,'Low Quality',NULL,NULL,18);",
"INSERT INTO `recordingprofiles` VALUES (74,'Default',NULL,NULL,19);",
"INSERT INTO `recordingprofiles` VALUES (75,'Live TV',NULL,NULL,19);",
"INSERT INTO `recordingprofiles` VALUES (76,'High Quality',NULL,NULL,19);",
"INSERT INTO `recordingprofiles` VALUES (77,'Low Quality',NULL,NULL,19);",
"INSERT INTO `settings` VALUES ('mythfilldatabaseLastRunStart','',NULL);",
"INSERT INTO `settings` VALUES ('mythfilldatabaseLastRunEnd','',NULL);",
"INSERT INTO `settings` VALUES ('mythfilldatabaseLastRunStatus','',NULL);",
"INSERT INTO `settings` VALUES ('DataDirectMessage','',NULL);",
"INSERT INTO `settings` VALUES ('HaveRepeats','0',NULL);",
"INSERT INTO `settings` VALUES ('DBSchemaVer','1382',NULL);",
"INSERT INTO `settings` VALUES ('HardwareProfileEnabled','0',NULL);",
"INSERT INTO `settings` VALUES ('ImageStorageGroupName','Images',NULL);",
"INSERT INTO `settings` VALUES ('ImageSortOrder','0',NULL);",
"INSERT INTO `settings` VALUES ('ImageShowHiddenFiles','0',NULL);",
"INSERT INTO `settings` VALUES ('ImageSlideShowTime','3500',NULL);",
"INSERT INTO `settings` VALUES ('ImageTransitionType','1',NULL);",
"INSERT INTO `settings` VALUES ('ImageTransitionTime','1000',NULL);",
"INSERT INTO `sportsapi` VALUES (1,1,'Major League Baseball','baseball','mlb');",
"INSERT INTO `sportsapi` VALUES (2,1,'NCAA Men\\'s Baseball','baseball','college-baseball');",
"INSERT INTO `sportsapi` VALUES (3,1,'NCAA Women\\'s Softball','baseball','college-softball');",
"INSERT INTO `sportsapi` VALUES (4,1,'Olympic Men\\'s Baseball','baseball','olympics-baseball');",
"INSERT INTO `sportsapi` VALUES (5,1,'World Baseball Classic','baseball','world-baseball-classic');",
"INSERT INTO `sportsapi` VALUES (6,1,'Little League Baseball','baseball','llb');",
"INSERT INTO `sportsapi` VALUES (7,1,'Caribbean Series','baseball','caribbean-series');",
"INSERT INTO `sportsapi` VALUES (8,1,'Dominican Winter League','baseball','dominican-winter-league');",
"INSERT INTO `sportsapi` VALUES (9,1,'Venezuelan Winter League','baseball','venezuelan-winter-league');",
"INSERT INTO `sportsapi` VALUES (10,1,'Mexican League','baseball','mexican-winter-league');",
"INSERT INTO `sportsapi` VALUES (11,1,'Puerto Rican Winter League','baseball','puerto-rican-winter-league');",
"INSERT INTO `sportsapi` VALUES (20,1,'National Football League','football','nfl');",
"INSERT INTO `sportsapi` VALUES (21,1,'NCAA - Football','football','college-football');",
"INSERT INTO `sportsapi` VALUES (22,1,'XFL','football','xfl');",
"INSERT INTO `sportsapi` VALUES (23,1,'Canadian Football League','football','cfl');",
"INSERT INTO `sportsapi` VALUES (40,1,'National Basketball Association','basketball','nba');",
"INSERT INTO `sportsapi` VALUES (41,1,'Women\\'s National Basketball Association','basketball','wnba');",
"INSERT INTO `sportsapi` VALUES (42,1,'NCAA Men\\'s Basketball','basketball','mens-college-basketball');",
"INSERT INTO `sportsapi` VALUES (43,1,'NCAA Women\\'s Basketball','basketball','womens-college-basketball');",
"INSERT INTO `sportsapi` VALUES (44,1,'Olympics Men\\'s Basketball','basketball','mens-olympic-basketball');",
"INSERT INTO `sportsapi` VALUES (45,1,'Olympics Women\\'s Basketball','basketball','womens-olympic-basketball');",
"INSERT INTO `sportsapi` VALUES (46,1,'National Basketball Association Summer League Las Vegas','basketball','nba-summer-las-vegas');",
"INSERT INTO `sportsapi` VALUES (47,1,'National Basketball Association Summer League Utah','basketball','nba-summer-utah');",
"INSERT INTO `sportsapi` VALUES (48,1,'National Basketball Association Summer League Orlando','basketball','nba-summer-orlando');",
"INSERT INTO `sportsapi` VALUES (49,1,'National Basketball Association Summer League Sacramento','basketball','nba-summer-sacramento');",
"INSERT INTO `sportsapi` VALUES (50,1,'NBA G League','basketball','nba-development');",
"INSERT INTO `sportsapi` VALUES (51,1,'International Basketball Federation','basketball','fiba');",
"INSERT INTO `sportsapi` VALUES (60,1,'National Hockey League','hockey','nfl');",
"INSERT INTO `sportsapi` VALUES (61,1,'NCAA Men\\'s Ice Hockey','hockey','mens-college-hockey');",
"INSERT INTO `sportsapi` VALUES (62,1,'NCAA Women\\'s Hockey','hockey','womens-college-hockey');",
"INSERT INTO `sportsapi` VALUES (63,1,'World Cup of Hockey','hockey','hockey-world-cup');",
"INSERT INTO `sportsapi` VALUES (64,1,'Men\\'s Olympic Ice Hockey','hockey','mens-olympic-hockey');",
"INSERT INTO `sportsapi` VALUES (65,1,'Women\\'s Olympic Ice Hockey','hockey','womens-olympic-hockey');",
"INSERT INTO `sportsapi` VALUES (66,1,'NCAA Women\\'s Field Hockey','field-hockey','womens-college-field-hockey');",
"INSERT INTO `sportsapi` VALUES (80,1,'NCAA Men\\'s Volleyball','volleyball','mens-college-volleyball');",
"INSERT INTO `sportsapi` VALUES (81,1,'NCAA Women\\'s Volleyball','volleyball','womens-college-volleyball');",
"INSERT INTO `sportsapi` VALUES (100,1,'NCAA Men\\'s Lacrosse','lacrosse','mens-college-lacrosse');",
"INSERT INTO `sportsapi` VALUES (101,1,'NCAA Women\\'s Lacrosse','lacrosse','womens-college-lacrosse');",
"INSERT INTO `sportsapi` VALUES (120,1,'NCAA Men\\'s Water Polo','water-polo','mens-college-water-polo');",
"INSERT INTO `sportsapi` VALUES (121,1,'NCAA Women\\'s Water Polo','water-polo','womens-college-water-polo');",
"INSERT INTO `sportsapi` VALUES (200,1,'NCAA Men\\'s Soccer','soccer','usa.ncaa.m.1');",
"INSERT INTO `sportsapi` VALUES (201,1,'NCAA Women\\'s Soccer','soccer','usa.ncaa.w.1');",
"INSERT INTO `sportsapi` VALUES (202,1,'Major League Soccer','soccer','usa.1');",
"INSERT INTO `sportsapi` VALUES (203,1,'English Premier League','soccer','eng.1');",
"INSERT INTO `sportsapi` VALUES (204,1,'English League Championship','soccer','eng.2');",
"INSERT INTO `sportsapi` VALUES (205,1,'Italian Serie A','soccer','ita.1');",
"INSERT INTO `sportsapi` VALUES (206,1,'French Ligue 1','soccer','fra.1');",
"INSERT INTO `sportsapi` VALUES (207,1,'French Ligue 2','soccer','fra.2');",
"INSERT INTO `sportsapi` VALUES (208,1,'Spanish LaLiga','soccer','esp.1');",
"INSERT INTO `sportsapi` VALUES (209,1,'German Bundesliga','soccer','ger.1');",
"INSERT INTO `sportsapi` VALUES (210,1,'German 2. Bundesliga','soccer','ger.2');",
"INSERT INTO `sportsapi` VALUES (211,1,'Mexican Liga BBVA MX','soccer','mex.1');",
"INSERT INTO `sportsapi` VALUES (212,1,'Copa Do Brasil','soccer','bra.copa_do_brazil');",
"INSERT INTO `sportsapi` VALUES (213,1,'CONCACAF Leagues Cup','soccer','concacaf.leagues.cup');",
"INSERT INTO `sportsapi` VALUES (214,1,'CONCACAF League','soccer','concacaf.league');",
"INSERT INTO `sportsapi` VALUES (215,1,'CONCACAF Champions League','soccer','concacaf.champions');",
"INSERT INTO `sportsapi` VALUES (216,1,'CONCACAF Nations League','soccer','concacaf.nations.league');",
"INSERT INTO `sportsapi` VALUES (217,1,'CONCACAF Gold Cup','soccer','concacaf.gold');",
"INSERT INTO `sportsapi` VALUES (218,1,'FIFA World Cup','soccer','fifa.world');",
"INSERT INTO `sportsapi` VALUES (219,1,'FIFA World Cup Qualifying - UEFA','soccer','fifa.worldq.uefa');",
"INSERT INTO `sportsapi` VALUES (220,1,'FIFA World Cup Qualifying - CONCACAF','soccer','fifa.worldq.concacaf');",
"INSERT INTO `sportsapi` VALUES (221,1,'FIFA World Cup Qualifying - CONMEBOL','soccer','fifa.worldq.conmebol');",
"INSERT INTO `sportsapi` VALUES (222,1,'FIFA World Cup Qualifying - AFC','soccer','fifa.worldq.afc');",
"INSERT INTO `sportsapi` VALUES (223,1,'FIFA World Cup Qualifying - CAF','soccer','fifa.worldq.caf');",
"INSERT INTO `sportsapi` VALUES (224,1,'FIFA World Cup Qualifying - OFC','soccer','fifa.worldq.ofc');",
"INSERT INTO `sportsapi` VALUES (225,1,'UEFA Champions League','soccer','uefa.champions');",
"INSERT INTO `sportsapi` VALUES (226,1,'UEFA Europa League','soccer','uefa.europa');",
"INSERT INTO `sportsapi` VALUES (227,1,'UEFA Europa Conference League','soccer','uefa.europa.conf');",
"INSERT INTO `sportsapi` VALUES (228,1,'English Carabao Cup','soccer','eng.league_cup');",
"INSERT INTO `sportsapi` VALUES (229,1,'USL Championship','soccer','usa.usl.1');",
"INSERT INTO `sportsapi` VALUES (230,1,'United States NWSL','soccer','usa.nwsl');",
"INSERT INTO `sportsapi` VALUES (231,1,'FA Women\\'s Super League','soccer','eng.w.1');",
"INSERT INTO `sportsapi` VALUES (232,1,'English FA Cup','soccer','eng.fa');",
"INSERT INTO `sportsapi` VALUES (233,1,'Spanish Copa del Rey','soccer','esp.copa_del_rey');",
"INSERT INTO `sportsapi` VALUES (234,1,'German DFB Pokal','soccer','ger.dfb_pokal');",
"INSERT INTO `sportsapi` VALUES (235,1,'Italian Coppa Italia','soccer','ita.coppa_italia');",
"INSERT INTO `sportsapi` VALUES (236,1,'French Coupe de France','soccer','fra.coupe_de_france');",
"INSERT INTO `sportsapi` VALUES (237,1,'AFC Champions League','soccer','afc.champions');",
"INSERT INTO `sportsapi` VALUES (238,1,'Dutch KNVB Beker','soccer','ned.cup');",
"INSERT INTO `sportsapi` VALUES (239,1,'Dutch Eredivisie','soccer','ned.1');",
"INSERT INTO `sportsapi` VALUES (240,1,'Portuguese Liga','soccer','por.1');",
"INSERT INTO `sportsapi` VALUES (241,1,'Russian Premier League','soccer','rus.1');",
"INSERT INTO `sportsapi` VALUES (242,1,'Mexican Liga de Expansión MX','soccer','mex.2');",
"INSERT INTO `sportsapi` VALUES (243,1,'Mexican Copa MX','soccer','mex.copa_mx');",
"INSERT INTO `sportsapi` VALUES (244,1,'Campeones Cup','soccer','campeones.cup');",
"INSERT INTO `sportsapi` VALUES (245,1,'United States Open Cup','soccer','usa.open');",
"INSERT INTO `sportsapi` VALUES (246,1,'USL League One','soccer','usa.usl.l1');",
"INSERT INTO `sportsapi` VALUES (247,1,'Scottish Premiership','soccer','sco.1');",
"INSERT INTO `sportsapi` VALUES (248,1,'Chinese Super League','soccer','chn.1');",
"INSERT INTO `sportsapi` VALUES (249,1,'Australian A-League','soccer','aus.1');",
"INSERT INTO `sportsapi` VALUES (250,1,'International Friendly','soccer','fifa.friendly');",
"INSERT INTO `sportsapi` VALUES (251,1,'Women\\'s International Friendly','soccer','fifa.friendly.w');",
"INSERT INTO `sportsapi` VALUES (252,1,'UEFA European Under-21 Championship Qualifying','soccer','uefa.euro_u21_qual');",
"INSERT INTO `sportsapi` VALUES (253,1,'FIFA Women\\'s World Cup','soccer','fifa.wwc');",
"INSERT INTO `sportsapi` VALUES (254,1,'FIFA Club World Cup','soccer','fifa.cwc');",
"INSERT INTO `sportsapi` VALUES (255,1,'CONCACAF Gold Cup Qualifying','soccer','concacaf.gold_qual');",
"INSERT INTO `sportsapi` VALUES (256,1,'CONCACAF Nations League Qualifying','soccer','concacaf.nations.league_qual');",
"INSERT INTO `sportsapi` VALUES (257,1,'CONCACAF Cup','soccer','concacaf.confederations_playoff');",
"INSERT INTO `sportsapi` VALUES (258,1,'UEFA Nations League','soccer','uefa.nations');",
"INSERT INTO `sportsapi` VALUES (259,1,'UEFA European Championship','soccer','uefa.euro');",
"INSERT INTO `sportsapi` VALUES (260,1,'UEFA European Championship Qualifying','soccer','uefa.euroq');",
"INSERT INTO `sportsapi` VALUES (261,1,'Copa America','soccer','conmebol.america');",
"INSERT INTO `sportsapi` VALUES (262,1,'AFC Asian Cup','soccer','afc.asian.cup');",
"INSERT INTO `sportsapi` VALUES (263,1,'AFC Asian Cup Qualifiers','soccer','afc.cupq');",
"INSERT INTO `sportsapi` VALUES (264,1,'Africa Cup of Nations qualifying','soccer','caf.nations_qual');",
"INSERT INTO `sportsapi` VALUES (265,1,'Africa Cup of Nations','soccer','caf.nations');",
"INSERT INTO `sportsapi` VALUES (266,1,'Africa Nations Championship','soccer','caf.championship');",
"INSERT INTO `sportsapi` VALUES (267,1,'WAFU Cup of Nations','soccer','wafu.nations');",
"INSERT INTO `sportsapi` VALUES (268,1,'SheBelieves Cup','soccer','fifa.shebelieves');",
"INSERT INTO `sportsapi` VALUES (269,1,'FIFA Confederations Cup','soccer','fifa.confederations');",
"INSERT INTO `sportsapi` VALUES (270,1,'Non-FIFA Friendly','soccer','nonfifa');",
"INSERT INTO `sportsapi` VALUES (271,1,'Spanish LaLiga 2','soccer','esp.2');",
"INSERT INTO `sportsapi` VALUES (272,1,'Spanish Supercopa','soccer','esp.super_cup');",
"INSERT INTO `sportsapi` VALUES (273,1,'Portuguese Liga Promotion/Relegation Playoffs','soccer','por.1.promotion.relegation');",
"INSERT INTO `sportsapi` VALUES (274,1,'Belgian First Division A - Promotion/Relegation Playoffs','soccer','bel.promotion.relegation');",
"INSERT INTO `sportsapi` VALUES (275,1,'Belgian First Division A','soccer','bel.1');",
"INSERT INTO `sportsapi` VALUES (276,1,'Austrian Bundesliga','soccer','aut.1');",
"INSERT INTO `sportsapi` VALUES (277,1,'Turkish Super Lig','soccer','tur.1');",
"INSERT INTO `sportsapi` VALUES (278,1,'Austrian Bundesliga Promotion/Relegation Playoffs','soccer','aut.promotion.relegation');",
"INSERT INTO `sportsapi` VALUES (279,1,'Greek Super League','soccer','gre.1');",
"INSERT INTO `sportsapi` VALUES (280,1,'Greek Super League Promotion/Relegation Playoffs','soccer','gre.1.promotion.relegation');",
"INSERT INTO `sportsapi` VALUES (281,1,'Swiss Super League','soccer','sui.1');",
"INSERT INTO `sportsapi` VALUES (282,1,'Swiss Super League Promotion/Relegation Playoffs','soccer','sui.1.promotion.relegation');",
"INSERT INTO `sportsapi` VALUES (283,1,'UEFA Women\\'s Champions League','soccer','uefa.wchampions');",
"INSERT INTO `sportsapi` VALUES (284,1,'International Champions Cup','soccer','global.champs_cup');",
"INSERT INTO `sportsapi` VALUES (285,1,'Women\\'s International Champions Cup','soccer','global.wchamps_cup');",
"INSERT INTO `sportsapi` VALUES (286,1,'Club Friendly','soccer','club.friendly');",
"INSERT INTO `sportsapi` VALUES (287,1,'UEFA Champions League Qualifying','soccer','uefa.champions_qual');",
"INSERT INTO `sportsapi` VALUES (288,1,'UEFA Europa Conference League Qualifying','soccer','uefa.europa.conf_qual');",
"INSERT INTO `sportsapi` VALUES (289,1,'UEFA Europa League Qualifying','soccer','uefa.europa_qual');",
"INSERT INTO `sportsapi` VALUES (290,1,'UEFA Super Cup','soccer','uefa.super_cup');",
"INSERT INTO `sportsapi` VALUES (291,1,'English FA Community Shield','soccer','eng.charity');",
"INSERT INTO `sportsapi` VALUES (292,1,'Supercoppa Italiana','soccer','ita.super_cup');",
"INSERT INTO `sportsapi` VALUES (293,1,'French Trophee des Champions','soccer','fra.super_cup');",
"INSERT INTO `sportsapi` VALUES (294,1,'Dutch Johan Cruyff Shield','soccer','ned.supercup');",
"INSERT INTO `sportsapi` VALUES (295,1,'Trofeo Joan Gamper','soccer','esp.joan_gamper');",
"INSERT INTO `sportsapi` VALUES (296,1,'German DFL-Supercup','soccer','ger.super_cup');",
"INSERT INTO `sportsapi` VALUES (297,1,'Audi Cup','soccer','ger.audi_cup');",
"INSERT INTO `sportsapi` VALUES (298,1,'Premier League Asia Trophy','soccer','eng.asia_trophy');",
"INSERT INTO `sportsapi` VALUES (299,1,'Emirates Cup','soccer','friendly.emirates_cup');",
"INSERT INTO `sportsapi` VALUES (300,1,'Japanese J League World Challenge','soccer','jpn.world_challenge');",
"INSERT INTO `sportsapi` VALUES (301,1,'SuperCopa Euroamericana','soccer','euroamericana.supercopa');",
"INSERT INTO `sportsapi` VALUES (302,1,'Men\\'s Olympic Tournament','soccer','fifa.olympics');",
"INSERT INTO `sportsapi` VALUES (303,1,'Women\\'s Olympic Tournament','soccer','fifa.w.olympics');",
"INSERT INTO `sportsapi` VALUES (304,1,'CONMEBOL Pre-Olympic Tournament','soccer','fifa.conmebol.olympicsq');",
"INSERT INTO `sportsapi` VALUES (305,1,'CONCACAF Men\\'s Olympic Qualifying','soccer','fifa.concacaf.olympicsq');",
"INSERT INTO `sportsapi` VALUES (306,1,'CONCACAF Women\\'s Olympic Qualifying Tournament','soccer','fifa.w.concacaf.olympicsq');",
"INSERT INTO `sportsapi` VALUES (307,1,'CONCACAF Women\\'s Championship','soccer','concacaf.womens.championship');",
"INSERT INTO `sportsapi` VALUES (308,1,'FIFA Under-20 World Cup','soccer','fifa.world.u20');",
"INSERT INTO `sportsapi` VALUES (309,1,'FIFA Under-17 World Cup','soccer','fifa.world.u17');",
"INSERT INTO `sportsapi` VALUES (310,1,'Toulon Tournament','soccer','global.toulon');",
"INSERT INTO `sportsapi` VALUES (311,1,'UEFA European Under-21 Championship','soccer','uefa.euro_u21');",
"INSERT INTO `sportsapi` VALUES (312,1,'UEFA European Under-19 Championship','soccer','uefa.euro.u19');",
"INSERT INTO `sportsapi` VALUES (313,1,'Under-21 International Friendly','soccer','fifa.friendly_u21');",
"INSERT INTO `sportsapi` VALUES (314,1,'UEFA Women\\'s European Championship','soccer','uefa.weuro');",
"INSERT INTO `sportsapi` VALUES (315,1,'German Bundesliga Promotion/Relegation Playoff','soccer','ger.playoff.relegation');",
"INSERT INTO `sportsapi` VALUES (316,1,'German 2. Bundesliga Promotion/Relegation Playoffs','soccer','ger.2.promotion.relegation');",
"INSERT INTO `sportsapi` VALUES (317,1,'English Women\\'s FA Cup','soccer','eng.w.fa');",
"INSERT INTO `sportsapi` VALUES (318,1,'English Women\\'s FA Community Shield','soccer','eng.w.charity');",
"INSERT INTO `sportsapi` VALUES (319,1,'English EFL Trophy','soccer','eng.trophy');",
"INSERT INTO `sportsapi` VALUES (320,1,'English National League','soccer','eng.5');",
"INSERT INTO `sportsapi` VALUES (321,1,'English League One','soccer','eng.3');",
"INSERT INTO `sportsapi` VALUES (322,1,'English League Two','soccer','eng.4');",
"INSERT INTO `sportsapi` VALUES (323,1,'Scottish Cup','soccer','sco.tennents');",
"INSERT INTO `sportsapi` VALUES (324,1,'Scottish League Cup','soccer','sco.cis');",
"INSERT INTO `sportsapi` VALUES (325,1,'Scottish Premiership Promotion/Relegation Playoffs','soccer','sco.1.promotion.relegation');",
"INSERT INTO `sportsapi` VALUES (326,1,'Scottish League One','soccer','sco.3');",
"INSERT INTO `sportsapi` VALUES (327,1,'Scottish Championship','soccer','sco.2');",
"INSERT INTO `sportsapi` VALUES (328,1,'Scottish Championship Promotion/Relegation Playoffs','soccer','sco.2.promotion.relegation');",
"INSERT INTO `sportsapi` VALUES (329,1,'Scottish League One Promotion/Relegation Playoffs','soccer','sco.3.promotion.relegation');",
"INSERT INTO `sportsapi` VALUES (330,1,'Scottish League Two Promotion/Relegation Playoffs','soccer','sco.4.promotion.relegation');",
"INSERT INTO `sportsapi` VALUES (331,1,'Scottish League Two','soccer','sco.4');",
"INSERT INTO `sportsapi` VALUES (332,1,'Scottish League Challenge Cup','soccer','sco.challenge');",
"INSERT INTO `sportsapi` VALUES (333,1,'Dutch Eredivisie Promotion/Relegation Playoffs','soccer','ned.playoff.relegation');",
"INSERT INTO `sportsapi` VALUES (334,1,'Dutch Eredivisie Cup','soccer','ned.w.eredivisie_cup');",
"INSERT INTO `sportsapi` VALUES (335,1,'Dutch Keuken Kampioen Divisie','soccer','ned.2');",
"INSERT INTO `sportsapi` VALUES (336,1,'Dutch Tweede Divisie','soccer','ned.3');",
"INSERT INTO `sportsapi` VALUES (337,1,'Dutch KNVB Beker Vrouwen','soccer','ned.w.knvb_cup');",
"INSERT INTO `sportsapi` VALUES (338,1,'Dutch Vrouwen Eredivisie','soccer','ned.w.1');",
"INSERT INTO `sportsapi` VALUES (339,1,'Italian Serie B','soccer','ita.2');",
"INSERT INTO `sportsapi` VALUES (340,1,'French Ligue 1 Promotion/Relegation Playoffs','soccer','fra.1.promotion.relegation');",
"INSERT INTO `sportsapi` VALUES (341,1,'French Ligue 2 Promotion/Relegation Playoffs','soccer','fra.2.promotion.relegation');",
"INSERT INTO `sportsapi` VALUES (342,1,'Swedish Allsvenskan','soccer','swe.1');",
"INSERT INTO `sportsapi` VALUES (343,1,'Swedish Allsvenskanliga Promotion/Relegation Playoffs','soccer','swe.1.promotion.relegation');",
"INSERT INTO `sportsapi` VALUES (344,1,'Danish Superliga','soccer','den.1');",
"INSERT INTO `sportsapi` VALUES (345,1,'Danish SAS-Ligaen Promotion/Relegation Playoffs','soccer','den.promotion.relegation');",
"INSERT INTO `sportsapi` VALUES (346,1,'Romanian Liga 1 Promotion/Relegation Playoffs','soccer','rou.1.promotion.relegation');",
"INSERT INTO `sportsapi` VALUES (347,1,'Romanian Liga 1','soccer','rou.1');",
"INSERT INTO `sportsapi` VALUES (348,1,'Norwegian Eliteserien Promotion/Relegation Playoffs','soccer','nor.1.promotion.relegation');",
"INSERT INTO `sportsapi` VALUES (349,1,'Norwegian Eliteserien','soccer','nor.1');",
"INSERT INTO `sportsapi` VALUES (350,1,'Maltese Premier League','soccer','mlt.1');",
"INSERT INTO `sportsapi` VALUES (351,1,'Israeli Premier League','soccer','isr.1');",
"INSERT INTO `sportsapi` VALUES (352,1,'Irish Premier Division Promotion/Relegation Playoffs','soccer','ir1.1.promotion.relegation');",
"INSERT INTO `sportsapi` VALUES (353,1,'Irish Premier Division','soccer','irl.1');",
"INSERT INTO `sportsapi` VALUES (354,1,'Welsh Premier League','soccer','wal.1');",
"INSERT INTO `sportsapi` VALUES (355,1,'Northern Irish Premiership','soccer','nir.1');",
"INSERT INTO `sportsapi` VALUES (356,1,'CONMEBOL Copa Libertadores','soccer','conmebol.libertadores');",
"INSERT INTO `sportsapi` VALUES (357,1,'CONMEBOL Copa Sudamericana','soccer','conmebol.sudamericana');",
"INSERT INTO `sportsapi` VALUES (358,1,'CONMEBOL Recopa Sudamericana','soccer','conmebol.recopa');",
"INSERT INTO `sportsapi` VALUES (359,1,'Argentine Liga Profesional de Fútbol','soccer','arg.1');",
"INSERT INTO `sportsapi` VALUES (360,1,'Copa Argentina','soccer','arg.copa');",
"INSERT INTO `sportsapi` VALUES (361,1,'Argentine Copa de la Liga Profesional','soccer','arg.copa_lpf');",
"INSERT INTO `sportsapi` VALUES (362,1,'Argentine Copa de la Superliga','soccer','arg.copa_de_la_superliga');",
"INSERT INTO `sportsapi` VALUES (363,1,'Argentine Trofeo de Campeones de la Superliga','soccer','arg.trofeo_de_la_campeones');",
"INSERT INTO `sportsapi` VALUES (364,1,'Argentine Supercopa','soccer','arg.supercopa');",
"INSERT INTO `sportsapi` VALUES (365,1,'Argentine Nacional B','soccer','arg.2');",
"INSERT INTO `sportsapi` VALUES (366,1,'Argentine Primera B','soccer','arg.3');",
"INSERT INTO `sportsapi` VALUES (367,1,'Argentine Primera C','soccer','arg.4');",
"INSERT INTO `sportsapi` VALUES (368,1,'Argentine Primera D','soccer','arg.5');",
"INSERT INTO `sportsapi` VALUES (369,1,'Supercopa Do Brazil','soccer','bra.supercopa_do_brazil');",
"INSERT INTO `sportsapi` VALUES (370,1,'Brazilian Serie A','soccer','bra.1');",
"INSERT INTO `sportsapi` VALUES (371,1,'Brazilian Serie B','soccer','bra.2');",
"INSERT INTO `sportsapi` VALUES (372,1,'Brazilian Serie C','soccer','bra.3');",
"INSERT INTO `sportsapi` VALUES (373,1,'Copa Do Nordeste','soccer','bra.copa_do_nordeste');",
"INSERT INTO `sportsapi` VALUES (374,1,'Brazilian Campeonato Carioca','soccer','bra.camp.carioca');",
"INSERT INTO `sportsapi` VALUES (375,1,'Brazilian Campeonato Paulista','soccer','bra.camp.paulista');",
"INSERT INTO `sportsapi` VALUES (376,1,'Brazilian Campeonato Gaucho','soccer','bra.camp.gaucho');",
"INSERT INTO `sportsapi` VALUES (377,1,'Brazilian Campeonato Mineiro','soccer','bra.camp.mineiro');",
"INSERT INTO `sportsapi` VALUES (378,1,'Chilean Primera División','soccer','chi.1');",
"INSERT INTO `sportsapi` VALUES (379,1,'Copa Chile','soccer','chi.copa_chi');",
"INSERT INTO `sportsapi` VALUES (380,1,'International U20 Friendly','soccer','fifa.u20.friendly');",
"INSERT INTO `sportsapi` VALUES (381,1,'Segunda División de Chile','soccer','chi.2');",
"INSERT INTO `sportsapi` VALUES (382,1,'Chilean Supercopa','soccer','chi.super_cup');",
"INSERT INTO `sportsapi` VALUES (383,1,'Uruguayan Primera Division','soccer','uru.1');",
"INSERT INTO `sportsapi` VALUES (384,1,'Segunda División de Uruguay','soccer','uru.2');",
"INSERT INTO `sportsapi` VALUES (385,1,'Colombian SuperLiga','soccer','col.superliga');",
"INSERT INTO `sportsapi` VALUES (386,1,'Colombian Primera A','soccer','col.1');",
"INSERT INTO `sportsapi` VALUES (387,1,'Colombian Primera B','soccer','col.2');",
"INSERT INTO `sportsapi` VALUES (388,1,'Peruvian Supercopa','soccer','per.supercopa');",
"INSERT INTO `sportsapi` VALUES (389,1,'Copa Colombia','soccer','col.copa');",
"INSERT INTO `sportsapi` VALUES (390,1,'Peruvian Primera Profesional','soccer','per.1');",
"INSERT INTO `sportsapi` VALUES (391,1,'Paraguayan Primera Division','soccer','par.1');",
"INSERT INTO `sportsapi` VALUES (392,1,'Ecuadoran Primera A','soccer','ecu.1');",
"INSERT INTO `sportsapi` VALUES (393,1,'Ecuadoran Supercopa','soccer','ecu.supercopa');",
"INSERT INTO `sportsapi` VALUES (394,1,'Ecuador Serie B','soccer','ecu.2');",
"INSERT INTO `sportsapi` VALUES (395,1,'Venezuelan Primera Profesional','soccer','ven.1');",
"INSERT INTO `sportsapi` VALUES (396,1,'United States NWSL Challenge Cup','soccer','usa.nwsl.cup');",
"INSERT INTO `sportsapi` VALUES (397,1,'Segunda División de Venezuela','soccer','ven.2');",
"INSERT INTO `sportsapi` VALUES (398,1,'Bolivian Liga Profesional','soccer','bol.1');",
"INSERT INTO `sportsapi` VALUES (399,1,'Mexican Supercopa MX','soccer','mex.supercopa');",
"INSERT INTO `sportsapi` VALUES (400,1,'Mexican Campeon de Campeones','soccer','mex.campeon');",
"INSERT INTO `sportsapi` VALUES (401,1,'CONCACAF Champions Cup','soccer','concacaf.champions_cup');",
"INSERT INTO `sportsapi` VALUES (402,1,'CONCACAF U23 Tournament','soccer','concacaf.u23');",
"INSERT INTO `sportsapi` VALUES (403,1,'Honduran Primera Division','soccer','hon.1');",
"INSERT INTO `sportsapi` VALUES (404,1,'Costa Rican Primera Division','soccer','crc.1');",
"INSERT INTO `sportsapi` VALUES (405,1,'Jamaican Premier League','soccer','jam.1');",
"INSERT INTO `sportsapi` VALUES (406,1,'Guatemalan Liga Nacional','soccer','gua.1');",
"INSERT INTO `sportsapi` VALUES (407,1,'Australian W-League','soccer','aus.w.1');",
"INSERT INTO `sportsapi` VALUES (408,1,'Salvadoran Primera Division','soccer','slv.1');",
"INSERT INTO `sportsapi` VALUES (409,1,'AFF Cup','soccer','aff.championship');",
"INSERT INTO `sportsapi` VALUES (410,1,'AFC Cup','soccer','afc.cup');",
"INSERT INTO `sportsapi` VALUES (411,1,'SAFF Championship','soccer','afc.saff.championship');",
"INSERT INTO `sportsapi` VALUES (412,1,'Chinese Super League Promotion/Relegation Playoffs','soccer','chn.1.promotion.relegation');",
"INSERT INTO `sportsapi` VALUES (413,1,'Japanese J League','soccer','jpn.1');",
"INSERT INTO `sportsapi` VALUES (414,1,'Indonesian Liga 1','soccer','idn.1');",
"INSERT INTO `sportsapi` VALUES (415,1,'Indian Super League','soccer','ind.1');",
"INSERT INTO `sportsapi` VALUES (416,1,'Indian I-League','soccer','ind.2');",
"INSERT INTO `sportsapi` VALUES (417,1,'Malaysian Super League','soccer','mys.1');",
"INSERT INTO `sportsapi` VALUES (418,1,'Singaporean Premier League','soccer','sgp.1');",
"INSERT INTO `sportsapi` VALUES (419,1,'Thai League 1','soccer','tha.1');",
"INSERT INTO `sportsapi` VALUES (420,1,'Bangabandhu Cup','soccer','bangabandhu.cup');",
"INSERT INTO `sportsapi` VALUES (421,1,'COSAFA Cup','soccer','caf.cosafa');",
"INSERT INTO `sportsapi` VALUES (422,1,'CAF Champions League','soccer','caf.champions');",
"INSERT INTO `sportsapi` VALUES (423,1,'South African Premiership Promotion/Relegation Playoffs','soccer','rsa.1.promotion.relegation');",
"INSERT INTO `sportsapi` VALUES (424,1,'CAF Confederations Cup','soccer','caf.confed');",
"INSERT INTO `sportsapi` VALUES (425,1,'South African Premiership','soccer','rsa.1');",
"INSERT INTO `sportsapi` VALUES (426,1,'South African First Division','soccer','rsa.2');",
"INSERT INTO `sportsapi` VALUES (427,1,'South African Telkom Knockout','soccer','rsa.telkom_knockout');",
"INSERT INTO `sportsapi` VALUES (428,1,'South African Nedbank Cup','soccer','rsa.nedbank');",
"INSERT INTO `sportsapi` VALUES (429,1,'MTN 8 Cup','soccer','rsa.mtn8');",
"INSERT INTO `sportsapi` VALUES (430,1,'Nigerian Professional League','soccer','nga.1');",
"INSERT INTO `sportsapi` VALUES (431,1,'Ghanaian Premier League','soccer','gha.1');",
"INSERT INTO `sportsapi` VALUES (432,1,'Zambian Super League','soccer','zam.1');",
"INSERT INTO `sportsapi` VALUES (433,1,'Kenyan Premier League','soccer','ken.1');",
"INSERT INTO `sportsapi` VALUES (434,1,'Zimbabwean Premier Soccer League','soccer','zim.1');",
"INSERT INTO `sportsapi` VALUES (435,1,'Ugandan Premier League','soccer','uga.1');",
"INSERT INTO `sportsapi` VALUES (436,1,'Misc. U.S. Soccer Games','soccer','generic.ussf');",
"INSERT INTO `sportsapi` VALUES (1000,2,'Major League Baseball','baseball','mlb');",
"INSERT INTO `sportscleanup` VALUES (1,1,1000,'soccer','(SE)','\\\\(\\\\w+\\\\)',0,'');",
"INSERT INTO `sportscleanup` VALUES (2,1,1000,'soccer','AFC','\\\\AA\\\\.?F\\\\.?C\\\\.?|\\\\bA\\\\.?F\\\\.?C\\\\.?\\\\Z',0,'');",
"INSERT INTO `sportscleanup` VALUES (3,1,1000,'soccer','AC etc.','\\\\AA\\\\.?[AC]\\\\.?|\\\\bA\\\\.?[AC]\\\\.?\\\\Z',0,'');",
"INSERT INTO `sportscleanup` VALUES (4,1,1000,'soccer','BK','\\\\AB\\\\.?K\\\\.?|\\\\bB\\\\.?K\\\\.?\\\\Z',0,'');",
"INSERT INTO `sportscleanup` VALUES (5,1,1000,'soccer','BSC','\\\\AB\\\\.?S\\\\.?C\\\\.?|\\\\bB\\\\.?S\\\\.?C\\\\.?\\\\Z',0,'');",
"INSERT INTO `sportscleanup` VALUES (6,1,1000,'soccer','CSyD','\\\\AC\\\\.?S\\\\.?( y )?D\\\\.?|\\\\bC\\\\.?S\\\\.?( y )?D\\\\.?\\\\Z',0,'');",
"INSERT INTO `sportscleanup` VALUES (7,1,1000,'soccer','CD etc.','\\\\AC\\\\.?[ADFRS]\\\\.?|\\\\bC\\\\.?[ADFRS]\\\\.?\\\\Z',0,'');",
"INSERT INTO `sportscleanup` VALUES (8,1,1000,'soccer','FC','\\\\AF\\\\.?C\\\\.?|\\\\bF\\\\.?C\\\\.?\\\\Z',0,'');",
"INSERT INTO `sportscleanup` VALUES (9,1,1000,'soccer','HSC','\\\\AH\\\\.?S\\\\.?C\\\\.?|\\\\bH\\\\.?S\\\\.?C\\\\.?\\\\Z',0,'');",
"INSERT INTO `sportscleanup` VALUES (10,1,1000,'soccer','RC etc.','\\\\AR\\\\.?[BC]\\\\.?|\\\\bR\\\\.?[BC]\\\\.?\\\\Z',0,'');",
"INSERT INTO `sportscleanup` VALUES (11,1,1000,'soccer','SC etc.','\\\\AS\\\\.?[ACV]\\\\.?|\\\\bS\\\\.?[ACV]\\\\.?\\\\Z',0,'');",
"INSERT INTO `sportscleanup` VALUES (12,1,1000,'soccer','TSG','\\\\AT\\\\.?S\\\\.?G\\\\.?|\\\\bT\\\\.?S\\\\.?G\\\\.?\\\\Z',0,'');",
"INSERT INTO `sportscleanup` VALUES (13,1,1000,'soccer','VFB etc.','\\\\AV\\\\.?F\\\\.?[BL]\\\\.?|\\\\bV\\\\.?F\\\\.?[BL]\\\\.?\\\\Z',0,'');",
"INSERT INTO `sportscleanup` VALUES (14,1,2000,'all','','Inglaterra',0,'England');",
"INSERT INTO `sportscleanup` VALUES (15,1,2000,'all','','Munchen',0,'Munich');",
"INSERT INTO `sportscleanup` VALUES (16,1,1100,'basketball','Cal State','Cal State',0,'CSU');",
"INSERT INTO `sportscleanup` VALUES (17,1,1000,'basketball','Grambling','Grambling State',0,'Grambling');",
"INSERT INTO `sportscleanup` VALUES (18,1,1000,'basketball','Hawaii','Hawaii',0,'Hawai\\'i');",
"INSERT INTO `sportscleanup` VALUES (19,1,1000,'basketball','LIU','LIU',0,'Long Island University');",
"INSERT INTO `sportscleanup` VALUES (20,1,1100,'basketball','Loyola','Loyola-',0,'Loyola ');",
"INSERT INTO `sportscleanup` VALUES (21,1,1000,'basketball','Loyola (Md.)','Loyola (Md.)',0,'Loyola (MD)');",
"INSERT INTO `sportscleanup` VALUES (22,1,1000,'basketball','McNeese','McNeese State',0,'McNeese');",
"INSERT INTO `sportscleanup` VALUES (23,1,1000,'basketball','Miami (OH)','Miami (Ohio)',0,'Miami (OH)');",
"INSERT INTO `sportscleanup` VALUES (24,1,1000,'basketball','UAB','Alabama-Birmingham',0,'UAB');",
"INSERT INTO `sportscleanup` VALUES (25,1,1000,'basketball','UConn','Connecticut',0,'UConn');",
"INSERT INTO `sportscleanup` VALUES (26,1,1000,'basketball','UMass','Massachusetts',0,'UMass');",
"INSERT INTO `sportscleanup` VALUES (27,1,1100,'basketball','UNC','UNC-',0,'UNC ');",
"INSERT INTO `sportscleanup` VALUES (28,1,1000,'basketball','UTEP','Texas-El Paso',0,'UTEP');",
"INSERT INTO `sportscleanup` VALUES (29,1,1100,'basketball','Texas','Texas-',0,'UT ');",
"INSERT INTO `sportscleanup` VALUES (30,1,1000,'basketball','Chattanooga','UT-Chattanooga',0,'Chattanooga');",
"INSERT INTO `sportscleanup` VALUES (31,1,1100,'basketball','UT','UT-',0,'UT ');",
"INSERT INTO `sportslisting` VALUES (1,1,'\\\\A(?:MLB Baseball)\\\\z');",
"INSERT INTO `sportslisting` VALUES (2,1,'\\\\A(?:Béisbol MLB)\\\\z');",
"INSERT INTO `sportslisting` VALUES (3,1,'\\\\A(?:MLB All-Star Game)\\\\z');",
"INSERT INTO `sportslisting` VALUES (4,1,'\\\\A(?:World Series)\\\\z');",
"INSERT INTO `sportslisting` VALUES (5,2,'\\\\A(?:College Baseball)\\\\z');",
"INSERT INTO `sportslisting` VALUES (6,2,'\\\\A(?:College World Series)\\\\z');",
"INSERT INTO `sportslisting` VALUES (7,3,'\\\\A(?:College Softball)\\\\z');",
"INSERT INTO `sportslisting` VALUES (8,4,'\\\\A(?:Women\\'s College World Series)\\\\z');",
"INSERT INTO `sportslisting` VALUES (9,10,'\\\\A(?:Béisbol Liga Mexicana)\\\\z');",
"INSERT INTO `sportslisting` VALUES (10,20,'\\\\A(?:N\\\\w+ Football)\\\\z');",
"INSERT INTO `sportslisting` VALUES (11,20,'\\\\A(?:Super Bowl( [CLXVI]+)?)\\\\z');",
"INSERT INTO `sportslisting` VALUES (12,20,'\\\\A(?:Fútbol Americano NFL)\\\\z');",
"INSERT INTO `sportslisting` VALUES (13,21,'\\\\A(?:College Football)\\\\z');",
"INSERT INTO `sportslisting` VALUES (14,21,'\\\\A(?:\\\\w+ Bowl)\\\\z');",
"INSERT INTO `sportslisting` VALUES (15,21,'\\\\A(?:Fútbol Americano de Universitario)\\\\z');",
"INSERT INTO `sportslisting` VALUES (16,40,'\\\\A(?:NBA Basketball)\\\\z');",
"INSERT INTO `sportslisting` VALUES (17,40,'\\\\A(?:NBA Finals)\\\\z');",
"INSERT INTO `sportslisting` VALUES (18,41,'\\\\A(?:WNBA Basketball)\\\\z');",
"INSERT INTO `sportslisting` VALUES (19,41,'\\\\A(?:WNBA Finals)\\\\z');",
"INSERT INTO `sportslisting` VALUES (20,42,'\\\\A(?:College Basketball)\\\\z');",
"INSERT INTO `sportslisting` VALUES (21,42,'\\\\A(?:NCAA Basketball Tournament)\\\\z');",
"INSERT INTO `sportslisting` VALUES (22,43,'\\\\A(?:Women\\'s College Basketball)\\\\z');",
"INSERT INTO `sportslisting` VALUES (23,43,'\\\\A(?:NCAA Women\\'s Basketball Tournament)\\\\z');",
"INSERT INTO `sportslisting` VALUES (24,60,'\\\\A(?:NHL Hockey)\\\\z');",
"INSERT INTO `sportslisting` VALUES (25,60,'\\\\A(?:NHL Winter Classic)\\\\z');",
"INSERT INTO `sportslisting` VALUES (26,60,'\\\\A(?:NHL \\\\w+ Conference Final)\\\\z');",
"INSERT INTO `sportslisting` VALUES (27,60,'\\\\A(?:Stanley Cup Finals)\\\\z');",
"INSERT INTO `sportslisting` VALUES (28,61,'\\\\A(?:College Hockey)\\\\z');",
"INSERT INTO `sportslisting` VALUES (29,61,'\\\\A(?:Frozen Four)\\\\z');",
"INSERT INTO `sportslisting` VALUES (30,62,'\\\\A(?:Women\\'s College Hockey)\\\\z');",
"INSERT INTO `sportslisting` VALUES (31,66,'\\\\A(?:College Field Hockey)\\\\z');",
"INSERT INTO `sportslisting` VALUES (32,80,'\\\\A(?:College Volleyball)\\\\z');",
"INSERT INTO `sportslisting` VALUES (33,81,'\\\\A(?:Women\\'s College Volleyball)\\\\z');",
"INSERT INTO `sportslisting` VALUES (34,100,'\\\\A(?:College Lacrosse)\\\\z');",
"INSERT INTO `sportslisting` VALUES (35,101,'\\\\A(?:Women\\'s College Lacrosse)\\\\z');",
"INSERT INTO `sportslisting` VALUES (36,120,'\\\\A(?:College Water Polo)\\\\z');",
"INSERT INTO `sportslisting` VALUES (37,121,'\\\\A(?:Women\\'s College Water Polo)\\\\z');",
"INSERT INTO `sportslisting` VALUES (38,200,'\\\\A(?:College Soccer)\\\\z');",
"INSERT INTO `sportslisting` VALUES (39,201,'\\\\A(?:Women\\'s College Soccer)\\\\z');",
"INSERT INTO `sportslisting` VALUES (40,202,'\\\\A(?:MLS Soccer|Fútbol MLS)\\\\z');",
"INSERT INTO `sportslisting` VALUES (41,203,'\\\\A(?:(Premier League|EPL) Soccer)\\\\z');",
"INSERT INTO `sportslisting` VALUES (42,203,'\\\\A(?:English Premier League)\\\\z');",
"INSERT INTO `sportslisting` VALUES (43,203,'\\\\A(?:Fútbol Premier League)\\\\z');",
"INSERT INTO `sportslisting` VALUES (44,205,'\\\\A(?:Italian Serie A Soccer)\\\\z');",
"INSERT INTO `sportslisting` VALUES (45,339,'\\\\A(?:Italian Serie B Soccer)\\\\z');",
"INSERT INTO `sportslisting` VALUES (46,206,'\\\\A(?:French Ligue 1 Soccer|Fútbol Ligue 1|Fútbol Liga 1)\\\\z');",
"INSERT INTO `sportslisting` VALUES (47,207,'\\\\A(?:French Ligue 2 Soccer|Fútbol Ligue 2|Fútbol Liga 2)\\\\z');",
"INSERT INTO `sportslisting` VALUES (48,208,'\\\\A(?:Fútbol LaLiga)\\\\z');",
"INSERT INTO `sportslisting` VALUES (49,208,'\\\\A(?:Fútbol Español Primera Division)\\\\z');",
"INSERT INTO `sportslisting` VALUES (50,208,'\\\\A(?:Spanish Primera Division Soccer)\\\\z');",
"INSERT INTO `sportslisting` VALUES (51,209,'\\\\A(?:(German )?Bundesliga Soccer)\\\\z');",
"INSERT INTO `sportslisting` VALUES (52,209,'\\\\A(?:Fútbol Bundesliga)\\\\z');",
"INSERT INTO `sportslisting` VALUES (53,209,'\\\\A(?:Fútbol Copa de Alemania)\\\\z');",
"INSERT INTO `sportslisting` VALUES (54,210,'\\\\A(?:German 2. Bundesliga Soccer)\\\\z');",
"INSERT INTO `sportslisting` VALUES (55,211,'\\\\A(?:Fútbol Mexicano Primera División|Fútbol Mexicano Liga Premier|Fútbol Mexicano)\\\\z');",
"INSERT INTO `sportslisting` VALUES (56,211,'\\\\A(?:Mexico Primera Division Soccer)\\\\z');",
"INSERT INTO `sportslisting` VALUES (57,212,'\\\\A(?:Copa do Brazil Soccer)\\\\z');",
"INSERT INTO `sportslisting` VALUES (58,214,'\\\\A(?:CONCACAF League Soccer)\\\\z');",
"INSERT INTO `sportslisting` VALUES (59,215,'\\\\A(?:CONCACAF Champions League Soccer)\\\\z');",
"INSERT INTO `sportslisting` VALUES (60,216,'\\\\A(?:CONCACAF Nations League Soccer)\\\\z');",
"INSERT INTO `sportslisting` VALUES (61,217,'\\\\A(?:CONCACAF Gold Cup Soccer)\\\\z');",
"INSERT INTO `sportslisting` VALUES (62,218,'\\\\A(?:FIFA World Cup Soccer)\\\\z');",
"INSERT INTO `sportslisting` VALUES (63,219,'\\\\A(?:FIFA World Cup Qualifying( Soccer)?|FIFA Eliminatorias Copa Mundial)\\\\z');",
"INSERT INTO `sportslisting` VALUES (64,220,'\\\\A(?:FIFA World Cup Qualifying( Soccer)?|FIFA Eliminatorias Copa Mundial)\\\\z');",
"INSERT INTO `sportslisting` VALUES (65,221,'\\\\A(?:FIFA World Cup Qualifying( Soccer)?|FIFA Eliminatorias Copa Mundial)\\\\z');",
"INSERT INTO `sportslisting` VALUES (66,222,'\\\\A(?:FIFA World Cup Qualifying( Soccer)?|FIFA Eliminatorias Copa Mundial)\\\\z');",
"INSERT INTO `sportslisting` VALUES (67,223,'\\\\A(?:FIFA World Cup Qualifying( Soccer)?|FIFA Eliminatorias Copa Mundial)\\\\z');",
"INSERT INTO `sportslisting` VALUES (68,224,'\\\\A(?:FIFA World Cup Qualifying( Soccer)?|FIFA Eliminatorias Copa Mundial)\\\\z');",
"INSERT INTO `sportslisting` VALUES (69,225,'\\\\A(?:Fútbol UEFA Champions League)\\\\z');",
"INSERT INTO `sportslisting` VALUES (70,225,'\\\\A(?:UEFA Champions League Soccer)\\\\z');",
"INSERT INTO `sportslisting` VALUES (71,226,'\\\\A(?:Fútbol UEFA Europa League)\\\\z');",
"INSERT INTO `sportslisting` VALUES (72,229,'\\\\A(?:Fútbol USL Championship)\\\\z');",
"INSERT INTO `sportslisting` VALUES (73,229,'\\\\A(?:USL Championship Soccer)\\\\z');",
"INSERT INTO `sportslisting` VALUES (74,230,'\\\\A(?:NWSL Soccer)\\\\z');",
"INSERT INTO `sportslisting` VALUES (75,231,'\\\\A(?:FA Women\\'s Super League)\\\\z');",
"INSERT INTO `sportslisting` VALUES (76,242,'\\\\A(?:Fútbol Mexicano Liga Expansión)\\\\z');",
"INSERT INTO `sportslisting` VALUES (77,258,'\\\\A(?:UEFA Nations League Soccer)\\\\z');",
"INSERT INTO `sportslisting` VALUES (78,258,'\\\\A(?:Fútbol UEFA Nations League)\\\\z');",
"INSERT INTO `sportslisting` VALUES (79,271,'\\\\A(?:Fútbol Español Segunda Division)\\\\z');",
"INSERT INTO `sportslisting` VALUES (80,277,'\\\\A(?:Fútbol Turco Superliga)\\\\z');",
"INSERT INTO `sportslisting` VALUES (81,277,'\\\\A(?:Turkish Super Lig Soccer)\\\\z');",
"INSERT INTO `sportslisting` VALUES (82,279,'\\\\A(?:Superleague Greek Soccer)\\\\z');",
"INSERT INTO `sportslisting` VALUES (83,356,'\\\\A(?:Fútbol CONMEBOL Libertadores)\\\\z');",
"INSERT INTO `sportslisting` VALUES (84,357,'\\\\A(?:Fútbol CONMEBOL Sudamericana)\\\\z');",
"INSERT INTO `sportslisting` VALUES (85,359,'\\\\A(?:Fútbol Argentino Primera División)\\\\z');",
"INSERT INTO `sportslisting` VALUES (86,360,'\\\\A(?:Fútbol Copa Argentina)\\\\z');",
"INSERT INTO `sportslisting` VALUES (87,365,'\\\\A(?:Fútbol Argentino Primera Nacional( B)?)\\\\z');",
"INSERT INTO `sportslisting` VALUES (88,366,'\\\\A(?:Fútbol Argentino Primera B)\\\\z');",
"INSERT INTO `sportslisting` VALUES (89,367,'\\\\A(?:Fútbol Argentino Primera C)\\\\z');",
"INSERT INTO `sportslisting` VALUES (90,368,'\\\\A(?:Fútbol Argentino Primera D)\\\\z');",
"INSERT INTO `sportslisting` VALUES (91,386,'\\\\A(?:Fútbol Columbiano Primera División)\\\\z');",
"INSERT INTO `sportslisting` VALUES (92,403,'\\\\A(?:Fútbol Hondureño Primera División)\\\\z');",
"INSERT INTO `sportslisting` VALUES (93,404,'\\\\A(?:Fútbol Costarricense Primera División)\\\\z');",
"INSERT INTO `sportslisting` VALUES (94,1000,'\\\\A(?:MLB Baseball)\\\\z');",
"INSERT INTO `sportslisting` VALUES (95,1000,'\\\\A(?:Béisbol MLB)\\\\z');",
"INSERT INTO `users` VALUES (1,'admin','bcd911b2ecb15ffbd6d8e6e744d60cf6','0000-00-00 00:00:00');",
"INSERT INTO `videotypes` VALUES (1,'txt','',1,0);",
"INSERT INTO `videotypes` VALUES (2,'log','',1,0);",
"INSERT INTO `videotypes` VALUES (3,'mpg','Internal',0,0);",
"INSERT INTO `videotypes` VALUES (4,'avi','',0,1);",
"INSERT INTO `videotypes` VALUES (5,'vob','Internal',0,0);",
"INSERT INTO `videotypes` VALUES (6,'mpeg','Internal',0,0);",
"INSERT INTO `videotypes` VALUES (8,'iso','Internal',0,0);",
"INSERT INTO `videotypes` VALUES (9,'img','Internal',0,0);",
"INSERT INTO `videotypes` VALUES (10,'mkv','Internal',0,0);",
"INSERT INTO `videotypes` VALUES (11,'mp4','Internal',0,0);",
"INSERT INTO `videotypes` VALUES (12,'m2ts','Internal',0,0);",
"INSERT INTO `videotypes` VALUES (13,'evo','Internal',0,0);",
"INSERT INTO `videotypes` VALUES (14,'divx','Internal',0,0);",
"INSERT INTO `videotypes` VALUES (15,'mov','Internal',0,0);",
"INSERT INTO `videotypes` VALUES (16,'qt','Internal',0,0);",
"INSERT INTO `videotypes` VALUES (17,'wmv','Internal',0,0);",
"INSERT INTO `videotypes` VALUES (18,'3gp','Internal',0,0);",
"INSERT INTO `videotypes` VALUES (19,'asf','Internal',0,0);",
"INSERT INTO `videotypes` VALUES (20,'ogg','Internal',0,0);",
"INSERT INTO `videotypes` VALUES (21,'ogm','Internal',0,0);",
"INSERT INTO `videotypes` VALUES (22,'flv','Internal',0,0);",
"INSERT INTO `videotypes` VALUES (23,'ogv','Internal',0,0);",
"INSERT INTO `videotypes` VALUES (25,'nut','Internal',0,0);",
"INSERT INTO `videotypes` VALUES (26,'mxf','Internal',0,0);",
"INSERT INTO `videotypes` VALUES (27,'m4v','Internal',0,0);",
"INSERT INTO `videotypes` VALUES (28,'rm','Internal',0,0);",
"INSERT INTO `videotypes` VALUES (29,'ts','Internal',0,0);",
"INSERT INTO `videotypes` VALUES (30,'swf','Internal',0,0);",
"INSERT INTO `videotypes` VALUES (31,'f4v','Internal',0,0);",
"INSERT INTO `videotypes` VALUES (32,'nuv','Internal',0,0);"

// NOLINTEND(modernize-raw-string-literal)

};

    QString dbver = "";
    if (!performActualUpdate("MythTV", "DBSchemaVer",
                             updates, "1382", dbver))
        return false;

    GetMythDB()->SetHaveSchema(true);

    return true;
}

DBUpdates getRecordingExtenderDbInfo (int version)
{
    switch (version)
    {
      case -1:
        return {
            R"(DROP TABLE IF EXISTS sportscleanup;)",
            R"(DROP TABLE IF EXISTS sportslisting;)",
            R"(DROP TABLE IF EXISTS sportsapi;)",
        };

      case 0:
        return {
            R"(ALTER TABLE record ADD COLUMN autoextend
                TINYINT UNSIGNED DEFAULT 0;)",
        };

      case 1:
        return {
            R"(CREATE TABLE sportsapi (
              id INT UNSIGNED PRIMARY KEY,
              provider TINYINT UNSIGNED DEFAULT 0,
              name VARCHAR(128) NOT NULL,
              key1 VARCHAR(64) NOT NULL,
              key2 VARCHAR(64) NOT NULL,
              UNIQUE(provider,key1(25),key2(50))
              ) ENGINE=MyISAM DEFAULT CHARSET=utf8;)",
            R"(INSERT INTO sportsapi
            VALUES
              (   1,1,'Major League Baseball','baseball','mlb'),
              (   2,1,'NCAA Men''s Baseball','baseball','college-baseball'),
              (   3,1,'NCAA Women''s Softball','baseball','college-softball'),
              (   4,1,'Olympic Men''s Baseball','baseball','olympics-baseball'),
              (   5,1,'World Baseball Classic','baseball','world-baseball-classic'),
              (   6,1,'Little League Baseball','baseball','llb'),
              (   7,1,'Caribbean Series','baseball','caribbean-series'),
              (   8,1,'Dominican Winter League','baseball','dominican-winter-league'),
              (   9,1,'Venezuelan Winter League','baseball','venezuelan-winter-league'),
              (  10,1,'Mexican League','baseball','mexican-winter-league'),
              (  11,1,'Puerto Rican Winter League','baseball','puerto-rican-winter-league');)",

            R"(INSERT INTO sportsapi
            VALUES
              (  20,1,'National Football League','football','nfl'),
              (  21,1,'NCAA - Football','football','college-football'),
              (  22,1,'XFL','football','xfl'),
              (  23,1,'Canadian Football League','football','cfl');)",

            R"(INSERT INTO sportsapi
            VALUES
              (  40,1,'National Basketball Association','basketball','nba'),
              (  41,1,'Women''s National Basketball Association','basketball','wnba'),
              (  42,1,'NCAA Men''s Basketball','basketball','mens-college-basketball'),
              (  43,1,'NCAA Women''s Basketball','basketball','womens-college-basketball'),
              (  44,1,'Olympics Men''s Basketball','basketball','mens-olympic-basketball'),
              (  45,1,'Olympics Women''s Basketball','basketball','womens-olympic-basketball'),
              (  46,1,'National Basketball Association Summer League Las Vegas','basketball','nba-summer-las-vegas'),
              (  47,1,'National Basketball Association Summer League Utah','basketball','nba-summer-utah'),
              (  48,1,'National Basketball Association Summer League Orlando','basketball','nba-summer-orlando'),
              (  49,1,'National Basketball Association Summer League Sacramento','basketball','nba-summer-sacramento'),
              (  50,1,'NBA G League','basketball','nba-development'),
              (  51,1,'International Basketball Federation','basketball','fiba');)",

            R"(INSERT INTO sportsapi
            VALUES
              (  60,1,'National Hockey League','hockey','nfl'),
              (  61,1,'NCAA Men''s Ice Hockey','hockey','mens-college-hockey'),
              (  62,1,'NCAA Women''s Hockey','hockey','womens-college-hockey'),
              (  63,1,'World Cup of Hockey','hockey','hockey-world-cup'),
              (  64,1,'Men''s Olympic Ice Hockey','hockey','mens-olympic-hockey'),
              (  65,1,'Women''s Olympic Ice Hockey','hockey','womens-olympic-hockey'),
              (  66,1,'NCAA Women''s Field Hockey','field-hockey','womens-college-field-hockey');)",

            R"(INSERT INTO sportsapi
            VALUES
              (  80,1,'NCAA Men''s Volleyball','volleyball','mens-college-volleyball'),
              (  81,1,'NCAA Women''s Volleyball','volleyball','womens-college-volleyball');)",

            R"(INSERT INTO sportsapi
            VALUES
              ( 100,1,'NCAA Men''s Lacrosse','lacrosse','mens-college-lacrosse'),
              ( 101,1,'NCAA Women''s Lacrosse','lacrosse','womens-college-lacrosse');)",

            R"(INSERT INTO sportsapi
            VALUES
              ( 120,1,'NCAA Men''s Water Polo','water-polo','mens-college-water-polo'),
              ( 121,1,'NCAA Women''s Water Polo','water-polo','womens-college-water-polo');)",

            R"(INSERT INTO sportsapi
            VALUES
              ( 200,1,'NCAA Men''s Soccer','soccer','usa.ncaa.m.1'),
              ( 201,1,'NCAA Women''s Soccer','soccer','usa.ncaa.w.1'),
              ( 202,1,'Major League Soccer','soccer','usa.1'),
              ( 203,1,'English Premier League','soccer','eng.1'),
              ( 204,1,'English League Championship','soccer','eng.2'),
              ( 205,1,'Italian Serie A','soccer','ita.1'),
              ( 206,1,'French Ligue 1','soccer','fra.1'),
              ( 207,1,'French Ligue 2','soccer','fra.2'),
              ( 208,1,'Spanish LaLiga','soccer','esp.1'),
              ( 209,1,'German Bundesliga','soccer','ger.1'),
              ( 210,1,'German 2. Bundesliga','soccer','ger.2'),
              ( 211,1,'Mexican Liga BBVA MX','soccer','mex.1'),
              ( 212,1,'Copa Do Brasil','soccer','bra.copa_do_brazil'),
              ( 213,1,'CONCACAF Leagues Cup','soccer','concacaf.leagues.cup'),
              ( 214,1,'CONCACAF League','soccer','concacaf.league'),
              ( 215,1,'CONCACAF Champions League','soccer','concacaf.champions'),
              ( 216,1,'CONCACAF Nations League','soccer','concacaf.nations.league'),
              ( 217,1,'CONCACAF Gold Cup','soccer','concacaf.gold'),
              ( 218,1,'FIFA World Cup','soccer','fifa.world'),
              ( 219,1,'FIFA World Cup Qualifying - UEFA','soccer','fifa.worldq.uefa'),
              ( 220,1,'FIFA World Cup Qualifying - CONCACAF','soccer','fifa.worldq.concacaf'),
              ( 221,1,'FIFA World Cup Qualifying - CONMEBOL','soccer','fifa.worldq.conmebol'),
              ( 222,1,'FIFA World Cup Qualifying - AFC','soccer','fifa.worldq.afc'),
              ( 223,1,'FIFA World Cup Qualifying - CAF','soccer','fifa.worldq.caf'),
              ( 224,1,'FIFA World Cup Qualifying - OFC','soccer','fifa.worldq.ofc'),
              ( 225,1,'UEFA Champions League','soccer','uefa.champions'),
              ( 226,1,'UEFA Europa League','soccer','uefa.europa'),
              ( 227,1,'UEFA Europa Conference League','soccer','uefa.europa.conf'),
              ( 228,1,'English Carabao Cup','soccer','eng.league_cup'),
              ( 229,1,'USL Championship','soccer','usa.usl.1'),
              ( 230,1,'United States NWSL','soccer','usa.nwsl'),
              ( 231,1,'FA Women''s Super League','soccer','eng.w.1'),
              ( 232,1,'English FA Cup','soccer','eng.fa'),
              ( 233,1,'Spanish Copa del Rey','soccer','esp.copa_del_rey'),
              ( 234,1,'German DFB Pokal','soccer','ger.dfb_pokal'),
              ( 235,1,'Italian Coppa Italia','soccer','ita.coppa_italia'),
              ( 236,1,'French Coupe de France','soccer','fra.coupe_de_france'),
              ( 237,1,'AFC Champions League','soccer','afc.champions'),
              ( 238,1,'Dutch KNVB Beker','soccer','ned.cup'),
              ( 239,1,'Dutch Eredivisie','soccer','ned.1'),
              ( 240,1,'Portuguese Liga','soccer','por.1'),
              ( 241,1,'Russian Premier League','soccer','rus.1'),
              ( 242,1,'Mexican Liga de Expansión MX','soccer','mex.2'),
              ( 243,1,'Mexican Copa MX','soccer','mex.copa_mx'),
              ( 244,1,'Campeones Cup','soccer','campeones.cup'),
              ( 245,1,'United States Open Cup','soccer','usa.open'),
              ( 246,1,'USL League One','soccer','usa.usl.l1'),
              ( 247,1,'Scottish Premiership','soccer','sco.1'),
              ( 248,1,'Chinese Super League','soccer','chn.1'),
              ( 249,1,'Australian A-League','soccer','aus.1'),
              ( 250,1,'International Friendly','soccer','fifa.friendly'),
              ( 251,1,'Women''s International Friendly','soccer','fifa.friendly.w'),
              ( 252,1,'UEFA European Under-21 Championship Qualifying','soccer','uefa.euro_u21_qual'),
              ( 253,1,'FIFA Women''s World Cup','soccer','fifa.wwc'),
              ( 254,1,'FIFA Club World Cup','soccer','fifa.cwc'),
              ( 255,1,'CONCACAF Gold Cup Qualifying','soccer','concacaf.gold_qual'),
              ( 256,1,'CONCACAF Nations League Qualifying','soccer','concacaf.nations.league_qual'),
              ( 257,1,'CONCACAF Cup','soccer','concacaf.confederations_playoff'),
              ( 258,1,'UEFA Nations League','soccer','uefa.nations'),
              ( 259,1,'UEFA European Championship','soccer','uefa.euro'),
              ( 260,1,'UEFA European Championship Qualifying','soccer','uefa.euroq'),
              ( 261,1,'Copa America','soccer','conmebol.america'),
              ( 262,1,'AFC Asian Cup','soccer','afc.asian.cup'),
              ( 263,1,'AFC Asian Cup Qualifiers','soccer','afc.cupq'),
              ( 264,1,'Africa Cup of Nations qualifying','soccer','caf.nations_qual'),
              ( 265,1,'Africa Cup of Nations','soccer','caf.nations'),
              ( 266,1,'Africa Nations Championship','soccer','caf.championship'),
              ( 267,1,'WAFU Cup of Nations','soccer','wafu.nations'),
              ( 268,1,'SheBelieves Cup','soccer','fifa.shebelieves'),
              ( 269,1,'FIFA Confederations Cup','soccer','fifa.confederations'),
              ( 270,1,'Non-FIFA Friendly','soccer','nonfifa'),
              ( 271,1,'Spanish LaLiga 2','soccer','esp.2'),
              ( 272,1,'Spanish Supercopa','soccer','esp.super_cup'),
              ( 273,1,'Portuguese Liga Promotion/Relegation Playoffs','soccer','por.1.promotion.relegation'),
              ( 274,1,'Belgian First Division A - Promotion/Relegation Playoffs','soccer','bel.promotion.relegation'),
              ( 275,1,'Belgian First Division A','soccer','bel.1'),
              ( 276,1,'Austrian Bundesliga','soccer','aut.1'),
              ( 277,1,'Turkish Super Lig','soccer','tur.1'),
              ( 278,1,'Austrian Bundesliga Promotion/Relegation Playoffs','soccer','aut.promotion.relegation'),
              ( 279,1,'Greek Super League','soccer','gre.1'),
              ( 280,1,'Greek Super League Promotion/Relegation Playoffs','soccer','gre.1.promotion.relegation'),
              ( 281,1,'Swiss Super League','soccer','sui.1'),
              ( 282,1,'Swiss Super League Promotion/Relegation Playoffs','soccer','sui.1.promotion.relegation'),
              ( 283,1,'UEFA Women''s Champions League','soccer','uefa.wchampions'),
              ( 284,1,'International Champions Cup','soccer','global.champs_cup'),
              ( 285,1,'Women''s International Champions Cup','soccer','global.wchamps_cup'),
              ( 286,1,'Club Friendly','soccer','club.friendly'),
              ( 287,1,'UEFA Champions League Qualifying','soccer','uefa.champions_qual'),
              ( 288,1,'UEFA Europa Conference League Qualifying','soccer','uefa.europa.conf_qual'),
              ( 289,1,'UEFA Europa League Qualifying','soccer','uefa.europa_qual'),
              ( 290,1,'UEFA Super Cup','soccer','uefa.super_cup'),
              ( 291,1,'English FA Community Shield','soccer','eng.charity'),
              ( 292,1,'Supercoppa Italiana','soccer','ita.super_cup'),
              ( 293,1,'French Trophee des Champions','soccer','fra.super_cup'),
              ( 294,1,'Dutch Johan Cruyff Shield','soccer','ned.supercup'),
              ( 295,1,'Trofeo Joan Gamper','soccer','esp.joan_gamper'),
              ( 296,1,'German DFL-Supercup','soccer','ger.super_cup'),
              ( 297,1,'Audi Cup','soccer','ger.audi_cup'),
              ( 298,1,'Premier League Asia Trophy','soccer','eng.asia_trophy'),
              ( 299,1,'Emirates Cup','soccer','friendly.emirates_cup'),
              ( 300,1,'Japanese J League World Challenge','soccer','jpn.world_challenge'),
              ( 301,1,'SuperCopa Euroamericana','soccer','euroamericana.supercopa'),
              ( 302,1,'Men''s Olympic Tournament','soccer','fifa.olympics'),
              ( 303,1,'Women''s Olympic Tournament','soccer','fifa.w.olympics'),
              ( 304,1,'CONMEBOL Pre-Olympic Tournament','soccer','fifa.conmebol.olympicsq'),
              ( 305,1,'CONCACAF Men''s Olympic Qualifying','soccer','fifa.concacaf.olympicsq'),
              ( 306,1,'CONCACAF Women''s Olympic Qualifying Tournament','soccer','fifa.w.concacaf.olympicsq'),
              ( 307,1,'CONCACAF Women''s Championship','soccer','concacaf.womens.championship'),
              ( 308,1,'FIFA Under-20 World Cup','soccer','fifa.world.u20'),
              ( 309,1,'FIFA Under-17 World Cup','soccer','fifa.world.u17'),
              ( 310,1,'Toulon Tournament','soccer','global.toulon'),
              ( 311,1,'UEFA European Under-21 Championship','soccer','uefa.euro_u21'),
              ( 312,1,'UEFA European Under-19 Championship','soccer','uefa.euro.u19'),
              ( 313,1,'Under-21 International Friendly','soccer','fifa.friendly_u21'),
              ( 314,1,'UEFA Women''s European Championship','soccer','uefa.weuro'),
              ( 315,1,'German Bundesliga Promotion/Relegation Playoff','soccer','ger.playoff.relegation'),
              ( 316,1,'German 2. Bundesliga Promotion/Relegation Playoffs','soccer','ger.2.promotion.relegation'),
              ( 317,1,'English Women''s FA Cup','soccer','eng.w.fa'),
              ( 318,1,'English Women''s FA Community Shield','soccer','eng.w.charity'),
              ( 319,1,'English EFL Trophy','soccer','eng.trophy'),
              ( 320,1,'English National League','soccer','eng.5'),
              ( 321,1,'English League One','soccer','eng.3'),
              ( 322,1,'English League Two','soccer','eng.4'),
              ( 323,1,'Scottish Cup','soccer','sco.tennents'),
              ( 324,1,'Scottish League Cup','soccer','sco.cis'),
              ( 325,1,'Scottish Premiership Promotion/Relegation Playoffs','soccer','sco.1.promotion.relegation'),
              ( 326,1,'Scottish League One','soccer','sco.3'),
              ( 327,1,'Scottish Championship','soccer','sco.2'),
              ( 328,1,'Scottish Championship Promotion/Relegation Playoffs','soccer','sco.2.promotion.relegation'),
              ( 329,1,'Scottish League One Promotion/Relegation Playoffs','soccer','sco.3.promotion.relegation'),
              ( 330,1,'Scottish League Two Promotion/Relegation Playoffs','soccer','sco.4.promotion.relegation'),
              ( 331,1,'Scottish League Two','soccer','sco.4'),
              ( 332,1,'Scottish League Challenge Cup','soccer','sco.challenge'),
              ( 333,1,'Dutch Eredivisie Promotion/Relegation Playoffs','soccer','ned.playoff.relegation'),
              ( 334,1,'Dutch Eredivisie Cup','soccer','ned.w.eredivisie_cup'),
              ( 335,1,'Dutch Keuken Kampioen Divisie','soccer','ned.2'),
              ( 336,1,'Dutch Tweede Divisie','soccer','ned.3'),
              ( 337,1,'Dutch KNVB Beker Vrouwen','soccer','ned.w.knvb_cup'),
              ( 338,1,'Dutch Vrouwen Eredivisie','soccer','ned.w.1'),
              ( 339,1,'Italian Serie B','soccer','ita.2'),
              ( 340,1,'French Ligue 1 Promotion/Relegation Playoffs','soccer','fra.1.promotion.relegation'),
              ( 341,1,'French Ligue 2 Promotion/Relegation Playoffs','soccer','fra.2.promotion.relegation'),
              ( 342,1,'Swedish Allsvenskan','soccer','swe.1'),
              ( 343,1,'Swedish Allsvenskanliga Promotion/Relegation Playoffs','soccer','swe.1.promotion.relegation'),
              ( 344,1,'Danish Superliga','soccer','den.1'),
              ( 345,1,'Danish SAS-Ligaen Promotion/Relegation Playoffs','soccer','den.promotion.relegation'),
              ( 346,1,'Romanian Liga 1 Promotion/Relegation Playoffs','soccer','rou.1.promotion.relegation'),
              ( 347,1,'Romanian Liga 1','soccer','rou.1'),
              ( 348,1,'Norwegian Eliteserien Promotion/Relegation Playoffs','soccer','nor.1.promotion.relegation'),
              ( 349,1,'Norwegian Eliteserien','soccer','nor.1'),
              ( 350,1,'Maltese Premier League','soccer','mlt.1'),
              ( 351,1,'Israeli Premier League','soccer','isr.1'),
              ( 352,1,'Irish Premier Division Promotion/Relegation Playoffs','soccer','ir1.1.promotion.relegation'),
              ( 353,1,'Irish Premier Division','soccer','irl.1'),
              ( 354,1,'Welsh Premier League','soccer','wal.1'),
              ( 355,1,'Northern Irish Premiership','soccer','nir.1'),
              ( 356,1,'CONMEBOL Copa Libertadores','soccer','conmebol.libertadores'),
              ( 357,1,'CONMEBOL Copa Sudamericana','soccer','conmebol.sudamericana'),
              ( 358,1,'CONMEBOL Recopa Sudamericana','soccer','conmebol.recopa'),
              ( 359,1,'Argentine Liga Profesional de Fútbol','soccer','arg.1'),
              ( 360,1,'Copa Argentina','soccer','arg.copa'),
              ( 361,1,'Argentine Copa de la Liga Profesional','soccer','arg.copa_lpf'),
              ( 362,1,'Argentine Copa de la Superliga','soccer','arg.copa_de_la_superliga'),
              ( 363,1,'Argentine Trofeo de Campeones de la Superliga','soccer','arg.trofeo_de_la_campeones'),
              ( 364,1,'Argentine Supercopa','soccer','arg.supercopa'),
              ( 365,1,'Argentine Nacional B','soccer','arg.2'),
              ( 366,1,'Argentine Primera B','soccer','arg.3'),
              ( 367,1,'Argentine Primera C','soccer','arg.4'),
              ( 368,1,'Argentine Primera D','soccer','arg.5'),
              ( 369,1,'Supercopa Do Brazil','soccer','bra.supercopa_do_brazil'),
              ( 370,1,'Brazilian Serie A','soccer','bra.1'),
              ( 371,1,'Brazilian Serie B','soccer','bra.2'),
              ( 372,1,'Brazilian Serie C','soccer','bra.3'),
              ( 373,1,'Copa Do Nordeste','soccer','bra.copa_do_nordeste'),
              ( 374,1,'Brazilian Campeonato Carioca','soccer','bra.camp.carioca'),
              ( 375,1,'Brazilian Campeonato Paulista','soccer','bra.camp.paulista'),
              ( 376,1,'Brazilian Campeonato Gaucho','soccer','bra.camp.gaucho'),
              ( 377,1,'Brazilian Campeonato Mineiro','soccer','bra.camp.mineiro'),
              ( 378,1,'Chilean Primera División','soccer','chi.1'),
              ( 379,1,'Copa Chile','soccer','chi.copa_chi'),
              ( 380,1,'International U20 Friendly','soccer','fifa.u20.friendly'),
              ( 381,1,'Segunda División de Chile','soccer','chi.2'),
              ( 382,1,'Chilean Supercopa','soccer','chi.super_cup'),
              ( 383,1,'Uruguayan Primera Division','soccer','uru.1'),
              ( 384,1,'Segunda División de Uruguay','soccer','uru.2'),
              ( 385,1,'Colombian SuperLiga','soccer','col.superliga'),
              ( 386,1,'Colombian Primera A','soccer','col.1'),
              ( 387,1,'Colombian Primera B','soccer','col.2'),
              ( 388,1,'Peruvian Supercopa','soccer','per.supercopa'),
              ( 389,1,'Copa Colombia','soccer','col.copa'),
              ( 390,1,'Peruvian Primera Profesional','soccer','per.1'),
              ( 391,1,'Paraguayan Primera Division','soccer','par.1'),
              ( 392,1,'Ecuadoran Primera A','soccer','ecu.1'),
              ( 393,1,'Ecuadoran Supercopa','soccer','ecu.supercopa'),
              ( 394,1,'Ecuador Serie B','soccer','ecu.2'),
              ( 395,1,'Venezuelan Primera Profesional','soccer','ven.1'),
              ( 396,1,'United States NWSL Challenge Cup','soccer','usa.nwsl.cup'),
              ( 397,1,'Segunda División de Venezuela','soccer','ven.2'),
              ( 398,1,'Bolivian Liga Profesional','soccer','bol.1'),
              ( 399,1,'Mexican Supercopa MX','soccer','mex.supercopa'),
              ( 400,1,'Mexican Campeon de Campeones','soccer','mex.campeon'),
              ( 401,1,'CONCACAF Champions Cup','soccer','concacaf.champions_cup'),
              ( 402,1,'CONCACAF U23 Tournament','soccer','concacaf.u23'),
              ( 403,1,'Honduran Primera Division','soccer','hon.1'),
              ( 404,1,'Costa Rican Primera Division','soccer','crc.1'),
              ( 405,1,'Jamaican Premier League','soccer','jam.1'),
              ( 406,1,'Guatemalan Liga Nacional','soccer','gua.1'),
              ( 407,1,'Australian W-League','soccer','aus.w.1'),
              ( 408,1,'Salvadoran Primera Division','soccer','slv.1'),
              ( 409,1,'AFF Cup','soccer','aff.championship'),
              ( 410,1,'AFC Cup','soccer','afc.cup'),
              ( 411,1,'SAFF Championship','soccer','afc.saff.championship'),
              ( 412,1,'Chinese Super League Promotion/Relegation Playoffs','soccer','chn.1.promotion.relegation'),
              ( 413,1,'Japanese J League','soccer','jpn.1'),
              ( 414,1,'Indonesian Liga 1','soccer','idn.1'),
              ( 415,1,'Indian Super League','soccer','ind.1'),
              ( 416,1,'Indian I-League','soccer','ind.2'),
              ( 417,1,'Malaysian Super League','soccer','mys.1'),
              ( 418,1,'Singaporean Premier League','soccer','sgp.1'),
              ( 419,1,'Thai League 1','soccer','tha.1'),
              ( 420,1,'Bangabandhu Cup','soccer','bangabandhu.cup'),
              ( 421,1,'COSAFA Cup','soccer','caf.cosafa'),
              ( 422,1,'CAF Champions League','soccer','caf.champions'),
              ( 423,1,'South African Premiership Promotion/Relegation Playoffs','soccer','rsa.1.promotion.relegation'),
              ( 424,1,'CAF Confederations Cup','soccer','caf.confed'),
              ( 425,1,'South African Premiership','soccer','rsa.1'),
              ( 426,1,'South African First Division','soccer','rsa.2'),
              ( 427,1,'South African Telkom Knockout','soccer','rsa.telkom_knockout'),
              ( 428,1,'South African Nedbank Cup','soccer','rsa.nedbank'),
              ( 429,1,'MTN 8 Cup','soccer','rsa.mtn8'),
              ( 430,1,'Nigerian Professional League','soccer','nga.1'),
              ( 431,1,'Ghanaian Premier League','soccer','gha.1'),
              ( 432,1,'Zambian Super League','soccer','zam.1'),
              ( 433,1,'Kenyan Premier League','soccer','ken.1'),
              ( 434,1,'Zimbabwean Premier Soccer League','soccer','zim.1'),
              ( 435,1,'Ugandan Premier League','soccer','uga.1'),
              ( 436,1,'Misc. U.S. Soccer Games','soccer','generic.ussf');)",

            R"(INSERT INTO sportsapi
            VALUES
              (1000,2,'Major League Baseball','baseball','mlb');)",

            R"(CREATE TABLE sportslisting (
              id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
              api INT UNSIGNED NOT NULL,
              title VARCHAR(128) NOT NULL
            ) ENGINE=MyISAM DEFAULT CHARSET=utf8;)",

            R"(INSERT INTO sportslisting (api,title)
            VALUES
              (   1, '\\A(?:MLB Baseball)\\z'),
              (   1, '\\A(?:Béisbol MLB)\\z'),
              (   1, '\\A(?:MLB All-Star Game)\\z'),
              (   1, '\\A(?:World Series)\\z'),
              (   2, '\\A(?:College Baseball)\\z'),
              (   2, '\\A(?:College World Series)\\z'),
              (   3, '\\A(?:College Softball)\\z'),
              (   4, '\\A(?:Women''s College World Series)\\z'),
              (  10, '\\A(?:Béisbol Liga Mexicana)\\z'),

              (  20, '\\A(?:N\\w+ Football)\\z'),
              (  20, '\\A(?:Super Bowl( [CLXVI]+)?)\\z'),
              (  20, '\\A(?:Fútbol Americano NFL)\\z'),
              (  21, '\\A(?:College Football)\\z'),
              (  21, '\\A(?:\\w+ Bowl)\\z'),
              (  21, '\\A(?:Fútbol Americano de Universitario)\\z'),

              (  40, '\\A(?:NBA Basketball)\\z'),
              (  40, '\\A(?:NBA Finals)\\z'),
              (  41, '\\A(?:WNBA Basketball)\\z'),
              (  41, '\\A(?:WNBA Finals)\\z'),
              (  42, '\\A(?:College Basketball)\\z'),
              (  42, '\\A(?:NCAA Basketball Tournament)\\z'),
              (  43, '\\A(?:Women''s College Basketball)\\z'),
              (  43, '\\A(?:NCAA Women''s Basketball Tournament)\\z'),

              (  60, '\\A(?:NHL Hockey)\\z'),
              (  60, '\\A(?:NHL Winter Classic)\\z'),
              (  60, '\\A(?:NHL \\w+ Conference Final)\\z'),
              (  60, '\\A(?:Stanley Cup Finals)\\z'),
              (  61, '\\A(?:College Hockey)\\z'),
              (  61, '\\A(?:Frozen Four)\\z'),
              (  62, '\\A(?:Women''s College Hockey)\\z'),
              (  66, '\\A(?:College Field Hockey)\\z'),

              (  80, '\\A(?:College Volleyball)\\z'),
              (  81, '\\A(?:Women''s College Volleyball)\\z'),

              ( 100, '\\A(?:College Lacrosse)\\z'),
              ( 101, '\\A(?:Women''s College Lacrosse)\\z'),

              ( 120, '\\A(?:College Water Polo)\\z'),
              ( 121, '\\A(?:Women''s College Water Polo)\\z'),

              ( 200, '\\A(?:College Soccer)\\z'),
              ( 201, '\\A(?:Women''s College Soccer)\\z'),
              ( 202, '\\A(?:MLS Soccer|Fútbol MLS)\\z'),
              ( 203, '\\A(?:(Premier League|EPL) Soccer)\\z'),
              ( 203, '\\A(?:English Premier League)\\z'),
              ( 203, '\\A(?:Fútbol Premier League)\\z'),
              ( 205, '\\A(?:Italian Serie A Soccer)\\z'),
              ( 339, '\\A(?:Italian Serie B Soccer)\\z'),
              ( 206, '\\A(?:French Ligue 1 Soccer|Fútbol Ligue 1|Fútbol Liga 1)\\z'),
              ( 207, '\\A(?:French Ligue 2 Soccer|Fútbol Ligue 2|Fútbol Liga 2)\\z'),
              ( 208, '\\A(?:Fútbol LaLiga)\\z'),
              ( 208, '\\A(?:Fútbol Español Primera Division)\\z'),
              ( 208, '\\A(?:Spanish Primera Division Soccer)\\z'),
              ( 209, '\\A(?:(German )?Bundesliga Soccer)\\z'),
              ( 209, '\\A(?:Fútbol Bundesliga)\\z'),
              ( 209, '\\A(?:Fútbol Copa de Alemania)\\z'),
              ( 210, '\\A(?:German 2. Bundesliga Soccer)\\z'),
              ( 211, '\\A(?:Fútbol Mexicano Primera División|Fútbol Mexicano Liga Premier|Fútbol Mexicano)\\z'),
              ( 211, '\\A(?:Mexico Primera Division Soccer)\\z'),
              ( 212, '\\A(?:Copa do Brazil Soccer)\\z'),
              ( 214, '\\A(?:CONCACAF League Soccer)\\z'),
              ( 215, '\\A(?:CONCACAF Champions League Soccer)\\z'),
              ( 216, '\\A(?:CONCACAF Nations League Soccer)\\z'),
              ( 217, '\\A(?:CONCACAF Gold Cup Soccer)\\z'),
              ( 218, '\\A(?:FIFA World Cup Soccer)\\z'),
              ( 219, '\\A(?:FIFA World Cup Qualifying( Soccer)?|FIFA Eliminatorias Copa Mundial)\\z'),
              ( 220, '\\A(?:FIFA World Cup Qualifying( Soccer)?|FIFA Eliminatorias Copa Mundial)\\z'),
              ( 221, '\\A(?:FIFA World Cup Qualifying( Soccer)?|FIFA Eliminatorias Copa Mundial)\\z'),
              ( 222, '\\A(?:FIFA World Cup Qualifying( Soccer)?|FIFA Eliminatorias Copa Mundial)\\z'),
              ( 223, '\\A(?:FIFA World Cup Qualifying( Soccer)?|FIFA Eliminatorias Copa Mundial)\\z'),
              ( 224, '\\A(?:FIFA World Cup Qualifying( Soccer)?|FIFA Eliminatorias Copa Mundial)\\z'),
              ( 225, '\\A(?:Fútbol UEFA Champions League)\\z'),
              ( 225, '\\A(?:UEFA Champions League Soccer)\\z'),
              ( 226, '\\A(?:Fútbol UEFA Europa League)\\z'),
              ( 229, '\\A(?:Fútbol USL Championship)\\z'),
              ( 229, '\\A(?:USL Championship Soccer)\\z'),
              ( 230, '\\A(?:NWSL Soccer)\\z'),
              ( 231, '\\A(?:FA Women''s Super League)\\z'),
              ( 242, '\\A(?:Fútbol Mexicano Liga Expansión)\\z'),
              ( 258, '\\A(?:UEFA Nations League Soccer)\\z'),
              ( 258, '\\A(?:Fútbol UEFA Nations League)\\z'),
              ( 271, '\\A(?:Fútbol Español Segunda Division)\\z'),
              ( 277, '\\A(?:Fútbol Turco Superliga)\\z'),
              ( 277, '\\A(?:Turkish Super Lig Soccer)\\z'),
              ( 279, '\\A(?:Superleague Greek Soccer)\\z'),
              ( 356, '\\A(?:Fútbol CONMEBOL Libertadores)\\z'),
              ( 357, '\\A(?:Fútbol CONMEBOL Sudamericana)\\z'),
              ( 359, '\\A(?:Fútbol Argentino Primera División)\\z'),
              ( 360, '\\A(?:Fútbol Copa Argentina)\\z'),
              ( 365, '\\A(?:Fútbol Argentino Primera Nacional( B)?)\\z'),
              ( 366, '\\A(?:Fútbol Argentino Primera B)\\z'),
              ( 367, '\\A(?:Fútbol Argentino Primera C)\\z'),
              ( 368, '\\A(?:Fútbol Argentino Primera D)\\z'),
              ( 386, '\\A(?:Fútbol Columbiano Primera División)\\z'),
              ( 403, '\\A(?:Fútbol Hondureño Primera División)\\z'),
              ( 404, '\\A(?:Fútbol Costarricense Primera División)\\z'),

              (1000, '\\A(?:MLB Baseball)\\z'),
              (1000, '\\A(?:Béisbol MLB)\\z');
              )",

            R"(CREATE TABLE sportscleanup (
              id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
              provider TINYINT UNSIGNED DEFAULT 0,
              weight INT UNSIGNED NOT NULL,
              key1 VARCHAR(256) NOT NULL,
              name VARCHAR(256) NOT NULL,
              pattern VARCHAR(256) NOT NULL,
              nth TINYINT UNSIGNED NOT NULL,
              replacement VARCHAR(128) NOT NULL
            ) ENGINE=MyISAM DEFAULT CHARSET=utf8;)",

            // Sigh. It would be nice if these patterns could use the
            // '\b' boundary matching sequence in the first part of
            // the match, but the period is not part of the set of
            // characters that make up a word, so trying to optionally
            // match the final period in the string 'F.C. Foo' with
            // the pattern '\.?\b' always fails because period and
            // space are both non-word characters and therefore there
            // is no word boundary between them. This will always
            // match on 'F.C' and never on 'F.C.'.
            //
            // I have also seen a TV listing where the team name was
            // 'F.C.Something' without a space, so the first part of
            // these patterns doesn't require a following space in
            // order to match. These patterns are case sensitive, so
            // the first part shouldn't match part of an ordinary team
            // name unless the team name is in all caps.
            R"A(INSERT INTO sportscleanup (provider,weight,key1,name,pattern,nth,replacement)
            VALUES
              (1,1000,'soccer', '(SE)',     '\\(\\w+\\)', 0, ''),
              (1,1000,'soccer', 'AFC',      '\\AA\\.?F\\.?C\\.?|\\bA\\.?F\\.?C\\.?\\Z', 0, ''),
              (1,1000,'soccer', 'AC etc.',  '\\AA\\.?[AC]\\.?|\\bA\\.?[AC]\\.?\\Z', 0, ''),
              (1,1000,'soccer', 'BK',       '\\AB\\.?K\\.?|\\bB\\.?K\\.?\\Z', 0, ''),
              (1,1000,'soccer', 'BSC',      '\\AB\\.?S\\.?C\\.?|\\bB\\.?S\\.?C\\.?\\Z', 0, ''),
              (1,1000,'soccer', 'CSyD',     '\\AC\\.?S\\.?( y )?D\\.?|\\bC\\.?S\\.?( y )?D\\.?\\Z', 0, ''),
              (1,1000,'soccer', 'CD etc.',  '\\AC\\.?[ADFRS]\\.?|\\bC\\.?[ADFRS]\\.?\\Z', 0, ''),
              (1,1000,'soccer', 'FC',       '\\AF\\.?C\\.?|\\bF\\.?C\\.?\\Z', 0, ''),
              (1,1000,'soccer', 'HSC',      '\\AH\\.?S\\.?C\\.?|\\bH\\.?S\\.?C\\.?\\Z', 0, ''),
              (1,1000,'soccer', 'RC etc.',  '\\AR\\.?[BC]\\.?|\\bR\\.?[BC]\\.?\\Z', 0, ''),
              (1,1000,'soccer', 'SC etc.',  '\\AS\\.?[ACV]\\.?|\\bS\\.?[ACV]\\.?\\Z', 0, ''),
              (1,1000,'soccer', 'TSG',      '\\AT\\.?S\\.?G\\.?|\\bT\\.?S\\.?G\\.?\\Z', 0, ''),
              (1,1000,'soccer', 'VFB etc.', '\\AV\\.?F\\.?[BL]\\.?|\\bV\\.?F\\.?[BL]\\.?\\Z', 0, ''),
              (1,2000,'all',    '',         'Inglaterra', 0, 'England'),
              (1,2000,'all',    '',         'Munchen', 0, 'Munich');
              )A",
        };
      case 2:
        return {
            // More TV listing name to API name mappings for college
            // basketball.  Using a weight of 1000 for specific
            // changes and 1100 for general changes.
            R"A(INSERT INTO sportscleanup (provider,weight,key1,name,pattern,nth,replacement)
            VALUES
              (1,1100,'basketball', 'Cal State',    'Cal State', 0, 'CSU'),
              (1,1000,'basketball', 'Grambling',    'Grambling State', 0, 'Grambling'),
              (1,1000,'basketball', 'Hawaii',       'Hawaii', 0, 'Hawai''i'),
              (1,1000,'basketball', 'LIU',          'LIU', 0, 'Long Island University'),
              (1,1100,'basketball', 'Loyola',       'Loyola-', 0, 'Loyola '),
              (1,1000,'basketball', 'Loyola (Md.)', 'Loyola \(Md.\)', 0, 'Loyola (MD)'),
              (1,1000,'basketball', 'McNeese',      'McNeese State', 0, 'McNeese'),
              (1,1000,'basketball', 'Miami (OH)',   'Miami \(Ohio\)', 0, 'Miami (OH)'),
              (1,1000,'basketball', 'UAB',          'Alabama-Birmingham', 0, 'UAB'),
              (1,1000,'basketball', 'UConn',        'Connecticut', 0, 'UConn'),
              (1,1000,'basketball', 'UMass',        'Massachusetts', 0, 'UMass'),
              (1,1100,'basketball', 'UNC',          'UNC-', 0, 'UNC '),
              (1,1000,'basketball', 'UTEP',         'Texas-El Paso', 0, 'UTEP'),
              (1,1100,'basketball', 'Texas',        'Texas-', 0, 'UT '),
              (1,1000,'basketball', 'Chattanooga',  'UT-Chattanooga', 0, 'Chattanooga'),
              (1,1100,'basketball', 'UT',           'UT-', 0, 'UT ');
              )A",
        };

      default:
        return {};
    }
}

/* vim: set expandtab tabstop=4 shiftwidth=4: */
