/*	Copyright (C) 2018-2024 Martin Guy <martinwguy@gmail.com>
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, either version 3 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * Program: spettro
 *	Play an audio file displaying a scrolling log-frequency spectrogram.
 *
 * File: main.c
 *	Main routine, parameter handling, reading and servicing key strokes,
 *	scheduling FFT calculations and receiving the results,
 *	the FFT result cache and drawing overlays on rows and columns.
 *
 * The audio file is given as a command-line argument
 * A window opens showing a graphical representation of the audio file:
 * each frame of audio samples is shown as a vertical bar whose colors are
 * taken from the "heat maps" of "sox spectrogram".
 *
 * The color at each point represents the energy in the sound at some
 * frequency (band) at a certain moment in time (over a short period).
 * The vertical axis, representing frequency, is logarithmic, giving an
 * equal number of pixel rows in each octave of the scale, by default
 * 9 octaves from 27.5 Hz (very bottom A) to 14080 Hz (the toppest A
 * we can hear.)
 *
 * At startup, the start of the piece is at the centre of the window and
 * the first seconds of the audio file are shown on the right half. The
 * left half is all grey.
 *
 * If you hit play (press 'space'), the audio starts playing and the
 * display scrolls left so that the current playing position remains at the
 * centre of the window. Another space pauses the playback, another makes it
 * continue from where it was. When it reaches the end of piece, the playback
 * stops; pressing space makes it start again from the beginning.
 *
 * If you resize the window the displayed image is zoomed.
 *
 * For command-line options and key bindings, see Usage in args.c.
 *
 * It runs in three types of thread:
 * - the main thread handles GUI events, starts/stops the audio player,
 *   tells the calc thread what to calculate, receives results, and
 *   displays them.
 * - The calc thread performs FFTs and reports back when they're done.
 * - the timer thread is called periodically to scroll the display in sync
 *   with the audio playback. The actual scrolling is done in the main loop
 *   in response to an event posted by the timer thread.
 *
 *	Martin Guy <martinwguy@gmail.com>, Dec 2016 - May 2017.
 */

#include "spettro.h"

/*
 * System header files
 */
#include <unistd.h>	/* for sleep() */
#include <time.h>	/* for time() */
#include <sys/time.h>	/* for gettimeofday() */
#if SDL_AUDIO && HAVE_LIBX11
#include <X11/Xlib.h>	/* for XInitThreads() */
#endif

#if HAVE_SIGNAL && HAVE_SETJMP
#include <signal.h>
#include <setjmp.h>

static jmp_buf segv_buf;
static void
segv_handler(int sig)
{
    longjmp(segv_buf, SIGSEGV);
}
#endif

/*
 * Local header files
 */
#include "args.h"
#include "audio.h"
#include "a_cache.h"
#include "axes.h"
#include "cacatext.h"
#include "r_cache.h"
#include "convert.h"
#include "gui.h"
#include "interpol.h"
#include "overlay.h"
#include "paint.h"
#include "schedule.h"
#include "text.h"
#include "timer.h"
#include "window.h"	/* for free_windows() */
#include "ui.h"
#include "ui_funcs.h"	/* for set_time_zoom() */

static char *progname;	/* executable name for stderr or re-exec */

static int remove_arg(int argc, char **argv, int optind);
static int insert_arg(int argc, char **argv, char *arg);

int
main(int argc, char **argv)
{
    audio_file_t *af = NULL;
    char *filename;
    extern int optind;

    progname = argv[0];

    process_spettroflags();

    optind = 1;	/* Reset getopt() for a new scan */
    process_args(argc, argv);
    if (optind == -1) {
	fprintf(stderr, "Failed to recognise command-line arguments.\n");
	exit(1);
    }

    /* They must supply at least one filename argument */
    if (argv[optind] == NULL) {
	fprintf(stderr, "You must supply at least one audio file name.\n");
	exit(1);
    }

#if HAVE_LIBCACA
    /* Initialize the caca font here so that text size measurements are OK */
    caca_init();
#endif

    /* Width of frequency axis on the left of the graph.
     * 22050- == pixel space + text + pixel space + 2 pixels for tick.
     * With the caca font, "2E+05" is one pixel wider than "00000" but
     * it just uses the single-pixel gap on the left and is uncommon.
     */
    frequency_axis_width = 1 + text_width("00000") + 1 + 2;

    /* Note name frequency axis on the right.
     * -A0 == Two pixels for tick, a space, two * (letter + blank column)
     */
    note_name_axis_width = 2 + 1 + text_width("A0") + 1;

    /* Top and bottom axes = space above, text, space below. */
    top_margin = 1 + text_height("A") + 1;
    bottom_margin = 1 + text_height("0") + 1;

    /* Process the filename argument */
    if (shuffle_mode) {
	struct timeval tv;
	long r;	/* random number */
	int i;

	/* Pick a random file */
	if (gettimeofday(&tv, NULL) == 0)
	    srand(tv.tv_sec ^ tv.tv_usec);
	else
	    srand((unsigned)time(NULL));
	/* Burn a semi-random number of random numbers */
	i = optind; do { r = rand(); } while (++i < argc);

	optind += r % (argc - optind);
    }
    filename = argv[optind];

    if ((af = open_audio_file(filename)) == NULL) {
	/* The libraries blurt error messages and errno is usually
	 * rubbish: No such file or directory or Success */
	perror(filename);
	nexting = TRUE;
	goto next_file;
    }

    /* Give console log of played files so you can see in shuffle mode
     * what a previous song was */
    printf("Playing %s\n", filename);

    if (audio_file_length() <= 0.0) {
	fprintf(stderr, "The audio file is of %s length.\n",
		audio_file_length() == 0.0 ? "zero" : "negative");
	nexting = TRUE;
	goto next_file;
    }

    /* If anything uses SDL, initialise it. The audio and video will be
     * enabled separately in audio_init() and gui_init()
     */
#if SDL_AUDIO || SDL_TIMER || SDL_VIDEO || SDL_MAIN

# if HAVE_LIBX11
    /* Without this, you get:
     * [xcb] Unknown request in queue while dequeuing
     * [xcb] Most likely this is a multi-threaded client and XInitThreads has not been called
     * [xcb] Aborting, sorry about that.
     */
    if (!XInitThreads()) {
	fprintf(stderr, "XInitThreads failed.\n");
	/* May be harmless */
    }
# endif

    {
	if (SDL_Init(
# if SDL_TIMER
		     SDL_INIT_TIMER |
# endif
# if SDL_MAIN
		     SDL_INIT_EVENTS |
# endif
		    0) != 0) {
	    fprintf(stderr, "Couldn't initialize SDL: %s.\n", SDL_GetError());
	    exit(1);
	}
	atexit(SDL_Quit);
    }
#endif

    /* Initialize the graphics subsystem */
    gui_init(filename);

    /* Initialize the audio subsystem */
    if (!output_file && !mute_mode) audio_init(af, filename);

    /* --fit (-=) argument is a special case for various settings and
     * has to happen after we know the audio file length. */
    if (start_up_fit) {
	/* Put time 0 at the left edge of min_x and
	 * audio_length-1 at the right edge of max_x */
	secs_t length = audio_file_length();
	int width = max_x - min_x + 1;

	if (start_time != 0.0)
	    fprintf(stderr, "Warning: -t/--start is overridden by -=/--fit\n");
	if (ppsec != DEFAULT_PPSEC)
	    fprintf(stderr, "Warning: -P/--ppsec is overridden by -=/--fit\n");
	if (length != 0.0) {
	    ppsec = width / length;
	    start_time = (disp_offset - min_x) / ppsec;
	    if (fft_freq == DEFAULT_FFT_FREQ)
		fft_freq = (width / length) / 2;
	    /* freq / 2 (twice the time) because window functions mostly cover
	     * the central half of the audio sample */
	}
    }

    /* If they set disp_time with -t or --start, check it's within the audio. */
    if (start_time >= audio_file_length()) {
	fprintf(stderr,
		"Starting time (%s) ", seconds_to_string(disp_time));
	fprintf(stderr, "is beyond the end of the audio (%s).\n",
		seconds_to_string(audio_file_length()));
	start_time = audio_file_length() - 1/sr;
    }

    set_disp_time(start_time);
    set_real_start_time(start_time);

    /* The row overlay (piano notes/staff lines) doesn't depend on
     * the sample rate, only on min/max_freq, so it doesn't change
     * unless they freq pan/zoom.
     */
    make_row_overlay();	

    /* Apply the -t flag */
    if (disp_time != 0.0) set_playing_time(disp_time);

    start_scheduler(max_threads);

    draw_axes();

    repaint_display(FALSE); /* Schedules the initial screen refresh */

    /* Start the screen-scrolling interrupt */
    if (!output_file) start_timer();

    /* Catch segfaults, probably inside libav, and exec to play next file */
# if HAVE_SIGNAL && HAVE_SETJMP
    if (setjmp(segv_buf) == 0)
	signal(SIGSEGV, segv_handler);
    else {
	nexting = TRUE;
	goto next_file;
    }
# endif

    /* Run the main event loop until we quit */
    gui_main();

#if HAVE_SIGNAL && HAVE_SETJMP
    signal(SIGSEGV, SIG_DFL);
#endif

    if (!output_file) stop_timer();
    stop_scheduler();
    if (!output_file && !mute_mode) audio_deinit();
    gui_deinit();

next_file:
    if (nexting && optind < argc-1) {
	int i;

	/* Shift the following filenames down,
	 * removing the current filename */
	argc = remove_arg(argc, argv, optind);

	/* Remove any existing -U or -F flags from the command line.
	 * Doesn't work if they originally supplied -U or -F in a
	 * multi-letter argument. */
	for (i=1; i < argc; i++) {
	    if (strcmp(argv[i], "-U") == 0 ||
		strcmp(argv[i], "-F") == 0) {
		argc = remove_arg(argc, argv, i);
		/* If it's before the next filename to play, shift that too */
		if (i < optind) optind--;
		i--;	/* Process the same arg again */
	    }
	}

	/* If we were minimized, make the new invocation minimised too */
	if (start_up_minimized)	argc = insert_arg(argc, argv, "-U");
	if (fullscreen_mode)	argc = insert_arg(argc, argv, "-F");

	/* Make sure argv[] is NULL-terminated for execvp() */
	argv[argc] = NULL;

	/* Close all file descriptors except stdio because they remain open
	 * across an exec() and it runs out after playing 1000+ tracks.
	 * Surprisingly, it has 1712 open fds, mostly shared memory open to
	 * /usr/lib/.../lib*.so.* - 7 of each one.
	 */
	for (i=3; i < 1024; i++) (void) close(i);

	execvp(progname, argv);
	fprintf(stderr, "Re-exec of %s failed.\n", progname);
	exit(1);
    }

    /* Free memory to make valgrind happier */
    drop_all_work();
    drop_all_results();
    free_interpolate_cache();
    free_row_overlay();
    free_windows();
    close_audio_file(af);

    return 0;
}

/* Drop an argument from the argv, returning the new argc */
static int
remove_arg(int argc, char **argv, int optind)
{
    int i;

    for (i=optind, argc--; i < argc; i++)
	argv[i] = argv[i+1]; /* Also copies the NULL */

    return argc;
}

/* Add an argument at the start of argv, returning the new argc */
static int
insert_arg(int argc, char **argv, char *arg)
{
    int i;	/* Indexes the arg we're copying to */

    for (i=argc; i > 1; --i)
	argv[i] = argv[i-1];
    argv[1] = arg;

    return argc + 1;
}
