/*	Copyright (C) 2018-2024 Martin Guy <martinwguy@gmail.com>
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, either version 3 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * spettro.h - definitions and declarations for everybody
 *	       should be the first file included by all *.c files
 */

#ifndef SPETTRO_H
#define SPETTRO_H

#include "configure.h"

/* Headers for everyone */
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <assert.h>
#include <sys/types.h>	/* For off_t */
#include "alloc.h"

typedef int bool;

typedef double secs_t;	 /* Time measured in seconds */
typedef off_t  frames_t; /* Time measured in sample frames */
typedef double freq_t;	 /* Anything in things per second */

#ifndef FALSE
# define FALSE 0
#endif
#ifndef TRUE
# define TRUE 1
#endif

/* Value for when a floating point value is not set.
 * INFINITY doesn't work in c90 so pick an improbable value */
#define UNDEFINED (-DELTA)

#ifndef MIN
#define MIN(a,b) ((a) < (b) ? (a) : (b))
#endif
#ifndef MAX
#define MAX(a,b) ((a) > (b) ? (a) : (b))
#endif

/* Slop factor for comparisons involving calculated floating point values. */
/* A one-bit difference in the mantissa of a 4-byte float is 1.19209e-07 */
#define DELTA (1.0e-06)
#define DELTA_GT(a, b) ((a) > (b) + DELTA)
#define DELTA_LT(a, b) ((a) < (b) - DELTA)
#define DELTA_GE(a, b) ((a) >= (b) - DELTA)
#define DELTA_LE(a, b) ((a) <= (b) + DELTA)
#define DELTA_EQ(a, b) ((a) > (b) - DELTA && (a) < (b) + DELTA)
#define DELTA_NE(a, b) ((a) < (b) - DELTA || (a) > (b) + DELTA)

/* Portability stuff */

#if HAVE_BZERO || HAVE_BCOPY || HAVE_RINDEX
# include <strings.h>
#endif
#if HAVE_MEMSET || HAVE_MEMMOVE || HAVE_STRRCHR || HAVE_STRDUP
# include <string.h>
#endif
#if !HAVE_BZERO && HAVE_MEMSET
# define bzero(p, n) memset((p), 0, (n))
#endif
#if !HAVE_MEMMOVE && HAVE_BCOPY
# define memmove(dest, src, n) bcopy((src), (dest), (n))
#endif
#if !HAVE_STRRCHR && HAVE_RINDEX
/* For some reason aclocal barfs if you check for index()
 * so we assume that rindex() implies index() */
# define strchr(s, c) index((s), (c))
# define strrchr(s, c) rindex((s), (c))
#endif
#if !HAVE_STRDUP
extern char *strdup(const char *s);
#endif
#if !HAVE_SQRTF
extern float sqrtf(float f);
#endif
#if !HAVE_LOG10F
extern float log10f(float f);
#endif

/* These are undeclared/undefined when CC=c99 for some reason */
#if HAVE_STRDUP
extern char *strdup(const char *s);
#endif
/* These are undeclared/undefined when CC=c89 */
#ifndef M_PI
# define M_PI (3.141592653589793)
#endif
#if HAVE_ROUND
extern double round(double d);
#endif
#if HAVE_FLOOR
extern double floor(double d);
#endif
#if HAVE_TRUNC
extern double trunc(double d);
#endif
#if HAVE_ACOSH
extern double acosh(double d);
#endif

/* Select toolkit components to use */

/* Shorter macro for having all necessary elements of libav */
#define HAVE_LIBAV (HAVE_LIBAVFORMAT && HAVE_LIBAVCODEC && HAVE_LIBAVUTIL)

/* Sanity checks */
#if !HAVE_LIBSNDFILE && !HAVE_LIBMPG123 && !HAVE_LIBSOX && !HAVE_LIBAV
# error "You must have one of LIBSNDFILE, LIBMPG123, LIBSOX and LIBAV"
#endif

/* Compile-time toolkit selectors */
#if USE_EFL
# define EMOTION_AUDIO	1
# define EVAS_VIDEO	1
# define ECORE_TIMER	1
# define ECORE_MAIN	1
# define ECORE_LOCKS	1
#elif USE_SDL
# define SDL_AUDIO	1
# define SDL_VIDEO	1
# define SDL_TIMER	1
# define SDL_MAIN	1
# define SDL_LOCKS	1
#elif USE_EFL_SDL
# define SDL_AUDIO	1
# define EVAS_VIDEO	1
# define ECORE_TIMER	1
# define ECORE_MAIN	1
# define ECORE_LOCKS	1
#else
# error "You must define one of USE_EFL, USE_EFL_SDL and USE_SDL"
#endif

#if USE_SDL || USE_EFL_SDL
# include <SDL.h>
# if SDL_MAJOR_VERSION == 1
#error "Spettro no longer supports SDL1.2. Use SDL2.0"
# else
#  define SDL_PUSHEVENT_SUCCESS 1	/* Was 0 in SDL1 */
# endif
#endif

#endif
