/*	Copyright (C) 2018-2024 Martin Guy <martinwguy@gmail.com>
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, either version 3 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * ui.c - The current state of the user interface - all that determines
 *	  what the screen looks like.
 *
 *	  One day this will probably all be in a struct.
 */

#include "spettro.h"
#include "ui.h"

#include "audio.h"

#include <errno.h>

/* UI state variables */

/* Size of display area in pixels */
unsigned disp_width	= DEFAULT_DISP_WIDTH;
unsigned disp_height	= DEFAULT_DISP_HEIGHT;

/* Range of frequencies to display */
freq_t min_freq	= DEFAULT_MIN_FREQ;
freq_t max_freq	= DEFAULT_MAX_FREQ;	

/* Dynamic range of color map (values below this are black) */
float dyn_range = DEFAULT_DYN_RANGE;

/* Highest value seen so far in spectrogram data. */
float logmax = DEFAULT_LOGMAX;

/* How many video output frames to generate per second while playing
 * and how many pixel columns to generate per second of the audio file.
 * If they are equal, the graphic should scroll by one pixel column at a time.
 */
freq_t fps	= DEFAULT_FPS;
freq_t max_fps  = 50.0;  /* Upper limit, will be the screen refresh rate */

freq_t ppsec	= DEFAULT_PPSEC;

/* The "FFT frequency": 1/fft_freq seconds of audio are windowed and FFT-ed */
freq_t fft_freq	= DEFAULT_FFT_FREQ;

/* Which window functions to apply to each audio sample before FFt-ing it */
window_function_t window_function = DEFAULT_WINDOW_FUNCTION;

/* The -t/--start time parameter */
secs_t start_time = 0.0;

bool piano_lines  = FALSE;	/* Draw lines where piano keys fall? */
bool staff_lines  = FALSE;	/* Draw manuscript score staff lines? */
bool guitar_lines = FALSE;	/* Draw guitar string lines? */

bool show_frequency_axes = FALSE; /* Are we to show/showing the v axes? */
bool show_time_axes = FALSE;	/* Are we to show/showing the h axes? */
bool use_tiny_font = FALSE;	/* Draw axes using the tiny builtint font? */

secs_t left_bar_time = UNDEFINED;
secs_t right_bar_time = UNDEFINED;
int beats_per_bar = DEFAULT_BEATS_PER_BAR;

/* Other option flags */
bool autoplay_mode = TRUE;	/* !-p Start playing on startup */
bool mute_mode = FALSE;		/* -m  Be mute */
bool shuffle_mode = FALSE;	/* -S  Play files in random order */
bool fullscreen_mode = FALSE;	/* -F  Start up in fullscreen mode?
				 *     and are-we-in-fullscreen-mode? */
bool start_up_minimized = FALSE;/* Start with window minimized? */
bool start_up_fit = FALSE;	/* Start showing whole audio file? */
bool green_line_off = FALSE;	/* Should we omit it when repainting? */
double softvol = 1.0;
double softvol_double = 0.0;	/* Increase softvol so it doubles in n secs */
int max_threads = 0;		/* 0 means use default (the number of CPUs) */
char *output_file = NULL;	/* Image file to write to and quit. This is done
				 * when the last result has come in from the
				 * FFT threads, in schedule.c:calc_notify() */

bool quitting = FALSE;		/* Has the user asked to quit? */
bool nexting = FALSE;		/* Has the user asked for the next file? */

/* Where in time and space is the current playing position on the screen? */
secs_t disp_time = 0.0;		/* When in the audio file is the crosshair? 
				 * Always a multiple of "secpp". */

#include "a_cache.h"
/* Only use this function to change disp_time */
void
set_disp_time(secs_t when)
{
    disp_time = round(when / secpp) * secpp;
    reposition_audio_cache();
    /* This is necessary to update the virtual playing time but makes the sound
     * output go bubbly at the fps rate in normal mode */
    if (mute_mode) set_playing_time(when);
}

int disp_offset;		/* Crosshair is in which display column? */

/* The width in pixels of the vertical axes, when they are present.
 * "int", not "unsigned", so that "(int)x < frequency_axis_width" works */
int frequency_axis_width, note_name_axis_width;

/* Space above and below for status info at the top, time axis at the bottom */
int top_margin, bottom_margin;

void
set_disp_offset(void)
{
    disp_offset = disp_width / 2;
    {
	char *s = getenv("DISP_OFFSET");
	long n;

#if HAVE_STRTOL
	if (s && (errno = 0, n = strtol(s, NULL, 10), errno == 0)) {
#elif HAVE_SSCANF
	if (s && sscanf(s, "%ld", &n) == 1 && n >= 0)) {
#elif HAVE_ATOL
	if (s && (n = atol(s), n >= 0)) {
#endif
	      if (n < min_x) disp_offset = min_x;
	      else if (n > max_x) disp_offset = max_x;
	      else disp_offset = n;
	} else
	    disp_offset = disp_width / 2;
    }
}
