/*	Copyright (C) 2018-2024 Martin Guy <martinwguy@gmail.com>
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, either version 3 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* cacatext.c: Draw text on the display using libcaca's smallest font */

#include "spettro.h"

#if HAVE_LIBCACA

#include "cacatext.h"

#include "gui.h"
#include "lock.h"
#include "ui.h"

#include <string.h>	/* for strlen() */
#include <ctype.h>	/* for toupper() */
#include <caca.h>

/* Digits and upper case letters have no ascenders or descenders so
 * we can trim three rows of pixels off the top and the bottom.
 * The only exceptions are comma and slash in the status line
 * but it's OK if they descend into the blank row below the text.
 *
 * For the larger "Monospace 12" font, these should be 3 and 4,
 * though the comma descends 2 pixels below the base line.
 */
#define STRIP_ABOVE 3
#define STRIP_BELOW 3

static bool caca_render_char(uint32_t ch, int at_x, int at_y);
static int  caca_get_char_width(caca_font_t *f, uint32_t ch);

static caca_font_t *caca_font = NULL;

/*
 * Return the width of a text in pixels.
 *
 * If something fails, returns a negative number.
 */
int
caca_text_width(const char *text)
{
    int i;
    int width = 0;

    for (i=0; text[i] != '\0'; i++) {
	int cw = caca_get_char_width(caca_font, (uint32_t)text[i]);
	if (cw < 0) continue;	/* Character not found */
	width += cw + 1; /* One pixel space between characters... */
    }
    if (width == 0) return 0;
    else return width - 1;	/* ...except after the last one */
}

int
caca_text_height(const char *text)
{
    int height;
    height = caca_get_font_height(caca_font) - (STRIP_ABOVE + STRIP_BELOW);
    return height;
}

/*
 * Draw the given text at the given coordinates.
 *
 * alignment_y can be
 *	TOP to puts the top pixel of the text at that y coordinate
 *	CENTER to put the middle pixel of the text at that y coordinate
 *	BOTTOM to put the bottom pixel of the text at that y coordinate
 * alignment_x can be
 *	LEFT to put the leftmost pixel of the text at that x coordinate
 *	RIGHT to put the rightmost pixel of the text at that x coordinate
 *	CENTER to center the text on that coordinate value.
 *
 * To save vertical space, we map all letters to upper case and trim a
 * row of pixels off the top nd the bottom, which is why there are
 * "- 2" and "- 1" all over the height and position calculations.
 */
void
caca_draw_text(const char *text, int at_x, int at_y,
	       alignment_t alignment_x, alignment_t alignment_y)
{
    int width, height;
    int x;	/* index into the string */
    char *upper_text = Malloc(strlen(text) + 1);
    int i;

    for (i=0; text[i] != '\0'; i++) upper_text[i] = toupper(text[i]);
    upper_text[i] = '\0';
    text = upper_text;

    width = caca_text_width(text);
    height = caca_text_height(text);

    /* Make at_x and at_y the position of the top left pixel of the text */
    switch (alignment_x) {
    case LEFT:	/* at_x = at_x;	*/	break;
    case RIGHT: at_x -= width - 1;	break; /* We strip blank left column */
    case CENTER: at_x -= width/2;	break;
    }

    switch (alignment_y) {
    case TOP:	 /* at_y = at_y; */	break;
    case BOTTOM: at_y += height - 1;	break;
    case CENTER: at_y += height/2;	break;
    }

    /* Draw text at that position */
    for (x=0; text[x]; x++) {
	int cw = caca_get_char_width(caca_font, (uint32_t)text[x]);;
	if (cw >= 0) {	/* If the character exists in the font */
	    caca_render_char(text[x], at_x, at_y);
	    at_x += cw + 1;
	}
    }

    free(upper_text);
}

/* Load the caca font.
 * This is called from main() once and at the beginning of every text
 * function so that, if libcaca is present but the font loading fails,
 * we revert to the spettro's build-in font.
 */
bool
caca_init(void)
{
    static bool init_failed = FALSE;

    if (use_tiny_font) return FALSE;

    if (init_failed) {
	/* Only try to initialise it once. Subsequent calls tell callers
	 * whether the original initialization succeeded or not. */
	return FALSE;
    }

    if (caca_font == NULL) {
	char const *const *fonts = caca_get_font_list(); /* Never fails */

	if (fonts[0] == NULL || 
	    (caca_font = caca_load_font(fonts[0], 0)) == NULL) {

	    fprintf(stdout, "Loading of caca font %s ", fonts[0]);
	    perror("failed");
	    init_failed = TRUE;
	    return FALSE;
	}
    }

    return TRUE;
}

/* The rest is adapted from the libcaca source code in caca/font.c */

/*
 *  libcaca     Colour ASCII-Art library
 *  Copyright © 2002—2021 Sam Hocevar <sam@hocevar.net>
 *              All Rights Reserved
 *
 *  This file is free software. It comes without any warranty, to
 *  the extent permitted by applicable law. You can redistribute it
 *  and/or modify it under the terms of the Do What the Fuck You Want
 *  to Public License, Version 2, as published by Sam Hocevar. See
 *  http://www.wtfpl.net/ for more details.
 */

/* Helper structures for font loading */
struct font_header
{
    uint32_t control_size, data_size;
    uint16_t version, blocks;
    uint32_t glyphs;
    uint16_t bpp, width, height, maxwidth, maxheight, flags;
};

struct block_info
{
    uint32_t start, stop, index;
};

struct glyph_info
{
    uint16_t width, height;
    uint32_t data_offset;
};

struct caca_font
{
    struct font_header header;
    struct block_info *block_list;
    uint32_t *user_block_list;
    struct glyph_info *glyph_list;
    uint8_t *font_data;
    uint8_t *private;
};

/* Our own version of glyph_info, including extra information.
 * We can't modify struct glyph_info because it mirrors the in-memory
 * font format. */
struct glyph
{
    struct glyph_info *info;
    uint32_t ch;	/* The char value this correcponds to */
    uint8_t *glyph;	/* pointer to unpacked data */
    uint8_t x_offset;	/* How many blank columns does it have on the left? */
    uint8_t x_width;	/* How many non-blank columns does it have? */
    uint8_t y_offset;	/* How many blank rows does it have on top? */
    uint8_t y_height;	/* How many rows does it consist of? */
    /* The x_offset and width are measured for each individual character
     * to make a proportionally-spaced font out of a fixed-width one.
     * The y_offset and height are used the same way but are calculated,
     * not measured.
     */
};

static struct glyph	 *caca_get_glyph(caca_font_t *f, uint32_t ch);
static void		  caca_free_glyph(caca_font_t *f, struct glyph *glyph);

/* Render a Unicode character with its top left corner at at_[xy] */
static bool
caca_render_char(uint32_t ch, int at_x, int at_y)
{
    struct glyph *glyph;	/* glyph_info, unpacked data and char metrics */
    int x, y;

    glyph = caca_get_glyph(caca_font, ch);
    if (glyph == NULL) return FALSE;

    /* render glyph */
    gui_lock();
    for(y = 0; y < glyph->y_height; y++)
    {
	for(x = 0; x < glyph->x_width; x++)
	{
	    uint32_t p;

	    p = glyph->glyph[(y + glyph->y_offset) * glyph->info->width +
			      x + glyph->x_offset];
	    /* p is the brightness of this pixel from 0(dark) to 255 (bright) */
	    if (p > 0) gui_putpixel(at_x + x,
				    at_y - y,
				    RGB_to_color(0, p, 0));
	}
    }
    gui_unlock();

    caca_free_glyph(caca_font, glyph);

    return TRUE;
}

#define DECLARE_UNPACKGLYPH(bpp) \
    static void \
      unpack_glyph ## bpp(uint8_t *glyph, uint8_t *packed_data, int n) \
{ \
    int i; \
    \
    for(i = 0; i < n; i++) \
    { \
        uint8_t pixel = packed_data[i / (8 / bpp)]; \
        pixel >>= bpp * ((8 / bpp) - 1 - (i % (8 / bpp))); \
        pixel %= (1 << bpp); \
        pixel *= 0xff / ((1 << bpp) - 1); \
        *glyph++ = pixel; \
    } \
}
DECLARE_UNPACKGLYPH(4)
DECLARE_UNPACKGLYPH(2)
DECLARE_UNPACKGLYPH(1)

static struct glyph_info *
caca_get_glyph_info(caca_font_t *f, uint32_t ch)
{
    int b;			/* Block index */

    /* Find the Unicode block where our glyph lies */
    for(b = 0; b < f->header.blocks; b++)
    {
	if(ch < f->block_list[b].start)
	{
	    b = f->header.blocks;
	    break;
	}

	if(ch < f->block_list[b].stop)
	    break;
    }

    /* Glyph not in font? */
    if(b == f->header.blocks) {
	fprintf(stdout, "Glyph %u is not in the libcaca font\n", ch);
	return NULL;
    }

    return &f->glyph_list[f->block_list[b].index
			+ ch - f->block_list[b].start];
}

static void caca_trim_glyph(struct glyph *glyph);

/* Return a pointer to the glyph data, a square array of 8-bit numbers */
static struct glyph *
caca_get_glyph(caca_font_t *f, uint32_t ch)
{
    struct glyph *glyph;	/* Unpacked data and font metrics */
    struct glyph_info *g = caca_get_glyph_info(f, ch);

    if (g == NULL) return NULL;
    glyph = Malloc(sizeof(struct glyph));
    glyph->info = g;
    if(f->header.bpp != 8) {
        glyph->glyph = Malloc(f->header.width * f->header.height * 2);
    }
    /* Starting values */
    glyph->ch = ch;		/* for our reference only */
    glyph->x_offset = 0;
    glyph->x_width = glyph->info->width;
    glyph->y_offset = STRIP_ABOVE;
    glyph->y_height = glyph->info->height - STRIP_ABOVE - 1;
			/* All characters have one blank row at the bottom */

    /* Unpack glyph */
    switch(f->header.bpp)
    {
    case 8:
	glyph->glyph = f->font_data + g->data_offset;
	break;
    case 4:
	unpack_glyph4(glyph->glyph, f->font_data + g->data_offset,
		      g->width * g->height);
	break;
    case 2:
	unpack_glyph2(glyph->glyph, f->font_data + g->data_offset,
		      g->width * g->height);
	break;
    case 1:
	unpack_glyph1(glyph->glyph, f->font_data + g->data_offset,
		      g->width * g->height);
	break;
    }

    /* Measure the character's actual width, with some exceptions:
     * - the figure 1 is narrower than the other digits and makes the
     *   figures on the time axis jiggle about so set it to the same
     *   geometry as the other digits; right-justifying it looks better.
     * - The space looks too wide, so we set it to half a character width.
     * - The @ character descends by two rows, not one like the comma,
     *   and it has one more blank row above it than the capital letters
     *   and digits so we move it up by one row.
     */
    switch (ch) {
	struct glyph *glyph0;
    case '1':
	glyph0 = caca_get_glyph(f, (uint32_t) '0');
	glyph->x_offset = glyph0->x_offset - 1;	/* right-justify */
	glyph->x_width = glyph0->x_width;
	caca_free_glyph(f, glyph0);
	break;
    case ' ':
	glyph->x_offset = 0;
	glyph->x_width = glyph->info->width / 2;
	break;
    case '@':
	glyph->y_offset++;
	break;
    default:
	caca_trim_glyph(glyph);
    }

    return glyph;
}

/* Trim blank columns off the left and right of the glyph by
 * setting x_offset and x_width to the meaasured values 
 */
static void
caca_trim_glyph(struct glyph *glyph)
{
    int x, y;

    /* See how many blank columns it has on the left */
    for (x = 0; x < glyph->info->width; x++)
	for (y=0; y < glyph->info->height; y++)
	    if (glyph->glyph[y * glyph->info->width + x] > 0)
		/* We've found the first non-blank column */
		goto break_left;
break_left:
    /* Trim blank columns off everything except the space */
    if (x < glyph->info->width) {
	glyph->x_offset = x;
	glyph->x_width = glyph->info->width - x;
    }

    /* See how many blank columns it has on the right */
    for (x = glyph->info->width - 1; x >= 0; x--)
	for (y=0; y < glyph->info->height; y++)
	    if (glyph->glyph[y * glyph->info->width + x] > 0)
		/* We've found the last non-blank column */
		goto break_right;
break_right:
    /* Trim blank columns off everything except the space */
    if (x > 0) {
	glyph->x_width -= (glyph->info->width - 1) - x;
    }
}

/* Return the width of a character in pixels.
 * If the character is not in the font, return -1
 */
static int
caca_get_char_width(caca_font_t *f, uint32_t ch)
{
    struct glyph *glyph = caca_get_glyph(f, ch);

    if (glyph == NULL) return -1;
    else return glyph->x_width;
}

static void
caca_free_glyph(caca_font_t *f, struct glyph *glyph)
{
    if(f->header.bpp != 8)
        free(glyph->glyph);
    free(glyph);
}

#endif /* HAVE_LIBCACA */
