/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.melange.jvmmodel;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import fr.inria.diverse.melange.adapters.AdaptersFactory;
import fr.inria.diverse.melange.adapters.EObjectAdapter;
import fr.inria.diverse.melange.adapters.ResourceAdapter;
import fr.inria.diverse.melange.ast.LanguageExtensions;
import fr.inria.diverse.melange.ast.ModelingElementExtensions;
import fr.inria.diverse.melange.ast.NamingHelper;
import fr.inria.diverse.melange.jvmmodel.MelangeTypesBuilder;
import fr.inria.diverse.melange.jvmmodel.MetaclassAdapterInferrer;
import fr.inria.diverse.melange.lib.EcoreExtensions;
import fr.inria.diverse.melange.lib.MappingExtensions;
import fr.inria.diverse.melange.metamodel.melange.ClassBinding;
import fr.inria.diverse.melange.metamodel.melange.Language;
import fr.inria.diverse.melange.metamodel.melange.Mapping;
import fr.inria.diverse.melange.metamodel.melange.ModelType;
import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.WeakHashMap;
import java.util.function.Consumer;
import org.eclipse.emf.codegen.ecore.genmodel.GenModel;
import org.eclipse.emf.codegen.ecore.genmodel.GenPackage;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EFactory;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.ETypeParameter;
import org.eclipse.emf.ecore.impl.EFactoryImpl;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtend2.lib.StringConcatenationClient;
import org.eclipse.xtext.common.types.JvmAnnotationReference;
import org.eclipse.xtext.common.types.JvmConstructor;
import org.eclipse.xtext.common.types.JvmField;
import org.eclipse.xtext.common.types.JvmFormalParameter;
import org.eclipse.xtext.common.types.JvmGenericType;
import org.eclipse.xtext.common.types.JvmMember;
import org.eclipse.xtext.common.types.JvmOperation;
import org.eclipse.xtext.common.types.JvmTypeParameter;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.common.types.TypesFactory;
import org.eclipse.xtext.naming.IQualifiedNameProvider;
import org.eclipse.xtext.util.internal.Stopwatches;
import org.eclipse.xtext.xbase.jvmmodel.IJvmDeclaredTypeAcceptor;
import org.eclipse.xtext.xbase.jvmmodel.JvmAnnotationReferenceBuilder;
import org.eclipse.xtext.xbase.jvmmodel.JvmTypeReferenceBuilder;
import org.eclipse.xtext.xbase.jvmmodel.JvmTypesBuilder;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * Generates the in-the-large artifacts between a {@link Language} and a
 * {@link ModelType} it implements, and then delegates the generation of
 * in-the-small adapters for each implemented meta-class to
 * {@link MetaclassAdapterInferrer}.
 * 
 * @see NamingHelper#factoryAdapterNameFor
 * @see NamingHelper#adapterNameFor
 */
@SuppressWarnings("all")
public class LanguageAdapterInferrer {
  @Inject
  @Extension
  private JvmTypesBuilder _jvmTypesBuilder;
  
  @Inject
  @Extension
  private IQualifiedNameProvider _iQualifiedNameProvider;
  
  @Inject
  @Extension
  private NamingHelper _namingHelper;
  
  @Inject
  @Extension
  private EcoreExtensions _ecoreExtensions;
  
  @Inject
  @Extension
  private MelangeTypesBuilder _melangeTypesBuilder;
  
  @Inject
  @Extension
  private ModelingElementExtensions _modelingElementExtensions;
  
  @Inject
  @Extension
  private MappingExtensions _mappingExtensions;
  
  @Inject
  @Extension
  private MetaclassAdapterInferrer _metaclassAdapterInferrer;
  
  @Inject
  @Extension
  private LanguageExtensions _languageExtensions;
  
  @Inject
  @Extension
  private JvmAnnotationReferenceBuilder.Factory jvmAnnRefBuilderFact;
  
  @Extension
  private JvmAnnotationReferenceBuilder jvmAnnRefBuilder;
  
  @Extension
  private JvmTypeReferenceBuilder typeRefBuilder;
  
  private IJvmDeclaredTypeAcceptor acceptor;
  
  private Mapping mapping;
  
  /**
   * @see #generateLanguageAdapter
   * @see #generateFactoryAdapter
   * @see #generateAdaptersFactory
   */
  public void generateAdapter(final Language l, final ModelType superType, final IJvmDeclaredTypeAcceptor acc, @Extension final JvmTypeReferenceBuilder builder) {
    final Stopwatches.StoppedTask task = Stopwatches.forTask("generate metamodel adapters");
    task.start();
    this.jvmAnnRefBuilder = this.jvmAnnRefBuilderFact.create(l.eResource().getResourceSet());
    this.acceptor = acc;
    this.typeRefBuilder = builder;
    final Function1<Mapping, Boolean> _function = new Function1<Mapping, Boolean>() {
      @Override
      public Boolean apply(final Mapping it) {
        ModelType _to = it.getTo();
        return Boolean.valueOf(Objects.equal(_to, superType));
      }
    };
    this.mapping = IterableExtensions.<Mapping>findFirst(l.getMappings(), _function);
    this.generateLanguageAdapter(l, superType);
    this.generateFactoryAdapters(l, superType);
    this.generateAdaptersFactory(l, superType);
    final Function1<EClass, Boolean> _function_1 = new Function1<EClass, Boolean>() {
      @Override
      public Boolean apply(final EClass it) {
        return Boolean.valueOf(LanguageAdapterInferrer.this._ecoreExtensions.isAbstractable(it));
      }
    };
    final Consumer<EClass> _function_2 = new Consumer<EClass>() {
      @Override
      public void accept(final EClass cls) {
        LanguageAdapterInferrer.this._metaclassAdapterInferrer.generateAdapter(l.getSyntax(), superType, cls, LanguageAdapterInferrer.this.acceptor, builder);
      }
    };
    IterableExtensions.<EClass>filter(this._modelingElementExtensions.getAllClasses(superType), _function_1).forEach(_function_2);
    task.stop();
  }
  
  /**
   * Generates an adapter around a EMF#Resource for the {@code l -> mt} pair.
   * 
   * @see ResourceAdapter
   * @see NamingHelper#adapterNameFor
   */
  private void generateLanguageAdapter(final Language l, final ModelType mt) {
    final Procedure1<JvmGenericType> _function = new Procedure1<JvmGenericType>() {
      @Override
      public void apply(final JvmGenericType it) {
        EList<JvmTypeReference> _superTypes = it.getSuperTypes();
        JvmTypeReference _typeRef = LanguageAdapterInferrer.this.typeRefBuilder.typeRef(ResourceAdapter.class);
        LanguageAdapterInferrer.this._jvmTypesBuilder.<JvmTypeReference>operator_add(_superTypes, _typeRef);
        EList<JvmTypeReference> _superTypes_1 = it.getSuperTypes();
        JvmTypeReference _typeRef_1 = LanguageAdapterInferrer.this.typeRefBuilder.typeRef(LanguageAdapterInferrer.this._iQualifiedNameProvider.getFullyQualifiedName(mt).toString());
        LanguageAdapterInferrer.this._jvmTypesBuilder.<JvmTypeReference>operator_add(_superTypes_1, _typeRef_1);
        EList<JvmMember> _members = it.getMembers();
        final Procedure1<JvmConstructor> _function = new Procedure1<JvmConstructor>() {
          @Override
          public void apply(final JvmConstructor it) {
            StringConcatenationClient _client = new StringConcatenationClient() {
              @Override
              protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                _builder.append("super(");
                String _adaptersFactoryNameFor = LanguageAdapterInferrer.this._namingHelper.getAdaptersFactoryNameFor(l.getSyntax(), mt);
                _builder.append(_adaptersFactoryNameFor);
                _builder.append(".getInstance());");
                _builder.newLineIfNotEmpty();
              }
            };
            LanguageAdapterInferrer.this._jvmTypesBuilder.setBody(it, _client);
          }
        };
        JvmConstructor _constructor = LanguageAdapterInferrer.this._jvmTypesBuilder.toConstructor(l, _function);
        LanguageAdapterInferrer.this._jvmTypesBuilder.<JvmConstructor>operator_add(_members, _constructor);
        final Function1<GenModel, EList<GenPackage>> _function_1 = new Function1<GenModel, EList<GenPackage>>() {
          @Override
          public EList<GenPackage> apply(final GenModel it) {
            return it.getUsedGenPackages();
          }
        };
        final List<GenPackage> excluded = IterableExtensions.<GenPackage>toList(Iterables.<GenPackage>concat(IterableExtensions.<GenModel, EList<GenPackage>>map(LanguageAdapterInferrer.this._modelingElementExtensions.getGenmodels(mt), _function_1)));
        final Function1<GenPackage, Boolean> _function_2 = new Function1<GenPackage, Boolean>() {
          @Override
          public Boolean apply(final GenPackage genPkg) {
            boolean _contains = excluded.contains(genPkg);
            return Boolean.valueOf((!_contains));
          }
        };
        final Consumer<GenPackage> _function_3 = new Consumer<GenPackage>() {
          @Override
          public void accept(final GenPackage genPkg) {
            EList<JvmMember> _members = it.getMembers();
            String _factoryName = genPkg.getFactoryName();
            String _plus = ("get" + _factoryName);
            final Procedure1<JvmOperation> _function = new Procedure1<JvmOperation>() {
              @Override
              public void apply(final JvmOperation it) {
                EList<JvmAnnotationReference> _annotations = it.getAnnotations();
                JvmAnnotationReference _annotationRef = LanguageAdapterInferrer.this.jvmAnnRefBuilder.annotationRef(Override.class);
                LanguageAdapterInferrer.this._jvmTypesBuilder.<JvmAnnotationReference>operator_add(_annotations, _annotationRef);
                StringConcatenationClient _client = new StringConcatenationClient() {
                  @Override
                  protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                    _builder.append("return new ");
                    String _factoryAdapterNameFor = LanguageAdapterInferrer.this._namingHelper.factoryAdapterNameFor(genPkg, l.getSyntax(), mt);
                    _builder.append(_factoryAdapterNameFor);
                    _builder.append("();");
                    _builder.newLineIfNotEmpty();
                  }
                };
                LanguageAdapterInferrer.this._jvmTypesBuilder.setBody(it, _client);
              }
            };
            JvmOperation _method = LanguageAdapterInferrer.this._jvmTypesBuilder.toMethod(l, _plus, LanguageAdapterInferrer.this.typeRefBuilder.typeRef(genPkg.getQualifiedFactoryInterfaceName()), _function);
            LanguageAdapterInferrer.this._jvmTypesBuilder.<JvmOperation>operator_add(_members, _method);
          }
        };
        IterableExtensions.<GenPackage>filter(LanguageAdapterInferrer.this._modelingElementExtensions.getAllGenPkgs(mt), _function_2).forEach(_function_3);
        EList<JvmMember> _members_1 = it.getMembers();
        final Procedure1<JvmOperation> _function_4 = new Procedure1<JvmOperation>() {
          @Override
          public void apply(final JvmOperation it) {
            EList<JvmAnnotationReference> _annotations = it.getAnnotations();
            JvmAnnotationReference _annotationRef = LanguageAdapterInferrer.this.jvmAnnRefBuilder.annotationRef(Override.class);
            LanguageAdapterInferrer.this._jvmTypesBuilder.<JvmAnnotationReference>operator_add(_annotations, _annotationRef);
            StringConcatenationClient _client = new StringConcatenationClient() {
              @Override
              protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                _builder.append("java.util.Set<org.eclipse.emf.ecore.EFactory> res = new java.util.HashSet<org.eclipse.emf.ecore.EFactory>();");
                _builder.newLine();
                {
                  final Function1<GenPackage, Boolean> _function = new Function1<GenPackage, Boolean>() {
                    @Override
                    public Boolean apply(final GenPackage genPkg) {
                      boolean _contains = excluded.contains(genPkg);
                      return Boolean.valueOf((!_contains));
                    }
                  };
                  Iterable<GenPackage> _filter = IterableExtensions.<GenPackage>filter(LanguageAdapterInferrer.this._modelingElementExtensions.getAllGenPkgs(mt), _function);
                  for(final GenPackage genPkg : _filter) {
                    _builder.append("res.add(get");
                    String _factoryName = genPkg.getFactoryName();
                    _builder.append(_factoryName);
                    _builder.append("());");
                    _builder.newLineIfNotEmpty();
                  }
                }
                _builder.append("return res;");
                _builder.newLine();
              }
            };
            LanguageAdapterInferrer.this._jvmTypesBuilder.setBody(it, _client);
          }
        };
        JvmOperation _method = LanguageAdapterInferrer.this._jvmTypesBuilder.toMethod(l, "getFactories", LanguageAdapterInferrer.this.typeRefBuilder.typeRef(Set.class), _function_4);
        LanguageAdapterInferrer.this._jvmTypesBuilder.<JvmOperation>operator_add(_members_1, _method);
        EList<JvmMember> _members_2 = it.getMembers();
        final Procedure1<JvmOperation> _function_5 = new Procedure1<JvmOperation>() {
          @Override
          public void apply(final JvmOperation it) {
            EList<JvmAnnotationReference> _annotations = it.getAnnotations();
            JvmAnnotationReference _annotationRef = LanguageAdapterInferrer.this.jvmAnnRefBuilder.annotationRef(Override.class);
            LanguageAdapterInferrer.this._jvmTypesBuilder.<JvmAnnotationReference>operator_add(_annotations, _annotationRef);
            EList<JvmFormalParameter> _parameters = it.getParameters();
            JvmFormalParameter _parameter = LanguageAdapterInferrer.this._jvmTypesBuilder.toParameter(l, "uri", LanguageAdapterInferrer.this.typeRefBuilder.typeRef(String.class));
            LanguageAdapterInferrer.this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters, _parameter);
            StringConcatenationClient _client = new StringConcatenationClient() {
              @Override
              protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                _builder.append("this.adaptee.setURI(");
                _builder.append(URI.class);
                _builder.append(".createURI(uri));");
                _builder.newLineIfNotEmpty();
                _builder.append("this.adaptee.save(null);");
                _builder.newLine();
              }
            };
            LanguageAdapterInferrer.this._jvmTypesBuilder.setBody(it, _client);
            EList<JvmTypeReference> _exceptions = it.getExceptions();
            JvmTypeReference _typeRef = LanguageAdapterInferrer.this.typeRefBuilder.typeRef(IOException.class);
            LanguageAdapterInferrer.this._jvmTypesBuilder.<JvmTypeReference>operator_add(_exceptions, _typeRef);
          }
        };
        JvmOperation _method_1 = LanguageAdapterInferrer.this._jvmTypesBuilder.toMethod(l, "save", LanguageAdapterInferrer.this.typeRefBuilder.typeRef(Void.TYPE), _function_5);
        LanguageAdapterInferrer.this._jvmTypesBuilder.<JvmOperation>operator_add(_members_2, _method_1);
      }
    };
    this.acceptor.<JvmGenericType>accept(this._jvmTypesBuilder.toClass(l, this._namingHelper.adapterNameFor(l.getSyntax(), mt)), _function);
  }
  
  private void generateFactoryAdapters(final Language l, final ModelType mt) {
    final Function1<GenModel, EList<GenPackage>> _function = new Function1<GenModel, EList<GenPackage>>() {
      @Override
      public EList<GenPackage> apply(final GenModel it) {
        return it.getUsedGenPackages();
      }
    };
    final List<GenPackage> excluded = IterableExtensions.<GenPackage>toList(Iterables.<GenPackage>concat(IterableExtensions.<GenModel, EList<GenPackage>>map(this._modelingElementExtensions.getGenmodels(mt), _function)));
    final Function1<GenPackage, Boolean> _function_1 = new Function1<GenPackage, Boolean>() {
      @Override
      public Boolean apply(final GenPackage genPkg) {
        boolean _contains = excluded.contains(genPkg);
        return Boolean.valueOf((!_contains));
      }
    };
    final Consumer<GenPackage> _function_2 = new Consumer<GenPackage>() {
      @Override
      public void accept(final GenPackage genPkg) {
        LanguageAdapterInferrer.this.generateFactoryAdapter(genPkg, l, mt);
      }
    };
    IterableExtensions.<GenPackage>filter(this._modelingElementExtensions.getAllGenPkgs(mt), _function_1).forEach(_function_2);
  }
  
  /**
   * Generates a factory adapter that implements the abstract factory of
   * {@code mt}, delegates elements creation to the factory of {@code l},
   * and encapsulate the newly created elements in the appropriate adapters.
   * 
   * @see NamingHelper#factoryAdapterNameFor
   * @see EFactory
   */
  private void generateFactoryAdapter(final GenPackage genPkg, final Language l, final ModelType mt) {
    final Procedure1<JvmGenericType> _function = new Procedure1<JvmGenericType>() {
      @Override
      public void apply(final JvmGenericType it) {
        final String adaptersFactoryFqn = LanguageAdapterInferrer.this._namingHelper.getAdaptersFactoryNameFor(l.getSyntax(), mt);
        EList<JvmTypeReference> _superTypes = it.getSuperTypes();
        JvmTypeReference _typeRef = LanguageAdapterInferrer.this.typeRefBuilder.typeRef(genPkg.getQualifiedFactoryInterfaceName());
        LanguageAdapterInferrer.this._jvmTypesBuilder.<JvmTypeReference>operator_add(_superTypes, _typeRef);
        EList<JvmTypeReference> _superTypes_1 = it.getSuperTypes();
        JvmTypeReference _typeRef_1 = LanguageAdapterInferrer.this.typeRefBuilder.typeRef(EFactoryImpl.class);
        LanguageAdapterInferrer.this._jvmTypesBuilder.<JvmTypeReference>operator_add(_superTypes_1, _typeRef_1);
        EList<JvmMember> _members = it.getMembers();
        final Procedure1<JvmField> _function = new Procedure1<JvmField>() {
          @Override
          public void apply(final JvmField it) {
            StringConcatenationClient _client = new StringConcatenationClient() {
              @Override
              protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                _builder.append(adaptersFactoryFqn);
                _builder.append(".getInstance()");
              }
            };
            LanguageAdapterInferrer.this._jvmTypesBuilder.setInitializer(it, _client);
          }
        };
        JvmField _field = LanguageAdapterInferrer.this._jvmTypesBuilder.toField(l, "adaptersFactory", LanguageAdapterInferrer.this.typeRefBuilder.typeRef(adaptersFactoryFqn), _function);
        LanguageAdapterInferrer.this._jvmTypesBuilder.<JvmField>operator_add(_members, _field);
        final String pkgFactoryFqn = LanguageAdapterInferrer.this._namingHelper.getFactoryFqnFor(l.getSyntax(), genPkg.getEcorePackage());
        EList<JvmMember> _members_1 = it.getMembers();
        StringConcatenation _builder = new StringConcatenation();
        String _name = genPkg.getEcorePackage().getName();
        _builder.append(_name);
        _builder.append("Adaptee");
        final Procedure1<JvmField> _function_1 = new Procedure1<JvmField>() {
          @Override
          public void apply(final JvmField it) {
            StringConcatenationClient _client = new StringConcatenationClient() {
              @Override
              protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                _builder.append(pkgFactoryFqn);
                _builder.append(".eINSTANCE");
              }
            };
            LanguageAdapterInferrer.this._jvmTypesBuilder.setInitializer(it, _client);
          }
        };
        JvmField _field_1 = LanguageAdapterInferrer.this._jvmTypesBuilder.toField(l, _builder.toString(), LanguageAdapterInferrer.this.typeRefBuilder.typeRef(pkgFactoryFqn), _function_1);
        LanguageAdapterInferrer.this._jvmTypesBuilder.<JvmField>operator_add(_members_1, _field_1);
        final Function1<EClass, Boolean> _function_2 = new Function1<EClass, Boolean>() {
          @Override
          public Boolean apply(final EClass it) {
            return Boolean.valueOf(LanguageAdapterInferrer.this._ecoreExtensions.isInstantiable(it));
          }
        };
        final Consumer<EClass> _function_3 = new Consumer<EClass>() {
          @Override
          public void accept(final EClass cls) {
            String _name = cls.getName();
            String _plus = ("create" + _name);
            final Procedure1<JvmOperation> _function = new Procedure1<JvmOperation>() {
              @Override
              public void apply(final JvmOperation m) {
                EList<JvmAnnotationReference> _annotations = m.getAnnotations();
                JvmAnnotationReference _annotationRef = LanguageAdapterInferrer.this.jvmAnnRefBuilder.annotationRef(Override.class);
                LanguageAdapterInferrer.this._jvmTypesBuilder.<JvmAnnotationReference>operator_add(_annotations, _annotationRef);
                final Function1<EPackage, Boolean> _function = new Function1<EPackage, Boolean>() {
                  @Override
                  public Boolean apply(final EPackage it) {
                    final Function1<EClassifier, Boolean> _function = new Function1<EClassifier, Boolean>() {
                      @Override
                      public Boolean apply(final EClassifier it) {
                        return Boolean.valueOf(LanguageAdapterInferrer.this._mappingExtensions.namesMatch(LanguageAdapterInferrer.this.mapping, it, cls));
                      }
                    };
                    return Boolean.valueOf(IterableExtensions.<EClassifier>exists(it.getEClassifiers(), _function));
                  }
                };
                final EPackage associatedPkg = IterableExtensions.<EPackage>findFirst(LanguageAdapterInferrer.this._modelingElementExtensions.getPkgs(l.getSyntax()), _function);
                String _elvis = null;
                EList<ClassBinding> _rules = null;
                if (LanguageAdapterInferrer.this.mapping!=null) {
                  _rules=LanguageAdapterInferrer.this.mapping.getRules();
                }
                ClassBinding _findFirst = null;
                if (_rules!=null) {
                  final Function1<ClassBinding, Boolean> _function_1 = new Function1<ClassBinding, Boolean>() {
                    @Override
                    public Boolean apply(final ClassBinding it) {
                      String _to = it.getTo();
                      String _name = cls.getName();
                      return Boolean.valueOf(Objects.equal(_to, _name));
                    }
                  };
                  _findFirst=IterableExtensions.<ClassBinding>findFirst(_rules, _function_1);
                }
                String _from = null;
                if (_findFirst!=null) {
                  _from=_findFirst.getFrom();
                }
                if (_from != null) {
                  _elvis = _from;
                } else {
                  String _name = cls.getName();
                  _elvis = _name;
                }
                final String associatedCls = _elvis;
                final Consumer<ETypeParameter> _function_2 = new Consumer<ETypeParameter>() {
                  @Override
                  public void accept(final ETypeParameter t) {
                    EList<JvmTypeParameter> _typeParameters = m.getTypeParameters();
                    JvmTypeParameter _createJvmTypeParameter = TypesFactory.eINSTANCE.createJvmTypeParameter();
                    final Procedure1<JvmTypeParameter> _function = new Procedure1<JvmTypeParameter>() {
                      @Override
                      public void apply(final JvmTypeParameter it) {
                        it.setName(t.getName());
                      }
                    };
                    JvmTypeParameter _doubleArrow = ObjectExtensions.<JvmTypeParameter>operator_doubleArrow(_createJvmTypeParameter, _function);
                    LanguageAdapterInferrer.this._jvmTypesBuilder.<JvmTypeParameter>operator_add(_typeParameters, _doubleArrow);
                  }
                };
                cls.getETypeParameters().forEach(_function_2);
                StringConcatenationClient _client = new StringConcatenationClient() {
                  @Override
                  protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                    _builder.append("return adaptersFactory.create");
                    String _simpleAdapterNameFor = LanguageAdapterInferrer.this._namingHelper.simpleAdapterNameFor(l.getSyntax(), mt, cls);
                    _builder.append(_simpleAdapterNameFor);
                    _builder.append("(");
                    String _name = associatedPkg.getName();
                    _builder.append(_name);
                    _builder.append("Adaptee.create");
                    _builder.append(associatedCls);
                    _builder.append("(), null);");
                    _builder.newLineIfNotEmpty();
                  }
                };
                LanguageAdapterInferrer.this._jvmTypesBuilder.setBody(m, _client);
              }
            };
            final JvmOperation newCreate = LanguageAdapterInferrer.this._jvmTypesBuilder.toMethod(l, _plus, null, _function);
            newCreate.setReturnType(LanguageAdapterInferrer.this._melangeTypesBuilder.typeRef(mt, cls, Collections.<JvmOperation>unmodifiableList(CollectionLiterals.<JvmOperation>newArrayList(newCreate))));
            EList<JvmMember> _members = it.getMembers();
            LanguageAdapterInferrer.this._jvmTypesBuilder.<JvmOperation>operator_add(_members, newCreate);
          }
        };
        IterableExtensions.<EClass>filter(Iterables.<EClass>filter(genPkg.getEcorePackage().getEClassifiers(), EClass.class), _function_2).forEach(_function_3);
        EList<JvmMember> _members_2 = it.getMembers();
        final Procedure1<JvmOperation> _function_4 = new Procedure1<JvmOperation>() {
          @Override
          public void apply(final JvmOperation it) {
            EList<JvmAnnotationReference> _annotations = it.getAnnotations();
            JvmAnnotationReference _annotationRef = LanguageAdapterInferrer.this.jvmAnnRefBuilder.annotationRef(Override.class);
            LanguageAdapterInferrer.this._jvmTypesBuilder.<JvmAnnotationReference>operator_add(_annotations, _annotationRef);
            StringConcatenationClient _client = new StringConcatenationClient() {
              @Override
              protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                _builder.append("return get");
                String _packageInterfaceName = genPkg.getPackageInterfaceName();
                _builder.append(_packageInterfaceName);
                _builder.append("();");
                _builder.newLineIfNotEmpty();
              }
            };
            LanguageAdapterInferrer.this._jvmTypesBuilder.setBody(it, _client);
          }
        };
        JvmOperation _method = LanguageAdapterInferrer.this._jvmTypesBuilder.toMethod(l, "getEPackage", LanguageAdapterInferrer.this.typeRefBuilder.typeRef(EPackage.class), _function_4);
        LanguageAdapterInferrer.this._jvmTypesBuilder.<JvmOperation>operator_add(_members_2, _method);
        EList<JvmMember> _members_3 = it.getMembers();
        String _packageInterfaceName = genPkg.getPackageInterfaceName();
        String _plus = ("get" + _packageInterfaceName);
        final Procedure1<JvmOperation> _function_5 = new Procedure1<JvmOperation>() {
          @Override
          public void apply(final JvmOperation it) {
            StringConcatenationClient _client = new StringConcatenationClient() {
              @Override
              protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                _builder.append("return ");
                String _qualifiedPackageInterfaceName = genPkg.getQualifiedPackageInterfaceName();
                _builder.append(_qualifiedPackageInterfaceName);
                _builder.append(".eINSTANCE;");
                _builder.newLineIfNotEmpty();
              }
            };
            LanguageAdapterInferrer.this._jvmTypesBuilder.setBody(it, _client);
          }
        };
        JvmOperation _method_1 = LanguageAdapterInferrer.this._jvmTypesBuilder.toMethod(l, _plus, LanguageAdapterInferrer.this.typeRefBuilder.typeRef(genPkg.getQualifiedPackageInterfaceName()), _function_5);
        LanguageAdapterInferrer.this._jvmTypesBuilder.<JvmOperation>operator_add(_members_3, _method_1);
      }
    };
    this.acceptor.<JvmGenericType>accept(this._jvmTypesBuilder.toClass(l, this._namingHelper.factoryAdapterNameFor(genPkg, l.getSyntax(), mt)), _function);
  }
  
  /**
   * Generates a factory of adapters between {@code l} and {@code mt}.
   * 
   * @see NamingHelper#getAdaptersFactoryNameFor
   */
  private void generateAdaptersFactory(final Language l, final ModelType mt) {
    final String adapFactName = this._namingHelper.getAdaptersFactoryNameFor(l.getSyntax(), mt);
    final Procedure1<JvmGenericType> _function = new Procedure1<JvmGenericType>() {
      @Override
      public void apply(final JvmGenericType it) {
        EList<JvmTypeReference> _superTypes = it.getSuperTypes();
        JvmTypeReference _typeRef = LanguageAdapterInferrer.this.typeRefBuilder.typeRef(AdaptersFactory.class);
        LanguageAdapterInferrer.this._jvmTypesBuilder.<JvmTypeReference>operator_add(_superTypes, _typeRef);
        EList<JvmMember> _members = it.getMembers();
        final Procedure1<JvmField> _function = new Procedure1<JvmField>() {
          @Override
          public void apply(final JvmField it) {
            it.setStatic(true);
          }
        };
        JvmField _field = LanguageAdapterInferrer.this._jvmTypesBuilder.toField(l, "instance", LanguageAdapterInferrer.this.typeRefBuilder.typeRef(adapFactName), _function);
        LanguageAdapterInferrer.this._jvmTypesBuilder.<JvmField>operator_add(_members, _field);
        EList<JvmMember> _members_1 = it.getMembers();
        final Procedure1<JvmOperation> _function_1 = new Procedure1<JvmOperation>() {
          @Override
          public void apply(final JvmOperation it) {
            it.setStatic(true);
            StringConcatenationClient _client = new StringConcatenationClient() {
              @Override
              protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                _builder.append("if (instance == null) {");
                _builder.newLine();
                _builder.append("\t");
                _builder.append("instance = new ");
                _builder.append(adapFactName, "\t");
                _builder.append("();");
                _builder.newLineIfNotEmpty();
                _builder.append("}");
                _builder.newLine();
                _builder.append("return instance;");
                _builder.newLine();
              }
            };
            LanguageAdapterInferrer.this._jvmTypesBuilder.setBody(it, _client);
          }
        };
        JvmOperation _method = LanguageAdapterInferrer.this._jvmTypesBuilder.toMethod(l, "getInstance", LanguageAdapterInferrer.this.typeRefBuilder.typeRef(adapFactName), _function_1);
        LanguageAdapterInferrer.this._jvmTypesBuilder.<JvmOperation>operator_add(_members_1, _method);
        EList<JvmMember> _members_2 = it.getMembers();
        JvmField _field_1 = LanguageAdapterInferrer.this._jvmTypesBuilder.toField(l, "register", 
          LanguageAdapterInferrer.this.typeRefBuilder.typeRef(WeakHashMap.class, LanguageAdapterInferrer.this.typeRefBuilder.typeRef(EObject.class), LanguageAdapterInferrer.this.typeRefBuilder.typeRef(EObjectAdapter.class)));
        LanguageAdapterInferrer.this._jvmTypesBuilder.<JvmField>operator_add(_members_2, _field_1);
        EList<JvmMember> _members_3 = it.getMembers();
        final Procedure1<JvmConstructor> _function_2 = new Procedure1<JvmConstructor>() {
          @Override
          public void apply(final JvmConstructor it) {
            StringConcatenationClient _client = new StringConcatenationClient() {
              @Override
              protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                _builder.append("register = new ");
                _builder.append(WeakHashMap.class);
                _builder.append("();");
                _builder.newLineIfNotEmpty();
              }
            };
            LanguageAdapterInferrer.this._jvmTypesBuilder.setBody(it, _client);
          }
        };
        JvmConstructor _constructor = LanguageAdapterInferrer.this._jvmTypesBuilder.toConstructor(l, _function_2);
        LanguageAdapterInferrer.this._jvmTypesBuilder.<JvmConstructor>operator_add(_members_3, _constructor);
        EList<JvmMember> _members_4 = it.getMembers();
        final Procedure1<JvmOperation> _function_3 = new Procedure1<JvmOperation>() {
          @Override
          public void apply(final JvmOperation it) {
            EList<JvmFormalParameter> _parameters = it.getParameters();
            JvmFormalParameter _parameter = LanguageAdapterInferrer.this._jvmTypesBuilder.toParameter(l, "o", LanguageAdapterInferrer.this.typeRefBuilder.typeRef(EObject.class));
            LanguageAdapterInferrer.this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters, _parameter);
            EList<JvmFormalParameter> _parameters_1 = it.getParameters();
            JvmFormalParameter _parameter_1 = LanguageAdapterInferrer.this._jvmTypesBuilder.toParameter(l, "res", LanguageAdapterInferrer.this.typeRefBuilder.typeRef(Resource.class));
            LanguageAdapterInferrer.this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_1, _parameter_1);
            final Function1<EClass, Boolean> _function = new Function1<EClass, Boolean>() {
              @Override
              public Boolean apply(final EClass it) {
                return Boolean.valueOf(((LanguageAdapterInferrer.this._languageExtensions.hasAdapterFor(l, mt, it) && LanguageAdapterInferrer.this._ecoreExtensions.isInstantiable(it)) && LanguageAdapterInferrer.this._ecoreExtensions.isAbstractable(it)));
              }
            };
            final Iterable<EClass> orderedCls = LanguageAdapterInferrer.this._ecoreExtensions.sortByClassInheritance(IterableExtensions.<EClass>filter(LanguageAdapterInferrer.this._modelingElementExtensions.getAllClasses(mt), _function));
            StringConcatenationClient _client = new StringConcatenationClient() {
              @Override
              protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                {
                  for(final EClass cls : orderedCls) {
                    final String clsFqn = LanguageAdapterInferrer.this._namingHelper.getFqnFor(l.getSyntax(), cls);
                    _builder.newLineIfNotEmpty();
                    _builder.append("if (o instanceof ");
                    _builder.append(clsFqn);
                    _builder.append("){");
                    _builder.newLineIfNotEmpty();
                    _builder.append("\t");
                    _builder.append("return create");
                    String _name = cls.getName();
                    _builder.append(_name, "\t");
                    _builder.append("Adapter((");
                    _builder.append(clsFqn, "\t");
                    _builder.append(") o, res);");
                    _builder.newLineIfNotEmpty();
                    _builder.append("}");
                    _builder.newLine();
                  }
                }
                _builder.newLine();
                _builder.append("return null;");
                _builder.newLine();
              }
            };
            LanguageAdapterInferrer.this._jvmTypesBuilder.setBody(it, _client);
          }
        };
        JvmOperation _method_1 = LanguageAdapterInferrer.this._jvmTypesBuilder.toMethod(l, "createAdapter", LanguageAdapterInferrer.this.typeRefBuilder.typeRef(EObjectAdapter.class), _function_3);
        LanguageAdapterInferrer.this._jvmTypesBuilder.<JvmOperation>operator_add(_members_4, _method_1);
        final Function1<EClass, Boolean> _function_4 = new Function1<EClass, Boolean>() {
          @Override
          public Boolean apply(final EClass it) {
            return Boolean.valueOf(LanguageAdapterInferrer.this._ecoreExtensions.isAbstractable(it));
          }
        };
        final Consumer<EClass> _function_5 = new Consumer<EClass>() {
          @Override
          public void accept(final EClass cls) {
            final String adapName = LanguageAdapterInferrer.this._namingHelper.adapterNameFor(l.getSyntax(), mt, cls);
            final Function1<EClass, Boolean> _function = new Function1<EClass, Boolean>() {
              @Override
              public Boolean apply(final EClass it) {
                return Boolean.valueOf(LanguageAdapterInferrer.this._mappingExtensions.namesMatch(LanguageAdapterInferrer.this.mapping, it, cls));
              }
            };
            final EClass mmCls = IterableExtensions.<EClass>findFirst(LanguageAdapterInferrer.this._modelingElementExtensions.getAllClasses(l.getSyntax()), _function);
            EList<JvmMember> _members = it.getMembers();
            StringConcatenation _builder = new StringConcatenation();
            _builder.append("create");
            String _name = cls.getName();
            _builder.append(_name);
            _builder.append("Adapter");
            final Procedure1<JvmOperation> _function_1 = new Procedure1<JvmOperation>() {
              @Override
              public void apply(final JvmOperation it) {
                EList<JvmFormalParameter> _parameters = it.getParameters();
                JvmFormalParameter _parameter = LanguageAdapterInferrer.this._jvmTypesBuilder.toParameter(l, "adaptee", LanguageAdapterInferrer.this.typeRefBuilder.typeRef(LanguageAdapterInferrer.this._namingHelper.getFqnFor(l.getSyntax(), mmCls)));
                LanguageAdapterInferrer.this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters, _parameter);
                EList<JvmFormalParameter> _parameters_1 = it.getParameters();
                JvmFormalParameter _parameter_1 = LanguageAdapterInferrer.this._jvmTypesBuilder.toParameter(l, "res", LanguageAdapterInferrer.this.typeRefBuilder.typeRef(Resource.class));
                LanguageAdapterInferrer.this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_1, _parameter_1);
                StringConcatenationClient _client = new StringConcatenationClient() {
                  @Override
                  protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                    _builder.append("if (adaptee == null)");
                    _builder.newLine();
                    _builder.append("\t");
                    _builder.append("return null;");
                    _builder.newLine();
                    _builder.append(EObjectAdapter.class);
                    _builder.append(" adapter = register.get(adaptee);");
                    _builder.newLineIfNotEmpty();
                    _builder.append("if(adapter != null)");
                    _builder.newLine();
                    _builder.append("\t ");
                    _builder.append("return (");
                    _builder.append(adapName, "\t ");
                    _builder.append(") adapter;");
                    _builder.newLineIfNotEmpty();
                    _builder.append("else {");
                    _builder.newLine();
                    _builder.append("\t");
                    _builder.append("adapter = new ");
                    _builder.append(adapName, "\t");
                    _builder.append("();");
                    _builder.newLineIfNotEmpty();
                    _builder.append("\t");
                    _builder.append("adapter.setAdaptee(adaptee);");
                    _builder.newLine();
                    _builder.append("\t");
                    _builder.append("adapter.setResource(res);");
                    _builder.newLine();
                    _builder.append("\t");
                    _builder.append("register.put(adaptee, adapter);");
                    _builder.newLine();
                    _builder.append("\t");
                    _builder.append("return (");
                    _builder.append(adapName, "\t");
                    _builder.append(") adapter;");
                    _builder.newLineIfNotEmpty();
                    _builder.append("}");
                    _builder.newLine();
                  }
                };
                LanguageAdapterInferrer.this._jvmTypesBuilder.setBody(it, _client);
              }
            };
            JvmOperation _method = LanguageAdapterInferrer.this._jvmTypesBuilder.toMethod(l, _builder.toString(), LanguageAdapterInferrer.this.typeRefBuilder.typeRef(adapName), _function_1);
            LanguageAdapterInferrer.this._jvmTypesBuilder.<JvmOperation>operator_add(_members, _method);
          }
        };
        IterableExtensions.<EClass>filter(LanguageAdapterInferrer.this._modelingElementExtensions.getAllClasses(mt), _function_4).forEach(_function_5);
      }
    };
    this.acceptor.<JvmGenericType>accept(this._jvmTypesBuilder.toClass(l, adapFactName), _function);
  }
}
