/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.melange.utils;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import fr.inria.diverse.melange.ast.AspectExtensions;
import fr.inria.diverse.melange.lib.EcoreExtensions;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EOperation;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EParameter;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.EcoreFactory;
import org.eclipse.xtext.common.types.JvmAnnotationReference;
import org.eclipse.xtext.common.types.JvmAnnotationValue;
import org.eclipse.xtext.common.types.JvmCustomAnnotationValue;
import org.eclipse.xtext.common.types.JvmDeclaredType;
import org.eclipse.xtext.common.types.JvmEnumerationLiteral;
import org.eclipse.xtext.common.types.JvmEnumerationType;
import org.eclipse.xtext.common.types.JvmField;
import org.eclipse.xtext.common.types.JvmFormalParameter;
import org.eclipse.xtext.common.types.JvmGenericType;
import org.eclipse.xtext.common.types.JvmMember;
import org.eclipse.xtext.common.types.JvmOperation;
import org.eclipse.xtext.common.types.JvmType;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.common.types.JvmVisibility;
import org.eclipse.xtext.xbase.XStringLiteral;
import org.eclipse.xtext.xbase.jvmmodel.JvmTypeReferenceBuilder;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure2;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * Infers the minimal Ecore file (an {@link EPackage}) corresponding to the
 * "modeling intention" of a K3 aspect. For example, from the following aspect:
 * 
 * <code>
 * \@Aspect(className = A)
 * class AspectA {
 *     public int foo
 *     def void bar() {}
 * }
 * </code>
 * 
 * it will infer a new {@link EPackage} containing an {@link EClass} {@code A}
 * with an {@link EAttribute} {@code foo} and an {@link EOperation} {@code foo}.
 */
@SuppressWarnings("all")
public class AspectToEcore {
  @Inject
  @Extension
  private AspectExtensions _aspectExtensions;

  @Inject
  @Extension
  private EcoreExtensions _ecoreExtensions;

  @Inject
  @Extension
  private TypeReferencesHelper _typeReferencesHelper;

  @Inject
  private JvmTypeReferenceBuilder.Factory typeRefBuilderFactory;

  private static final String CONTAINMENT_ANNOTATION_FQN = "fr.inria.diverse.melange.annotation.Containment";

  private static final String UNIQUE_ANNOTATION_FQN = "fr.inria.diverse.melange.annotation.Unique";

  private static final String OPPOSITE_ANNOTATION_FQN = "fr.inria.diverse.melange.annotation.Opposite";

  private static final List<String> K3_PREFIXES = Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("_privk3", "super_"));

  public static final String PROP_NAME = "AspectProperties";

  public static final String SELF_PARAM_NAME = "_self";

  /**
   * Analyzes the aspect {@code aspect}, woven on the {@link EClass}
   * {@code baseCls} contained in the {@link EPackage} {@code basePkg}, and
   * returns the corresponding {@link EPackage} describing its modeling intention.
   */
  public EPackage inferEcoreFragment(final JvmDeclaredType aspect, final EClass baseCls, final Set<EPackage> basePkgs) {
    EPackage _xifexpression = null;
    if ((baseCls != null)) {
      _xifexpression = this.copyPackage(baseCls);
    } else {
      EPackage _createEPackage = EcoreFactory.eINSTANCE.createEPackage();
      final Procedure1<EPackage> _function = (EPackage it) -> {
        it.setName(IterableExtensions.<EPackage>head(basePkgs).getName());
        it.setNsPrefix(IterableExtensions.<EPackage>head(basePkgs).getNsPrefix());
        it.setNsURI(IterableExtensions.<EPackage>head(basePkgs).getNsURI());
      };
      _xifexpression = ObjectExtensions.<EPackage>operator_doubleArrow(_createEPackage, _function);
    }
    final EPackage aspPkg = _xifexpression;
    final EPackage aspTopPkg = this._ecoreExtensions.getRootPackage(aspPkg);
    EClass _createEClass = EcoreFactory.eINSTANCE.createEClass();
    final Procedure1<EClass> _function_1 = (EClass cls) -> {
      String _xifexpression_1 = null;
      if ((baseCls != null)) {
        _xifexpression_1 = baseCls.getName();
      } else {
        _xifexpression_1 = aspect.getSimpleName();
      }
      cls.setName(_xifexpression_1);
      boolean _xifexpression_2 = false;
      if ((baseCls != null)) {
        _xifexpression_2 = baseCls.isAbstract();
      } else {
        _xifexpression_2 = aspect.isAbstract();
      }
      cls.setAbstract(_xifexpression_2);
      boolean _xifexpression_3 = false;
      if ((baseCls != null)) {
        _xifexpression_3 = baseCls.isInterface();
      } else {
        _xifexpression_3 = false;
      }
      cls.setInterface(_xifexpression_3);
      if ((baseCls == null)) {
        this._ecoreExtensions.addAspectAnnotation(cls);
        if (((aspect.getExtendedClass() != null) && (!Objects.equal(aspect.getExtendedClass().getSimpleName(), "Object")))) {
          EList<EClass> _eSuperTypes = cls.getESuperTypes();
          EClass _orCreateClass = this._ecoreExtensions.getOrCreateClass(aspTopPkg, 
            aspect.getExtendedClass().getQualifiedName());
          _eSuperTypes.add(_orCreateClass);
        }
      }
    };
    final EClass aspCls = ObjectExtensions.<EClass>operator_doubleArrow(_createEClass, _function_1);
    EList<EClassifier> _eClassifiers = aspPkg.getEClassifiers();
    _eClassifiers.add(aspCls);
    this.inferAspectFieldsEcoreFragment(aspect, aspCls, basePkgs, aspTopPkg);
    this.inferAspectMethodsEcoreFragment(aspect, aspCls, baseCls, aspPkg, basePkgs, aspTopPkg);
    return aspTopPkg;
  }

  /**
   * infers the fields to be added in the ecore fragment
   */
  private void inferAspectFieldsEcoreFragment(final JvmDeclaredType aspect, final EClass aspCls, final Set<EPackage> basePkgs, final EPackage aspTopPkg) {
    final Function1<JvmField, Boolean> _function = (JvmField it) -> {
      return Boolean.valueOf((Objects.equal(it.getVisibility(), JvmVisibility.PUBLIC) && (!it.isStatic())));
    };
    final Consumer<JvmField> _function_1 = (JvmField field) -> {
      final JvmTypeReference fieldType = field.getType();
      int _xifexpression = (int) 0;
      boolean _isList = this._typeReferencesHelper.isList(fieldType);
      if (_isList) {
        _xifexpression = (-1);
      } else {
        _xifexpression = 1;
      }
      final int upperB = _xifexpression;
      JvmType _xifexpression_1 = null;
      boolean _isList_1 = this._typeReferencesHelper.isList(fieldType);
      if (_isList_1) {
        _xifexpression_1 = this._typeReferencesHelper.getContainedElementsType(fieldType);
      } else {
        _xifexpression_1 = fieldType.getType();
      }
      final JvmType realType = _xifexpression_1;
      EClass _xifexpression_2 = null;
      String _qualifiedName = realType.getQualifiedName();
      String _uniqueId = this._ecoreExtensions.getUniqueId(aspCls);
      boolean _equals = Objects.equal(_qualifiedName, _uniqueId);
      if (_equals) {
        _xifexpression_2 = aspCls;
      } else {
        _xifexpression_2 = this._ecoreExtensions.findClass(basePkgs, realType.getQualifiedName());
      }
      final EClass find = _xifexpression_2;
      if ((find != null)) {
        EList<EStructuralFeature> _eStructuralFeatures = aspCls.getEStructuralFeatures();
        EReference _createEReference = EcoreFactory.eINSTANCE.createEReference();
        final Procedure1<EReference> _function_2 = (EReference it) -> {
          it.setName(field.getSimpleName());
          it.setEType(this._ecoreExtensions.getOrCreateClass(aspTopPkg, this.toQualifiedName(find)));
          it.setUpperBound(upperB);
          it.setContainment(this.isContainment(field));
          this._ecoreExtensions.addAspectAnnotation(it);
          it.setUnique(this.isUnique(field));
        };
        EReference _doubleArrow = ObjectExtensions.<EReference>operator_doubleArrow(_createEReference, _function_2);
        _eStructuralFeatures.add(_doubleArrow);
      } else {
        EList<EStructuralFeature> _eStructuralFeatures_1 = aspCls.getEStructuralFeatures();
        EAttribute _createEAttribute = EcoreFactory.eINSTANCE.createEAttribute();
        final Procedure1<EAttribute> _function_3 = (EAttribute it) -> {
          it.setName(field.getSimpleName());
          EClassifier _xifexpression_3 = null;
          if ((realType instanceof JvmEnumerationType)) {
            final Function1<JvmEnumerationLiteral, String> _function_4 = (JvmEnumerationLiteral it_1) -> {
              return it_1.getSimpleName();
            };
            _xifexpression_3 = this._ecoreExtensions.getOrCreateEnum(aspTopPkg, ((JvmEnumerationType)realType).getSimpleName(), 
              ListExtensions.<JvmEnumerationLiteral, String>map(((JvmEnumerationType)realType).getLiterals(), _function_4));
          } else {
            _xifexpression_3 = this._ecoreExtensions.getOrCreateDataType(aspTopPkg, realType.getSimpleName(), 
              realType.getQualifiedName());
          }
          it.setEType(_xifexpression_3);
          it.setUpperBound(upperB);
          this._ecoreExtensions.addAspectAnnotation(it);
          it.setUnique(this.isUnique(field));
        };
        EAttribute _doubleArrow_1 = ObjectExtensions.<EAttribute>operator_doubleArrow(_createEAttribute, _function_3);
        _eStructuralFeatures_1.add(_doubleArrow_1);
      }
    };
    IterableExtensions.<JvmField>filter(aspect.getDeclaredFields(), _function).forEach(_function_1);
  }

  /**
   * infers the methods to be added in the ecore fragment
   */
  private void inferAspectMethodsEcoreFragment(final JvmDeclaredType aspect, final EClass aspCls, final EClass baseCls, final EPackage aspPkg, final Set<EPackage> basePkgs, final EPackage aspTopPkg) {
    final JvmTypeReferenceBuilder typeRefBuilder = this.typeRefBuilderFactory.create(aspect.eResource().getResourceSet());
    final Function1<JvmOperation, Boolean> _function = (JvmOperation it) -> {
      return Boolean.valueOf(((!this.isK3Specific(it)) && Objects.equal(it.getVisibility(), JvmVisibility.PUBLIC)));
    };
    final Consumer<JvmOperation> _function_1 = (JvmOperation op) -> {
      final String featureName = this.findFeatureNameFor(aspect, op, typeRefBuilder);
      if ((featureName == null)) {
        this.inferAspectMethodsEcoreFragmentFromOperation(aspect, aspCls, baseCls, aspPkg, basePkgs, aspTopPkg, op);
      } else {
        final Function1<EStructuralFeature, Boolean> _function_2 = (EStructuralFeature it) -> {
          String _name = it.getName();
          return Boolean.valueOf(Objects.equal(_name, featureName));
        };
        boolean _exists = IterableExtensions.<EStructuralFeature>exists(aspCls.getEStructuralFeatures(), _function_2);
        boolean _not = (!_exists);
        if (_not) {
          this.inferAspectStructuralFeaturesEcoreFragmentFromOperation(aspect, aspCls, baseCls, aspPkg, basePkgs, aspTopPkg, featureName, op);
        }
      }
    };
    IterableExtensions.<JvmOperation>filter(aspect.getDeclaredOperations(), _function).forEach(_function_1);
  }

  private void inferAspectMethodsEcoreFragmentFromOperation(final JvmDeclaredType aspect, final EClass aspCls, final EClass baseCls, final EPackage aspPkg, final Set<EPackage> basePkgs, final EPackage aspTopPkg, final JvmOperation op) {
    int _xifexpression = (int) 0;
    boolean _isList = this._typeReferencesHelper.isList(op.getReturnType());
    if (_isList) {
      _xifexpression = (-1);
    } else {
      _xifexpression = 1;
    }
    final int upperB = _xifexpression;
    JvmType _xifexpression_1 = null;
    boolean _isList_1 = this._typeReferencesHelper.isList(op.getReturnType());
    if (_isList_1) {
      _xifexpression_1 = this._typeReferencesHelper.getContainedElementsType(op.getReturnType());
    } else {
      _xifexpression_1 = op.getReturnType().getType();
    }
    final JvmType realType = _xifexpression_1;
    EClass _xifexpression_2 = null;
    String _qualifiedName = realType.getQualifiedName();
    String _uniqueId = this._ecoreExtensions.getUniqueId(aspCls);
    boolean _equals = Objects.equal(_qualifiedName, _uniqueId);
    if (_equals) {
      _xifexpression_2 = aspCls;
    } else {
      _xifexpression_2 = this._ecoreExtensions.findClass(basePkgs, realType.getQualifiedName());
    }
    final EClass retCls = _xifexpression_2;
    final Function1<EOperation, Boolean> _function = (EOperation it) -> {
      String _name = it.getName();
      String _simpleName = op.getSimpleName();
      return Boolean.valueOf(Objects.equal(_name, _simpleName));
    };
    boolean _exists = IterableExtensions.<EOperation>exists(aspCls.getEOperations(), _function);
    boolean _not = (!_exists);
    if (_not) {
      EList<EOperation> _eOperations = aspCls.getEOperations();
      EOperation _createEOperation = EcoreFactory.eINSTANCE.createEOperation();
      final Procedure1<EOperation> _function_1 = (EOperation it) -> {
        it.setName(op.getSimpleName());
        final Procedure2<JvmFormalParameter, Integer> _function_2 = (JvmFormalParameter p, Integer i) -> {
          if (((((i).intValue() > 0) || ((((i).intValue() == 0) && (!Objects.equal(p.getName(), AspectToEcore.SELF_PARAM_NAME))) && (!op.isStatic()))) || 
            (!this._aspectExtensions.hasAspectAnnotation(aspect)))) {
            final JvmType pType = p.getParameterType().getType();
            int _xifexpression_3 = (int) 0;
            boolean _isList_2 = this._typeReferencesHelper.isList(p.getParameterType());
            if (_isList_2) {
              _xifexpression_3 = (-1);
            } else {
              _xifexpression_3 = 1;
            }
            final int upperBP = _xifexpression_3;
            JvmType _xifexpression_4 = null;
            boolean _isList_3 = this._typeReferencesHelper.isList(p.getParameterType());
            if (_isList_3) {
              _xifexpression_4 = this._typeReferencesHelper.getContainedElementsType(p.getParameterType());
            } else {
              _xifexpression_4 = pType;
            }
            final JvmType realTypeP = _xifexpression_4;
            EClass _xifexpression_5 = null;
            String _qualifiedName_1 = realTypeP.getQualifiedName();
            String _uniqueId_1 = this._ecoreExtensions.getUniqueId(aspCls);
            boolean _equals_1 = Objects.equal(_qualifiedName_1, _uniqueId_1);
            if (_equals_1) {
              _xifexpression_5 = aspCls;
            } else {
              _xifexpression_5 = this._ecoreExtensions.findClass(basePkgs, realTypeP.getQualifiedName());
            }
            final EClass attrCls = _xifexpression_5;
            EList<EParameter> _eParameters = it.getEParameters();
            EParameter _createEParameter = EcoreFactory.eINSTANCE.createEParameter();
            final Procedure1<EParameter> _function_3 = (EParameter pp) -> {
              pp.setName(p.getSimpleName());
              pp.setUpperBound(upperBP);
              EClassifier _xifexpression_6 = null;
              if ((attrCls != null)) {
                _xifexpression_6 = this._ecoreExtensions.getOrCreateClass(aspTopPkg, this._ecoreExtensions.getUniqueId(attrCls));
              } else {
                EClassifier _xifexpression_7 = null;
                if ((realTypeP instanceof JvmEnumerationType)) {
                  final Function1<JvmEnumerationLiteral, String> _function_4 = (JvmEnumerationLiteral it_1) -> {
                    return it_1.getSimpleName();
                  };
                  _xifexpression_7 = this._ecoreExtensions.getOrCreateEnum(aspTopPkg, ((JvmEnumerationType)realTypeP).getSimpleName(), 
                    ListExtensions.<JvmEnumerationLiteral, String>map(((JvmEnumerationType)realTypeP).getLiterals(), _function_4));
                } else {
                  _xifexpression_7 = this._ecoreExtensions.getOrCreateDataType(aspTopPkg, realTypeP.getSimpleName(), 
                    realTypeP.getQualifiedName());
                }
                _xifexpression_6 = _xifexpression_7;
              }
              pp.setEType(_xifexpression_6);
            };
            EParameter _doubleArrow = ObjectExtensions.<EParameter>operator_doubleArrow(_createEParameter, _function_3);
            _eParameters.add(_doubleArrow);
          }
        };
        IterableExtensions.<JvmFormalParameter>forEach(op.getParameters(), _function_2);
        if (((!Objects.equal(op.getReturnType().getSimpleName(), "void")) && (op.getReturnType().getSimpleName() != "null"))) {
          it.setUpperBound(upperB);
          EClassifier _xifexpression_3 = null;
          if ((retCls != null)) {
            _xifexpression_3 = this._ecoreExtensions.getOrCreateClass(aspTopPkg, this._ecoreExtensions.getUniqueId(retCls));
          } else {
            EClassifier _xifexpression_4 = null;
            if ((realType instanceof JvmEnumerationType)) {
              final Function1<JvmEnumerationLiteral, String> _function_3 = (JvmEnumerationLiteral it_1) -> {
                return it_1.getSimpleName();
              };
              _xifexpression_4 = this._ecoreExtensions.getOrCreateEnum(aspTopPkg, ((JvmEnumerationType)realType).getSimpleName(), 
                ListExtensions.<JvmEnumerationLiteral, String>map(((JvmEnumerationType)realType).getLiterals(), _function_3));
            } else {
              _xifexpression_4 = this._ecoreExtensions.getOrCreateDataType(aspTopPkg, realType.getSimpleName(), 
                realType.getQualifiedName());
            }
            _xifexpression_3 = _xifexpression_4;
          }
          it.setEType(_xifexpression_3);
        }
        this._ecoreExtensions.addAspectAnnotation(it);
      };
      EOperation _doubleArrow = ObjectExtensions.<EOperation>operator_doubleArrow(_createEOperation, _function_1);
      _eOperations.add(_doubleArrow);
    }
  }

  /**
   * the operation has been identified has implementing a Structural feature
   * create this feature if it doesn't exists yet in the class
   */
  private void inferAspectStructuralFeaturesEcoreFragmentFromOperation(final JvmDeclaredType aspect, final EClass aspCls, final EClass baseCls, final EPackage aspPkg, final Set<EPackage> basePkgs, final EPackage aspTopPkg, final String featureName, final JvmOperation op) {
    JvmTypeReference _xifexpression = null;
    if ((op.getSimpleName().startsWith("get") || (op.getParameters().size() == 1))) {
      _xifexpression = op.getReturnType();
    } else {
      _xifexpression = op.getParameters().get(1).getParameterType();
    }
    final JvmTypeReference retType = _xifexpression;
    int _xifexpression_1 = (int) 0;
    boolean _isList = this._typeReferencesHelper.isList(op.getReturnType());
    if (_isList) {
      _xifexpression_1 = (-1);
    } else {
      _xifexpression_1 = 1;
    }
    final int upperB = _xifexpression_1;
    JvmType _xifexpression_2 = null;
    boolean _isList_1 = this._typeReferencesHelper.isList(op.getReturnType());
    if (_isList_1) {
      _xifexpression_2 = this._typeReferencesHelper.getContainedElementsType(retType);
    } else {
      _xifexpression_2 = retType.getType();
    }
    final JvmType realType = _xifexpression_2;
    EClass _xifexpression_3 = null;
    String _qualifiedName = realType.getQualifiedName();
    String _uniqueId = this._ecoreExtensions.getUniqueId(aspCls);
    boolean _equals = Objects.equal(_qualifiedName, _uniqueId);
    if (_equals) {
      _xifexpression_3 = aspCls;
    } else {
      _xifexpression_3 = this._ecoreExtensions.findClass(basePkgs, realType.getQualifiedName());
    }
    final EClass find = _xifexpression_3;
    if ((find != null)) {
      EList<EStructuralFeature> _eStructuralFeatures = aspCls.getEStructuralFeatures();
      EReference _createEReference = EcoreFactory.eINSTANCE.createEReference();
      final Procedure1<EReference> _function = (EReference it) -> {
        it.setName(featureName);
        it.setEType(this._ecoreExtensions.getOrCreateClass(aspTopPkg, this.toQualifiedName(find)));
        it.setUpperBound(upperB);
        it.setContainment(this.isContainment(op));
        this._ecoreExtensions.addAspectAnnotation(it);
        it.setUnique(this.isUnique(op));
        boolean _isOpposite = this.isOpposite(op);
        if (_isOpposite) {
          this._ecoreExtensions.addOppositeAnnotation(it, this.getOppositeValue(op));
        }
      };
      EReference _doubleArrow = ObjectExtensions.<EReference>operator_doubleArrow(_createEReference, _function);
      _eStructuralFeatures.add(_doubleArrow);
    } else {
      EList<EStructuralFeature> _eStructuralFeatures_1 = aspCls.getEStructuralFeatures();
      EAttribute _createEAttribute = EcoreFactory.eINSTANCE.createEAttribute();
      final Procedure1<EAttribute> _function_1 = (EAttribute it) -> {
        it.setName(featureName);
        EClassifier _xifexpression_4 = null;
        if ((realType instanceof JvmEnumerationType)) {
          final Function1<JvmEnumerationLiteral, String> _function_2 = (JvmEnumerationLiteral it_1) -> {
            return it_1.getSimpleName();
          };
          _xifexpression_4 = this._ecoreExtensions.getOrCreateEnum(aspTopPkg, ((JvmEnumerationType)realType).getSimpleName(), 
            ListExtensions.<JvmEnumerationLiteral, String>map(((JvmEnumerationType)realType).getLiterals(), _function_2));
        } else {
          _xifexpression_4 = this._ecoreExtensions.getOrCreateDataType(aspTopPkg, realType.getSimpleName(), 
            realType.getQualifiedName());
        }
        it.setEType(_xifexpression_4);
        it.setUpperBound(upperB);
        this._ecoreExtensions.addAspectAnnotation(it);
        boolean _isContainment = this.isContainment(op);
        if (_isContainment) {
          this._ecoreExtensions.addContainmentAnnotation(it);
        }
        it.setUnique(this.isUnique(op));
      };
      EAttribute _doubleArrow_1 = ObjectExtensions.<EAttribute>operator_doubleArrow(_createEAttribute, _function_1);
      _eStructuralFeatures_1.add(_doubleArrow_1);
    }
  }

  /**
   * For the getter or setter {@code op} in {@code type}, infers the
   * corresponding feature name (eg. getXyz()/setXyz() associated to the
   * "xyz" feature).
   * 
   * @param type The {@link JvmDeclaredType} of an aspect.
   * @param op   A {@link JvmOperation} of {@code type}.
   * @return the corresponding feature name, or null if it cannot be determined.
   */
  public String findFeatureNameFor(final JvmDeclaredType type, final JvmOperation op, final JvmTypeReferenceBuilder typeRefBuilder) {
    if ((((((op.getSimpleName().startsWith("get") && Character.isUpperCase(op.getSimpleName().charAt(3))) && (op.getParameters().size() == 1)) && (!Objects.equal(op.getReturnType().getSimpleName(), "void"))) && IterableExtensions.<JvmOperation>exists(type.getDeclaredOperations(), ((Function1<JvmOperation, Boolean>) (JvmOperation opp) -> {
      return Boolean.valueOf(((Objects.equal(opp.getSimpleName(), op.getSimpleName().replaceFirst("get", "set")) && Objects.equal(opp.getParameters().get(1).getParameterType().getQualifiedName(), op.getReturnType().getQualifiedName())) && Objects.equal(opp.getReturnType().getSimpleName(), "void")));
    }))) || ((((op.getSimpleName().startsWith("set") && Character.isUpperCase(op.getSimpleName().charAt(3))) && (op.getParameters().size() == 2)) && Objects.equal(op.getReturnType().getSimpleName(), "void")) && IterableExtensions.<JvmOperation>exists(type.getDeclaredOperations(), ((Function1<JvmOperation, Boolean>) (JvmOperation opp) -> {
      return Boolean.valueOf((Objects.equal(opp.getSimpleName(), op.getSimpleName().replaceFirst("set", "get")) && Objects.equal(opp.getReturnType().getQualifiedName(), op.getParameters().get(1).getParameterType().getQualifiedName())));
    }))))) {
      return StringExtensions.toFirstLower(op.getSimpleName().substring(3, op.getSimpleName().length()));
    } else {
      final Function1<JvmOperation, Boolean> _function = (JvmOperation opp) -> {
        return Boolean.valueOf((((opp != op) && Objects.equal(opp.getSimpleName(), op.getSimpleName())) && 
          ((((((op.getParameters().size() == 1) && (!Objects.equal(op.getReturnType().getSimpleName(), "void"))) && (opp.getParameters().size() == 2)) && Objects.equal(opp.getReturnType().getSimpleName(), "void")) && Objects.equal(op.getReturnType().getQualifiedName(), opp.getParameters().get(1).getParameterType().getQualifiedName())) || 
            (((((opp.getParameters().size() == 1) && (!Objects.equal(opp.getReturnType().getSimpleName(), "void"))) && (op.getParameters().size() == 2)) && Objects.equal(op.getReturnType().getSimpleName(), "void")) && Objects.equal(op.getParameters().get(1).getParameterType().getQualifiedName(), opp.getReturnType().getQualifiedName())))));
      };
      boolean _exists = IterableExtensions.<JvmOperation>exists(type.getDeclaredOperations(), _function);
      if (_exists) {
        return op.getSimpleName();
      } else {
        boolean _isGetter = this.isGetter(op, typeRefBuilder);
        if (_isGetter) {
          return op.getSimpleName();
        } else {
          if ((((((op.getSimpleName().startsWith("get") && Character.isUpperCase(op.getSimpleName().charAt(3))) && (op.getParameters().size() == 0)) && (!Objects.equal(op.getReturnType().getSimpleName(), "void"))) && IterableExtensions.<JvmOperation>exists(type.getDeclaredOperations(), ((Function1<JvmOperation, Boolean>) (JvmOperation opp) -> {
            return Boolean.valueOf((((Objects.equal(opp.getSimpleName(), op.getSimpleName().replaceFirst("get", "set")) && (opp.getParameters().size() == 1)) && Objects.equal(opp.getParameters().get(0).getParameterType().getQualifiedName(), op.getReturnType().getQualifiedName())) && Objects.equal(opp.getReturnType().getSimpleName(), "void")));
          }))) || ((((op.getSimpleName().startsWith("set") && Character.isUpperCase(op.getSimpleName().charAt(3))) && (op.getParameters().size() == 1)) && Objects.equal(op.getReturnType().getSimpleName(), "void")) && IterableExtensions.<JvmOperation>exists(type.getDeclaredOperations(), ((Function1<JvmOperation, Boolean>) (JvmOperation opp) -> {
            return Boolean.valueOf(((Objects.equal(opp.getSimpleName(), op.getSimpleName().replaceFirst("set", "get")) && (opp.getParameters().size() == 0)) && Objects.equal(opp.getReturnType().getQualifiedName(), op.getParameters().get(0).getParameterType().getQualifiedName())));
          }))))) {
            return StringExtensions.toFirstLower(op.getSimpleName().substring(3, op.getSimpleName().length()));
          } else {
            return null;
          }
        }
      }
    }
  }

  /**
   * Checks whether the given field is annotated with @Containment
   */
  private boolean isContainment(final JvmMember field) {
    final Function1<JvmAnnotationReference, Boolean> _function = (JvmAnnotationReference it) -> {
      String _qualifiedName = it.getAnnotation().getQualifiedName();
      return Boolean.valueOf(Objects.equal(_qualifiedName, AspectToEcore.CONTAINMENT_ANNOTATION_FQN));
    };
    return IterableExtensions.<JvmAnnotationReference>exists(field.getAnnotations(), _function);
  }

  /**
   * Checks whether the given field is annotated with @Unique or @Containment or @Opposite
   */
  private boolean isUnique(final JvmMember field) {
    return ((this.isContainment(field) || this.isOpposite(field)) || IterableExtensions.<JvmAnnotationReference>exists(field.getAnnotations(), 
      ((Function1<JvmAnnotationReference, Boolean>) (JvmAnnotationReference it) -> {
        String _qualifiedName = it.getAnnotation().getQualifiedName();
        return Boolean.valueOf(Objects.equal(_qualifiedName, AspectToEcore.UNIQUE_ANNOTATION_FQN));
      })));
  }

  /**
   * Checks whether the given field is annotated with @Opposite
   */
  private boolean isOpposite(final JvmMember field) {
    return (this.isContainment(field) || IterableExtensions.<JvmAnnotationReference>exists(field.getAnnotations(), 
      ((Function1<JvmAnnotationReference, Boolean>) (JvmAnnotationReference it) -> {
        String _qualifiedName = it.getAnnotation().getQualifiedName();
        return Boolean.valueOf(Objects.equal(_qualifiedName, AspectToEcore.OPPOSITE_ANNOTATION_FQN));
      })));
  }

  /**
   * Return the 'value' parameter of the annotation @Opposite
   * or null if none
   */
  private String getOppositeValue(final JvmMember field) {
    Object _xblockexpression = null;
    {
      final Function1<JvmAnnotationReference, Boolean> _function = (JvmAnnotationReference it) -> {
        String _qualifiedName = it.getAnnotation().getQualifiedName();
        return Boolean.valueOf(Objects.equal(_qualifiedName, AspectToEcore.OPPOSITE_ANNOTATION_FQN));
      };
      final JvmAnnotationReference annot = IterableExtensions.<JvmAnnotationReference>findFirst(field.getAnnotations(), _function);
      EList<JvmAnnotationValue> _values = null;
      if (annot!=null) {
        _values=annot.getValues();
      }
      JvmAnnotationValue _findFirst = null;
      if (_values!=null) {
        final Function1<JvmAnnotationValue, Boolean> _function_1 = (JvmAnnotationValue it) -> {
          String _valueName = it.getValueName();
          return Boolean.valueOf(Objects.equal(_valueName, "value"));
        };
        _findFirst=IterableExtensions.<JvmAnnotationValue>findFirst(_values, _function_1);
      }
      final JvmAnnotationValue annotVal = _findFirst;
      if ((annotVal instanceof JvmCustomAnnotationValue)) {
        EObject _head = IterableExtensions.<EObject>head(((JvmCustomAnnotationValue) annotVal).getValues());
        final XStringLiteral opRef = ((XStringLiteral) _head);
        String _value = null;
        if (opRef!=null) {
          _value=opRef.getValue();
        }
        return _value;
      }
      _xblockexpression = null;
    }
    return ((String)_xblockexpression);
  }

  /**
   * Checks whether the given operation is some obscure K3 code or not
   */
  private boolean isK3Specific(final JvmOperation op) {
    final Function1<String, Boolean> _function = (String p) -> {
      return Boolean.valueOf(op.getSimpleName().startsWith(p));
    };
    return IterableExtensions.<String>exists(AspectToEcore.K3_PREFIXES, _function);
  }

  /**
   * Create a copy of the hierarchy of sub-packages containing {@link baseCls}
   * Return the deepest package
   */
  private EPackage copyPackage(final EClass baseCls) {
    EPackage res = null;
    EPackage currentPkg = baseCls.getEPackage();
    EPackage last = null;
    while ((currentPkg != null)) {
      {
        final EPackage pkgCopy = EcoreFactory.eINSTANCE.createEPackage();
        pkgCopy.setName(currentPkg.getName());
        pkgCopy.setNsPrefix(currentPkg.getNsPrefix());
        pkgCopy.setNsURI(currentPkg.getNsURI());
        if ((last != null)) {
          EList<EPackage> _eSubpackages = pkgCopy.getESubpackages();
          _eSubpackages.add(last);
        } else {
          res = pkgCopy;
        }
        last = pkgCopy;
        currentPkg = currentPkg.getESuperPackage();
      }
    }
    return res;
  }

  private String toQualifiedName(final EClass clazz) {
    final List<String> res = CollectionLiterals.<String>newArrayList();
    res.add(clazz.getName());
    EPackage pack = clazz.getEPackage();
    while ((pack != null)) {
      {
        res.add(pack.getName());
        pack = pack.getESuperPackage();
      }
    }
    return IterableExtensions.join(ListExtensions.<String>reverse(res), ".");
  }

  /**
   * Return true if {@link op} is an Aspect generated getter for final field
   */
  private boolean isGetter(final JvmOperation op, final JvmTypeReferenceBuilder typeRefBuilder) {
    boolean _xblockexpression = false;
    {
      try {
        int _size = op.getParameters().size();
        boolean _equals = (_size == 1);
        if (_equals) {
          final String eclass = op.getParameters().get(0).getParameterType().getSimpleName();
          final String className = op.getDeclaringType().getQualifiedName();
          final JvmTypeReference aspectProperties = typeRefBuilder.typeRef(((className + eclass) + AspectToEcore.PROP_NAME));
          JvmType _type = aspectProperties.getType();
          final JvmGenericType type = ((JvmGenericType) _type);
          final Function1<JvmField, Boolean> _function = (JvmField it) -> {
            return Boolean.valueOf((Objects.equal(it.getSimpleName(), op.getSimpleName()) && it.isFinal()));
          };
          return IterableExtensions.<JvmField>exists(Iterables.<JvmField>filter(type.getMembers(), JvmField.class), _function);
        }
      } catch (final Throwable _t) {
        if (_t instanceof Exception) {
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
      _xblockexpression = false;
    }
    return _xblockexpression;
  }
}
