/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.melange.typesystem;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import com.google.inject.Singleton;
import fr.inria.diverse.melange.metamodel.melange.ModelType;
import java.util.Collection;

/**
 * Stores the implementation relations between {@link Language}s and
 * {@link ModelType}s and the subtyping relations between {@link ModelType}s
 * to be used in the specialized Xbase type system and compiler.
 * 
 * @see MelangeSynonymTypesProvider
 */
@Singleton
@SuppressWarnings("all")
public class MelangeTypesRegistry {
  /**
   * <MM, MT> <=> MM implements MT
   */
  private Multimap<String, ModelType> implementationRelations = ArrayListMultimap.<String, ModelType>create();

  /**
   * <MT, MT'> <=> MT subtypeOf MT'
   */
  private Multimap<String, ModelType> subtypingRelations = ArrayListMultimap.<String, ModelType>create();

  public void registerImplementation(final String mm, final ModelType mt) {
    boolean _containsEntry = this.implementationRelations.containsEntry(mm, mt);
    boolean _not = (!_containsEntry);
    if (_not) {
      this.implementationRelations.put(mm, mt);
    }
  }

  public void registerSubtyping(final String subType, final ModelType superType) {
    boolean _containsEntry = this.subtypingRelations.containsEntry(subType, superType);
    boolean _not = (!_containsEntry);
    if (_not) {
      this.subtypingRelations.put(subType, superType);
    }
  }

  public Collection<ModelType> getImplementations(final String mmFqn) {
    return this.implementationRelations.get(mmFqn);
  }

  public Collection<ModelType> getSubtypings(final String mtFqn) {
    return this.subtypingRelations.get(mtFqn);
  }

  public void clear() {
    this.implementationRelations.clear();
    this.subtypingRelations.clear();
  }
}
