/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.kura.core.data;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.regex.Pattern;
import org.eclipse.kura.KuraConnectException;
import org.eclipse.kura.KuraException;
import org.eclipse.kura.KuraNotConnectedException;
import org.eclipse.kura.KuraStoreException;
import org.eclipse.kura.KuraTimeoutException;
import org.eclipse.kura.KuraTooManyInflightMessagesException;
import org.eclipse.kura.configuration.ConfigurableComponent;
import org.eclipse.kura.core.data.DataMessage;
import org.eclipse.kura.core.data.DataServiceListeners;
import org.eclipse.kura.core.data.DataStore;
import org.eclipse.kura.core.data.store.DbDataStore;
import org.eclipse.kura.data.DataService;
import org.eclipse.kura.data.DataServiceListener;
import org.eclipse.kura.data.DataTransportListener;
import org.eclipse.kura.data.DataTransportService;
import org.eclipse.kura.data.DataTransportToken;
import org.eclipse.kura.db.DbService;
import org.osgi.service.component.ComponentContext;
import org.osgi.service.component.ComponentException;
import org.osgi.util.tracker.ServiceTracker;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DataServiceImpl
implements DataService,
DataTransportListener,
ConfigurableComponent {
    private static final Logger s_logger = LoggerFactory.getLogger(DataServiceImpl.class);
    private static final int TRANSPORT_TASK_TIMEOUT = 1;
    private static final String AUTOCONNECT_PROP_NAME = "connect.auto-on-startup";
    private static final String CONNECT_DELAY_PROP_NAME = "connect.retry-interval";
    private static final String DISCONNECT_DELAY_PROP_NAME = "disconnect.quiesce-timeout";
    private static final String STORE_HOUSEKEEPER_INTERVAL_PROP_NAME = "store.housekeeper-interval";
    private static final String STORE_PURGE_AGE_PROP_NAME = "store.purge-age";
    private static final String STORE_CAPACITY_PROP_NAME = "store.capacity";
    private static final String REPUBLISH_IN_FLIGHT_MSGS_PROP_NAME = "in-flight-messages.republish-on-new-session";
    private static final String MAX_IN_FLIGHT_MSGS_PROP_NAME = "in-flight-messages.max-number";
    private static final String IN_FLIGHT_MSGS_CONGESTION_TIMEOUT_PROP_NAME = "in-flight-messages.congestion-timeout";
    private Map<String, Object> m_properties = new HashMap<String, Object>();
    private DataTransportService m_dataTransportService;
    private DbService m_dbService;
    private DataServiceListeners m_dataServiceListeners;
    protected ScheduledExecutorService m_reconnectExecutor;
    private ScheduledFuture<?> m_reconnectFuture;
    private ScheduledExecutorService m_publisherExecutor;
    private DataStore m_store;
    private Map<DataTransportToken, Integer> m_inFlightMsgIds;
    private ScheduledExecutorService m_congestionExecutor;
    private ScheduledFuture<?> m_congestionFuture;

    protected void activate(ComponentContext componentContext, Map<String, Object> properties) {
        s_logger.info("Activating...");
        this.m_reconnectExecutor = Executors.newSingleThreadScheduledExecutor();
        this.m_publisherExecutor = Executors.newSingleThreadScheduledExecutor();
        this.m_congestionExecutor = Executors.newSingleThreadScheduledExecutor();
        this.m_properties.putAll(properties);
        this.m_store = new DbDataStore();
        try {
            this.m_store.start(this.m_dbService, (Integer)this.m_properties.get(STORE_HOUSEKEEPER_INTERVAL_PROP_NAME), (Integer)this.m_properties.get(STORE_PURGE_AGE_PROP_NAME), (Integer)this.m_properties.get(STORE_CAPACITY_PROP_NAME));
            List<DataMessage> inFlightMsgs = this.m_store.allInFlightMessagesNoPayload();
            this.m_inFlightMsgIds = new ConcurrentHashMap<DataTransportToken, Integer>();
            if (inFlightMsgs != null) {
                for (DataMessage message : inFlightMsgs) {
                    DataTransportToken token = new DataTransportToken(message.getPublishedMessageId(), message.getSessionId());
                    this.m_inFlightMsgIds.put(token, message.getId());
                    s_logger.debug("Restored in-fligh messages from store. Topic: {}, ID: {}, MQTT message ID: {}", new Object[]{message.getTopic(), message.getId(), message.getPublishedMessageId()});
                }
            }
        }
        catch (KuraStoreException e) {
            s_logger.error("Failed to start store", (Throwable)e);
            throw new ComponentException("Failed to start store", (Throwable)e);
        }
        ServiceTracker listenersTracker = new ServiceTracker(componentContext.getBundleContext(), DataServiceListener.class, null);
        this.m_dataServiceListeners = new DataServiceListeners((ServiceTracker<DataServiceListener, DataServiceListener>)listenersTracker);
        this.startReconnectTask();
    }

    public void updated(Map<String, Object> properties) {
        s_logger.info("Updating...");
        this.stopReconnectTask();
        this.m_properties.clear();
        this.m_properties.putAll(properties);
        this.m_store.update((Integer)this.m_properties.get(STORE_HOUSEKEEPER_INTERVAL_PROP_NAME), (Integer)this.m_properties.get(STORE_PURGE_AGE_PROP_NAME), (Integer)this.m_properties.get(STORE_CAPACITY_PROP_NAME));
        if (!this.m_dataTransportService.isConnected()) {
            this.startReconnectTask();
        }
    }

    protected void deactivate(ComponentContext componentContext) {
        s_logger.info("Deactivating...");
        this.m_congestionExecutor.shutdownNow();
        try {
            this.m_publisherExecutor.awaitTermination(1L, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            s_logger.info("Interrupted", (Throwable)e);
        }
        this.m_publisherExecutor.shutdownNow();
        this.stopReconnectTask();
        this.m_reconnectExecutor.shutdownNow();
        this.disconnect();
        this.m_dataServiceListeners.close();
        this.m_store.stop();
    }

    public void setDataTransportService(DataTransportService dataTransportService) {
        this.m_dataTransportService = dataTransportService;
    }

    public void unsetDataTransportService(DataTransportService dataTransportService) {
        this.m_dataTransportService = null;
    }

    public void setDbService(DbService dbService) {
        this.m_dbService = dbService;
    }

    public void unsetDbService(DbService dbService) {
        this.m_dbService = null;
    }

    public void onConnectionEstablished(boolean newSession) {
        s_logger.info("Notified connected");
        if (newSession) {
            Boolean unpublishInFlightMsgs = (Boolean)this.m_properties.get(REPUBLISH_IN_FLIGHT_MSGS_PROP_NAME);
            if (unpublishInFlightMsgs.booleanValue()) {
                s_logger.info("New session established. Unpublishing all in-flight messages. Disregarding the QoS level, this may cause duplicate messages.");
                try {
                    this.m_store.unpublishAllInFlighMessages();
                    this.m_inFlightMsgIds.clear();
                }
                catch (KuraStoreException e) {
                    s_logger.error("Failed to unpublish in-flight messages", (Throwable)e);
                }
            } else {
                s_logger.info("New session established. Dropping all in-flight messages.");
                try {
                    this.m_store.dropAllInFlightMessages();
                    this.m_inFlightMsgIds.clear();
                }
                catch (KuraStoreException e) {
                    s_logger.error("Failed to drop in-flight messages", (Throwable)e);
                }
            }
        }
        this.m_dataServiceListeners.onConnectionEstablished();
        this.submitPublishingWork();
    }

    public void onDisconnecting() {
        s_logger.info("Notified disconnecting");
        this.m_dataServiceListeners.onDisconnecting();
        Future<?> future = this.submitPublishingWork();
        try {
            future.get(1L, TimeUnit.SECONDS);
        }
        catch (InterruptedException interruptedException) {
            s_logger.info("Interrupted while waiting for the publishing work to complete");
        }
        catch (ExecutionException e) {
            s_logger.warn("ExecutionException while waiting for the publishing work to complete", (Throwable)e);
        }
        catch (TimeoutException timeoutException) {
            s_logger.warn("Timeout while waiting for the publishing work to complete");
        }
    }

    public void onDisconnected() {
        s_logger.info("Notified disconnected");
        this.m_dataServiceListeners.onDisconnected();
    }

    public void onConfigurationUpdating(boolean wasConnected) {
        s_logger.info("Notified DataTransportService configuration updating...");
        this.stopReconnectTask();
        this.disconnect(0L);
    }

    public void onConfigurationUpdated(boolean wasConnected) {
        s_logger.info("Notified DataTransportService configuration updated.");
        boolean autoConnect = this.startReconnectTask();
        if (!autoConnect && wasConnected) {
            try {
                this.connect();
            }
            catch (KuraConnectException e) {
                s_logger.error("Error during re-connect after configuration update.", (Throwable)e);
            }
        }
    }

    public void onConnectionLost(Throwable cause) {
        s_logger.info("connectionLost");
        this.stopReconnectTask();
        this.startReconnectTask();
        this.m_dataServiceListeners.onConnectionLost(cause);
    }

    public void onMessageArrived(String topic, byte[] payload, int qos, boolean retained) {
        s_logger.debug("Message arrived on topic: {}", (Object)topic);
        this.m_dataServiceListeners.onMessageArrived(topic, payload, qos, retained);
        this.submitPublishingWork();
    }

    public synchronized void onMessageConfirmed(DataTransportToken token) {
        s_logger.debug("Confirmed message with MQTT message ID: {} on session ID: {}", (Object)token.getMessageId(), (Object)token.getSessionId());
        Integer messageId = this.m_inFlightMsgIds.remove(token);
        if (messageId == null) {
            s_logger.info("Confirmed message published with MQTT message ID: {} not tracked in the map of in-flight messages", (Object)token.getMessageId());
        } else {
            DataMessage confirmedMessage = null;
            try {
                s_logger.info("Confirmed message ID: {} to store", (Object)messageId);
                this.m_store.confirmed(messageId);
                confirmedMessage = this.m_store.get(messageId);
            }
            catch (KuraStoreException e) {
                s_logger.error("Cannot confirm message to store", (Throwable)e);
            }
            if (confirmedMessage != null) {
                String topic = confirmedMessage.getTopic();
                this.m_dataServiceListeners.onMessageConfirmed(messageId, topic);
            } else {
                s_logger.error("Confirmed Message with ID {} could not be loaded from the DataStore.", (Object)messageId);
            }
        }
        if (this.m_inFlightMsgIds.size() < (Integer)this.m_properties.get(MAX_IN_FLIGHT_MSGS_PROP_NAME)) {
            this.handleInFlightDecongestion();
        }
        this.submitPublishingWork();
    }

    public void connect() throws KuraConnectException {
        this.stopReconnectTask();
        if (!this.m_dataTransportService.isConnected()) {
            this.m_dataTransportService.connect();
        }
    }

    public boolean isConnected() {
        return this.m_dataTransportService.isConnected();
    }

    public boolean isAutoConnectEnabled() {
        return (Boolean)this.m_properties.get(AUTOCONNECT_PROP_NAME);
    }

    public int getRetryInterval() {
        return (Integer)this.m_properties.get(CONNECT_DELAY_PROP_NAME);
    }

    public void disconnect(long quiesceTimeout) {
        this.stopReconnectTask();
        this.m_dataTransportService.disconnect(quiesceTimeout);
    }

    public void subscribe(String topic, int qos) throws KuraTimeoutException, KuraException, KuraNotConnectedException {
        this.m_dataTransportService.subscribe(topic, qos);
    }

    public void unsubscribe(String topic) throws KuraTimeoutException, KuraException, KuraNotConnectedException {
        this.m_dataTransportService.unsubscribe(topic);
    }

    public int publish(String topic, byte[] payload, int qos, boolean retain, int priority) throws KuraStoreException {
        s_logger.info("Storing message on topic :{}, priority: {}", (Object)topic, (Object)priority);
        DataMessage dataMsg = this.m_store.store(topic, payload, qos, retain, priority);
        s_logger.info("Stored message on topic :{}, priority: {}", (Object)topic, (Object)priority);
        this.submitPublishingWork();
        return dataMsg.getId();
    }

    public List<Integer> getUnpublishedMessageIds(String topicRegex) throws KuraStoreException {
        List<DataMessage> messages = this.m_store.allUnpublishedMessagesNoPayload();
        return this.buildMessageIds(messages, topicRegex);
    }

    public List<Integer> getInFlightMessageIds(String topicRegex) throws KuraStoreException {
        List<DataMessage> messages = this.m_store.allInFlightMessagesNoPayload();
        return this.buildMessageIds(messages, topicRegex);
    }

    public List<Integer> getDroppedInFlightMessageIds(String topicRegex) throws KuraStoreException {
        List<DataMessage> messages = this.m_store.allDroppedInFlightMessagesNoPayload();
        return this.buildMessageIds(messages, topicRegex);
    }

    private boolean startReconnectTask() {
        if (this.m_reconnectFuture != null && !this.m_reconnectFuture.isDone()) {
            s_logger.error("Reconnect task already running");
            throw new IllegalStateException("Reconnect task already running");
        }
        boolean autoConnect = (Boolean)this.m_properties.get(AUTOCONNECT_PROP_NAME);
        int reconnectInterval = (Integer)this.m_properties.get(CONNECT_DELAY_PROP_NAME);
        if (autoConnect) {
            int maxDelay = reconnectInterval / 5;
            maxDelay = maxDelay > 0 ? maxDelay : 1;
            int initialDelay = new Random().nextInt(maxDelay);
            s_logger.info("Starting reconnect task with initial delay {}", (Object)initialDelay);
            this.m_reconnectFuture = this.m_reconnectExecutor.scheduleAtFixedRate(new Runnable(){

                @Override
                public void run() {
                    String originalName = Thread.currentThread().getName();
                    Thread.currentThread().setName("DataServiceImpl:ReconnectTask");
                    boolean connected = false;
                    try {
                        try {
                            s_logger.info("Connecting...");
                            if (DataServiceImpl.this.m_dataTransportService.isConnected()) {
                                s_logger.info("Already connected. Reconnect task will be terminated.");
                            } else {
                                DataServiceImpl.this.m_dataTransportService.connect();
                                s_logger.info("Connected. Reconnect task will be terminated.");
                            }
                            connected = true;
                        }
                        catch (Exception e) {
                            s_logger.warn("Connect failed", (Throwable)e);
                            Thread.currentThread().setName(originalName);
                            if (connected) {
                                throw new RuntimeException("Connected. Reconnect task will be terminated.");
                            }
                        }
                        catch (Error e) {
                            s_logger.error("Unexpected Error. Task will be terminated", (Throwable)e);
                            throw e;
                        }
                    }
                    finally {
                        Thread.currentThread().setName(originalName);
                        if (connected) {
                            throw new RuntimeException("Connected. Reconnect task will be terminated.");
                        }
                    }
                }
            }, initialDelay, reconnectInterval, TimeUnit.SECONDS);
        }
        return autoConnect;
    }

    private void stopReconnectTask() {
        if (this.m_reconnectFuture != null && !this.m_reconnectFuture.isDone()) {
            s_logger.info("Reconnect task running. Stopping it");
            this.m_reconnectFuture.cancel(true);
        }
    }

    private void disconnect() {
        long millis = (long)((Integer)this.m_properties.get(DISCONNECT_DELAY_PROP_NAME)).intValue() * 1000L;
        this.m_dataTransportService.disconnect(millis);
    }

    private Future<?> submitPublishingWork() {
        return this.m_publisherExecutor.submit(new Runnable(){

            @Override
            public void run() {
                Thread.currentThread().setName("DataServiceImpl:Submit");
                if (!DataServiceImpl.this.m_dataTransportService.isConnected()) {
                    s_logger.info("DataPublisherService not connected");
                    return;
                }
                try {
                    DataMessage message = null;
                    while ((message = DataServiceImpl.this.m_store.getNextMessage()) != null) {
                        if (message.getQos() > 0 && DataServiceImpl.this.m_inFlightMsgIds.size() >= (Integer)DataServiceImpl.this.m_properties.get(DataServiceImpl.MAX_IN_FLIGHT_MSGS_PROP_NAME)) {
                            s_logger.warn("The configured maximum number of in-flight messages has been reached");
                            DataServiceImpl.this.handleInFlightCongestion();
                            break;
                        }
                        DataServiceImpl.this.publishInternal(message);
                        DataServiceImpl.this.m_dataServiceListeners.onMessagePublished(message.getId(), message.getTopic());
                    }
                }
                catch (KuraConnectException e) {
                    s_logger.info("DataPublisherService is not connected", (Throwable)e);
                }
                catch (KuraTooManyInflightMessagesException e) {
                    s_logger.info("Too many in-flight messages", (Throwable)e);
                    DataServiceImpl.this.handleInFlightCongestion();
                }
                catch (Exception e) {
                    s_logger.error("Probably an unrecoverable exception", (Throwable)e);
                }
            }
        });
    }

    private synchronized void publishInternal(DataMessage message) throws KuraConnectException, KuraTooManyInflightMessagesException, KuraStoreException, KuraException {
        String topic = message.getTopic();
        byte[] payload = message.getPayload();
        int qos = message.getQos();
        boolean retain = message.isRetain();
        int msgId = message.getId();
        s_logger.debug("Publishing message with ID: {} on topic: {}, priority: {}", new Object[]{msgId, topic, message.getPriority()});
        DataTransportToken token = this.m_dataTransportService.publish(topic, payload, qos, retain);
        if (token == null) {
            this.m_store.published(msgId);
            s_logger.debug("Published message with ID: {}", (Object)msgId);
        } else {
            Integer trackedMsgId = this.m_inFlightMsgIds.get(token);
            if (trackedMsgId != null) {
                s_logger.error("Token already tracked: " + token.getSessionId() + "-" + token.getMessageId());
            }
            this.m_inFlightMsgIds.put(token, msgId);
            this.m_store.published(msgId, token.getMessageId(), token.getSessionId());
            s_logger.debug("Published message with ID: {} and MQTT message ID: {}", (Object)msgId, (Object)token.getMessageId());
        }
    }

    private List<Integer> buildMessageIds(List<DataMessage> messages, String topicRegex) {
        Pattern topicPattern = Pattern.compile(topicRegex);
        ArrayList<Integer> ids = new ArrayList<Integer>();
        if (messages != null) {
            for (DataMessage message : messages) {
                String topic = message.getTopic();
                if (!topicPattern.matcher(topic).matches()) continue;
                ids.add(message.getId());
            }
        }
        return ids;
    }

    private void handleInFlightCongestion() {
        int timeout = (Integer)this.m_properties.get(IN_FLIGHT_MSGS_CONGESTION_TIMEOUT_PROP_NAME);
        if (timeout != 0 && (this.m_congestionFuture == null || this.m_congestionFuture.isDone())) {
            s_logger.warn("In-flight message congestion timeout started");
            this.m_congestionFuture = this.m_congestionExecutor.schedule(new Runnable(){

                @Override
                public void run() {
                    Thread.currentThread().setName("DataServiceImpl:InFlightCongestion");
                    s_logger.warn("In-flight message congestion timeout elapsed. Disconnecting and reconnecting again");
                    DataServiceImpl.this.disconnect();
                    DataServiceImpl.this.startReconnectTask();
                }
            }, (long)timeout, TimeUnit.SECONDS);
        }
    }

    private void handleInFlightDecongestion() {
        if (this.m_congestionFuture != null && !this.m_congestionFuture.isDone()) {
            this.m_congestionFuture.cancel(true);
        }
    }
}

