/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm;

import javax.persistence.AccessType;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;

/**
 * This interface defines the root-level contract for providing ORM.xml metadata. The properties
 * defined by this interface define all root and subsequently nested information available in an
 * ORM.xml.
 * <p>
 * Provisional API: This interface is part of an interim API that is still under development and
 * expected to change significantly before reaching stability. It is available at this early stage
 * to solicit feedback from pioneering adopters on the understanding that any code that uses this
 * API will almost certainly be broken (repeatedly) as the API evolves.<p>
 *
 * @see oracle.toplink.workbench.mappingsmodel.spi.jpa.orm.ExternalORMRepositoryFactory
 * @see oracle.toplink.workbench.mappingsmodel.spi.SPIManager
 * @see http://jcp.org/aboutJava/communityprocess/final/jsr220/index.html, persistence specificaton
 *
 * @version 2.5
 * @author John Bracken
 */
public interface ExternalORMConfiguration extends ExternalTenantDiscriminatorColumnProvider {

	/**
	 * Adds a new converter.
	 */
	ExternalConverter addConverter(int index);
	
	/**
	 * Adds a embedded entity with the given name.
	 */
	ExternalEmbeddableEntity addEmbeddableEntity(String embeddableClassName);

	/**
	 * Adds a embedded entity with the given name.
	 */
	void addEmbeddableEntity(int index, String embeddableClassName);

	/**
	 * Adds a entity for the given class name.
	 */
	void addEntity(int index, String entityClassName);

	
	/**
	 * Adds a entity for the given class name.
	 */
	ExternalEntity addEntity(String entityClassName);
	
	
	/**
	 * Adds a mapped super class entity with the given name
	 */
	void addMappedSuperClass(int index, String mappedClassName);

	/**
	 * Adds a native query with the given name.
	 */
	void addNamedNativeQuery(int index, String name);

	/**
	 * Adds a named query with the given name.
	 */
	ExternalNamedQuery addNamedQuery(int index, String name);

	/**
	 * Adds a named query with the given name.
	 */
	ExternalNamedQuery addNamedQuery(String name);

	/**
	 * Adds an object type converter.
	 */
	ExternalObjectTypeConverter addObjectTypeConverter(int index);

	/**
	 * Adds a sequence generator with the given name
	 */
	void addSequenceGenerator(int index, String name);

	/**
	 * Adds a SQL result set mapping with the given name
	 */
	void addSqlResultSetMapping(int index, String name);

	/**
	 * Adds a stored procedure query with the given name
	 */
	void addStoredProcedureQuery(int index, String name, String procedureName);

	/**
	 * Adds a new struct converter.
	 */
	ExternalStructConverter addStructConverter(int index);

	/**
	 * Adds a table generator with the given name
	 */
	void addTableGenerator(int index, String name);

	/**
	 * Adds a new type converter.
	 */
	ExternalTypeConverter addTypeConverter(int index);

	/**
	 * Creates new persistence unit representing this ORM's persistence unit metadata replacing
	 * the old one.
	 */
	ExternalPersistenceUnit buildPersistenceUnitMetaData();

	/**
	 * Returns a list of all converters defined globally for this ORM configuration.
	 */
	ListIterable<ExternalConverter> converters();

	/**
	 * Returns the count of all converters defined globally for this ORM configuration.
	 */
	int convertersSize();

	/**
	 * Returns a list of all embedded entities associated with this ORM configuration.
	 */
	ListIterable<ExternalEmbeddableEntity> embeddableEntities();

	/**
	 * Returns the count of embedded entities associated with this ORM configuration.
	 */
	int embeddableEntitiesSize();

	/**
	 * Returns a list of all entities associated with this ORM configuration.
	 */
	ListIterable<ExternalEntity> entities();

	/**
	 * Returns the count of entities associated with this ORM configuration.
	 */
	int entitiesSize();

	/**
	 * Returns the access type to be applied to all entities belonging to this ORM xml.
	 */
	AccessType getAccessType();

	/**
	 * Returns the current build version of the product.
	 *
	 * @return The product's current version
	 */
	String getBuildVersion();

	/**
	 * Returns the catalog name to be applied to all entities in this ORM xml if applicable.
	 */
	String getCatalogName();

	/**
	 * Returns the converter with at the given index.
	 */
	ExternalConverter getConverter(int index);

	/**
	 * Returns the description noted in the ORM.xml.
	 */
	String getDescription();

	/**
	 * Returns the document type of this ORM configuration.
	 */
	ORMDocumentType getDocumentType();

	/**
	 * Returns the embeddable entity for the given class name
	 */
	ExternalEmbeddableEntity getEmbeddableEntity(int index);

	/**
	 * Returns the entity associated with this ORM configuration identified by the given 
	 * index.
	 */
	ExternalEntity getEntity(int index);
	
	/**
	 * Returns the entity associated with this ORM configuration identified by 
	 * the given class name.
	 */
	ExternalEntity getEntity(String className);

	/**
	 * Returns the mapped superclass for the given class name
	 */
	ExternalMappedSuperClassEntity getMappedSuperClass(int index);

	/**
	 * Returns a native named query with the given name defined in this ORM configuration.
	 */
	ExternalNativeQuery getNamedNativeQuery(int index);

	/**
	 * Returns the named query with the provided name.
	 */
	ExternalBasicNamedQuery getNamedQuery(int index);

	/**
	 * Returns the named query with the given name if it exists.
	 */
	ExternalBasicNamedQuery getNamedQuery(String queryName);

	/**
	 * Returns the object type converter with at the given index.
	 */
	ExternalObjectTypeConverter getObjectTypeConverter(int index);

	/**
	 * Returns the default package name to be applied to all entities.
	 */
	String getPackageName();

	/**
	 * Returns the persistent unit representing this ORM's persistence unit metadata.
	 */
	ExternalPersistenceUnit getPersistenceUnitMetaData();

	/**
	 * Returns the default schema name to be applied to all referenced tables.
	 */
	String getSchemaName();

	/**
	 * Returns the sequence generator with the given name.
	 */
	ExternalSequenceGenerator getSequenceGenerator(int index);

	/**
	 * Returns the SQL Result Set Mapping at the given position.
	 */
	ExternalSQLResultSetMapping getSqlResultSetMapping(int index);

	/**
	 * Returns the stored procedure query at the index
	 */
	ExternalStoredProcedureQuery getStoredProcedureQuery(int index);

	/**
	 * Returns the stored procedure query with the given name
	 */
	ExternalStoredProcedureQuery getStoredProcedureQuery(int index, String name);

	/**
	 * Returns the struct converter with at the given index.
	 */
	ExternalStructConverter getStructConverter(int index);

	/**
	 * Returns a table generator with the specified name if it exists.
	 */
	ExternalTableGenerator getTableGenerator(int index);

	/**
	 * Returns the type converter with at the given index.
	 */
	ExternalTypeConverter getTypeConverter(int index);

	/**
	 * Returns a list of all mapped super class entities associated with this ORM configuration.
	 */
	ListIterable<ExternalMappedSuperClassEntity> mappedSuperClasses();

	/**
	 * Returns the count of mapped super class entities associated with this ORM configuration.
	 */
	int mappedSuperClassesSize();

	/**
	 * Returns a list of all native queries declared globally for this ORM configuration.
	 */
	ListIterable<ExternalNativeQuery> namedNativeQueries();

	/**
	 * Returns the count of all native queries declared globally for this ORM configuration.
	 */
	int namedNativeQueriesSize();

	/**
	 * Returns a list of all named queries declared globally for this ORM configuration.
	 */
	ListIterable<ExternalNamedQuery> namedQueries();

	/**
	 * Returns the count of all named queries declared globally for this ORM configuration.
	 */
	int namedQueriesSize();

	/**
	 * Returns a list of all converters defined globally for this ORM configuration.
	 */
	ListIterable<ExternalObjectTypeConverter> objectTypeConverters();

	/**
	 * Returns the count of all converters defined globally for this ORM configuration.
	 */
	int objectTypeConvertersSize();

	/**
	 * Removes the converter at index.
	 */
	void removeConverter(int index);

	/**
	 * Removes the embedded entity named.
	 */
	void removeEmbeddableEntity(int index);

	/**
	 * Removes the entity for the given class name.
	 */
	void removeEntity(int index);

	/**
	 * Removes the entity with the given class name.
	 */
	void removeEntity(String entityClassName);

	/**
	 * Removes the mapped super class entity named.
	 */
	void removeMappedSuperClass(int index);

	/**
	 * Removes the named query named
	 */
	void removeNamedQuery(int index);

	/**
	 * Removes the named query named
	 */
	void removeNamedQuery(String queryName);

	/**
	 * Removes the native query named.
	 */
	void removeNativeQuery(int index);

	/**
	 * Removes the converter at index.
	 */
	void removeObjectTypeConverter(int index);

	/**
	 * Removes the persistence unit representing this ORM's persistence unit metadata.
	 */
	void removePersistenceUnitMetaData();

	/**
	 * Removes the sequence generator named.
	 */
	void removeSequenceGenerator(int index);

	/**
	 * Removes the SQL result set mapping named.
	 */
	void removeSqlResultSetMapping(int index);

	/**
	 * Removes the stored procedure query named.
	 */
	void removeStoredProcedureQuery(int index);

	/**
	 * Removes the converter at index.
	 */
	void removeStructConverter(int index);

	/**
	 * Removes the table generator named.
	 */
	void removeTableGenerator(int index);

	/**
	 * Removes the converter at index.
	 */
	void removeTypeConverter(int index);

	/**
	 * Returns a list of all sequence generators defined globally for this ORM configuration.
	 */
	ListIterable<ExternalSequenceGenerator> sequenceGenerators();

	/**
	 * Returns the count of all sequence generators defined globally for this ORM configuration.
	 */
	int sequenceGeneratorsSize();

	/**
	 * Set the access type.
	 */
	void setAccessType(AccessType type);

	/**
	 * Set the catalog name.
	 */
	void setCatalogName(String catalog);

	/**
	 * Set the description
	 */
	void setDescription(String description);

	/**
	 * Sets the type of this ORM.
	 */
	void setDocumentType(ORMDocumentType type);

	/**
	 * Set the default package name to be applied to all entities.
	 */
	void setPackageName(String packageName);

	/**
	 * Set the default schema name to be applied to all referenced tables.
	 */
	void setSchemaName(String schema);

	/**
	 * Returns a list of all SQL result set mappings declared globally for this ORM configuration.
	 */
	ListIterable<ExternalSQLResultSetMapping> sqlResultSetMappings();

	/**
	 * Returns the count of all SQL result set mappings declared globally for this ORM configuration.
	 */
	int sqlResultSetMappingsSize();

	/**
	 * Returns a list of named stored procedure queries defined for this entity.
	 */
	ListIterable<ExternalStoredProcedureQuery> storedProcedureQueries();

	/**
	 * Returns the count of named stored procedure queried defined for this entity
	 */
	int storedProcedureQueriesSize();

	/**
	 * Returns a list of all converters defined globally for this ORM configuration.
	 */
	ListIterable<ExternalStructConverter> structConverters();

	/**
	 * Returns the count of all converters defined globally for this ORM configuration.
	 */
	int structConvertersSize();

	/**
	 * Returns a list of all table generators defined globally for this ORM configuration.
	 */
	ListIterable<ExternalTableGenerator> tableGenerators();

	/**
	 * Returns the count of all table generators defined globally for this ORM configuration.
	 */
	int tableGeneratorsSize();

	/**
	 * Derives and returns the XML representation of this configuration.
	 */
	String getXML();

	/**
	 * Returns a list of all converters defined globally for this ORM configuration.
	 */
	ListIterable<ExternalTypeConverter> typeConverters();

	/**
	 * Returns the count of all converters defined globally for this ORM configuration.
	 */
	int typeConvertersSize();
}