/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.orm.ExternalConverter;
import org.eclipse.persistence.tools.mapping.orm.ExternalConvertibleMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalObjectTypeConverter;
import org.eclipse.persistence.tools.mapping.orm.ExternalStructConverter;
import org.eclipse.persistence.tools.mapping.orm.ExternalTypeConverter;

/**
 * The external form for a convertible mapping, which is a child of an entity.
 *
 * @see EmbeddableEntity
 *
 * @version 2.5
 * @author Les Davis
 */
@SuppressWarnings("nls")
abstract class ConvertibleMapping extends NonTransientMapping
                                  implements ExternalConvertibleMapping {

	/**
	 * The element name used to store and retrieve the convert child text node.
	 */
	static final String CONVERT = "convert";

	/**
	 * Creates a new <code>ConvertibleMapping</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	ConvertibleMapping(EmbeddableEntity parent, int index) {
		super(parent, index);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(ACCESS);
		return names;
	}

	private Converter buildConverter() {
		return new Converter(this, -1);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(CONVERT);
		names.add(Converter.CONVERTER);
		names.add(TypeConverter.TYPE_CONVERTER);
		names.add(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		names.add(StructConverter.STRUCT_CONVERTER);
		names.add(Property.PROPERTY);
		names.add(AccessMethods.ACCESS_METHODS);
		return names;
	}

	private ObjectTypeConverter buildObjectTypeConverter() {
		return new ObjectTypeConverter(this, -1);
	}

	private StructConverter buildStructConverter() {
		return new StructConverter(this, -1);
	}

	private TypeConverter buildTypeConverter() {
		return new TypeConverter(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getConvert() {
		return getChildTextNode(CONVERT);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalObjectTypeConverter getObjectTypeConverter() {

		if (hasChild(ObjectTypeConverter.OBJECT_TYPE_CONVERTER)) {
			return buildObjectTypeConverter();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalStructConverter getStructConverter() {

		if (hasChild(StructConverter.STRUCT_CONVERTER)) {
			return buildStructConverter();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalTypeConverter getTypeConverter() {

		if (hasChild(TypeConverter.TYPE_CONVERTER)) {
			return buildTypeConverter();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalConverter getValueConverter() {

		if (hasChild(Converter.CONVERTER)) {
			return buildConverter();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeValueConverter() {
		removeChild(Converter.CONVERTER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeValueObjectTypeConverter() {
		removeChild(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeValueStructConverter() {
		removeChild(StructConverter.STRUCT_CONVERTER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeValueTypeConverter() {
		removeChild(TypeConverter.TYPE_CONVERTER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setAsValueConverter() {

		removeChild(TypeConverter.TYPE_CONVERTER);
		removeChild(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		removeChild(StructConverter.STRUCT_CONVERTER);

		Converter valueConverter = buildConverter();
		valueConverter.addSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setAsValueObjectTypeConverter() {

		removeChild(TypeConverter.TYPE_CONVERTER);
		removeChild(Converter.CONVERTER);
		removeChild(StructConverter.STRUCT_CONVERTER);

		ObjectTypeConverter valueConverter = buildObjectTypeConverter();
		valueConverter.addSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setAsValueStructConverter() {

		removeChild(TypeConverter.TYPE_CONVERTER);
		removeChild(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		removeChild(Converter.CONVERTER);

		StructConverter valueConverter = buildStructConverter();
		valueConverter.addSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setAsValueTypeConverter() {

		removeChild(Converter.CONVERTER);
		removeChild(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		removeChild(StructConverter.STRUCT_CONVERTER);

		TypeConverter valueConverter = buildTypeConverter();
		valueConverter.addSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setConvert(String value) {
		updateChildTextNode(CONVERT, value);
	}
}