/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import javax.persistence.TemporalType;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityIDGeneratedValue;
import org.eclipse.persistence.tools.mapping.orm.ExternalIDMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalNoSqlField;
import org.eclipse.persistence.tools.mapping.orm.ExternalSequenceGenerator;
import org.eclipse.persistence.tools.mapping.orm.ExternalTableGenerator;

/**
 * The external form for a ID mapping, which is a child of an entity.
 *
 * @see MappedSuperClassEntity
 *
 * @version 2.5
 * @author Les Davis
 * @author Pascal Filion
 */
@SuppressWarnings("nls")
final class IdMapping extends ConvertibleMapping
                      implements ExternalIDMapping {

	/**
	 * The node name used to store and retrieve the {@link Element} encapsulated by this external form.
	 */
	static final String ID = "id";

	/**
	 * The attribute name used to store and retrieve the mutable property.
	 */
	static final String MUTABLE = "mutable";

	/**
	 * The element name used to store and retrieve the temporal child text node.
	 */
	static final String TEMPORAL = "temporal";

	/**
	 * Creates a new <code>IdMapping</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	IdMapping(MappedSuperClassEntity parent, int index) {
		super(parent, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityIDGeneratedValue addGeneratedValue() {
		EntityIDGeneratedValue generatedValue = buildGeneratedValue();
		generatedValue.addSelf();
		return generatedValue;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalSequenceGenerator addSequenceGenerator() {
		SequenceGenerator sequenceGenerator = buildSequenceGenerator();
		sequenceGenerator.addSelf();
		return sequenceGenerator;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalTableGenerator addTableGenerator() {
		TableGenerator tableGenerator = buildTableGenerator();
		tableGenerator.addSelf();
		return tableGenerator;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(ACCESS);
		names.add(MUTABLE);
		names.add(ATTRIBUTE_TYPE);
		return names;
	}

	private EntityColumn buildColumn() {
		return new EntityColumn(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {

		List<String> names = new ArrayList<String>();
		names.add(EntityColumn.COLUMN);
		names.add(NoSqlField.FIELD);
		names.add(EntityIDGeneratedValue.GENERATED_VALUE);
		names.add(TEMPORAL);
		names.add(CONVERT);
		names.add(Converter.CONVERTER);
		names.add(TypeConverter.TYPE_CONVERTER);
		names.add(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		names.add(StructConverter.STRUCT_CONVERTER);
		names.add(TableGenerator.TABLE_GENERATOR);
		names.add(SequenceGenerator.SEQUENCE_GENERATOR);
		names.add(Property.PROPERTY);
		names.add(AccessMethods.ACCESS_METHODS);

		return names;
	}

	private EntityIDGeneratedValue buildGeneratedValue() {
		return new EntityIDGeneratedValue(this);
	}

	private ExternalNoSqlField buildNoSqlField() {
		return new NoSqlField(this);
	}

	private SequenceGenerator buildSequenceGenerator() {
		return new SequenceGenerator(this);
	}

	private TableGenerator buildTableGenerator() {
		return new TableGenerator(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public EntityColumn getColumn() {

		if (hasChild(EntityColumn.COLUMN)) {
			return buildColumn();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return ID;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityIDGeneratedValue getGeneratedValue() {

		if (hasChild(EntityIDGeneratedValue.GENERATED_VALUE)) {
			return buildGeneratedValue();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getNoSqlField() {
		ExternalNoSqlField field = buildNoSqlField();
		return field.getName();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalSequenceGenerator getSequenceGenerator() {

		if (hasChild(SequenceGenerator.SEQUENCE_GENERATOR)) {
			return buildSequenceGenerator();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalTableGenerator getTableGenerator() {

		if (hasChild(TableGenerator.TABLE_GENERATOR)) {
			return buildTableGenerator();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TemporalType getTemporalType() {
		return getChildEnumNode(TEMPORAL, TemporalType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final boolean isIdMapping() {
		return true;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean isMutable() {
		return getBooleanAttribute(MUTABLE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeGeneratedValue() {
		EntityIDGeneratedValue generatedValue = buildGeneratedValue();
		generatedValue.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeSequenceGenerator() {
		SequenceGenerator generator = buildSequenceGenerator();
		generator.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeTableGenerator() {
		TableGenerator generator = buildTableGenerator();
		generator.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAttributeType(String attributeType) {
		setAttribute(ATTRIBUTE_TYPE, attributeType);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	@SuppressWarnings("null")
	public void setColumn(String columnName) {

		AbstractColumn column = getColumn();

		if ((column == null) && (columnName == null)) {
			return;
		}

		if ((column == null) && (columnName != null)) {
			column = buildColumn();
			column.addSelf();
		}

		if (columnName != null) {
			column.setName(columnName);
		}
		else {
			column.removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMutable(Boolean mutable) {
		setAttribute(MUTABLE, mutable);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setNoSqlField(String field) {
		ExternalNoSqlField noSqlField = buildNoSqlField();
		noSqlField.setName(field);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setSequenceGenerator(String generatorName) {
		updateChildTextNode(SequenceGenerator.SEQUENCE_GENERATOR, generatorName);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setTableGenerator(String generatorName) {
		updateChildTextNode(TableGenerator.TABLE_GENERATOR, generatorName);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setTemporalType(TemporalType type) {
		updateChildTextNode(TEMPORAL, type);
	}
}