/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.orm.ExternalPrimaryKeyGenerator;
import org.w3c.dom.Element;

/**
 * The external form of a generator, which can either be a sequence generator or a table generator.
 *
 * @see SequenceGenerator
 * @see TableGenerator
 *
 * @version 2.5
 * @author Les Davis
 * @author Pascal Filion
 */
@SuppressWarnings("nls")
abstract class PrimaryKeyGenerator extends AbstractExternalForm
                                   implements ExternalPrimaryKeyGenerator {

	/**
	 * The position of the element within the list of children with the same type owned by the parent.
	 */
	private int index;

	/**
	 * The attribute name used to store and retrieve the allocation-size of the property.
	 */
	static final String ALLOCATION_SIZE = "allocation-size";

	/**
	 * The attribute name used to store and retrieve the temporal property.
	 */
	static final String CATALOG = "catalog";

	/**
	 * The element name of the child text node for the description.
	 */
	static final String DESCRIPTION = "description";

	/**
	 * The attribute name used to store and retrieve the initial-value of the property.
	 */
	static final String INITIAL_VALUE = "initial-value";

	/**
	 * The attribute name used to store and retrieve the name property.
	 */
	static final String NAME = "name";

	/**
	 * The attribute name used to store and retrieve the temporal property.
	 */
	static final String SCHEMA = "schema";

	/**
	 * Creates a new <code>PrimaryKeyGenerator</code>.
	 *
	 * @param parent The parent of this external form
	 */
	PrimaryKeyGenerator(AbstractExternalForm parent) {
		super(parent);
		index = -1;
	}

	/**
	 * Creates a new <code>PrimaryKeyGenerator</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	PrimaryKeyGenerator(AbstractExternalForm parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected final Element addSelf(String elementName, List<String> elementNamesOrder) {

		if (index == -1) {
			return super.addSelf(elementName, elementNamesOrder);
		}

		return addChild(getParent(), elementName, index, elementNamesOrder);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(DESCRIPTION);
		return names;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final Integer getAllocationSize() {
		return getIntegerAttribute(ALLOCATION_SIZE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getCatalogName() {
		return getAttribute(CATALOG);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getDescription() {
		return getChildTextNode(DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final Element getElement() {

		if (index == -1) {
			return super.getElement();
		}

		return getChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final int getIndex() {
		return index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final Integer getInitialValue() {
		return getIntegerAttribute(INITIAL_VALUE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getName() {
		return getAttribute(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getSchemaName() {
		return getAttribute(SCHEMA);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeSelf() {

		if (index == -1) {
			super.removeSelf();
		}
		else {
			removeChild(getParent(), getElementName(), index);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setAllocationSize(Integer size) {

		setAttribute(ALLOCATION_SIZE, size);

		if (shouldBeRemoved()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setCatalogName(String name) {

		setAttribute(CATALOG, name);

		if (shouldBeRemoved()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setDescription(String description) {
		updateChildTextNode(DESCRIPTION, description);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setInitialValue(Integer size) {

		setAttribute(INITIAL_VALUE, size);

		if (shouldBeRemoved()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setName(String name) {

		setAttribute(NAME, name);

		if (shouldBeRemoved()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setSchemaName(String name) {

		setAttribute(SCHEMA, name);

		if (shouldBeRemoved()) {
			removeSelf();
		}
	}

	final boolean shouldBeRemoved() {
		return (index == -1) && !hasAnyChildren() && !hasAttributes();
	}
}