/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.List;
import java.util.ListIterator;
import org.eclipse.persistence.tools.mapping.orm.ExternalUniqueConstraint;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;
import org.eclipse.persistence.tools.utility.iterable.ListListIterable;
import org.w3c.dom.Element;

/**
 * The external form of a unique constraint.
 *
 * @see TableGenerator
 *
 * @version 2.5
 * @author Les Davis
 * @author Pascal Filion
 */
@SuppressWarnings("nls")
final class UniqueConstraint extends AbstractExternalForm
                             implements ExternalUniqueConstraint {

	/**
	 * The position of the element within the list of children with the same type owned by the parent.
	 */
	private int index;

	/**
	 * The element name used to store and retrieve the column-name child element.
	 */
	static final String COLUMN_NAME = "column-name";

	/**
	 * The node name used to store and retrieve the {@link Element} encapsulated by this external form.
	 */
	static final String UNIQUE_CONSTRAINT = "unique-constraint";

	/**
	 * Creates a new <code>UniqueConstraint</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	UniqueConstraint(Table parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * Creates a new <code>UniqueConstraint</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	UniqueConstraint(TableGenerator parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Element addSelf(String elementName, List<String> elementNamesOrder) {
		return addChild(getParent(), elementName, index, elementNamesOrder);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<String> columnNames() {
		return new ListListIterable<String>(getChildrenTextNode(COLUMN_NAME));
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int columnNamesSize() {
		return getChildrenSize(COLUMN_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Element getElement() {
		return getChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return UNIQUE_CONSTRAINT;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getIndex() {
		return index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeSelf() {
		removeChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setColumnNames(ListIterator<String> columnNames) {

		removeChildren(COLUMN_NAME);

		while (columnNames.hasNext()) {
			addChildTextNode(COLUMN_NAME, columnNames.next());
		}
	}
}