/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.persistence.dom;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.eclipse.persistence.tools.mapping.orm.dom.AbstractExternalForm;
import org.eclipse.persistence.tools.mapping.persistence.ExternalPersistenceConfiguration;
import org.eclipse.persistence.tools.mapping.persistence.ExternalPersistenceUnit;
import org.eclipse.persistence.tools.mapping.persistence.PersistenceDocumentType;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;
import org.eclipse.persistence.tools.utility.iterable.ListListIterable;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * The external form interacting with the XML document for the Persistence Configuration file.
 *
 * @version 2.5
 * @author Les Davis
 * @author Pascal Filion
 */
@SuppressWarnings("nls")
final class PersistenceConfiguration extends AbstractExternalForm
                                     implements ExternalPersistenceConfiguration {

	/**
	 * The root document that represents this persistence.xml.
	 */
	private Document document;

	/**
	 * The URI of the ORM namespace.
	 */
	static final String NAMESPACE_URI = "http://java.sun.com/xml/ns/persistence";

	/**
	 * The node name used to store and retrieve the {@link Element} encapsulated by this external form.
	 */
	static final String PERSISTENCE = "persistence";

	/**
	 * The URI of the Persistence Configuration schema, JPA 1.0.
	 */
	private static final String PERSISTENCE_XSD_URI_1_0 = "http://java.sun.com/xml/ns/persistence/persistence_1_0.xsd";

	/**
	 * The URI of the Persistence Configuration schema, JPA 2.0.
	 */
	private static final String PERSISTENCE_XSD_URI_2_0 = "http://java.sun.com/xml/ns/persistence/persistence_2_0.xsd";

	/**
	 * The URI of the Persistence Configuration schema, JPA 2.1.
	 */
	private static final String PERSISTENCE_XSD_URI_2_1 = "http://java.sun.com/xml/ns/persistence/persistence_2_1.xsd";

	/**
	 * The element name of the child text node for version of the Persistence Configuration file.
	 */
	static final String VERSION = "version";

	/**
	 * Creates a new <code>PersistenceConfiguration</code>.
	 *
	 * @param document The root document that represents this persistence.xml
	 */
	PersistenceConfiguration(Document document) {
		super(null);
		this.document = document;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addPersistenceUnit(int index, String name) {
		PersistenceUnit persistenceUnit = buildPersistenceUnit(index);
		persistenceUnit.addSelf();
		persistenceUnit.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Element addSelf(String elementName) {
		return addSelf(elementName, Collections.<String>emptyList());
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Element addSelf(String elementName, List<String> elementNamesOrder) {

		Element element = document.createElementNS(NAMESPACE_URI, elementName);
		element.setAttributeNS("http://www.w3.org/2000/xmlns/", "xmlns", NAMESPACE_URI);
		addXmlns(element, "xsi", XSI_URI);
		element.setAttributeNS(XSI_URI, "xsi:schemaLocation", buildSchemaLocation(PersistenceDocumentType.JPA_2_0));
		element.setAttribute(VERSION, getBuildVersion());

		document.appendChild(element);

		return element;
	}

	private PersistenceUnit buildPersistenceUnit(int index) {
		return new PersistenceUnit(this, index);
	}

	private String buildSchemaLocation(PersistenceDocumentType type) {

		StringBuilder sb = new StringBuilder();
		sb.append(NAMESPACE_URI);
		sb.append(" ");

		switch (type) {
			case JPA_1_0: sb.append(PERSISTENCE_XSD_URI_1_0); break;
			case JPA_2_0: sb.append(PERSISTENCE_XSD_URI_2_0); break;
			case JPA_2_1: sb.append(PERSISTENCE_XSD_URI_2_1); break;
		}

		return sb.toString();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getBuildVersion() {
		return PersistenceDocumentType.JPA_2_0.getVersion();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Element getElement() {
		return getRootElement();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return PERSISTENCE;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalPersistenceUnit getPersistenceUnit(int index) {

		Element element = getChild(PersistenceUnit.PERSISTENCE_UNIT, index);

		if (element == null) {
			return null;
		}

		return buildPersistenceUnit(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public PersistenceDocumentType getDocumentType() {
		return PersistenceDocumentType.value(getNamespace(), getVersion(), getSchemaLocation());
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalPersistenceUnit> persistenceUnits() {

		int count = getChildrenSize(PersistenceUnit.PERSISTENCE_UNIT);
		List<ExternalPersistenceUnit> persistenceUnits = new ArrayList<ExternalPersistenceUnit>(count);

		for (int index = count; --index >= 0; ) {
			ExternalPersistenceUnit persistenceUnit = buildPersistenceUnit(index);
			persistenceUnits.add(0, persistenceUnit);
		}

		return new ListListIterable<ExternalPersistenceUnit>(persistenceUnits);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int persistenceUnitsSize() {
		return getChildrenSize(PersistenceUnit.PERSISTENCE_UNIT);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removePersistenceUnit(int index) {
		PersistenceUnit persistenceUnit = buildPersistenceUnit(index);
		persistenceUnit.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDocumentType(PersistenceDocumentType type) {
		setAttribute(VERSION, type.getVersion());
		setSchemaLocation(buildSchemaLocation(type));
	}
}