/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.persistence.dom;

import java.util.List;
import org.eclipse.persistence.tools.mapping.orm.dom.AbstractExternalForm;
import org.eclipse.persistence.tools.mapping.persistence.ExternalProperty;
import org.w3c.dom.Element;

/**
 * The external form of a persistence unit property, which is a child of the properties element.
 *
 * @see PersistenceUnit
 *
 * @version 2.5
 * @author Pascal Filion
 */
@SuppressWarnings("nls")
final class Property extends AbstractExternalForm
                     implements ExternalProperty {

	/**
	 * The index of this external form within the list of properties.
	 */
	private int index;

	/**
	 * The attribute name used to store the key mapping the value.
	 */
	static final String NAME = "name";

	/**
	 * The element name used of the XML node.
	 */
	static final String PROPERTIES = "properties";

	/**
	 * The element name used of the XML node.
	 */
	static final String PROPERTY = "property";

	/**
	 * The attribute name used to store the value mapped to a key.
	 */
	static final String VALUE = "value";

	/**
	 * Creates a new <code>Property</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	Property(PersistenceUnit parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Element addSelf(String elementName, List<String> elementNamesOrder) {

		Element element = getChild(getParent(), elementName);

		if (element == null) {
			element = addChild(getParent(), elementName, elementNamesOrder);
		}

		return addChild(element, PROPERTY, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Element getElement() {

		Element element = getChild(getParent(), PROPERTIES);

		if (element == null) {
			return null;
		}

		return getChild(element, PROPERTY, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return PROPERTIES;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getIndex() {
		return index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getName() {
		return getAttribute(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getValue() {
		return getAttribute(VALUE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeSelf() {

		Element element = getChild(getParent(), getElementName());

		if (element != null) {
			removeChild(element, PROPERTY, index);

			if (!hasAnyChildren(element)) {
				remove(getParentElement(), element);
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setName(String name) {
		setAttribute(NAME, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setValue(String value) {
		setAttribute(VALUE, value);
	}
}