/*******************************************************************************
 * Copyright (c) 2009, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.utility.command;

import org.eclipse.persistence.tools.utility.ObjectTools;
import org.eclipse.persistence.tools.utility.reference.SynchronizedBoolean;

/**
 * Straightforward implementation of {@link StatefulCommandExecutor}
 * that executes commands immediately by default. This executor can
 * also be used to adapt simple {@link CommandExecutor}s to the
 * {@link StatefulCommandExecutor} interface, providing support for
 * lifecycle state.
 */
@SuppressWarnings("nls")
public abstract class AbstractStatefulCommandExecutor<E extends CommandExecutor>
	implements StatefulCommandExecutor
{
	protected final SynchronizedBoolean active = new SynchronizedBoolean(false);
	protected final E commandExecutor;


	protected AbstractStatefulCommandExecutor(E commandExecutor) {
		super();
		if (commandExecutor == null) {
			throw new NullPointerException();
		}
		this.commandExecutor = commandExecutor;
	}

	@Override
	public synchronized void start() {
		if (this.active.isTrue()) {
			throw new IllegalStateException("Not stopped.");
		}
		this.active.setTrue();
	}

	/**
	 * If the command executor is inactive the command is simply ignored.
	 */
	@Override
	public void execute(Command command) {
		if (this.active.isTrue()) {
			this.commandExecutor.execute(command);
		}
	}

	@Override
	public synchronized void stop() {
		if (this.active.isFalse()) {
			throw new IllegalStateException("Not started.");
		}
		this.active.setFalse();
	}

	@Override
	public String toString() {
		return ObjectTools.toString(this, this.commandExecutor);
	}
}