/*******************************************************************************
 * Copyright (c) 2007, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.utility.model.value;

import org.eclipse.persistence.tools.utility.model.event.CollectionAddEvent;
import org.eclipse.persistence.tools.utility.model.event.CollectionChangeEvent;
import org.eclipse.persistence.tools.utility.model.event.CollectionClearEvent;
import org.eclipse.persistence.tools.utility.model.event.CollectionRemoveEvent;
import org.eclipse.persistence.tools.utility.model.listener.CollectionChangeListener;

/**
 * This abstract class provides the infrastructure needed to wrap
 * another collection value model, "lazily" listen to it, and propagate
 * its change notifications. Subclasses must implement the appropriate
 * {@link CollectionValueModel}.
 *
 * @param <E> the type of elements held by the model
 */
@SuppressWarnings("nls")
public abstract class CollectionValueModelWrapper<E>
	extends AbstractCollectionValueModel
{
	/** The wrapped collection value model. */
	protected final CollectionValueModel<? extends E> collectionModel;

	/** A listener that allows us to sync with changes to the wrapped collection model. */
	protected final CollectionChangeListener collectionChangeListener;


	// ********** constructors **********

	/**
	 * Construct a collection value model with the specified wrapped
	 * collection value model.
	 */
	protected CollectionValueModelWrapper(CollectionValueModel<? extends E> collectionModel) {
		super();
		if (collectionModel == null) {
			throw new NullPointerException();
		}
		this.collectionModel = collectionModel;
		this.collectionChangeListener = this.buildCollectionChangeListener();
	}


	// ********** initialization **********

	protected CollectionChangeListener buildCollectionChangeListener() {
		return new CollectionChangeListener() {
			@Override
			public void itemsAdded(CollectionAddEvent event) {
				CollectionValueModelWrapper.this.itemsAdded(event);
			}
			@Override
			public void itemsRemoved(CollectionRemoveEvent event) {
				CollectionValueModelWrapper.this.itemsRemoved(event);
			}
			@Override
			public void collectionCleared(CollectionClearEvent event) {
				CollectionValueModelWrapper.this.collectionCleared(event);
			}
			@Override
			public void collectionChanged(CollectionChangeEvent event) {
				CollectionValueModelWrapper.this.collectionChanged(event);
			}
			@Override
			public String toString() {
				return "collection change listener";
			}
		};
	}


	// ********** AbstractCollectionValueModel implementation **********

	/**
	 * Start listening to the collection holder.
	 */
	@Override
	protected void engageModel() {
		this.collectionModel.addCollectionChangeListener(CollectionValueModel.VALUES, this.collectionChangeListener);
	}

	/**
	 * Stop listening to the collection holder.
	 */
	@Override
	protected void disengageModel() {
		this.collectionModel.removeCollectionChangeListener(CollectionValueModel.VALUES, this.collectionChangeListener);
	}


	// ********** helper methods **********

	// minimize scope of suppressed warnings
	@SuppressWarnings("unchecked")
	protected Iterable<E> getItems(CollectionAddEvent event) {
		return (Iterable<E>) event.getItems();
	}

	// minimize scope of suppressed warnings
	@SuppressWarnings("unchecked")
	protected Iterable<E> getItems(CollectionRemoveEvent event) {
		return (Iterable<E>) event.getItems();
	}


	// ********** collection change support **********

	/**
	 * Items were added to the wrapped collection holder;
	 * propagate the change notification appropriately.
	 */
	protected abstract void itemsAdded(CollectionAddEvent event);

	/**
	 * Items were removed from the wrapped collection holder;
	 * propagate the change notification appropriately.
	 */
	protected abstract void itemsRemoved(CollectionRemoveEvent event);

	/**
	 * The wrapped collection holder was cleared;
	 * propagate the change notification appropriately.
	 */
	protected abstract void collectionCleared(CollectionClearEvent event);

	/**
	 * The value of the wrapped collection holder has changed;
	 * propagate the change notification appropriately.
	 */
	protected abstract void collectionChanged(CollectionChangeEvent event);
}