/*******************************************************************************
 * Copyright (c) 2007, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.utility.model.value;

import java.util.Arrays;
import org.eclipse.persistence.tools.utility.model.Model;
import org.eclipse.persistence.tools.utility.model.event.CollectionAddEvent;
import org.eclipse.persistence.tools.utility.model.event.CollectionChangeEvent;
import org.eclipse.persistence.tools.utility.model.event.CollectionClearEvent;
import org.eclipse.persistence.tools.utility.model.event.CollectionRemoveEvent;
import org.eclipse.persistence.tools.utility.model.listener.CollectionChangeListener;

/**
 * Extend {@link ValueAspectAdapter} to listen to one or more collection
 * aspects of the value in the wrapped value model.
 */
@SuppressWarnings("nls")
public class ValueCollectionAdapter<V extends Model>
	extends ValueAspectAdapter<V>
{
	/** The names of the value's collections that we listen to. */
	protected final String[] collectionNames;

	/** Listener that listens to the value. */
	protected final CollectionChangeListener valueCollectionListener;


	// ********** constructors **********

	/**
	 * Construct an adapter for the specified value collections.
	 */
	public ValueCollectionAdapter(ModifiablePropertyValueModel<V> valueHolder, String... collectionNames) {
		super(valueHolder);
		if (collectionNames == null) {
			throw new NullPointerException();
		}
		this.collectionNames = collectionNames;
		this.valueCollectionListener = this.buildValueCollectionListener();
	}


	// ********** initialization **********

	protected CollectionChangeListener buildValueCollectionListener() {
		return new CollectionChangeListener() {
			@Override
			public void itemsAdded(CollectionAddEvent event) {
				ValueCollectionAdapter.this.itemsAdded(event);
			}
			@Override
			public void itemsRemoved(CollectionRemoveEvent event) {
				ValueCollectionAdapter.this.itemsRemoved(event);
			}
			@Override
			public void collectionCleared(CollectionClearEvent event) {
				ValueCollectionAdapter.this.collectionCleared(event);
			}
			@Override
			public void collectionChanged(CollectionChangeEvent event) {
				ValueCollectionAdapter.this.collectionChanged(event);
			}
			@Override
			public String toString() {
				return "value collection listener: " + Arrays.asList(ValueCollectionAdapter.this.collectionNames);
			}
		};
	}


	// ********** ValueAspectAdapter implementation **********

	@Override
	protected void engageValue_() {
		for (String collectionName : this.collectionNames) {
			this.value.addCollectionChangeListener(collectionName, this.valueCollectionListener);
		}
	}

	@Override
	protected void disengageValue_() {
		for (String collectionName : this.collectionNames) {
			this.value.removeCollectionChangeListener(collectionName, this.valueCollectionListener);
		}
	}


	// ********** change events **********

	protected void itemsAdded(@SuppressWarnings("unused") CollectionAddEvent event) {
		this.valueAspectChanged();
	}

	protected void itemsRemoved(@SuppressWarnings("unused") CollectionRemoveEvent event) {
		this.valueAspectChanged();
	}

	protected void collectionCleared(@SuppressWarnings("unused") CollectionClearEvent event) {
		this.valueAspectChanged();
	}

	protected void collectionChanged(@SuppressWarnings("unused") CollectionChangeEvent event) {
		this.valueAspectChanged();
	}
}