/*******************************************************************************
 * Copyright (c) 2007, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.utility.tests.model.value;

import java.util.Collection;
import java.util.Iterator;
import org.eclipse.persistence.tools.utility.collection.Bag;
import org.eclipse.persistence.tools.utility.collection.CollectionTools;
import org.eclipse.persistence.tools.utility.collection.HashBag;
import org.eclipse.persistence.tools.utility.model.event.CollectionAddEvent;
import org.eclipse.persistence.tools.utility.model.event.CollectionChangeEvent;
import org.eclipse.persistence.tools.utility.model.event.CollectionClearEvent;
import org.eclipse.persistence.tools.utility.model.event.CollectionRemoveEvent;
import org.eclipse.persistence.tools.utility.model.listener.CollectionChangeListener;
import org.eclipse.persistence.tools.utility.model.value.CollectionValueModel;

/**
 * Helper class that keeps an internal collection in synch with the
 * collection held by a collection value model.
 */
class CoordinatedBag<E> implements Bag<E>, CollectionChangeListener {

	private Bag<E> bag = new HashBag<E>();

	CoordinatedBag(CollectionValueModel<E> cvm) {
		cvm.addCollectionChangeListener(CollectionValueModel.VALUES, this);
	}


	// ********** Collection implementation **********

	@Override
	public boolean add(E o) {
		return this.bag.add(o);
	}

	@Override
	public boolean addAll(Collection<? extends E> c) {
		return this.bag.addAll(c);
	}

	@Override
	public void clear() {
		this.bag.clear();
	}

	@Override
	public boolean contains(Object o) {
		return this.bag.contains(o);
	}

	@Override
	public boolean containsAll(Collection<?> c) {
		return this.bag.containsAll(c);
	}

	@Override
	public boolean isEmpty() {
		return this.bag.isEmpty();
	}

	@Override
	public Iterator<E> iterator() {
		return this.bag.iterator();
	}

	@Override
	public boolean remove(Object o) {
		return this.bag.remove(o);
	}

	@Override
	public boolean removeAll(Collection<?> c) {
		return this.bag.removeAll(c);
	}

	@Override
	public boolean retainAll(Collection<?> c) {
		return this.bag.retainAll(c);
	}

	@Override
	public int size() {
		return this.bag.size();
	}

	@Override
	public Object[] toArray() {
		return this.bag.toArray();
	}

	@Override
	public <T> T[] toArray(T[] a) {
		return this.bag.toArray(a);
	}


	// ********** Bag implementation **********

	@Override
	public int count(Object o) {
		return this.bag.count(o);
	}

	@Override
	public boolean add(E o, int count) {
		return this.bag.add(o, count);
	}

	@Override
	public boolean remove(Object o, int count) {
		return this.bag.remove(o, count);
	}

	@Override
	public Iterator<E> uniqueIterator() {
		return this.bag.uniqueIterator();
	}

	@Override
	public int uniqueCount() {
		return this.bag.uniqueCount();
	}

	@Override
	public Iterator<Bag.Entry<E>> entries() {
		return this.bag.entries();
	}

	// ********** CollectionChangeListener implementation **********

	@Override
	@SuppressWarnings("unchecked")
	public void itemsAdded(CollectionAddEvent event) {
		for (E item : (Iterable<E>) event.getItems()) {
			this.bag.add(item);
		}
	}

	@Override
	@SuppressWarnings("unchecked")
	public void itemsRemoved(CollectionRemoveEvent event) {
		for (E item : (Iterable<E>) event.getItems()) {
			this.bag.remove(item);
		}
	}

	@Override
	public void collectionCleared(CollectionClearEvent event) {
		this.bag.clear();
	}

	@Override
	@SuppressWarnings("unchecked")
	public void collectionChanged(CollectionChangeEvent event) {
		this.bag.clear();
		CollectionTools.addAll(this.bag, ((CollectionValueModel<E>) event.getSource()).iterator());
	}


	// ********** standard methods **********

	@Override
	public boolean equals(Object o) {
		return this.bag.equals(o);
	}

	@Override
	public int hashCode() {
		return this.bag.hashCode();
	}

	@Override
	public String toString() {
		return this.bag.toString();
	}
}