/*******************************************************************************
 * Copyright (c) 2005, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.utility.iterable;

import java.util.Iterator;
import org.eclipse.persistence.tools.utility.collection.ListTools;
import org.eclipse.persistence.tools.utility.iterator.ChainIterator;

/**
 * A <code>ChainIterable</code> provides a pluggable {@link Iterable}
 * that loops over a chain of arbitrarily linked objects. The chain
 * should be null-terminated (i.e. a call to the {@link #nextLink(Object)}
 * method should return <code>null</code> when it is passed the last
 * link of the chain).
 * To use, supply a starting link and supply a
 * {@link org.eclipse.jpt.common.utility.internal.iterator.ChainIterator.Linker}
 * or subclass <code>ChainIterable</code> and override the
 * {@link #nextLink(Object)} method.
 * The starting link will be the first object returned by the iterable's iterator.
 * If the starting link is <code>null</code>, the iterable will be empty.
 * Note this iterable does not support <code>null</code> elements.
 *
 * @param <E> the type of elements returned by the iterable's iterator
 *
 * @see ChainIterator
 */
@SuppressWarnings("nls")
public class ChainIterable<E>
	implements Iterable<E>
{
	private final E startLink;
	private final ChainIterator.Linker<E> linker;


	/**
	 * Construct an iterable with the specified starting link
	 * and a default linker that calls back to the iterable.
	 * Use this constructor if you want to override the
	 * {@link #nextLink(Object)} method instead of building a
	 * {@link org.eclipse.jpt.common.utility.internal.iterator.ChainIterator.Linker}.
	 */
	public ChainIterable(E startLink) {
		super();
		this.startLink = startLink;
		this.linker = this.buildDefaultLinker();
	}

	/**
	 * Construct an iterator with the specified starting link
	 * and linker.
	 */
	public ChainIterable(E startLink, ChainIterator.Linker<E> linker) {
		super();
		if (linker == null) {
			throw new NullPointerException();
		}
		this.startLink = startLink;
		this.linker = linker;
	}

	protected ChainIterator.Linker<E> buildDefaultLinker() {
		return new DefaultLinker();
	}

	@Override
	public Iterator<E> iterator() {
		return new ChainIterator<E>(this.startLink, this.linker);
	}

	/**
	 * Return the next link in the chain; null if there are no more links.
	 * <p>
	 * This method can be overridden by a subclass as an alternative to
	 * building a {@link org.eclipse.jpt.common.utility.internal.iterator.ChainIterator.Linker}.
	 */
	protected E nextLink(@SuppressWarnings("unused") E currentLink) {
		throw new RuntimeException("This method was not overridden.");
	}

	@Override
	public String toString() {
		return ListTools.list(this).toString();
	}


	//********** default linker **********

	protected class DefaultLinker implements ChainIterator.Linker<E> {
		@Override
		public E nextLink(E currentLink) {
			return ChainIterable.this.nextLink(currentLink);
		}
	}
}