/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.orm.ExternalAssociationOverride;
import org.eclipse.persistence.tools.mapping.orm.ExternalJoinColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalJoinTable;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;
import org.eclipse.persistence.tools.utility.iterable.ListListIterable;
import org.w3c.dom.Element;

/**
 * The external form of an association override.
 *
 * @see ElementCollectionMapping
 * @see Entity
 *
 * @version 2.5
 * @author Les Davis
 * @author Pascal Filion
 */
@SuppressWarnings("nls")
final class AssociationOverride extends AbstractExternalForm
                                implements ExternalAssociationOverride {

	/**
	 * The node name used to store and retrieve the {@link Element} encapsulated by this external form.
	 */
	static final String ASSOCIATION_OVERRIDE = "association-override";

	/**
	 * The element name of the child text node for the description.
	 */
	static final String DESCRIPTION = "description";

	/**
	 * The attribute name used to store and retrieve the name property.
	 */
	static final String NAME = "name";

	/**
	 * The position of the element within the list of children with the same type owned by the parent.
	 */
	private int index;

	/**
	 * Creates a new <code>AssociationOverride</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	AssociationOverride(ElementCollectionMapping parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * Creates a new <code>AssociationOverride</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	AssociationOverride(EmbeddedMapping parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * Creates a new <code>AssociationOverride</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	AssociationOverride(Entity parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * Creates a new <code>AssociationOverride</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	AssociationOverride(ObjectCollectionMapping parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addJoinColumn(int index, String name) {
		JoinColumn joinColumn = buildJoinColumn(index);
		joinColumn.addSelf();
		joinColumn.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void addJoinTable(String tableName) {
		JoinTable joinTable = buildJoinTable();
		joinTable.addSelf();
		joinTable.setName(tableName);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Element addSelf(String elementName, List<String> elementNamesOrder) {
		return addChild(getParent(), elementName, index, elementNamesOrder);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(DESCRIPTION);
		names.add(JoinColumn.JOIN_COLUMN);
		names.add(JoinTable.JOIN_TABLE);
		return names;
	}

	private JoinColumn buildJoinColumn(int index) {
		return new JoinColumn(this, index);
	}

	private JoinTable buildJoinTable() {
		return new JoinTable(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDescription() {
		return getChildTextNode(DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Element getElement() {
		return getChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return ASSOCIATION_OVERRIDE;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalJoinColumn getJoinColumn(int index) {

		Element element = getChild(JoinColumn.JOIN_COLUMN, index);

		if (element != null) {
			return buildJoinColumn(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalJoinTable getJoinTable() {

		Element element = getChild(JoinTable.JOIN_TABLE);

		if (element != null) {
			return buildJoinTable();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getName() {
		return getAttribute(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalJoinColumn> joinColumns() {

		int count = joinColumnsSize();
		List<ExternalJoinColumn> joinColumns = new ArrayList<ExternalJoinColumn>(count);

		for (int index = count; --index >= 0;) {
			ExternalJoinColumn joinColumn = buildJoinColumn(index);
			joinColumns.add(0, joinColumn);
		}

		return new ListListIterable<ExternalJoinColumn>(joinColumns);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int joinColumnsSize() {
		return getChildrenSize(JoinColumn.JOIN_COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeJoinColumn(int index) {
		JoinColumn joinColumn = buildJoinColumn(index);
		joinColumn.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeSelf() {
		removeChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDescription(String description) {
		updateChildTextNode(DESCRIPTION, description);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setName(String name) {
		setAttribute(NAME, name);
	}
}