/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import javax.persistence.AccessType;
import org.eclipse.persistence.tools.mapping.orm.ExternalAccessMethods;
import org.eclipse.persistence.tools.mapping.orm.ExternalNonTransientMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalProperty;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;
import org.eclipse.persistence.tools.utility.iterable.ListListIterable;
import org.w3c.dom.Element;

/**
 * The external form for a non-transient mapping, which is a child of an entity.
 *
 * @see EmbeddableEntity
 *
 * @version 2.5
 * @author Les Davis
 */
@SuppressWarnings("nls")
abstract class NonTransientMapping extends Mapping
                                   implements ExternalNonTransientMapping {

	/**
	 * The attribute name used to store and retrieve the access property.
	 */
	static final String ACCESS = "access";

	/**
	 * Creates a new <code>NonTransientMapping</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	NonTransientMapping(EmbeddableEntity parent, int index) {
		super(parent, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void addAccessMethods(String getMethodName, String setMethodName) {
		AccessMethods accessMethods = buildAccessMethods();
		accessMethods.addSelf();
		accessMethods.setGetMethod(getMethodName);
		accessMethods.setSetMethod(setMethodName);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void addProperty(int index, String name, String value) {
		Property property = buildProperty(index);
		property.addSelf();
		property.setName(name);
		property.setValue(value);
	}

	private AccessMethods buildAccessMethods() {
		return new AccessMethods(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(Property.PROPERTY);
		names.add(AccessMethods.ACCESS_METHODS);
		return names;
	}
	
	private Property buildProperty(int index) {
		return new Property(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalAccessMethods getAccessMethods() {

		Element element = getChild(AccessMethods.ACCESS_METHODS);

		if (element == null) {
			return null;
		}

		return buildAccessMethods();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final AccessType getAccessType() {
		return getEnumAttribute(ACCESS, AccessType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalProperty getProperty(int index) {

		Element element = getChild(Property.PROPERTY, index);

		if (element == null) {
			return null;
		}

		return buildProperty(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ListIterable<ExternalProperty> properties() {

		int count = propertiesSize();
		List<ExternalProperty> properties = new ArrayList<ExternalProperty>(count);

		for (int index = count; --index >= 0;) {
			ExternalProperty associationOverride = buildProperty(index);
			properties.add(0, associationOverride);
		}

		return new ListListIterable<ExternalProperty>(properties);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final int propertiesSize() {
		return getChildrenSize(Property.PROPERTY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeAccessMethods() {
		removeChild(AccessMethods.ACCESS_METHODS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeProperty(int index) {
		Property property = buildProperty(index);
		property.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setAccessType(AccessType type) {
		setAttribute(ACCESS, type);
	}
}