/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import javax.persistence.CascadeType;
import javax.persistence.FetchType;
import org.eclipse.persistence.annotations.JoinFetchType;
import org.eclipse.persistence.tools.mapping.orm.ExternalBatchFetch;
import org.eclipse.persistence.tools.mapping.orm.ExternalJoinTable;
import org.eclipse.persistence.tools.mapping.orm.ExternalRelationshipMapping;
import org.eclipse.persistence.tools.utility.ObjectTools;
import org.eclipse.persistence.tools.utility.iterable.EmptyListIterable;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;
import org.eclipse.persistence.tools.utility.iterable.ListListIterable;
import org.w3c.dom.Element;

/**
 * The external form for a relationship mapping, which is a child of an entity.
 *
 * @see MappedSuperClassEntity
 *
 * @version 2.5
 * @author Les Davis
 * @author Pascal Filion
 */
@SuppressWarnings("nls")
abstract class RelationshipMapping extends NonTransientMapping
                                   implements ExternalRelationshipMapping {

	/**
	 * The list of ordered element names required for insertion of children at the right location.
	 */
	private List<String> cascadeTypeIndices;

	/**
	 * The attribute name used to store and retrieve the access property.
	 */
	static final String ACCESS = "access";

	/**
	 * The element name used to store and retrieve the cascade child element.
	 */
	static final String CASCADE = "cascade";

	/**
	 * The element name used to store and retrieve the cascade-all child node of the cascade child.
	 */
	static final String CASCADE_ALL = "cascade-all";

	/**
	 * The element name used to store and retrieve the cascade-detach child node of the cascade child.
	 */
	static final String CASCADE_DETACH = "cascade-detach";

	/**
	 * The element name used to store and retrieve the cascade-merge child node of the cascade child.
	 */
	static final String CASCADE_MERGE = "cascade-merge";

	/**
	 * The element name used to store and retrieve the cascade-persist child node of the cascade child.
	 */
	static final String CASCADE_PERSIST = "cascade-persist";

	/**
	 * The element name used to store and retrieve the cascade-refresh child node of the cascade child.
	 */
	static final String CASCADE_REFRESH = "cascade-refresh";

	/**
	 * The element name used to store and retrieve the cascade-remove child node of the cascade child.
	 */
	static final String CASCADE_REMOVE = "cascade-remove";

	/**
	 * The attribute name used to store and retrieve the fetch property.
	 */
	static final String FETCH = "fetch";

	/**
	 * The element name used to store and retrieve the join-fetch child text node.
	 */
	static final String JOIN_FETCH = "join-fetch";

	/**
	 * The attribute name used to store and retrieve the optional property.
	 */
	static final String OPTIONAL = "optional";

	/**
	 * The attribute name used to store and retrieve the target-entity property.
	 */
	static final String TARGET_ENTITY = "target-entity";

	/**
	 * Creates a new <code>RelationshipMapping</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	RelationshipMapping(EmbeddableEntity parent, int index) {
		super(parent, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void addBatchFetch() {
		BatchFetch batchFetch = buildBatchFetch();
		batchFetch.addSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void addCascadeType(CascadeType type) {

		Element element = getChild(CASCADE);

		if (element == null) {
			element = addChild(CASCADE);
		}

		addChild(element, cascadeType(type), cascadeTypeIndices);
	}

	/**
	 * {@inheritDoc}
	 *
	 * @return
	 */
	@Override
	public final ExternalJoinTable addJoinTable() {
		JoinTable joinTable = buildJoinTable();
		joinTable.addSelf();
		return joinTable;
	}

	private BatchFetch buildBatchFetch() {
		return new BatchFetch(this);
	}

	private List<String> buildCascadeTypeIndices() {
		List<String> names = new ArrayList<String>();
		names.add(CASCADE_ALL);
		names.add(CASCADE_PERSIST);
		names.add(CASCADE_MERGE);
		names.add(CASCADE_REMOVE);
		names.add(CASCADE_REFRESH);
		names.add(CASCADE_DETACH);
		return names;
	}

	private JoinTable buildJoinTable() {
		return new JoinTable(this);
	}

	private String cascadeType(CascadeType type) {
		switch (type) {
			case ALL:     return CASCADE_ALL;
			case DETACH:  return CASCADE_DETACH;
			case MERGE:   return CASCADE_MERGE;
			case PERSIST: return CASCADE_PERSIST;
			case REFRESH: return CASCADE_REFRESH;
			case REMOVE:  return CASCADE_REMOVE;
			default:      return null;
		}
	}

	private CascadeType cascadeType(Element element) {

		String elementName = getNodeName(element);

		if (ObjectTools.equals(elementName, CASCADE_ALL)) {
			return CascadeType.ALL;
		}

		if (ObjectTools.equals(elementName, CASCADE_DETACH)) {
			return CascadeType.DETACH;
		}

		if (ObjectTools.equals(elementName, CASCADE_MERGE)) {
			return CascadeType.MERGE;
		}

		if (ObjectTools.equals(elementName, CASCADE_PERSIST)) {
			return CascadeType.PERSIST;
		}

		if (ObjectTools.equals(elementName, CASCADE_REFRESH)) {
			return CascadeType.REFRESH;
		}

		if (ObjectTools.equals(elementName, CASCADE_REMOVE)) {
			return CascadeType.REMOVE;
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ListIterable<CascadeType> cascadeTypes() {

		Element element = getChild(CASCADE);

		if (element == null) {
			return EmptyListIterable.instance();
		}

		List<Element> children = getChildren(element);
		List<CascadeType> cascadeTypes = new ArrayList<CascadeType>(children.size());

		for (Element childElement : children) {
			CascadeType cascadeType = cascadeType(childElement);
			if (cascadeType != null) {
				cascadeTypes.add(cascadeType);
			}
		}

		return new ListListIterable<CascadeType>(cascadeTypes);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalBatchFetch getBatchFetch() {

		if (hasChild(BatchFetch.BATCH_FETCH)) {
			return buildBatchFetch();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final FetchType getFetchType() {
		return getEnumAttribute(FETCH, FetchType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final JoinFetchType getJoinFetchType() {
		return getChildEnumNode(JOIN_FETCH, JoinFetchType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalJoinTable getJoinTable() {

		if (hasChild(JoinTable.JOIN_TABLE)) {
			return buildJoinTable();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getTargetEntityName() {
		return getAttribute(TARGET_ENTITY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void initialize() {
		super.initialize();
		cascadeTypeIndices = buildCascadeTypeIndices();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final Boolean isOptional() {
		return getBooleanAttribute(OPTIONAL);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeBatchFetch() {
		removeChild(BatchFetch.BATCH_FETCH);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeCascadeType(CascadeType type) {

		Element element = getChild(CASCADE);

		if (element != null) {
			removeChild(element, cascadeType(type));

			if (!hasAnyChildren(element)) {
				remove(element);
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeJoinTable() {

		Element element = getChild(JoinTable.JOIN_TABLE);

		if (element != null) {
			remove(element);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setFetchType(FetchType type) {
		setAttribute(FETCH, type);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setJoinFetchType(JoinFetchType type) {
		updateChildTextNode(JOIN_FETCH, type);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setOptional(Boolean optional) {
		setAttribute(OPTIONAL, optional);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setTargetEntityName(String entityName) {
		setAttribute(TARGET_ENTITY, entityName);
	}
}