/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import javax.persistence.TemporalType;
import org.eclipse.persistence.tools.mapping.orm.ExternalVersionMapping;

/**
 * The external form for a version mapping, which is a child of an entity.
 *
 * @see MappedSuperClassEntity
 *
 * @version 2.5
 * @author Les Davis
 * @author Pascal Filion
 */
@SuppressWarnings("nls")
final class VersionMapping extends ConvertibleMapping
                           implements ExternalVersionMapping {

	/**
	 * The element name used to store and retrieve the convert child text node.
	 */
	static final String CONVERT = "convert";

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getNoSqlField() {
		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setNoSqlField(String field) {
		// No supported
	}

	/**
	 * The attribute name used to store and retrieve the mutable property.
	 */
	static final String MUTABLE = "mutable";

	/**
	 * The element name used to store and retrieve the temporal child text node.
	 */
	static final String TEMPORAL = "temporal";

	/**
	 * The node name used to store and retrieve the {@link Element} encapsulated by this external form.
	 */
	static final String VERSION = "version";

	/**
	 * Creates a new <code>VersionMapping</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	VersionMapping(MappedSuperClassEntity parent, int index) {
		super(parent, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(ACCESS);
		names.add(MUTABLE);
		names.add(ATTRIBUTE_TYPE);
		return names;
	}

	private EntityColumn buildColumn() {
		return new EntityColumn(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(EntityColumn.COLUMN);
		names.add(TEMPORAL);
		names.add(CONVERT);
		names.add(Converter.CONVERTER);
		names.add(TypeConverter.TYPE_CONVERTER);
		names.add(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		names.add(StructConverter.STRUCT_CONVERTER);
		names.add(Property.PROPERTY);
		names.add(AccessMethods.ACCESS_METHODS);
		return names;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public EntityColumn getColumn() {

		if (hasChild(EntityColumn.COLUMN)) {
			return buildColumn();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getConvert() {
		return getChildTextNode(CONVERT);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return VERSION;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TemporalType getTemporalType() {
		return getChildEnumNode(TEMPORAL, TemporalType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean isMutable() {
		return getBooleanAttribute(MUTABLE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAttributeType(String attributeType) {
		setAttribute(ATTRIBUTE_TYPE, attributeType);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	@SuppressWarnings("null")
	public void setColumn(String columnName) {

		EntityColumn column = getColumn();

		if ((column == null) && (columnName != null)) {
			column = buildColumn();
			column.addSelf();
		}
		else if ((column != null) && (columnName == null)) {
			column.removeSelf();
		}

		if (columnName != null) {
			column.setName(columnName);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setConvert(String convert) {
		updateChildTextNode(CONVERT, convert);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMutable(Boolean mutable) {
		setAttribute(MUTABLE, mutable);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setTemporalType(TemporalType type) {
		updateChildTextNode(TEMPORAL, type);
	}
}