/*******************************************************************************
 * Copyright (c) 2007, 2012 Oracle. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.gen.internal;

import org.eclipse.persistence.tools.gen.db.Column;
import org.eclipse.persistence.tools.gen.db.ForeignKey;
import org.eclipse.persistence.tools.gen.db.Table;
import org.eclipse.persistence.tools.utility.ObjectTools;

/**
 * Provide a pluggable way to determine whether and how the entity generator prints the names of
 * various database objects.
 * <p>
 * Provisional API: This interface is part of an interim API that is still under development and
 * expected to change significantly before reaching stability. It is available at this early stage
 * to solicit feedback from pioneering adopters on the understanding that any code that uses this
 * API will almost certainly be broken (repeatedly) as the API evolves.<p>
 *
 * @version 2.6
 */
public interface DatabaseAnnotationNameBuilder {

	/**
	 * Given the name of an entity and the table to which it is mapped,
	 * Returns a string to be used as the value for the entity's
	 * Returns null if the entity
	 * maps to the table by default.
	 */
	String buildTableAnnotationName(String entityName, Table table);

	/**
	 * Given the name of an attribute (field or property) and the column
	 * to which it is mapped,
	 * Returns a string to be used as the value for the attribute's
	 * Returns null if the attribute
	 * maps to the column by default.
	 */
	String buildColumnAnnotationName(String attributeName, Column column);

	/**
	 * Given the name of an attribute (field or property) and the
	 * many-to-one or many-to-many foreign key to which it is mapped,
	 * Returns a string to be used as the value for the attribute's
	 * Returns null if the attribute
	 * maps to the join column by default.
	 * The specified foreign key consists of a single column pair whose
	 * referenced column is the single-column primary key of the foreign
	 * key's referenced table.
	 */
	String buildJoinColumnAnnotationName(String attributeName, ForeignKey foreignKey);

	/**
	 * Returns a string to be used as the value for a JoinColumn
	 * annotation's 'name' or 'referencedColumnName' element.
	 * This is called for many-to-one and many-to-many mappings when
	 * the default join column name and/or referenced column name are/is
	 * not applicable.
	 * @see buildJoinColumnAnnotationName(String, ForeignKey)
	 */
	String buildJoinColumnAnnotationName(Column column);

	/**
	 * Returns a string to be used as the value for a JoinTable
	 * annotation's 'name' element.
	 * This is called for many-to-many mappings when the default
	 * join table name is not applicable.
	 */
	String buildJoinTableAnnotationName(Table table);


	/**
	 * The default implementation simple returns the database object's name,
	 * unaltered.
	 */
	final class Default
		implements DatabaseAnnotationNameBuilder
	{
		public static final DatabaseAnnotationNameBuilder INSTANCE = new Default();
		public static DatabaseAnnotationNameBuilder instance() {
			return INSTANCE;
		}
		// ensure single instance
		private Default() {
			super();
		}
		@Override
		public String buildTableAnnotationName(String entityName, Table table) {
			return table.getName();
		}
		@Override
		public String buildColumnAnnotationName(String attributeName, Column column) {
			return column.getName();
		}
		@Override
		public String buildJoinColumnAnnotationName(String attributeName, ForeignKey foreignKey) {
			return foreignKey.getColumnPair().getBaseColumn().getName();
		}
		@Override
		public String buildJoinColumnAnnotationName(Column column) {
			return column.getName();
		}
		@Override
		public String buildJoinTableAnnotationName(Table table) {
			return table.getName();
		}
		@Override
		public String toString() {
			return ObjectTools.toString(this);
		}
	}
}