/*******************************************************************************
 * Copyright (c) 2012, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.utility.command;

import org.eclipse.persistence.tools.utility.ExceptionHandler;
import org.eclipse.persistence.tools.utility.ObjectTools;

/**
 * This command executor wraps another command executor and uses an exception
 * handler to handle any exceptions thrown by an executing command.
 */
public abstract class AbstractSafeCommandExecutor<E extends CommandExecutor>
	implements CommandExecutor
{
	protected final E commandExecutor;
	protected final ExceptionHandler exceptionHandler;


	/**
	 * <strong>NB:</strong> The default exception handler simply
	 * <em>ignores</em> any and all exceptions.
	 */
	protected AbstractSafeCommandExecutor(E commandExecutor) {
		this(commandExecutor, ExceptionHandler.Null.instance());
	}

	protected AbstractSafeCommandExecutor(E commandExecutor, ExceptionHandler exceptionHandler) {
		super();
		if ((commandExecutor == null) || (exceptionHandler == null)) {
			throw new NullPointerException();
		}
		this.commandExecutor = commandExecutor;
		this.exceptionHandler = exceptionHandler;
	}

	@Override
	public void execute(Command command) {
		try {
			this.commandExecutor.execute(command);
		} catch (Throwable ex) {
			this.exceptionHandler.handleException(ex);
		}
	}

	@Override
	public String toString() {
		return ObjectTools.toString(this, this.commandExecutor);
	}
}