/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.AbstractExternalForm;
import org.eclipse.persistence.tools.mapping.orm.ExternalAssociationOverride;
import org.eclipse.persistence.tools.mapping.orm.ExternalJoinColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalJoinTable;
import org.eclipse.persistence.tools.utility.TextRange;
import org.w3c.dom.Element;

/**
 * The external form of an association override.
 *
 * @see ElementCollectionMapping
 * @see Entity
 *
 * @version 2.6
 */
final class AssociationOverride extends AbstractExternalForm
                                implements ExternalAssociationOverride {

	/**
	 * The position of the element within the list of children with the same type owned by the parent.
	 */
	private int index;

	/**
	 * Creates a new <code>AssociationOverride</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	AssociationOverride(AbstractExternalForm parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalJoinColumn addJoinColumn(String name) {
		JoinColumn joinColumn = buildJoinColumn(-1);
		joinColumn.addSelf();
		joinColumn.setName(name);
		return joinColumn;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalJoinTable addJoinTable(String tableName) {
		JoinTable joinTable = buildJoinTable();
		joinTable.addSelf();
		joinTable.setName(tableName);
		return joinTable;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(DESCRIPTION);
		names.add(JoinColumn.JOIN_COLUMN);
		names.add(JoinTable.JOIN_TABLE);
		return names;
	}

	private JoinColumn buildJoinColumn(int index) {
		return new JoinColumn(this, JoinColumn.JOIN_COLUMN, index);
	}

	private JoinTable buildJoinTable() {
		return new JoinTable(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void calculateInsertionIndex(Element parent, Element child, String elementName) {
		if (elementName == getElementName()) {
			index = index(parent, child, elementName);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDescription() {
		return getChildTextNode(DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Element getElement() {
		return getChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return ASSOCIATION_OVERRIDE;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalJoinColumn getJoinColumn(int index) {

		if (hasChild(JoinColumn.JOIN_COLUMN, index)) {
			return buildJoinColumn(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalJoinTable getJoinTable() {

		if (hasChild(JoinTable.JOIN_TABLE)) {
			return buildJoinTable();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getName() {
		return getAttribute(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getNameTextRange() {
		return getAttributeTextRange(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalJoinColumn> joinColumns() {

		int count = joinColumnsSize();
		List<ExternalJoinColumn> joinColumns = new ArrayList<ExternalJoinColumn>(count);

		for (int index = 0; index < count; index++) {
			joinColumns.add(buildJoinColumn(index));
		}

		return joinColumns;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int joinColumnsSize() {
		return getChildrenSize(JoinColumn.JOIN_COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeJoinColumn(int index) {
		JoinColumn joinColumn = buildJoinColumn(index);
		joinColumn.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeSelf() {
		removeChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDescription(String description) {
		updateChildTextNode(DESCRIPTION, description);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setName(String name) {
		setAttribute(NAME, name);
	}
}