/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import javax.persistence.EnumType;
import javax.persistence.FetchType;
import javax.persistence.TemporalType;
import org.eclipse.persistence.annotations.JoinFetchType;
import org.eclipse.persistence.tools.mapping.orm.ExternalAssociationOverride;
import org.eclipse.persistence.tools.mapping.orm.ExternalAttributeOverride;
import org.eclipse.persistence.tools.mapping.orm.ExternalBatchFetch;
import org.eclipse.persistence.tools.mapping.orm.ExternalClassConverter;
import org.eclipse.persistence.tools.mapping.orm.ExternalCollectionTable;
import org.eclipse.persistence.tools.mapping.orm.ExternalElementCollectionMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalJoinColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalMappingVisitor;
import org.eclipse.persistence.tools.mapping.orm.ExternalNoSqlField;
import org.eclipse.persistence.tools.mapping.orm.ExternalObjectTypeConverter;
import org.eclipse.persistence.tools.mapping.orm.ExternalOrderColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalStructConverter;
import org.eclipse.persistence.tools.mapping.orm.ExternalTypeConverter;
import org.eclipse.persistence.tools.utility.ClassNameTools;
import org.eclipse.persistence.tools.utility.TextRange;

/**
 * The external form for a element collection mapping, which is a child of an entity.
 *
 * @see EmbeddableEntity
 *
 * @version 2.6
 */
final class ElementCollectionMapping extends ConvertibleMapping
                                     implements ExternalElementCollectionMapping {

	/**
	 * Creates a new <code>ElementCollectionMapping</code>.
	 *
	 * @param parent The parent of this external form
	 */
	ElementCollectionMapping(Embeddable parent) {
		super(parent);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void accept(ExternalMappingVisitor visitor) {
		visitor.visit(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalAssociationOverride addAssociationOverride(String name) {
		AssociationOverride associationOverride = buildAssociationOverride(-1);
		associationOverride.addSelf();
		associationOverride.setName(name);
		return associationOverride;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalAttributeOverride addAttributeOverride(String name) {
		AttributeOverride attributeOverride = buildAttributeOverride(-1);
		attributeOverride.addSelf();
		attributeOverride.setName(name);
		return attributeOverride;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalBatchFetch addBatchFetch() {
		BatchFetch batchFetch = buildBatchFetch();
		batchFetch.addSelf();
		return batchFetch;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalCollectionTable addCollectionTable() {
		CollectionTable collectionTable = buildCollectionTable();
		collectionTable.addSelf();
		return collectionTable;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityColumn addColumn() {
		EntityColumn column = buildColumn(EntityColumn.COLUMN);
		column.addSelf();
		return column;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalAssociationOverride addMapKeyAssociationOverride() {
		AssociationOverride associationOverride = buildMapKeyAssociationOverride(-1);
		associationOverride.addSelf();
		return associationOverride;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalAttributeOverride addMapKeyAttributeOverride() {
		AttributeOverride attributeOverride = buildMapKeyAttributeOverride(-1);
		attributeOverride.addSelf();
		return attributeOverride;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityColumn addMapKeyColumn() {
		EntityColumn column = buildColumn(MAP_KEY_COLUMN);
		column.addSelf();
		return column;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalJoinColumn addMapKeyJoinColumn(String name) {
		JoinColumn joinColumn = buildJoinColumn(MAP_KEY_JOIN_COLUMN, -1);
		joinColumn.addSelf();
		joinColumn.setName(name);
		return joinColumn;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalOrderColumn addOrderColumn() {
		OrderColumn orderColumn = buildOrderColumn();
		orderColumn.addSelf();
		return orderColumn;
	}

	private AssociationOverride buildAssociationOverride(int index) {
		return new AssociationOverride(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(TARGET_CLASS);
		names.add(FETCH);
		names.add(ACCESS);
		names.add(ATTRIBUTE_TYPE);
		return names;
	}

	private AttributeOverride buildAttributeOverride(int index) {
		return new AttributeOverride(this, index);
	}

	private BatchFetch buildBatchFetch() {
		return new BatchFetch(this);
	}

	private CollectionTable buildCollectionTable() {
		return new CollectionTable(this);
	}

	private EntityColumn buildColumn(String elementName) {
		return new EntityColumn(this, elementName, -1);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(EntityColumn.COLUMN);
		names.add(NoSqlField.FIELD);
		names.add(AttributeOverride.ATTRIBUTE_OVERRIDE);
		names.add(AssociationOverride.ASSOCIATION_OVERRIDE);
		names.add(ORDER_BY);
		names.add(OrderColumn.ORDER_COLUMN);
		names.add(MAP_KEY);
		names.add(MAP_KEY_CLASS);
		names.add(MAP_KEY_TEMPORAL);
		names.add(MAP_KEY_ENUMERATED);
		names.add(MAP_KEY_CONVERT);
		names.add(MAP_KEY_ATTRIBUTE_OVERRIDE);
		names.add(MAP_KEY_ASSOCIATION_OVERRIDE);
		names.add(MAP_KEY_COLUMN);
		names.add(MAP_KEY_JOIN_COLUMN);
		names.add(CollectionTable.COLLECTION_TABLE);
		names.add(BatchFetch.BATCH_FETCH);
		names.add(CONVERT);
		names.add(ClassConverter.CONVERTER);
		names.add(TypeConverter.TYPE_CONVERTER);
		names.add(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		names.add(StructConverter.STRUCT_CONVERTER);
		names.add(Property.PROPERTY);
		names.add(AccessMethods.ACCESS_METHODS);
		return names;
	}

	private JoinColumn buildJoinColumn(String elementName, int index) {
		return new JoinColumn(this, elementName, index);
	}

	private ClassConverter buildKeyConverter() {
		return new ClassConverter(this, -1);
	}

	private ObjectTypeConverter buildKeyObjectTypeConverter() {
		return new ObjectTypeConverter(this, -1);
	}

	private StructConverter buildKeyStructConverter() {
		return new StructConverter(this, -1);
	}

	private TypeConverter buildKeyTypeConverter() {
		return new TypeConverter(this, -1);
	}

	private AssociationOverride buildMapKeyAssociationOverride(int index) {
		return new AssociationOverride(this, index);
	}

	private AttributeOverride buildMapKeyAttributeOverride(int index) {
		return new AttributeOverride(this, index);
	}

	private NoSqlField buildNoSqlField() {
		return new NoSqlField(this);
	}

	private OrderColumn buildOrderColumn() {
		return new OrderColumn(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalBatchFetch getBatchFetch() {

		if (hasChild(BatchFetch.BATCH_FETCH)) {
			return buildBatchFetch();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalCollectionTable getCollectionTable() {

		if (hasChild(CollectionTable.COLLECTION_TABLE)) {
			return buildCollectionTable();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityColumn getColumn() {

		if (hasChild(EntityColumn.COLUMN)) {
			return buildColumn(EntityColumn.COLUMN);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getConvertKey() {
		return getChildTextNode(CONVERT_KEY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getConvertKeyTextRange() {
		return getChildTextNodeTextRange(CONVERT_KEY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return ELEMENT_COLLECTION;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public EnumType getEnumeratedType() {
		return getChildEnumNode(ENUMERATED, EnumType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getEnumeratedTypeTextRange() {
		return getChildTextNodeTextRange(ENUMERATED);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FetchType getFetchType() {
		return getEnumAttribute(FETCH, FetchType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getFetchTypeTextRange() {
		return getAttributeTextRange(FETCH);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public JoinFetchType getJoinFetchType() {
		return getChildEnumNode(JOIN_FETCH, JoinFetchType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getJoinFetchTypeTextRange() {
		return getChildTextNodeTextRange(JOIN_FETCH);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalClassConverter getKeyConverter() {

		if (hasChild(ClassConverter.CONVERTER)) {
			return buildKeyConverter();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalObjectTypeConverter getKeyObjectTypeConverter() {

		if (hasChild(ObjectTypeConverter.OBJECT_TYPE_CONVERTER)) {
			return buildKeyObjectTypeConverter();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalStructConverter getKeyStructConverter() {

		if (hasChild(StructConverter.STRUCT_CONVERTER)) {
			return buildKeyStructConverter();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalTypeConverter getKeyTypeConverter() {

		if (hasChild(TypeConverter.TYPE_CONVERTER)) {
			return buildKeyTypeConverter();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getLobTextRange() {
		return getChildTextNodeTextRange(LOB);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getMapKey() {
		return getChildAttribute(MAP_KEY, NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalAssociationOverride getMapKeyAssociationOverride(int index) {

		if (hasChild(AssociationOverride.ASSOCIATION_OVERRIDE, index)) {
			return buildAssociationOverride(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalAttributeOverride getMapKeyAttributeOverride(int index) {

		if (hasChild(AttributeOverride.ATTRIBUTE_OVERRIDE, index)) {
			return buildAttributeOverride(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getMapKeyClassName() {
		return getChildAttribute(MAP_KEY_CLASS, CLASS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getMapKeyClassNameTextRange() {
		return getChildAttributeTextRange(MAP_KEY_CLASS, CLASS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityColumn getMapKeyColumn() {

		if (hasChild(MAP_KEY_COLUMN)) {
			return buildColumn(MAP_KEY_COLUMN);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getMapKeyConverterName() {
		return getChildTextNode(MAP_KEY_CONVERT);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getMapKeyConverterNameTextRange() {
		return getChildTextNodeTextRange(MAP_KEY_CONVERT);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public EnumType getMapKeyEnumeratedType() {
		return getChildEnumNode(MAP_KEY_ENUMERATED, EnumType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getMapKeyEnumeratedTypeTextRange() {
		return getChildTextNodeTextRange(MAP_KEY_ENUMERATED);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalJoinColumn getMapKeyJoinColumn(int index) {

		if (hasChild(MAP_KEY_JOIN_COLUMN, index)) {
			return buildJoinColumn(MAP_KEY_JOIN_COLUMN, index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TemporalType getMapKeyTemporalType() {
		return getChildEnumNode(MAP_KEY_TEMPORAL, TemporalType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getMapKeyTemporalTypeTextRange() {
		return getChildTextNodeTextRange(MAP_KEY_TEMPORAL);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getMapKeyTextRange() {
		return getChildAttributeTextRange(MAP_KEY, NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getMappingTextRange() {
		return getElementNameTextRange(getElement());
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getNoSqlField() {
		ExternalNoSqlField field = buildNoSqlField();
		return field.getName();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getOrderByFieldName() {
		return getChildTextNode(ORDER_BY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getOrderByFieldNameTextRange() {
		return getAttributeTextRange(ORDER_BY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalOrderColumn getOrderColumn() {

		if (hasChild(OrderColumn.ORDER_COLUMN)) {
			return buildOrderColumn();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getTargetClassName() {
		return getAttribute(TARGET_CLASS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getTargetClassNameTextRange() {
		return getAttributeTextRange(TARGET_CLASS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getTargetClassShortName() {
		return ClassNameTools.simpleName(getTargetClassName());
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TemporalType getTemporalType() {
		return getChildEnumNode(TEMPORAL, TemporalType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getTemporalTypeTextRange() {
		return getChildTextNodeTextRange(TEMPORAL);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean hasColumn() {
		return hasChild(EntityColumn.COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean hasOrderByClause() {
		return hasChild(ORDER_BY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isElementCollectionMapping() {
		return true;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean isLob() {
		return getChildBooleanNode(LOB);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalAssociationOverride> mapKeyAssociationOverrides() {

		int count = mapKeyAssociationOverridesSize();
		List<ExternalAssociationOverride> associationOverrides = new ArrayList<ExternalAssociationOverride>(count);

		for (int index = 0; index < count; index++) {
			associationOverrides.add(buildAssociationOverride(index));
		}

		return associationOverrides;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int mapKeyAssociationOverridesSize() {
		return getChildrenSize(AssociationOverride.ASSOCIATION_OVERRIDE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalAttributeOverride> mapKeyAttributeOverrides() {

		int count = mapKeyAttributeOverridesSize();
		List<ExternalAttributeOverride> attributeOverrides = new ArrayList<ExternalAttributeOverride>(count);

		for (int index = 0; index < count; index++) {
			attributeOverrides.add(buildAttributeOverride(index));
		}

		return attributeOverrides;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int mapKeyAttributeOverridesSize() {
		return getChildrenSize(AttributeOverride.ATTRIBUTE_OVERRIDE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalJoinColumn> mapKeyJoinColumns() {

		int count = mapKeyJoinColumnsSize();
		List<ExternalJoinColumn> joinColumns = new ArrayList<ExternalJoinColumn>(count);

		for (int index = 0; index < count; index++) {
			joinColumns.add(buildJoinColumn(MAP_KEY_JOIN_COLUMN, index));
		}

		return joinColumns;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int mapKeyJoinColumnsSize() {
		return getChildrenSize(MAP_KEY_JOIN_COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeBatchFetch() {
		removeChild(BatchFetch.BATCH_FETCH);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeCollectionTable() {
		CollectionTable collectionTable = buildCollectionTable();
		collectionTable.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeColumn() {
		EntityColumn column = buildColumn(EntityColumn.COLUMN);
		column.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeKeyConverter() {
		removeChild(ClassConverter.CONVERTER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeKeyObjectTypeConverter() {
		removeChild(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeKeyStructConverter() {
		removeChild(StructConverter.STRUCT_CONVERTER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeKeyTypeConverter() {
		removeChild(TypeConverter.TYPE_CONVERTER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeMapKeyAssociationOverride(int index) {
		AssociationOverride associationOverride = buildAssociationOverride(index);
		associationOverride.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeMapKeyAttributeOverride(int index) {
		AttributeOverride attributeOverride = buildMapKeyAttributeOverride(index);
		attributeOverride.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeMapKeyColumn() {
		EntityColumn column = buildColumn(MAP_KEY_COLUMN);
		column.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeMapKeyJoinColumn(int index) {
		JoinColumn joinColumn = buildJoinColumn(MAP_KEY_JOIN_COLUMN, index);
		joinColumn.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeOrderColumn() {
		OrderColumn orderColumn = buildOrderColumn();
		orderColumn.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAsKeyConverter(String name, String className) {

		removeChild(TypeConverter.TYPE_CONVERTER);
		removeChild(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		removeChild(StructConverter.STRUCT_CONVERTER);

		ClassConverter valueConverter = buildKeyConverter();
		valueConverter.addSelf();
		valueConverter.setName(name);
		valueConverter.setClassName(className);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAsKeyObjectTypeConverter(String name) {

		removeChild(TypeConverter.TYPE_CONVERTER);
		removeChild(ClassConverter.CONVERTER);
		removeChild(StructConverter.STRUCT_CONVERTER);

		ObjectTypeConverter valueConverter = buildKeyObjectTypeConverter();
		valueConverter.addSelf();
		valueConverter.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAsKeyStructConverter(String name, String converter) {

		removeChild(TypeConverter.TYPE_CONVERTER);
		removeChild(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		removeChild(ClassConverter.CONVERTER);

		StructConverter valueConverter = buildKeyStructConverter();
		valueConverter.addSelf();
		valueConverter.setName(name);
		valueConverter.setConverter(converter);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAsKeyTypeConverter(String name) {

		removeChild(ClassConverter.CONVERTER);
		removeChild(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		removeChild(StructConverter.STRUCT_CONVERTER);

		TypeConverter valueConverter = buildKeyTypeConverter();
		valueConverter.addSelf();
		valueConverter.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setColumn(String columnName) {
		EntityColumn column = buildColumn(EntityColumn.COLUMN);
		column.addSelf();
		column.setName(columnName);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setConvertKey(String value) {
		updateChildTextNode(CONVERT_KEY, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setEnumeratedType(EnumType type) {
		updateChildTextNode(ENUMERATED, type);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setFetchType(FetchType fetchType) {
		setAttribute(FETCH, fetchType);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setJoinFetchType(JoinFetchType type) {
		updateChildTextNode(JOIN_FETCH, type);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setLob(Boolean lob) {
		updateChildTextNode(LOB, lob);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMapKey(String mapKey) {
		updateChildAttribute(MAP_KEY, NAME, mapKey);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMapKeyClassName(String className) {
		updateChildAttribute(MAP_KEY_CLASS, CLASS, className);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMapKeyConverterName(String converterName) {
		updateChildTextNode(MAP_KEY_CONVERT, converterName);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMapKeyEnumeratedType(EnumType enumType) {
		updateChildTextNode(MAP_KEY_ENUMERATED, enumType);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMapKeyTemporalType(TemporalType temporalType) {
		updateChildTextNode(MAP_KEY_TEMPORAL, temporalType);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setNoSqlField(String field) {
		NoSqlField noSqlField = buildNoSqlField();
		noSqlField.addSelf();
		noSqlField.setName(field);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setOrderByFieldName(String orderBy) {
		updateChildTextNode(ORDER_BY, orderBy);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setTargetClassName(String className) {
		setAttribute(TARGET_CLASS, className);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setTemporalType(TemporalType type) {
		updateChildTextNode(TEMPORAL, type);
	}
}