/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.annotations.MultitenantType;
import org.eclipse.persistence.tools.mapping.AbstractExternalForm;
import org.eclipse.persistence.tools.mapping.orm.ExternalMultitenancyPolicy;
import org.eclipse.persistence.tools.mapping.orm.ExternalTenantDiscriminatorColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalTenantTableDiscriminator;
import org.eclipse.persistence.tools.utility.TextRange;
import org.w3c.dom.Element;

/**
 * The external form of the multitenant.
 *
 * @version 2.6
 */
final class MultitenancyPolicy extends AbstractExternalForm
                               implements ExternalMultitenancyPolicy {

	/**
	 * Creates a new <code>MultitenancyPolicy</code>.
	 *
	 * @param parent The parent of this external form
	 */
	public MultitenancyPolicy(MappedSuperclass parent) {
		super(parent);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalTenantDiscriminatorColumn addDiscriminatorColumn() {
		TenantDiscriminatorColumn column = buildDiscriminatorColumn(-1);
		column.addSelf();
		return column;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalTenantTableDiscriminator addTenantTableDiscriminator() {
		TenantTableDiscriminator discriminator = buildTenantTableDiscriminator();
		discriminator.addSelf();
		return discriminator;
	}

	private TenantDiscriminatorColumn buildDiscriminatorColumn(int index) {
		return new TenantDiscriminatorColumn(this, index);
	}

	private TenantTableDiscriminator buildTenantTableDiscriminator() {
		return new TenantTableDiscriminator(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalTenantDiscriminatorColumn> discriminatorColumns() {

		int count = discriminatorColumnSize();
		List<ExternalTenantDiscriminatorColumn> columns = new ArrayList<ExternalTenantDiscriminatorColumn>(count);

		for (int index = 0; index < count; index++) {
			columns.add(buildDiscriminatorColumn(index));
		}

		return columns;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int discriminatorColumnSize() {
		return getChildrenSize(AbstractTenantDiscriminatorColumn.TENANT_DISCRIMINATOR_COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalTenantDiscriminatorColumn getDiscriminatorColumn(int index) {

		if (hasChild(AbstractTenantDiscriminatorColumn.TENANT_DISCRIMINATOR_COLUMN, index)) {
			return buildDiscriminatorColumn(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return MULTITENANT;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getIncludeCriteriaTextRange() {
		return getAttributeTextRange(INCLUDE_CRITERIA);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public MultitenantType getMultitenantType() {
		return getEnumAttribute(TYPE, MultitenantType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getMultitenantTypeTextRange() {
		return getAttributeTextRange(TYPE);
	}

	public Element getParentEntityElement() {
		return getParentElement();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalTenantTableDiscriminator getTenantTableDiscriminator() {

		if (hasChild(TenantTableDiscriminator.TENANT_TABLE_DISCRIMINATOR)) {
			return buildTenantTableDiscriminator();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getTenantTableDiscriminatorTextRange() {
		return getChildTextRange(TenantTableDiscriminator.TENANT_TABLE_DISCRIMINATOR);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean isCriteriaIncluded() {
		return getBooleanAttribute(INCLUDE_CRITERIA);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeDiscriminatorColumn(int index) {
		AbstractTenantDiscriminatorColumn column = buildDiscriminatorColumn(index);
		column.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeTenantTableDiscriminator() {
		TenantTableDiscriminator table = buildTenantTableDiscriminator();
		table.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setIncludeCriteria(Boolean value) {
		setAttribute(INCLUDE_CRITERIA, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMultitenantType(MultitenantType type) {
		setAttribute(TYPE, type);
	}
}