/*******************************************************************************
 * Copyright (c) 2012, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.db;

import java.util.Iterator;
import org.eclipse.persistence.tools.db.model.ELDatabase;
import org.eclipse.persistence.tools.db.model.ELTable;
import org.eclipse.persistence.tools.gen.db.ConnectionProfile;
import org.eclipse.persistence.tools.gen.db.Schema;
import org.eclipse.persistence.tools.gen.db.SchemaContainer;
import org.eclipse.persistence.tools.gen.db.Sequence;
import org.eclipse.persistence.tools.gen.db.Table;
import org.eclipse.persistence.tools.utility.ObjectTools;
import org.eclipse.persistence.tools.utility.iterable.TransformationIterable;
import org.eclipse.persistence.tools.utility.iterator.FilteringIterator;
import org.eclipse.persistence.tools.utility.iterator.IteratorTools;

/**
 * The concrete implementation of {@link Schema}.
 * <p>
 * Provisional API: This interface is part of an interim API that is still under development and
 * expected to change significantly before reaching stability. It is available at this early stage
 * to solicit feedback from pioneering adopters on the understanding that any code that uses this
 * API will almost certainly be broken (repeatedly) as the API evolves.<p>
 *
 * @version 2.6
 */
@SuppressWarnings("nls")
public class EclipseLinkSchema implements Schema {

	private EclipseLinkDatabase database;
	private ELDatabase mwDatabase;
	private String schemaName;

	public EclipseLinkSchema(EclipseLinkDatabase database, ELDatabase mwDatabase, String schemaName) {
		super();
		this.database = database;
		this.mwDatabase = mwDatabase;
		this.schemaName = schemaName;

	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ConnectionProfile getConnectionProfile() {
		throw new UnsupportedOperationException("Not supported!");
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public SchemaContainer getContainer() {
		return getDatabase();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public EclipseLinkDatabase getDatabase() {
		return this.database;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getIdentifier() {
		return getDatabase().convertNameToIdentifier(this.schemaName);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getIdentifier(String defaultName) {
		return getDatabase().getIdentifier(getName(), defaultName);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getName() {
		return this.schemaName;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Sequence getSequenceForIdentifier(String identifier) {
		throw new UnsupportedOperationException("Sequences not supported.");
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Sequence getSequenceNamed(String name) {
		throw new UnsupportedOperationException("Sequences not supported.");
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Iterable<Sequence> getSequences() {
		throw new UnsupportedOperationException("Sequences not supported.");
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getSequencesSize() {
		throw new UnsupportedOperationException("Sequences not supported.");
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Iterable<String> getSortedSequenceIdentifiers() {
		throw new UnsupportedOperationException("Sequences not supported.");
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Iterable<String> getSortedTableIdentifiers() {
		return new TransformationIterable<Table, String>(getTables()) {
			@Override
			protected String transform(Table table) {
				return table.getName();
			}
		};
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Table getTableForIdentifier(String identifier) {
		return getTableNamed(identifier);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Table getTableNamed(String name) {

		for (Table table : getTables()) {
			if (table.getName().equals(name)) {
				return table;
			}
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Iterable<Table> getTables() {
		return new TransformationIterable<ELTable, Table>(new Iterable<ELTable>() {
			@Override
			public Iterator<ELTable> iterator() {
				return new FilteringIterator<ELTable>(EclipseLinkSchema.this.mwDatabase.tables()) {
					@Override
					protected boolean accept(ELTable table)
					{
						return ObjectTools.equals(EclipseLinkSchema.this.schemaName, table.getSchema());
					};
				};
			}
		}) {
			@Override
			protected Table transform(ELTable table) {
				return new EclipseLinkTable(EclipseLinkSchema.this, table);
			}
		};
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getTablesSize() {
		return IteratorTools.size(getTables().iterator());
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void refresh() {
		// Nothing to do since info is dynamically derived from the db.
	}
}