/*******************************************************************************
 * Copyright (c) 1998, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation from Oracle TopLink
******************************************************************************/
package org.eclipse.persistence.tools.db.relational;

import java.util.List;
import org.eclipse.persistence.tools.db.relational.handles.MWColumnHandle;
import org.eclipse.persistence.tools.db.relational.handles.MWHandle;
import org.eclipse.persistence.tools.db.relational.handles.MWHandle.NodeReferenceScrubber;
import org.eclipse.persistence.tools.db.relational.spi.ExternalForeignKeyColumnPair;
import org.eclipse.persistence.tools.utility.node.Node;

/**
 * Simple class that pairs a source column and a target column for a reference.
 *
 * @version 2.6
 */
@SuppressWarnings("nls")
public final class ELColumnPair extends ELModel {

	private MWColumnHandle sourceColumnHandle;
		public static final String SOURCE_COLUMN_PROPERTY = "sourceColumn";

	private MWColumnHandle targetColumnHandle;
		public static final String TARGET_COLUMN_PROPERTY = "targetColumn";

	// ********** constructors **********

	ELColumnPair(ELReference parent, ELColumn sourceColumn, ELColumn targetColumn) {
		super(parent);
		this.sourceColumnHandle.setColumn(sourceColumn);
		this.targetColumnHandle.setColumn(targetColumn);
	}


	// ********** initialization **********

	/**
	 * initialize persistent state
	 */
	@Override
	protected void initialize(Node parent) {
		super.initialize(parent);
		this.sourceColumnHandle = new MWColumnHandle(this, this.buildSourceColumnScrubber());
		this.targetColumnHandle = new MWColumnHandle(this, this.buildTargetColumnScrubber());
	}


	// ********** accessors **********

	public ELReference getReference() {
		return (ELReference) this.getParent();
	}

	public ELColumn getSourceColumn() {
		return this.sourceColumnHandle.getColumn();
	}

	public void setSourceColumn(ELColumn sourceColumn) {
		if ((sourceColumn != null) && (sourceColumn.getTable() != this.getReference().getSourceTable())) {
			throw new IllegalArgumentException();
		}
		Object old = this.sourceColumnHandle.getColumn();
		this.sourceColumnHandle.setColumn(sourceColumn);
		this.firePropertyChanged(SOURCE_COLUMN_PROPERTY, old, sourceColumn);
	}

	public ELColumn getTargetColumn() {
		return this.targetColumnHandle.getColumn();
	}

	public void setTargetColumn(ELColumn targetColumn) {
		if ((targetColumn != null) && (targetColumn.getTable() != this.getReference().getTargetTable())) {
			throw new IllegalArgumentException();
		}
		Object old = this.targetColumnHandle.getColumn();
		this.targetColumnHandle.setColumn(targetColumn);
		this.firePropertyChanged(TARGET_COLUMN_PROPERTY, old, targetColumn);
	}


	// ********** queries **********

	/**
	 * the pair's name is calculated from the two column names
	 */
	public String getName() {
		StringBuffer sb = new StringBuffer(200);
		this.appendColumn(sb, this.getSourceColumn());
		sb.append('=');
		this.appendColumn(sb, this.getTargetColumn());
		return sb.toString();
	}

	private void appendColumn(StringBuffer sb, ELColumn column) {
		if (column != null) {
			sb.append(column.qualifiedName());
		}
	}

	/**
	 * Returns whether the column pair matches up the specified columns
	 */
	boolean pairs(ELColumn sourceColumn, ELColumn targetColumn) {
		return (this.getSourceColumn() == sourceColumn) &&
			(this.getTargetColumn() == targetColumn);
	}

	public ELTable sourceTable() {
		return this.tableFrom(this.getSourceColumn());
	}

	public ELTable targetTable() {
		return this.tableFrom(this.getTargetColumn());
	}

	private ELTable tableFrom(ELColumn column) {
		return (column == null) ? null : column.getTable();
	}


	// ********** behavior **********

	public void setSourceAndTargetColumns(ELColumn sourceColumn, ELColumn targetColumn) {
		this.setSourceColumn(sourceColumn);
		this.setTargetColumn(targetColumn);
	}

	// ********** containment hierarchy **********

	@Override
	protected void addChildrenTo(List<Node> children) {
		super.addChildrenTo(children);
		children.add(this.sourceColumnHandle);
		children.add(this.targetColumnHandle);
	}

	private NodeReferenceScrubber buildSourceColumnScrubber() {
		return new NodeReferenceScrubber() {
			@Override
			public void nodeReferenceRemoved(Node node, MWHandle handle) {
				ELColumnPair.this.setSourceColumn(null);
			}
			@Override
			public String toString() {
				return "MWColumnPair.buildSourceColumnScrubber()";
			}
		};
	}

	private NodeReferenceScrubber buildTargetColumnScrubber() {
		return new NodeReferenceScrubber() {
			@Override
			public void nodeReferenceRemoved(Node node, MWHandle handle) {
				ELColumnPair.this.setTargetColumn(null);
			}
			@Override
			public String toString() {
				return "MWColumnPair.buildTargetColumnScrubber()";
			}
		};
	}


	// ********** importing/refreshing **********

	/**
	 * Returns whether the column pair corresponds to the specified
	 * "external" column pair;
	 * if the column pair matches an "external" column pair we keep,
	 * otherwise, we remove it from the reference
	 */
	boolean matches(ExternalForeignKeyColumnPair externalPair) {
		return (this.getSourceColumn() == this.sourceColumn(externalPair)) &&
				(this.getTargetColumn() == this.targetColumn(externalPair));
	}

	/**
	 * Returns the column in the source table with the same name as the
	 * "external" column pair's source column
	 */
	private ELColumn sourceColumn(ExternalForeignKeyColumnPair externalPair) {
		return this.getReference().sourceColumn(externalPair);
	}

	/**
	 * Returns the column in the target table with the same name as the
	 * "external" column pair's target column
	 */
	private ELColumn targetColumn(ExternalForeignKeyColumnPair externalPair) {
		return this.getReference().targetColumn(externalPair);
	}

	// ********** displaying and printing **********

	@Override
	public String displayString() {
		return this.getName();
	}

	public void toString(StringBuffer sb) {
		this.printColumnNameOn(this.getSourceColumn(), sb);
		sb.append("=>");
		this.printColumnNameOn(this.getTargetColumn(), sb);
	}

	private void printColumnNameOn(ELColumn column, StringBuffer sb) {
		sb.append((column == null) ? "null" : column.getName());
	}
}
