"use strict";
// tslint:disable:file-header
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
// Based on: https://github.com/Microsoft/vscode/blob/dd3e2d94f81139f9d18ba15a24c16c6061880b93/extensions/git/src/askpass.ts
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.Askpass = void 0;
const inversify_1 = require("@theia/core/shared/inversify");
const path = require("path");
const http = require("http");
const logger_1 = require("@theia/core/lib/common/logger");
const promise_util_1 = require("@theia/core/lib/common/promise-util");
const git_prompt_1 = require("../../common/git-prompt");
const dugite_git_prompt_1 = require("../../node/dugite-git-prompt");
let Askpass = class Askpass {
    constructor() {
        this.ready = new promise_util_1.Deferred();
    }
    init() {
        this.server = http.createServer((req, res) => this.onRequest(req, res));
        this.setup().then(serverAddress => {
            if (serverAddress) {
                this.serverAddress = serverAddress;
                const { address, port } = this.serverAddress;
                this.logger.info(`Git askpass helper is listening on http://${address}:${port}.`);
                this.ready.resolve(true);
            }
            else {
                this.logger.warn("Couldn't start the HTTP server for the Git askpass helper.");
                this.ready.resolve(false);
            }
        }).catch(() => {
            this.ready.resolve(false);
        });
    }
    async setup() {
        try {
            return new Promise(resolve => {
                this.server.on('error', err => this.logger.error(err));
                this.server.listen(0, this.hostname(), () => {
                    resolve(this.server.address());
                });
            });
        }
        catch (err) {
            this.logger.error('Could not launch Git askpass helper.', err);
            return undefined;
        }
    }
    onRequest(req, res) {
        const chunks = [];
        req.setEncoding('utf8');
        req.on('data', (d) => chunks.push(d));
        req.on('end', () => {
            const { gitRequest, gitHost } = JSON.parse(chunks.join(''));
            this.prompt(gitHost, gitRequest).then(result => {
                res.writeHead(200);
                res.end(JSON.stringify(result));
            }, err => {
                this.logger.error(err);
                res.writeHead(500);
                res.end();
            });
        });
    }
    async prompt(requestingHost, request) {
        try {
            const answer = await this.promptServer.ask({
                password: /password/i.test(request),
                text: request,
                details: `Git: ${requestingHost} (Press 'Enter' to confirm or 'Escape' to cancel.)`
            });
            if (git_prompt_1.GitPrompt.Success.is(answer) && typeof answer.result === 'string') {
                return answer.result;
            }
            else if (git_prompt_1.GitPrompt.Cancel.is(answer)) {
                return '';
            }
            else if (git_prompt_1.GitPrompt.Failure.is(answer)) {
                const { error } = answer;
                throw error;
            }
            throw new Error('Unexpected answer.'); // Do not ever log the `answer`, it might contain the password.
        }
        catch (e) {
            this.logger.error(`An unexpected error occurred when requesting ${request} by ${requestingHost}.`, e);
            return '';
        }
    }
    async getEnv() {
        const ok = await this.ready.promise;
        if (!ok) {
            return {
                GIT_ASKPASS: path.join(__dirname, '..', '..', '..', 'src', 'electron-node', 'askpass', 'askpass-empty.sh')
            };
        }
        const [ELECTRON_RUN_AS_NODE, GIT_ASKPASS, THEIA_GIT_ASKPASS_NODE, THEIA_GIT_ASKPASS_MAIN, THEIA_GIT_ASKPASS_HANDLE] = await Promise.all([
            this.ELECTRON_RUN_AS_NODE(),
            this.GIT_ASKPASS(),
            this.THEIA_GIT_ASKPASS_NODE(),
            this.THEIA_GIT_ASKPASS_MAIN(),
            this.THEIA_GIT_ASKPASS_HANDLE()
        ]);
        return {
            ELECTRON_RUN_AS_NODE,
            GIT_ASKPASS,
            THEIA_GIT_ASKPASS_NODE,
            THEIA_GIT_ASKPASS_MAIN,
            THEIA_GIT_ASKPASS_HANDLE
        };
    }
    dispose() {
        this.server.close();
    }
    hostname() {
        return 'localhost';
    }
    GIT_ASKPASS() {
        return path.join(__dirname, '..', '..', '..', 'src', 'electron-node', 'askpass', 'askpass.sh');
    }
    ELECTRON_RUN_AS_NODE() {
        return '1';
    }
    THEIA_GIT_ASKPASS_NODE() {
        return process.execPath;
    }
    THEIA_GIT_ASKPASS_MAIN() {
        return path.join(__dirname, 'askpass-main.js');
    }
    THEIA_GIT_ASKPASS_HANDLE() {
        if (this.serverAddress) {
            return `http://${this.hostname()}:${this.serverAddress.port}`;
        }
        return undefined;
    }
};
__decorate([
    inversify_1.inject(logger_1.ILogger),
    __metadata("design:type", Object)
], Askpass.prototype, "logger", void 0);
__decorate([
    inversify_1.inject(dugite_git_prompt_1.DugiteGitPromptServer),
    __metadata("design:type", dugite_git_prompt_1.DugiteGitPromptServer)
], Askpass.prototype, "promptServer", void 0);
__decorate([
    inversify_1.postConstruct(),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", void 0)
], Askpass.prototype, "init", null);
Askpass = __decorate([
    inversify_1.injectable()
], Askpass);
exports.Askpass = Askpass;
//# sourceMappingURL=askpass.js.map