/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __values = (this && this.__values) || function(o) {
    var s = typeof Symbol === "function" && Symbol.iterator, m = s && o[s], i = 0;
    if (m) return m.call(o);
    if (o && typeof o.length === "number") return {
        next: function () {
            if (o && i >= o.length) o = void 0;
            return { value: o && o[i++], done: !o };
        }
    };
    throw new TypeError(s ? "Object is not iterable." : "Symbol.iterator is not defined.");
};
import { TokenMetadata } from '../modes';
import { LogLevel } from '../../../platform/log/common/log';
import { MultilineTokens2, SparseEncodedTokens } from '../model/tokensStore';
var SemanticTokensProviderStyling = /** @class */ (function () {
    function SemanticTokensProviderStyling(_legend, _themeService, _logService) {
        this._legend = _legend;
        this._themeService = _themeService;
        this._logService = _logService;
        this._hashTable = new HashTable();
        this._hasWarnedOverlappingTokens = false;
    }
    SemanticTokensProviderStyling.prototype.getMetadata = function (tokenTypeIndex, tokenModifierSet, languageId) {
        var entry = this._hashTable.get(tokenTypeIndex, tokenModifierSet, languageId.id);
        var metadata;
        if (entry) {
            metadata = entry.metadata;
            if (this._logService.getLevel() === LogLevel.Trace) {
                this._logService.trace("SemanticTokensProviderStyling [CACHED] " + tokenTypeIndex + " / " + tokenModifierSet + ": foreground " + TokenMetadata.getForeground(metadata) + ", fontStyle " + TokenMetadata.getFontStyle(metadata).toString(2));
            }
        }
        else {
            var tokenType = this._legend.tokenTypes[tokenTypeIndex];
            var tokenModifiers = [];
            if (tokenType) {
                var modifierSet = tokenModifierSet;
                for (var modifierIndex = 0; modifierSet > 0 && modifierIndex < this._legend.tokenModifiers.length; modifierIndex++) {
                    if (modifierSet & 1) {
                        tokenModifiers.push(this._legend.tokenModifiers[modifierIndex]);
                    }
                    modifierSet = modifierSet >> 1;
                }
                if (modifierSet > 0 && this._logService.getLevel() === LogLevel.Trace) {
                    this._logService.trace("SemanticTokensProviderStyling: unknown token modifier index: " + tokenModifierSet.toString(2) + " for legend: " + JSON.stringify(this._legend.tokenModifiers));
                    tokenModifiers.push('not-in-legend');
                }
                var tokenStyle = this._themeService.getColorTheme().getTokenStyleMetadata(tokenType, tokenModifiers, languageId.language);
                if (typeof tokenStyle === 'undefined') {
                    metadata = 2147483647 /* NO_STYLING */;
                }
                else {
                    metadata = 0;
                    if (typeof tokenStyle.italic !== 'undefined') {
                        var italicBit = (tokenStyle.italic ? 1 /* Italic */ : 0) << 11 /* FONT_STYLE_OFFSET */;
                        metadata |= italicBit | 1 /* SEMANTIC_USE_ITALIC */;
                    }
                    if (typeof tokenStyle.bold !== 'undefined') {
                        var boldBit = (tokenStyle.bold ? 2 /* Bold */ : 0) << 11 /* FONT_STYLE_OFFSET */;
                        metadata |= boldBit | 2 /* SEMANTIC_USE_BOLD */;
                    }
                    if (typeof tokenStyle.underline !== 'undefined') {
                        var underlineBit = (tokenStyle.underline ? 4 /* Underline */ : 0) << 11 /* FONT_STYLE_OFFSET */;
                        metadata |= underlineBit | 4 /* SEMANTIC_USE_UNDERLINE */;
                    }
                    if (tokenStyle.foreground) {
                        var foregroundBits = (tokenStyle.foreground) << 14 /* FOREGROUND_OFFSET */;
                        metadata |= foregroundBits | 8 /* SEMANTIC_USE_FOREGROUND */;
                    }
                    if (metadata === 0) {
                        // Nothing!
                        metadata = 2147483647 /* NO_STYLING */;
                    }
                }
            }
            else {
                if (this._logService.getLevel() === LogLevel.Trace) {
                    this._logService.trace("SemanticTokensProviderStyling: unknown token type index: " + tokenTypeIndex + " for legend: " + JSON.stringify(this._legend.tokenTypes));
                }
                metadata = 2147483647 /* NO_STYLING */;
                tokenType = 'not-in-legend';
            }
            this._hashTable.add(tokenTypeIndex, tokenModifierSet, languageId.id, metadata);
            if (this._logService.getLevel() === LogLevel.Trace) {
                this._logService.trace("SemanticTokensProviderStyling " + tokenTypeIndex + " (" + tokenType + ") / " + tokenModifierSet + " (" + tokenModifiers.join(' ') + "): foreground " + TokenMetadata.getForeground(metadata) + ", fontStyle " + TokenMetadata.getFontStyle(metadata).toString(2));
            }
        }
        return metadata;
    };
    SemanticTokensProviderStyling.prototype.warnOverlappingSemanticTokens = function (lineNumber, startColumn) {
        if (!this._hasWarnedOverlappingTokens) {
            this._hasWarnedOverlappingTokens = true;
            console.warn("Overlapping semantic tokens detected at lineNumber " + lineNumber + ", column " + startColumn);
        }
    };
    return SemanticTokensProviderStyling;
}());
export { SemanticTokensProviderStyling };
export function toMultilineTokens2(tokens, styling, languageId) {
    var srcData = tokens.data;
    var tokenCount = (tokens.data.length / 5) | 0;
    var tokensPerArea = Math.max(Math.ceil(tokenCount / 1024 /* DesiredMaxAreas */), 400 /* DesiredTokensPerArea */);
    var result = [];
    var tokenIndex = 0;
    var lastLineNumber = 1;
    var lastStartCharacter = 0;
    while (tokenIndex < tokenCount) {
        var tokenStartIndex = tokenIndex;
        var tokenEndIndex = Math.min(tokenStartIndex + tokensPerArea, tokenCount);
        // Keep tokens on the same line in the same area...
        if (tokenEndIndex < tokenCount) {
            var smallTokenEndIndex = tokenEndIndex;
            while (smallTokenEndIndex - 1 > tokenStartIndex && srcData[5 * smallTokenEndIndex] === 0) {
                smallTokenEndIndex--;
            }
            if (smallTokenEndIndex - 1 === tokenStartIndex) {
                // there are so many tokens on this line that our area would be empty, we must now go right
                var bigTokenEndIndex = tokenEndIndex;
                while (bigTokenEndIndex + 1 < tokenCount && srcData[5 * bigTokenEndIndex] === 0) {
                    bigTokenEndIndex++;
                }
                tokenEndIndex = bigTokenEndIndex;
            }
            else {
                tokenEndIndex = smallTokenEndIndex;
            }
        }
        var destData = new Uint32Array((tokenEndIndex - tokenStartIndex) * 4);
        var destOffset = 0;
        var areaLine = 0;
        var prevLineNumber = 0;
        var prevStartCharacter = 0;
        var prevEndCharacter = 0;
        while (tokenIndex < tokenEndIndex) {
            var srcOffset = 5 * tokenIndex;
            var deltaLine = srcData[srcOffset];
            var deltaCharacter = srcData[srcOffset + 1];
            var lineNumber = lastLineNumber + deltaLine;
            var startCharacter = (deltaLine === 0 ? lastStartCharacter + deltaCharacter : deltaCharacter);
            var length_1 = srcData[srcOffset + 2];
            var tokenTypeIndex = srcData[srcOffset + 3];
            var tokenModifierSet = srcData[srcOffset + 4];
            var metadata = styling.getMetadata(tokenTypeIndex, tokenModifierSet, languageId);
            if (metadata !== 2147483647 /* NO_STYLING */) {
                if (areaLine === 0) {
                    areaLine = lineNumber;
                }
                if (prevLineNumber === lineNumber && prevEndCharacter > startCharacter) {
                    styling.warnOverlappingSemanticTokens(lineNumber, startCharacter + 1);
                    if (prevStartCharacter < startCharacter) {
                        // the previous token survives after the overlapping one
                        destData[destOffset - 4 + 2] = startCharacter;
                    }
                    else {
                        // the previous token is entirely covered by the overlapping one
                        destOffset -= 4;
                    }
                }
                destData[destOffset] = lineNumber - areaLine;
                destData[destOffset + 1] = startCharacter;
                destData[destOffset + 2] = startCharacter + length_1;
                destData[destOffset + 3] = metadata;
                destOffset += 4;
                prevLineNumber = lineNumber;
                prevStartCharacter = startCharacter;
                prevEndCharacter = startCharacter + length_1;
            }
            lastLineNumber = lineNumber;
            lastStartCharacter = startCharacter;
            tokenIndex++;
        }
        if (destOffset !== destData.length) {
            destData = destData.subarray(0, destOffset);
        }
        var tokens_1 = new MultilineTokens2(areaLine, new SparseEncodedTokens(destData));
        result.push(tokens_1);
    }
    return result;
}
var HashTableEntry = /** @class */ (function () {
    function HashTableEntry(tokenTypeIndex, tokenModifierSet, languageId, metadata) {
        this.tokenTypeIndex = tokenTypeIndex;
        this.tokenModifierSet = tokenModifierSet;
        this.languageId = languageId;
        this.metadata = metadata;
        this.next = null;
    }
    return HashTableEntry;
}());
var HashTable = /** @class */ (function () {
    function HashTable() {
        this._elementsCount = 0;
        this._currentLengthIndex = 0;
        this._currentLength = HashTable._SIZES[this._currentLengthIndex];
        this._growCount = Math.round(this._currentLengthIndex + 1 < HashTable._SIZES.length ? 2 / 3 * this._currentLength : 0);
        this._elements = [];
        HashTable._nullOutEntries(this._elements, this._currentLength);
    }
    HashTable._nullOutEntries = function (entries, length) {
        for (var i = 0; i < length; i++) {
            entries[i] = null;
        }
    };
    HashTable.prototype._hash2 = function (n1, n2) {
        return (((n1 << 5) - n1) + n2) | 0; // n1 * 31 + n2, keep as int32
    };
    HashTable.prototype._hashFunc = function (tokenTypeIndex, tokenModifierSet, languageId) {
        return this._hash2(this._hash2(tokenTypeIndex, tokenModifierSet), languageId) % this._currentLength;
    };
    HashTable.prototype.get = function (tokenTypeIndex, tokenModifierSet, languageId) {
        var hash = this._hashFunc(tokenTypeIndex, tokenModifierSet, languageId);
        var p = this._elements[hash];
        while (p) {
            if (p.tokenTypeIndex === tokenTypeIndex && p.tokenModifierSet === tokenModifierSet && p.languageId === languageId) {
                return p;
            }
            p = p.next;
        }
        return null;
    };
    HashTable.prototype.add = function (tokenTypeIndex, tokenModifierSet, languageId, metadata) {
        var e_1, _a;
        this._elementsCount++;
        if (this._growCount !== 0 && this._elementsCount >= this._growCount) {
            // expand!
            var oldElements = this._elements;
            this._currentLengthIndex++;
            this._currentLength = HashTable._SIZES[this._currentLengthIndex];
            this._growCount = Math.round(this._currentLengthIndex + 1 < HashTable._SIZES.length ? 2 / 3 * this._currentLength : 0);
            this._elements = [];
            HashTable._nullOutEntries(this._elements, this._currentLength);
            try {
                for (var oldElements_1 = __values(oldElements), oldElements_1_1 = oldElements_1.next(); !oldElements_1_1.done; oldElements_1_1 = oldElements_1.next()) {
                    var first = oldElements_1_1.value;
                    var p = first;
                    while (p) {
                        var oldNext = p.next;
                        p.next = null;
                        this._add(p);
                        p = oldNext;
                    }
                }
            }
            catch (e_1_1) { e_1 = { error: e_1_1 }; }
            finally {
                try {
                    if (oldElements_1_1 && !oldElements_1_1.done && (_a = oldElements_1.return)) _a.call(oldElements_1);
                }
                finally { if (e_1) throw e_1.error; }
            }
        }
        this._add(new HashTableEntry(tokenTypeIndex, tokenModifierSet, languageId, metadata));
    };
    HashTable.prototype._add = function (element) {
        var hash = this._hashFunc(element.tokenTypeIndex, element.tokenModifierSet, element.languageId);
        element.next = this._elements[hash];
        this._elements[hash] = element;
    };
    HashTable._SIZES = [3, 7, 13, 31, 61, 127, 251, 509, 1021, 2039, 4093, 8191, 16381, 32749, 65521, 131071, 262139, 524287, 1048573, 2097143];
    return HashTable;
}());
