"use strict";
/********************************************************************************
 * Copyright (C) 2017-2018 TypeFox and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
exports.bindFileResource = void 0;
require("../../src/browser/style/index.css");
const inversify_1 = require("@theia/core/shared/inversify");
const common_1 = require("@theia/core/lib/common");
const browser_1 = require("@theia/core/lib/browser");
const file_resource_1 = require("./file-resource");
const filesystem_preferences_1 = require("./filesystem-preferences");
const filesystem_watcher_1 = require("./filesystem-watcher");
const filesystem_frontend_contribution_1 = require("./filesystem-frontend-contribution");
const file_upload_service_1 = require("./file-upload-service");
const file_tree_label_provider_1 = require("./file-tree/file-tree-label-provider");
const file_service_1 = require("./file-service");
const remote_file_system_provider_1 = require("../common/remote-file-system-provider");
const filesystem_1 = require("../common/filesystem");
const uri_1 = require("@theia/core/lib/common/uri");
const files_1 = require("../common/files");
const env_variables_1 = require("@theia/core/lib/common/env-variables");
const contribution_provider_1 = require("@theia/core/lib/common/contribution-provider");
const remote_file_service_contribution_1 = require("./remote-file-service-contribution");
const filesystem_watcher_error_handler_1 = require("./filesystem-watcher-error-handler");
const encodings_1 = require("@theia/core/lib/common/encodings");
exports.default = new inversify_1.ContainerModule(bind => {
    filesystem_preferences_1.bindFileSystemPreferences(bind);
    contribution_provider_1.bindContributionProvider(bind, file_service_1.FileServiceContribution);
    bind(file_service_1.FileService).toSelf().inSingletonScope();
    bind(remote_file_system_provider_1.RemoteFileSystemServer).toDynamicValue(ctx => browser_1.WebSocketConnectionProvider.createProxy(ctx.container, remote_file_system_provider_1.remoteFileSystemPath, new remote_file_system_provider_1.RemoteFileSystemProxyFactory()));
    bind(remote_file_system_provider_1.RemoteFileSystemProvider).toSelf().inSingletonScope();
    bind(remote_file_service_contribution_1.RemoteFileServiceContribution).toSelf().inSingletonScope();
    bind(file_service_1.FileServiceContribution).toService(remote_file_service_contribution_1.RemoteFileServiceContribution);
    bind(filesystem_watcher_1.FileSystemWatcher).toSelf().inSingletonScope();
    bind(filesystem_watcher_error_handler_1.FileSystemWatcherErrorHandler).toSelf().inSingletonScope();
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    bind(filesystem_1.FileSystem).toDynamicValue(({ container }) => {
        const fileService = container.get(file_service_1.FileService);
        const environments = container.get(env_variables_1.EnvVariablesServer);
        const convertStat = stat => {
            var _a;
            return ({
                uri: stat.resource.toString(),
                lastModification: stat.mtime,
                size: stat.size,
                isDirectory: 'isDirectory' in stat && stat.isDirectory,
                children: 'children' in stat ? (_a = stat.children) === null || _a === void 0 ? void 0 : _a.map(convertStat) : undefined
            });
        };
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        const rethrowError = (uri, error) => {
            if (error instanceof files_1.FileOperationError) {
                if (error.fileOperationResult === 1 /* FILE_NOT_FOUND */) {
                    throw filesystem_1.FileSystemError.FileNotFound(uri);
                }
                if (error.fileOperationResult === 0 /* FILE_IS_DIRECTORY */) {
                    throw filesystem_1.FileSystemError.FileIsDirectory(uri);
                }
                if (error.fileOperationResult === 10 /* FILE_NOT_DIRECTORY */) {
                    throw filesystem_1.FileSystemError.FileNotDirectory(uri);
                }
                if (error.fileOperationResult === 3 /* FILE_MODIFIED_SINCE */) {
                    throw filesystem_1.FileSystemError.FileIsOutOfSync(uri);
                }
            }
            throw error;
        };
        return new class {
            async getFileStat(uri) {
                try {
                    const stat = await fileService.resolve(new uri_1.default(uri), { resolveMetadata: true });
                    return convertStat(stat);
                }
                catch (e) {
                    if (e instanceof files_1.FileOperationError && e.fileOperationResult === 1 /* FILE_NOT_FOUND */) {
                        return undefined;
                    }
                    rethrowError(uri, e);
                }
            }
            exists(uri) {
                return fileService.exists(new uri_1.default(uri));
            }
            async resolveContent(uri, options) {
                try {
                    const content = await fileService.read(new uri_1.default(uri), options);
                    return {
                        stat: convertStat(content),
                        content: content.value
                    };
                }
                catch (e) {
                    rethrowError(uri, e);
                }
            }
            async setContent(file, content, options) {
                try {
                    const result = await fileService.write(new uri_1.default(file.uri), content, Object.assign(Object.assign({}, options), { mtime: file.lastModification }));
                    return convertStat(result);
                }
                catch (e) {
                    rethrowError(file.uri, e);
                }
            }
            async updateContent(file, contentChanges, options) {
                try {
                    const result = await fileService.update(new uri_1.default(file.uri), contentChanges, {
                        mtime: file.lastModification,
                        etag: files_1.etag({ size: file.size, mtime: file.lastModification }),
                        readEncoding: (options === null || options === void 0 ? void 0 : options.encoding) || encodings_1.UTF8,
                        encoding: options === null || options === void 0 ? void 0 : options.overwriteEncoding,
                        overwriteEncoding: !!(options === null || options === void 0 ? void 0 : options.overwriteEncoding)
                    });
                    return convertStat(result);
                }
                catch (e) {
                    rethrowError(file.uri, e);
                }
            }
            async move(sourceUri, targetUri, options) {
                try {
                    const result = await fileService.move(new uri_1.default(sourceUri), new uri_1.default(targetUri), options);
                    return convertStat(result);
                }
                catch (e) {
                    rethrowError(sourceUri, e);
                }
            }
            async copy(sourceUri, targetUri, options) {
                try {
                    const result = await fileService.copy(new uri_1.default(sourceUri), new uri_1.default(targetUri), options);
                    return convertStat(result);
                }
                catch (e) {
                    rethrowError(sourceUri, e);
                }
            }
            async createFile(uri, options) {
                try {
                    const result = await fileService.create(new uri_1.default(uri), options === null || options === void 0 ? void 0 : options.content, { encoding: options === null || options === void 0 ? void 0 : options.encoding });
                    return convertStat(result);
                }
                catch (e) {
                    rethrowError(uri, e);
                }
            }
            async createFolder(uri) {
                try {
                    const result = await fileService.createFolder(new uri_1.default(uri));
                    return convertStat(result);
                }
                catch (e) {
                    rethrowError(uri, e);
                }
            }
            touchFile(uri) {
                throw new Error('Method not implemented.');
            }
            async delete(uri, options) {
                try {
                    return await fileService.delete(new uri_1.default(uri), { useTrash: options === null || options === void 0 ? void 0 : options.moveToTrash, recursive: true });
                }
                catch (e) {
                    rethrowError(uri, e);
                }
            }
            async getEncoding(uri) {
                const { encoding } = await fileService.read(new uri_1.default(uri));
                return encoding;
            }
            async guessEncoding(uri) {
                const { encoding } = await fileService.read(new uri_1.default(uri), { autoGuessEncoding: true });
                return encoding;
            }
            async getRoots() {
                const drives = await environments.getDrives();
                const roots = await Promise.all(drives.map(uri => this.getFileStat(uri)));
                return roots.filter(root => !!root);
            }
            async getCurrentUserHome() {
                return this.getFileStat(await environments.getHomeDirUri());
            }
            getDrives() {
                return environments.getDrives();
            }
            access(uri, mode) {
                return fileService.access(new uri_1.default(uri), mode);
            }
            getFsPath(uri) {
                return fileService.fsPath(new uri_1.default(uri));
            }
        };
    }).inSingletonScope();
    bindFileResource(bind);
    bind(file_upload_service_1.FileUploadService).toSelf().inSingletonScope();
    bind(filesystem_frontend_contribution_1.FileSystemFrontendContribution).toSelf().inSingletonScope();
    bind(common_1.CommandContribution).toService(filesystem_frontend_contribution_1.FileSystemFrontendContribution);
    bind(browser_1.FrontendApplicationContribution).toService(filesystem_frontend_contribution_1.FileSystemFrontendContribution);
    bind(file_tree_label_provider_1.FileTreeLabelProvider).toSelf().inSingletonScope();
    bind(browser_1.LabelProviderContribution).toService(file_tree_label_provider_1.FileTreeLabelProvider);
});
function bindFileResource(bind) {
    bind(file_resource_1.FileResourceResolver).toSelf().inSingletonScope();
    bind(common_1.ResourceResolver).toService(file_resource_1.FileResourceResolver);
}
exports.bindFileResource = bindFileResource;
//# sourceMappingURL=filesystem-frontend-module.js.map