"use strict";
/********************************************************************************
 * Copyright (C) 2018 TypeFox and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
exports.WebSocketChannel = void 0;
const disposable_1 = require("../disposable");
const event_1 = require("../event");
class WebSocketChannel {
    constructor(id, doSend) {
        this.id = id;
        this.doSend = doSend;
        this.closeEmitter = new event_1.Emitter();
        this.toDispose = new disposable_1.DisposableCollection(this.closeEmitter);
        this.fireOpen = () => { };
        this.fireMessage = () => { };
        this.fireError = () => { };
        this.closing = false;
    }
    dispose() {
        this.toDispose.dispose();
    }
    checkNotDisposed() {
        if (this.toDispose.disposed) {
            throw new Error('The channel has been disposed.');
        }
    }
    handleMessage(message) {
        if (message.kind === 'ready') {
            this.fireOpen();
        }
        else if (message.kind === 'data') {
            this.fireMessage(message.content);
        }
        else if (message.kind === 'close') {
            this.fireClose(message.code, message.reason);
        }
    }
    open(path) {
        this.checkNotDisposed();
        this.doSend(JSON.stringify({
            kind: 'open',
            id: this.id,
            path
        }));
    }
    ready() {
        this.checkNotDisposed();
        this.doSend(JSON.stringify({
            kind: 'ready',
            id: this.id
        }));
    }
    send(content) {
        this.checkNotDisposed();
        this.doSend(JSON.stringify({
            kind: 'data',
            id: this.id,
            content
        }));
    }
    close(code = 1000, reason = '') {
        if (this.closing) {
            // Do not try to close the channel if it is already closing.
            return;
        }
        this.checkNotDisposed();
        this.doSend(JSON.stringify({
            kind: 'close',
            id: this.id,
            code,
            reason
        }));
        this.fireClose(code, reason);
    }
    tryClose(code = 1000, reason = '') {
        if (this.closing || this.toDispose.disposed) {
            // Do not try to close the channel if it is already closing or disposed.
            return;
        }
        this.doSend(JSON.stringify({
            kind: 'close',
            id: this.id,
            code,
            reason
        }));
        this.fireClose(code, reason);
    }
    onOpen(cb) {
        this.checkNotDisposed();
        this.fireOpen = cb;
        this.toDispose.push(disposable_1.Disposable.create(() => this.fireOpen = () => { }));
    }
    onMessage(cb) {
        this.checkNotDisposed();
        this.fireMessage = cb;
        this.toDispose.push(disposable_1.Disposable.create(() => this.fireMessage = () => { }));
    }
    onError(cb) {
        this.checkNotDisposed();
        this.fireError = cb;
        this.toDispose.push(disposable_1.Disposable.create(() => this.fireError = () => { }));
    }
    fireClose(code, reason) {
        if (this.closing) {
            return;
        }
        this.closing = true;
        try {
            this.closeEmitter.fire([code, reason]);
        }
        finally {
            this.closing = false;
        }
        this.dispose();
    }
    onClose(cb) {
        this.checkNotDisposed();
        return this.closeEmitter.event(([code, reason]) => cb(code, reason));
    }
}
exports.WebSocketChannel = WebSocketChannel;
WebSocketChannel.wsPath = '/services';
//# sourceMappingURL=web-socket-channel.js.map